/*
* CEBL : CSU EEG Brain-Computer Interface Lab
*
* Author: Jeshua Bratman - jeshuabratman@gmail.com
*
* This file is part of CEBL.
*
* CEBL is free software; you can redistribute it and/or modify it.
* We only ask that if you use our code that you cite the source in
* your project or publication.
*
* EEG Group (www.cs.colostate.edu/eeg)
* Department of Computer Science
* Colorado State University
* 
*/

/*!
 * SharedLoader.hpp
 * \author Jeshua Bratman
 *
 * Loads plugins from shared libraries
 */



#ifndef SHAREDLOADER_H
#define SHAREDLOADER_H


#include <dlfcn.h>
#include <iostream>

using std::cerr;
using std::cout;
using std::endl;


template < typename T >
class SharedLoader
{
private:
  bool loaded;
  T* (*create_func)();
  void (*destroy_func)(T *);
  void *cls;//opened library
public:
  SharedLoader();
  ~SharedLoader();
  void LoadLibrary(const char *);
  T *New();
  void Delete(T*);

  bool Loaded() { return loaded; }
  void Close() { if(!loaded) { dlclose(cls);loaded = false; }}

};

//Constructor
template <typename T>
SharedLoader<T>::SharedLoader()
{
  loaded = false;
}

//Destructor
template <typename T>
SharedLoader<T>::~SharedLoader()
{
  if(loaded)
    dlclose(cls);
}

//create a new object
template <typename T>
T* SharedLoader<T>::New()
{
  //printf("Inside SharedLoader::New(): returning value from create function: function pointer = %x\n",create_func);
  if(loaded)
    return create_func();
  else
    return NULL;
}

//delete an object
template <typename T>
void SharedLoader<T>::Delete(T* obj)
{
  if(obj != NULL)
    destroy_func(obj);
}

//load a shared library
template <typename T>
void SharedLoader<T>::LoadLibrary(const char *filename)
{
  if(loaded)
    {
      dlclose(cls);
    }
  loaded = false;
  //load the library
  cls = dlopen(filename, RTLD_NOW);
  if(!cls)
    {
      cerr << "Cannot load library: " << dlerror() << '\n';
      loaded = false;
      return;
    }

  //reset errors
  dlerror();

  //CREATE
  typedef T* create_t();
  create_func = (create_t*)dlsym(cls, "ObjectCreate");
  const char* dlsym_error = dlerror();
  if (dlsym_error)
    {
      cerr << "Cannot load ObjectCreate: " << dlsym_error << '\n';
      loaded = false;
      return;
    }

  //DESTROY
  typedef void destroy_t(T*);
  destroy_func = (destroy_t*)dlsym(cls, "ObjectDestroy");
  dlsym_error = dlerror();
  if (dlsym_error)
    {
      cerr << "Cannot load ObjectDestroy: " << dlsym_error << '\n';
      loaded = false;
      return;
    }
  loaded = true;


}
#endif
