/*
* CEBL : CSU EEG Brain-Computer Interface Lab
*
* Author: Jeshua Bratman - jeshuabratman@gmail.com
*
* This file is part of CEBL.
*
* CEBL is free software; you can redistribute it and/or modify it.
* We only ask that if you use our code that you cite the source in
* your project or publication.
*
* EEG Group (www.cs.colostate.edu/eeg)
* Department of Computer Science
* Colorado State University
* 
*/

/*! Training.hpp
 * \author Jeshua Bratman
 *
 * Contains training options including num classes, sequences.
 * Using data from data source, collects data for training.
 */

#ifndef TRAINING_H
#define TRAINING_H

#include "EEGTrainingData.hpp"
#include "Timer.hpp"
#include "TimeoutThread.hpp"
#include "SessionManager.hpp"

#include <string>
#include <vector>
using namespace std;

//forward declarations
class CEBLModel;

class Training : public TimeoutThread
{
private:
  //make SessionManager a friend
  friend class SessionManager;

  CEBLModel *model;

  //training options
  int num_classes;
  int num_sequences;
  int sequence_length;
  int pause_length;
  std::vector<string> class_labels;

  //feedback options
  bool classification_feedback;
  bool classifier_trained;
  std::vector<double> class_proportions;

  /*!  Classify samples and update class_proportions
    if there is a trained classifier
    \param samples should be processed data
  */
  void classifySamples(EEGData samples);

  /*! Train classifier and continue the training loop.
   */
  void trainClassifierAndContinue();


  //data
  EEGTrainingData training_data;
  EEGTrainingData training_data_filtered;
  bool data_is_loaded;

  //file loading
  bool data_file_loaded;
  string data_filename;

  //training process
  bool training_is_active;
  int current_training_class;
  int current_training_sequence;

  Timer training_timer;
  int wait_length; // in ms
  std::vector<int> training_class_ordering;
  std::vector<int> current_class_sequence;
  int training_index;
  bool waiting;
  bool training_failed;
  string failure_message;

  void timeoutFunction();
  void initializeTraining();

  //control of training
  void stopSuccess();
  void stopFailure(string msg);


  //private access functions
  void setTrainingData(EEGTrainingData data)
  {
    training_data = data;
  }

  void setTrainingDataFiltered(EEGTrainingData data)
  {
    training_data_filtered = data;
  }

  void setDataIsLoaded(bool is_loaded)
  {
    data_is_loaded = is_loaded;
  }

public:
  Training(CEBLModel *);
  ~Training();

  //GETTING OPERATIONS
  //! get the labels of the classes
  std::vector<string> getClassLabels();
  //! get the label of a specific class
  string getClassLabel(int class_num);
  //! get number of classes to train on
  int getNumClasses();
  //! get the number of sequences to train on
  int getNumSequences();
  //! get length of each training sequence in seconds
  int getSequenceLength();
  //! get length of pause between each sequence
  int getPauseLength();
  //! get the currently loaded data
  EEGTrainingData getData();
  //! is data ready, either from a file or from  session
  bool dataIsLoaded();
  //! is a trainig data file loaded
  bool isDataFileLoaded();
  //! get filename of loaded datafile
  string getDataFilename();
  //! is  training currently occuring
  bool isActive();
  //! did training fail
  bool failed();
  //! get failure message
  string getFailureMessage();
  //! is training paused between sequences
  bool isPaused();
  //! get the current class being trained
  int getTrainingClass();
  //! get the current sequence being trained
  int getTrainingSequence();

  //SETTING OPERATIONS
  //! start the training process
  void start();
  ///! stop the  process
  void stop();
  //! set number of classes to train on
  void setNumClasses(int);
  //! set the number of sequences to train on
  void setNumSequences(int);
  //! set the length of each  sequence in seconds
  void setSequenceLength(int);
  //! set the length of pause between each sequence
  void setPauseLength(int);
  //! set class labels
  void setClassLabels(std::vector<string> labels);
  //! set label for a specific class
  void setClassLabel(int class_number, string label);
  //! load a data file
  void loadData(string filename);
  //! clear loaded data
  void clearData();
  //! save data to a file
  void saveData(string filename);



  //FEEDBACK OPERATIONS
  bool feedbackEnabled() { return classification_feedback; }
  void setFeedbackEnabled(bool flag) { classification_feedback = flag; }
  bool isTrainingClassifier();
  std::vector<double> getClassProportions() { return class_proportions; }

};

#endif

