#include "SimpleMax.hpp"
#include "cppR/cppR.hpp"

using namespace cppR;

namespace CEBL {

  SimpleMax::SimpleMax()
  {
    this->growth_rate = .001;
    this->shrink_rate = 0.0;
    this->plugin_name = "SimpleMax";
  }

  SimpleMax::~SimpleMax()
  {
  }

  void SimpleMax::init(int num_classes)
  {
    this->num_classes = num_classes;
    this->proportions = cppR::asStdVector(cppR::rep(0.0, num_classes));
  }

  //! get list of parameters needed for decision process
  std::map<std::string, CEBL::Param> SimpleMax::getParamsList()
  {
    std::map<std::string, CEBL::Param> params;

    CEBL::Param g("Growth Rate",
                "How much to grow classes on each update based on their probability.",
                this->growth_rate);
    g.setStep(0.0001);
    g.setMax(0.5);
    g.setMin(0.0);
    params["g"] = g;

    CEBL::Param s("Shrink",
                "How much to shrink all classes on each update.",
                this->shrink_rate);
    s.setStep(0.0001);
    s.setMax(0.5);
    s.setMin(0.0);
    params["s"] = s;

    return params;
  }

  //! set list of parameters
  void SimpleMax::setParamsList( std::map<std::string, CEBL::Param> &p)
  {
    this->growth_rate = p["g"].getDouble();
    this->shrink_rate = p["s"].getDouble();
  }


  void SimpleMax::updateWithProbabilities(std::vector<double> probs)
  {
    if(probs.size() != unsigned(num_classes))
      {
        this->init(probs.size());
      }

    using namespace cppR;
    this->proportions.resize(num_classes);
    for(int i=0;i<num_classes;i++)
      {
        double growth = (growth_rate + shrink_rate) * probs[i]
          - shrink_rate;
        this->proportions[i] += growth;
        if(this->proportions[i] < 0)
          this->proportions[i] = 0;


      }
  }

  std::vector<double> SimpleMax::decideClasses()
  {
    std::vector<double> ret = this->proportions;
    ublas::vector<double> props = asUblasVector(this->proportions); 
    //check if one has reached 1.0
    if(max(props) >= 1.0)
      {
        props = rep(0.0, num_classes);
      }
    this->proportions = asStdVector(props);

    return ret;
  }

  //----------------------------------------------------------------------
  //SAVING and LOADING



  //! serialize for saving to archive
  map<string, SerializedObject> SimpleMax::save() const
  {
    map<string, SerializedObject> ret;
    ret["growth_rate"] = serialize(growth_rate);
    ret["shrink_rate"] = serialize(shrink_rate);
    return ret;
  }

  //! serialize class to load form archive
  void SimpleMax::load(map<string, SerializedObject> objects)
  {
    deserialize(objects["growth_rate"],growth_rate);
    deserialize(objects["shrink_rate"],shrink_rate);
  }
}



/*************************************************************/
//DYNAMIC LOADING

extern "C" CEBL::Decision* ObjectCreate()
{
  return new CEBL::SimpleMax;
}

extern "C" void ObjectDestroy(CEBL::Decision* p)
{
  delete p;
}

