#include "IncrementalSVD.hpp"
using namespace cppR;

typedef ublas::matrix<double> Matrix;
typedef ublas::vector<double> Vector;
namespace CEBL
{
  //default constructor
  IncrementalSVD::IncrementalSVD()
  {
    n_lags = 0;
    bufferEmpty = true;
    svd_state_set = false;
    lambda = 0.99;
    plugin_name = "Incremental SVD";
  }


  //! get list of parameters needed for feature
  std::map<std::string, CEBL::Param> IncrementalSVD::getParamsList()
  {
    std::map<std::string, CEBL::Param> params;
    CEBL::Param lags("Lags", "How many lags?", n_lags);
    lags.setMax(500);
    lags.setMin(0);
    params["lags"] = lags;

    CEBL::Param lambda_param("Lambda", "", lambda);
    lambda_param.setStep(0.001);
    lambda_param.setMax(1.0);
    lambda_param.setMin(0.0);

    params["lambda"] = lambda_param;

    return params;
  }


  //! set list of parameters
  void IncrementalSVD::setParamsList( std::map<std::string, CEBL::Param> &params)
  {
    std::map<std::string, CEBL::Param> p = params;
    int new_lags = p["lags"].getInt();
    if(new_lags != n_lags)
      {
        n_lags = new_lags;
        reset();
      }

    lambda = p["lambda"].getDouble();
  }


  //! featureize data and return the result
  Matrix IncrementalSVD::use(const Matrix &data)
  {
    should_halt = false;

    //lagged data
    Matrix lagged_data;

    if (n_lags > 0)
      {
        if (bufferEmpty)
          {
            buffer = data;
            bufferEmpty = false;
          }
        else
          {
            // cbind (concatentate) new data onto right side of buffer
            int nRowsBuffer = nrow(buffer);
            int nColsBuffer = ncol(buffer);
            int nColsData = ncol(data);
            Matrix keep =
              submatrix(buffer,0,nRowsBuffer-1,
        		nColsBuffer-n_lags, nColsBuffer-1);
            buffer = keep;
            buffer.resize(nRowsBuffer, n_lags+nColsData);
            nColsBuffer = ncol(buffer);
            submatrix(buffer, 0, nRowsBuffer-1,
        	      nColsBuffer-nColsData, nColsBuffer-1) = data;
          }

        // will produce error if n_lags >= ncol(buffer)
        this->inturruptionPoint();
        lagged_data = cppR::Lag(buffer,n_lags);
      }
    else
      {
        // n_lags == 0
        lagged_data = data;
      }

    //svd the lagged data;
    this->inturruptionPoint();
    Matrix svdized_data = ISVD(lagged_data);

    //write stuff to an R file
    /*r_writer.Add(data,"data");
    r_writer.Add(lagged_data,"lagged.data");
    r_writer.Add(svdized_data,"svdized.data");
    r_writer.Write("temp.R");*/
    return svdized_data;

  }


  map<string, SerializedObject> IncrementalSVD::save() const
  {
    map<string, SerializedObject> ret;
    ret["n_lags"] = serialize(n_lags);
    ret["lambda"] = serialize(lambda);

    return ret;
  }
  void IncrementalSVD::load(map<string, SerializedObject> objects)
  {
    deserialize(objects["n_lags"],n_lags);
    deserialize(objects["lambda"],lambda);
  }


  //! reset the feature
  void IncrementalSVD::reset()
  {
    buffer.resize(0,0);
    bufferEmpty = true;
    svd_state_set = false;
    cout << "reset called\n";
  }


  /**************************************************/


  Matrix IncrementalSVD::ISVD(Matrix data)
  {
    //make sure matrix dimensions are okay
    if(svd_state_set  && nrow(data)!=ncol(svd_state.U))
      {
        svd_state_set = false;
      }
    //create state variables
    if(!svd_state_set)
      {
        svd_state.U = diag(1,nrow(data));
        svd_state.s = createVector(0.1,nrow(data));
        svd_state_set = true;
      }
    //do the incremental svd
    Matrix features;
    features.resize(0,0);
    for(int i=0; i < ncol(data); i++)
      {
        this->inturruptionPoint();
        Matrix data_col = createMatrix(Vector(column(data,i)),0,1);
        svd_state = SVDUpdateRank(data_col, svd_state,lambda);
        Matrix U = createMatrix(createVector(svd_state.U),svd_state.U.size1()*svd_state.U.size2(),0);
        if(features.size1() == 0)
          features = U;
        else
          features = cbind(features,U);
      }

    return features;
  }

  //incremental svd function
  SVDState IncrementalSVD::SVDUpdateRank(Matrix sample, SVDState state, double lambda)
  {
    int dimen = ncol(state.U);
    if(state.s.size() < unsigned(dimen))
      dimen = state.s.size();
    //resize U and s
    state.U.resize(state.U.size1(),dimen);
    state.s.resize(dimen);



    Matrix m = prod(t(state.U),sample);
    Matrix p = sample - prod(state.U, m);

    double pnorm = sqrt(sum(apply(p,pow,2)));

    Matrix sdiag = diag(state.s);

    Matrix K;
    K.resize(nrow(m) + 1, ncol(sdiag) + ncol(m));


    //create K matrix
    submatrix(K,0,nrow(m)-1,0,0) = Matrix(cbind(sdiag,m));
    int last_row = nrow(K)-1;
    for(int i=0;i<ncol(K);i++)
      {
        this->inturruptionPoint();
        if(i < dimen)
          K(last_row,i) = 0;
        else
          K(last_row,i) = pnorm;
      }
    //run svd
    SvdStruct<double> r = svd(K);
    Matrix Up = r.u;
    Vector sp = r.d;

    Matrix P;
    if(pnorm==0)
      P = p * 0;
    else
      P = p * (1.0/pnorm);


    state.U = prod(cbind(state.U,P),Up);
    state.U.resize(state.U.size1(),dimen);


    //reverse direction of negative singular vectors

      for(int col=0;col<ncol(state.U);col++)
      {
        this->inturruptionPoint();
        bool s = state.U(0,col) < 0;
        for(int row=0;row<nrow(state.U);row++)
          if(s)
            state.U(row,col) = -state.U(row,col);
      }

    sp.resize(dimen);
    state.s = sp * lambda;

    return state;
  }

}

/*************************************************************/
//DYNAMIC LOADING

extern "C" CEBL::Feature* ObjectCreate()
{
  return new CEBL::IncrementalSVD;
}

extern "C" void ObjectDestroy(CEBL::Feature* p)
{
  delete p;
}
