#include "PassBand.hpp"
#include "WidgetUtils.hpp"
#include <iostream>
#include <fftw3.h>


using namespace std;
using namespace cppR;
typedef ublas::matrix<double> Matrix;



/**************************************************/

double const pi = M_PI;

template <typename T>
double logb(int base, T value)
{
  return log(value)/log(base);
}
template <typename T>
double log2(T value)
{
  return logb(2,value);
}

/**************************************************/

namespace CEBL
{

  //----------------------------------------
  //constructor

  PassBand::PassBand()
  {
    trained = false;
    gamma = .99;
    plugin_name = "Pass Band";
    //InitGUI();
  };




  //----------------------------------------
  //! featurize data and return result
  ublas::matrix<double> PassBand::use(const ublas::matrix<double> &data)
  {
    //if filter has not been trained, just return the data
    if(!trained)
      {
        return data;
      }
    //create variables for return of data
    int nfeatures = nrow(data);
    ublas::matrix<double> ret = createMatrix(0,0,0);
    ublas::matrix<double> var_ret = createMatrix(0,0,data.size2());
    bool ret_set = false;

    for(int i=0;i<num_bands;i++)
      {
        //make sure we have a filter made


        //create a state
        if(states.size() <= i)
          {
            FilterState new_state;
            new_state.empty = true;
            states.push_back(new_state);
          }

        //create means and vars for this band
        if(means.size() <= i)
          {
            vector<double> temp;
            vars.push_back(temp);
            means.push_back(temp);
          }
        //filter data
        FilterResult temp = filter(filters[i], data, states[i]);
        states[i] = temp.state;
        //initially resize ret if it hasn't been set yet
        if(!ret_set)
          {
            ret = temp.filtered;
            ret_set = true;
          }
        else
          {
            //bind result onto return matrix
            ret = rbind(ret,temp.filtered);
          }

        //now calulate variance

        //matrix to return vars
        Matrix var_matrix;
        var_matrix.resize(data.size1(),data.size2());
        //loop through features
        for(int feature=0;feature<data.size1();feature++)
          {

            //create means and vars for this feature
            if(means[i].size() <= feature)
              {
        	vars[i].push_back(0);
        	means[i].push_back(0);
              }
            ublas::vector<double> row = ublas::row(temp.filtered,feature);
            //calculate new means and vars
            for(int col=0; col<data.size2(); col++)
              {
        	means[i][feature] = gamma * means[i][feature]
        	  + (1.0-gamma)*(row[col]);
        	vars[i][feature] = gamma * vars[i][feature]
        	  + (1.0-gamma)*(pow(row[col]-means[i][feature],2));

        	var_matrix(feature,col) = vars[i][feature];
              }
          }
        var_ret = rbind(var_ret,var_matrix);


      }

    bool use_variance = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(check_variance));
    if(use_variance)
      return var_ret;
    else
      return ret;
  }






  //** function called from CEBL to load gui interface */
  GtkWidget *PassBand::getGuiInterface()
  {
    return gui_container;
  }






  /** serialize class for saving and loading feature */
  void PassBand::save(boost::archive::text_oarchive &ar, const unsigned int &version) const
  {
    ar & trained;
    ar & num_bands;
    ar & gamma;

  }
  void PassBand::load(boost::archive::text_iarchive &ar, const unsigned int &version)
  {
  }






  /** train filter from the gui parameters **/
  void PassBand::train()
  {

    for(int i=0;i<num_bands;i++)
      {
        ConfigRow row = config_rows[i];
        int pass = int((row.low + row.high) / 2.0);
        //check if we have already made a filter for this configuration
        if(configurations.count(sample_rate) == 0
           || configurations[sample_rate].count(row.low) == 0
           || configurations[sample_rate][row.low].count(row.high) == 0)
          {
            //we have not made this filter yet. make it now
            configurations[sample_rate][row.low][row.high]
              = makePassband(151,8,sample_rate,row.low, pass, row.high);

          }
        //set current filter for this row to the right one
        filters[i] = configurations[sample_rate][row.low][row.high];
      }
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check_trained),true);
    trained = true;

    //clear the state buffers
    states.resize(0);
    vars.resize(0);
    means.resize(0);
  }








  void PassBand::ChangeOptions(int pos)
  {
    bool new_config = false;
    for(int i=0; i<num_bands; i++)
      {

        //first go through and make sure all spin_lows are less than spin_high
        int low = int(gtk_spin_button_get_value(GTK_SPIN_BUTTON(widget_rows[i].spin_low)));
        int high = int(gtk_spin_button_get_value(GTK_SPIN_BUTTON(widget_rows[i].spin_high)));

        //save this config for easy access
        if(i < config_rows.size())
          {
            config_rows[i].low = low;
            config_rows[i].high = high;
          }
        else
          {
            ConfigRow temp;
            temp.low = low;
            temp.high = high;
            config_rows.push_back(temp);
          }

        //if bounds are wrong
        if(low >= high)
          {
            //low is changing
            if(pos==0)
              {
        	gtk_spin_button_set_value(GTK_SPIN_BUTTON(widget_rows[i].spin_low),high-1);
        	low = high-1;
              }
            //high is changing
            if(pos==2)
              {
        	gtk_spin_button_set_value(GTK_SPIN_BUTTON(widget_rows[i].spin_high),low+1);
        	high = low+1;
              }
          }

        //now check if this is a new configuration
        if(configurations.count(sample_rate) == 0
           || configurations[sample_rate].count(low) == 0
           || configurations[sample_rate][low].count(high) == 0)
          {
            new_config = true;
          }
        else
          {

            //filter[i] = configurations[low][high];
          }
      }
    if(new_config)
      {
        gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check_trained),false);
        trained = false;
      }
    else
      {
        gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check_trained),true);
        trained = true;
      }
  }






  //--------------------------------------------------
  //gui functions



  //initialize gui widgets and structure
  void PassBand::InitGUI()
  {
    //initialize values
    num_bands = 0;

    //create main interface container
    gui_container = gtk_vbox_new(false,0);

    //first row is a box containing a label and button for number of bands
    GtkWidget *hbox_row1 = gtk_hbox_new(false,0);

    gtk_box_pack_start(GTK_BOX(gui_container),hbox_row1, false, false, 0);

    //------------------------------
    //create spin button for number of bands

    //spin button for num bands
    spin_bands = gtk_spin_button_new_with_range(1,50,1);
    gtk_widget_set_size_request(spin_bands, 40, 18);
    g_signal_connect(G_OBJECT(spin_bands),
        	     "value-changed",
        	     G_CALLBACK(CB_ChangeNumBands),
        	     (gpointer)this);
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(spin_bands),1);
    gtk_box_pack_start(GTK_BOX(hbox_row1), gtk_label_new("Number of Bands: "), false, false,0);
    gtk_box_pack_start(GTK_BOX(hbox_row1), spin_bands, false, false,0);

    spin_gamma = gtk_spin_button_new_with_range(0,1,0.001);
    gtk_widget_set_size_request(spin_gamma, 80, 18);
    g_signal_connect(G_OBJECT(spin_gamma),
        	     "value-changed",
        	     G_CALLBACK(CB_ChangeGamma),
        	     (gpointer)this);
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(spin_gamma),gamma);
    gtk_box_pack_start(GTK_BOX(hbox_row1), gtk_label_new("Gamma: "), false, false,0);
    gtk_box_pack_start(GTK_BOX(hbox_row1), spin_gamma, false, false,0);

    //--------------------------------------------------
    //create a check box to set whether to return variance
    // or straight output from the filter

    check_variance = gtk_check_button_new_with_label("Use Variance");
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check_variance),true);
    gtk_box_pack_end(GTK_BOX(hbox_row1), check_variance, false, false, 0);

    //----------------------------------------
    //create check box to show trained status

    check_trained = gtk_check_button_new_with_label("Trained");
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check_trained), false);
    gtk_widget_set_sensitive(check_trained,false);

    gtk_box_pack_end(GTK_BOX(hbox_row1), check_trained, false, false, 0);


    //----------------------------------------
    //create table for bands


    table_bands = gtk_table_new(2,2, true);
    gtk_table_set_row_spacings(GTK_TABLE(table_bands),4);
    gtk_table_set_col_spacings(GTK_TABLE(table_bands),4);

    gtk_table_attach_defaults(GTK_TABLE(table_bands),
        		      gtk_label_new("Lower (Hz)"),
        		      0, 1,
        		      0, 1);
    gtk_table_attach_defaults(GTK_TABLE(table_bands),
        		      gtk_label_new("Upper (Hz)"),
        		      1, 2,
        		      0, 1);
    gtk_box_pack_start(GTK_BOX(gui_container),table_bands, false, false, 4);

    //create initial row of values
    ChangeNumBands(1);


    //show container
    gtk_widget_show_all(gui_container);


  }






  //CB functions
  void PassBand::ChangeNumBands(int force_bands)
  {
    //find out how many bands there are now

    int new_bands;
    if(force_bands==-1)
      new_bands = int(gtk_spin_button_get_value(GTK_SPIN_BUTTON(spin_bands)));
    else
      new_bands = force_bands;

    if(new_bands == num_bands)
      return;

    gtk_widget_show_all(table_bands);

    //remove rows
    for(int i=0;i<widget_rows.size();i++)
      {
        gtk_widget_hide(widget_rows[i].spin_low);
        gtk_widget_hide(widget_rows[i].spin_high);
      }

    //assign member variable to new band number
    num_bands = new_bands;

    //resize table accordingly
    if(num_bands > widget_rows.size())
      gtk_table_resize(GTK_TABLE(table_bands),num_bands+1,2);


    //add the rows to the table
    for(int i=0;i<num_bands;i++)
      {

        //create new widgets as necessary
        if(i >= widget_rows.size())
          {
            PassBandGUIRow temp;
            //low
            temp.spin_low = gtk_spin_button_new_with_range(0,1000,1);
            gtk_spin_button_set_value(GTK_SPIN_BUTTON(temp.spin_low),20);
            g_signal_connect(G_OBJECT(temp.spin_low),
        		     "value-changed",
        		     G_CALLBACK(CB_ChangeOptionsLow),
        		     (gpointer)this);



            //high
            temp.spin_high = gtk_spin_button_new_with_range(1,1000,1);
            gtk_spin_button_set_value(GTK_SPIN_BUTTON(temp.spin_high),30);
            g_signal_connect(G_OBJECT(temp.spin_high),
        		     "value-changed",
        		     G_CALLBACK(CB_ChangeOptionsHigh),
        		     (gpointer)this);
            widget_rows.push_back(temp);
            //attach new widget
            gtk_table_attach_defaults(GTK_TABLE(table_bands),
        			      widget_rows[i].spin_low,
        			      0, 1,
        			      i+1, i+2);
            gtk_table_attach_defaults(GTK_TABLE(table_bands),
        			      widget_rows[i].spin_high,
        			      1, 2,
        			      i+1, i+2);

            gtk_widget_show(widget_rows[i].spin_low);
            gtk_widget_show(widget_rows[i].spin_high);
          }
        else
          {
            gtk_widget_show(widget_rows[i].spin_low);
            gtk_widget_show(widget_rows[i].spin_high);
          }
      }

    //notify a change in options
    ChangeOptions();

  }








  //CALLBACKS
  void PassBand::CB_ChangeNumBands(GtkWidget *spin, gpointer data)
  {
    ((PassBand*)data)->ChangeNumBands();
  }
  void PassBand::CB_ChangeGamma(GtkWidget *spin, gpointer data)
  {
    ((PassBand*)data)->gamma = gtk_spin_button_get_value(GTK_SPIN_BUTTON(spin));
  }
  void PassBand::CB_ChangeOptionsLow(GtkWidget *spin, gpointer data)
  {
    ((PassBand*)data)->ChangeOptions(0);
  }
  void PassBand::CB_ChangeOptionsHigh(GtkWidget *spin, gpointer data)
  {
    ((PassBand*)data)->ChangeOptions(2);
  }
}//end of namespace



/************************************************************/
//DYNAMIC LOADING

extern "C" CEBL::Feature* ObjectCreate()
{
  printf("Shared Library Says: Creating a Lag object.\n");
  return new CEBL::PassBand;
}

extern "C" void ObjectDestroy(CEBL::Feature* p)
{
  delete p;
}
