#include "WindowedSVD.hpp"

using namespace cppR;

typedef ublas::matrix<double> Matrix;

namespace CEBL
{
  //default constructor
  WindowedSVD::WindowedSVD()
  {
    n_lags = 0;
    window_size = 64;
    window_overlap = 32;
    raw_buffer.resize(0,0);
    plugin_name = "Windowed SVD";
  }

  std::map<std::string, CEBL::Param> WindowedSVD::getParamsList()
  {
    std::map<std::string, CEBL::Param> params;
    CEBL::Param lags("Lags", "How many lags?", n_lags);
    CEBL::Param wsize("Window Size", "How many samples per window?", window_size);
    CEBL::Param wover("Window Overlap", "How many samples do windows overlap??", window_overlap);
    lags.setMax(500);
    lags.setMin(0);

    wsize.setMax(9999999);
    wsize.setMin(1);

    wover.setMax(9999999);
    wover.setMin(0);

    params["lags"] = lags;
    params["wsize"] = wsize;
    params["wover"] = wover;

    return params;
  }
  void WindowedSVD::setParamsList( std::map<std::string, CEBL::Param> &p)
  {
    int new_lags = p["lags"].getInt();
    int new_ws = p["wsize"].getInt();
    int new_wo = p["wover"].getInt();

    if(new_lags != n_lags || new_ws != window_size || new_wo != window_overlap)
      {
        n_lags = new_lags;
        window_size = new_ws;
        window_overlap = new_wo;
        reset();
      }
  }


  //! featurize the data and return the result
  ublas::matrix<double> WindowedSVD::use(const ublas::matrix<double> &data)
  {
    // make sure the new data coming in matches the buffer
    if(nrow(raw_buffer) > 0 && nrow(raw_buffer)!=nrow(data))
      reset();
    // make sure parameters will work with the data passed in
    if(nrow(data)*(n_lags+1) > window_size)
      throw("Window size must be larger than the number of features * (lags+1).");

    bool debug = false;
    if(debug)
      {
        cout << "\n\n\nReceived " << ncol(data) << " new samples.\n";
        cout << "Already have " << ncol(raw_buffer) << " buffered samples\n";
        //cout << "And " << ncol(lagged_buffer) << " left over lagged samples\n";
      }

    if(ncol(raw_buffer)==0) {
      raw_buffer = data;
    } else {
      raw_buffer = cbind(raw_buffer, data);
    }

    Matrix lagged_data;
    int nColsBuffer = ncol(raw_buffer);
    int window_shift = window_size - window_overlap;

    lagged_data = cppR::Lag(raw_buffer,n_lags);

    int num_windows = int((ncol(lagged_data)-window_size)/window_shift);
    if (num_windows < 0)
      num_windows = 0;
    // num_raw_samples_left used later to shrink raw_buffer to just unused samples, for next time
    int num_lagged_samples_left = ncol(lagged_data) - (num_windows-1) * window_shift - (num_windows>0 ? window_size : 0);
    int num_raw_samples_left = num_lagged_samples_left + n_lags;

    Matrix svdized;
    svdized.resize(0,0);

    if(debug)
      {
        cout << "After lagging there are " << ncol(raw_buffer) << " buffered samples\n";
        cout << "And " << ncol(lagged_data) << " samples of lagged data\n";
        cout << "From this, we can create " << num_windows << " windows of " << window_size
             << " samples shifted by " << window_shift << endl;
      }

    if(num_windows == 0)
      {
        Matrix t;
        t.resize(0,0);
        return(t);
      }

    for(int w=0; w < num_windows; w++)
      {
        this->inturruptionPoint();

        // window is all rows of lagged_data and window_size cols
        Matrix window = submatrix(lagged_data, 0, 0,
        			  w * window_shift, w * window_shift + window_size -1);

        SvdStruct<double> sv = svd(window);

        if (debug) {
          cout << "sv: ";
          for (unsigned int i = 0; i < sv.d.size(); i++)
            cout << sv.d[i] << " ";
          cout << endl;

          cout << "size of u " << nrow(sv.u) << "x" << ncol(sv.u) << endl;
          cout << "size of d " << sv.d.size() << endl;
          cout << "size of v " << nrow(sv.v) << "x" << ncol(sv.v) << endl;
        }

        Matrix singular_vectors = submatrix(sv.u, 0,0,0,nrow(window)-1);

        Matrix temp;
        temp.resize(0,1);
        for(int i=0;i<ncol(singular_vectors);i++)
          {
            this->inturruptionPoint();
            Matrix col = createMatrix(ublas::vector<double>(column(singular_vectors,i)),nrow(singular_vectors),1);
            temp = rbind(temp,Matrix(col));
          }
        if(ncol(svdized)!=0)
          svdized = cbind(svdized, temp);
        else
          svdized = temp;
      }

    // Set raw buffer to whatever is left
    raw_buffer = submatrix(raw_buffer, 0,0, nColsBuffer - num_raw_samples_left - 1, nColsBuffer-1);

    return(svdized);
  }

  map<string, SerializedObject> WindowedSVD::save() const
  {
    map<string, SerializedObject> ret;
    ret["n_lags"] = serialize(n_lags);
    ret["window_size"] = serialize(window_size);
    ret["window_overlap"] = serialize(window_overlap);

    return ret;
  }
  void WindowedSVD::load(map<string, SerializedObject> objects)
  {
    deserialize(objects["n_lags"],n_lags);
    deserialize(objects["window_size"],window_size);
    deserialize(objects["window_overlap"],window_overlap);
  }

  //! reset the feature
  void WindowedSVD::reset()
  {
    raw_buffer.resize(0,0);
  }

}

/*************************************************************/
//DYNAMIC LOADING

extern "C" CEBL::Feature* ObjectCreate()
{
  return new CEBL::WindowedSVD;
}

extern "C" void ObjectDestroy(CEBL::Feature* p)
{
  delete p;
}
