// This file contains a significantly modified version of code found in 
// the sdsl-lite library, as such, the following applies:

/////////////////////////////////////////////////////////////////////////
// Copyright (C) 2007-2014 Simon Gog  All Right Reserved.

// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program.  If not, see http://www.gnu.org/licenses/ .
/////////////////////////////////////////////////////////////////////////


// The modifications are naturally also released under the GPL v3, to
// which the following applies:

/////////////////////////////////////////////////////////////////////////
// Copyright (C) 2014 Martin Muggli  All Right Reserved.

// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program.  If not, see http://www.gnu.org/licenses/ .
/////////////////////////////////////////////////////////////////////////

#include <cmath>
#include <boost/math/distributions/chi_squared.hpp>

std::vector<std::vector<int> > offsets_stack;

int global_align;
// Assume the portion to the right of r has been matched
template<class t_csa, class t_pat_iter>
typename t_csa::size_type
mybs(
    const t_csa& csa,
    const t_pat_iter& begin, 
    t_pat_iter it,  // 'it' is the part matched already
    typename t_csa::size_type l, 
    typename t_csa::size_type r, 
    int accum_silico_len,
    int accum_opt_len,
    int opt_depth, // depth of the search, means the number of tokens we've matched (not counting skips) by the entry to this function
    int &match_count,
    std::vector<typename t_csa::char_type> & running_alignment,
    float chi_squared_sum,
    int deviation_sum


    ) 

    
{

    if (begin == it){ 
        if ( opt_depth >= 1) {
            int sa_interval_size = r + 1 - l;  
            assert(sa_interval_size > 0);  // Normally, we'd check if this is empty before continuing, but using the wavelet tree function we know it's not
            // emit each position in the optical map that matches
            for (int ii = 0 ; ii < sa_interval_size ; ++ii) {
                double chisqcdf = 0.0;
                boost::math::chi_squared cs(opt_depth);
                chisqcdf = boost::math::cdf(cs, chi_squared_sum);

                if (chisqcdf > max_chisquare_cdf) continue;
                float Fval =  (abs((accum_silico_len) - (accum_opt_len )) /   sqrt((opt_depth) * OM_STDDEV * OM_STDDEV));
                if (emit_alignment_pattern) {
                    std::cout << "Alignment pattern: ";
                    //FIXME: reverse the order when appropriate so these visually match up correctly
                    for (typename std::vector<typename t_csa::char_type>::iterator ofitr = running_alignment.begin(); ofitr !=running_alignment.end(); ++ofitr) {
                        if (*ofitr == 0) {
                            std::cout << "*\t" ;
                        } else {
                            std::cout << *ofitr << "\t";
                        }
                    }
                }
                std::cout << std::endl;
                std::cout << "Alignment stats: {'fragnum' : " << sa[l + ii] << ", ";
                std::cout << "'locus' : " << optfrag2locus[sa[l + ii]] << ", ";
                std::cout << "'fval' : " << Fval <<", ";
                std::cout << "'chi_square_sum' : " << chi_squared_sum << ", ";
                std::cout << "'deviation_sum' : " << deviation_sum<< ", ";
                std::cout << "'num_matched_frags' : " << opt_depth << ", ";
                std::cout << "'chi_square_cdf' : " << chisqcdf << "}" << std::endl;

            }
            match_count += sa_interval_size;

        }
    } else {

        // consider skipping over this silico frag completely in case the corresponding optical frag  was lost in the optical mapping process
        if (*(it-1) <= largest_maybe_frag) {
            // if this is the last piece, and we've matched at least one piece so far, then emit the alignment we have
            // else decrement the query symbol index and continue
            running_alignment.push_back(0);

            mybs(csa, begin, (it-1), l, r, accum_silico_len, accum_opt_len, opt_depth, match_count, running_alignment, chi_squared_sum, deviation_sum);
            running_alignment.pop_back();
        }

        // Determine the candidate match value that would allow a further depth match based sheerly on it being an existing prefix character to some suffix in the optical map
        std::vector<typename t_csa::char_type> hits;
        const sdsl::csa_wt<sdsl::wt_int<>, 64, 64, sdsl::sa_order_sa_sampling<>, 
                           sdsl::int_vector<>, 
                           sdsl::int_alphabet<>> * const 
            fm = dynamic_cast<const sdsl::csa_wt<sdsl::wt_int<>, 64, 64, sdsl::sa_order_sa_sampling<>, sdsl::int_vector<>, sdsl::int_alphabet<>> * const>(&csa);
        assert(fm != NULL);
        int min =  (*(it-1)) - search_radius;
        if (min < 0) min = 0;
        int max = (*(it-1)) + search_radius;
        hits = restricted_unique_range_values(fm->wavelet_tree, /*l, fm->wavelet_tree.size(),*/ l, r, min, max);

        // For each such candidate value
        typename std::vector<typename t_csa::char_type>::iterator hits_end = hits.end();
        typename std::vector<typename t_csa::char_type>::iterator c;
        for (c = hits.begin(); c != hits_end; ++c) {


            // First check to see that SOMA eq. 1 is still satisfied
            float Fval =  (abs((accum_silico_len + *(it-1)) - (accum_opt_len + *c)) /   sqrt((opt_depth + 1) * OM_STDDEV * OM_STDDEV));
            int deviation = abs(*(it-1) - *c);
            float chi_squared = std::pow((float)deviation / (float)OM_STDDEV, 2);

            bool frag_seq_matches =  Fval <= f_thresh  ; // SOMA eqn (1), evaluating whether a candidate would still allowed
            if (frag_seq_matches) {
                running_alignment.push_back(*c);                

                // and if so, compute the new SA range
                typename t_csa::size_type newl;
                typename t_csa::size_type newr;
                sdsl::backward_search(csa, l, r, *c, newl, newr); // single character backward search

                if (verbose) {
                    std::vector<int> offsets;
                    // first the silico fragment
                    offsets.push_back(*(it-1)); 
                    // then the candidate optical frag
                    offsets.push_back(*c);
                    int sa_interval_size = newr + 1 - newl;  
                    for (int ii = 0 ; ii < sa_interval_size ; ++ii) {
                        offsets.push_back(optfrag2locus[csa[newl + ii]] );
                    }

                    offsets_stack.push_back(offsets);
                    for (std::vector<std::vector<int> >::iterator frame_it = offsets_stack.begin();  offsets_stack.end() != frame_it; ++frame_it) {
                        std::vector<int> &frame = *frame_it;
                        std::vector<int>::iterator val_it = frame.begin();
                        std::cout << "(silico " << *val_it;
                        ++val_it;
                        std::cout << " ~> optical " << *val_it;
                        ++val_it;
                        std::cout << ", @posns: {";
                        for(; val_it != frame.end(); ++val_it) {
                            std::cout << *val_it << " ";
                        }
                        std::cout << "})   ";

                    }
                    std::cout << std::endl; 
                }

                // if this is the last symbol in the query
                mybs(csa, begin, (it-1), newl, newr, accum_silico_len + *(it-1), accum_opt_len + *c, opt_depth + 1, match_count, running_alignment, chi_squared_sum + chi_squared, deviation_sum + deviation);
                if (verbose) offsets_stack.pop_back();
                running_alignment.pop_back();
            } else {
                if (verbose ) {
                    std::cout << "(silico " << *(it-1) << " != optical " << *c << ")" << std::endl;
                }
            }
        }


    }
}


template<class t_csa, class t_pat_iter>
typename t_csa::size_type
my_backward_search(
    const t_csa& csa,
    typename t_csa::size_type l,
    typename t_csa::size_type r,
    t_pat_iter begin,
    t_pat_iter end,
    typename t_csa::size_type& l_res,
    typename t_csa::size_type& r_res,
    SDSL_UNUSED typename std::enable_if<std::is_same<sdsl::csa_tag, typename t_csa::index_category>::value, sdsl::csa_tag>::type x = sdsl::csa_tag()
    )
{


    std::vector<typename t_csa::char_type>  running_alignment;    

    int explored_count = 0;
    t_pat_iter it = end;



    mybs(csa, begin, it, l, r, 0, 0, 0, explored_count, running_alignment, 0, 0);

//    std::cout << "(explored " << explored_count << ")";
//    std::cout << std::endl;
    return explored_count;
}







template<class t_csa, class t_pat_iter>
typename t_csa::size_type mycount(
    const t_csa& csa,
    t_pat_iter begin,
    t_pat_iter end,
    sdsl::csa_tag
    )
{
    if (end - begin > (typename std::iterator_traits<t_pat_iter>::difference_type)csa.size())
        return 0;
    typename t_csa::size_type t=0; // dummy variable for the backward_search call
    typename t_csa::size_type result = my_backward_search(csa, 0, csa.size()-1, begin, end, t, t);
    return result;
}


// from sdsl-lite:
template<class t_csx, class t_pat_iter>
typename t_csx::size_type mycount1(
    const t_csx& csx,
    t_pat_iter begin,
    t_pat_iter end
    )
{
    typename t_csx::index_category tag;
    return mycount(csx, begin, end,  tag);
}
