function [binnedFeatures,remainingEEG] = freqBands(eeg,state)
% [features,remainingEEG] = freqBands(eeg,state)
%
%  eeg: nChannels x nSamples
% state.freqBands:  structure of parameter values, from freqVarsGUI
%   .channels  list of channel indices into rows of eeg
%   .freqRanges = pairs of beginning and ending frequencies
%   .width = central freq (angular) = usually 5
%   .windowSize: number of samples to calculate variance over
%   .windowShift: number of samples to shift for next window
%
%  features:
%    nFeatures x nWindows

mystate = state.freqBands;

if ~isfield(mystate,'channels') || ...
      ~isfield(mystate,'freqRanges') || ...
      ~isfield(mystate,'width') || ...
      ~isfield(mystate,'windowSize') || ...
      ~isfield(mystate,'windowShift') 
  fprintf(1,'Usage: mystate.channels = 1:6\n');
  fprintf(1,'       mystate.freqRanges = [10 13  20 30]\n');
  fprintf(1,'       mystate.width = 5\n');
  fprintf(1,'       mystate.windowSize = 64\n');
  fprintf(1,'       mystate.windowShift = 32\n');
  fprintf(1,'       features = freqBand(eeg,mystate)\n');
  fprintf(1,'       %  eeg contains one sample per row\n');
  error('               freqBands incorrectly called.');
end

numSamples = size(eeg,2);
data = eeg(mystate.channels,:);
[numChannels,numSamples] = size(data);
numBands = length(mystate.freqRanges)/2;
numFeatures = numChannels * numBands;

  nwin = 0;
  for wi = 1:mystate.windowShift:numSamples - mystate.windowSize+1
    nwin = nwin + 1;
  end

if nwin > 0

  features = zeros(numFeatures,numSamples);

  Fs = 256;

  for b = 1:numBands
    freqs = mystate.freqRanges((b-1)*2+1):mystate.freqRanges(b*2);
    numFreqs = length(freqs);
    for c = 1:length(mystate.channels)
      onechannel = zeros(numFreqs,numSamples);
      for fi = 1:numFreqs
        onechannel(fi,:) = energyvec(freqs(fi),data(c,:),Fs,mystate.width);
      end
      features((b-1)*numChannels+c,:) = mean(onechannel); %mean over freqs in this band
    end %channels loop
  end %bands loop

  binnedFeatures = zeros(numFeatures,nwin);

  for w = 1:nwin
    start = (w-1)*mystate.windowShift+1;
    binnedFeatures(:,w) = mean(features(:,start:start+mystate.windowSize-1),2);
  end

  nUnused = size(data,2) - ( (nwin-1) * mystate.windowShift + ...
      mystate.windowSize);

%binnedFeatures(2,:) = binnedFeatures(1,:) - binnedFeatures(2,:);

else
  binnedFeatures = [];
  nUnused = size(eeg,2);
end


remainingEEG = eeg(:,end-nUnused+1:end);

  
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function y = energyvec(f,s,Fs,width)
% function y = energyvec(f,s,Fs,width)
%
% Return a vector containing the energy as a
% function of time for frequency f. The energy
% is calculated using Morlet's wavelets.
% s : signal
% Fs: sampling frequency
% width : width of Morlet wavelet (>= 5 suggested).
%
%

dt = 1/Fs;
sf = f/width;
st = 1/(2*pi*sf);

t=-3.5*st:dt:3.5*st;
m = morlet(f,t,width);

y = conv(s,m);

y = (2*abs(y)/Fs).^2;
y = y(ceil(length(m)/2):length(y)-floor(length(m)/2));


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function y = morlet(f,t,width)
% function y = morlet(f,t,width)
%
% Morlet's wavelet for frequency f and time t.
% The wavelet will be normalized so the total energy is 1.
% width defines the ``width'' of the wavelet.
% A value >= 5 is suggested.
%
% Ref: Tallon-Baudry et al., J. Neurosci. 15, 722-734 (1997)
%
%
% Ole Jensen, August 1998

sf = f/width;
st = 1/(2*pi*sf);
A = 1/(st*sqrt(2*pi));
y = A*exp(-t.^2/(2*st^2)).*exp(i*2*pi*f.*t);


