#include <math.h>
#include "functions.h"
#include <iostream>
#include <utility>
#include <fstream>
#include <sstream>
#include <iterator>
#include <assert.h>
#include <string>
using namespace std;

// Global Variables
// _____________________________________________________________________________
#define PI (double)(3.14159)

bool bNeedInit = true;
vector<vector<double> > myGeoData;
vector<vector<double> > myGeoOffset;
typedef pair<double, double> pdouble;
vector<double> coord_transform( vector<double> args, pair<double,double> range );
vector<vector<double> > rotMatCached;

// Load the static corrections data
// _____________________________________________________________________________
void loadGeoData()
{
  // Configuration file name
  string baseFile   = "base";
  string offsetFile = "offset";

  ifstream base(baseFile.c_str());
  ifstream offset(offsetFile.c_str());
  istringstream instream;
  double myPatternLength, myProfilelength;
  string s;
  
  // Get the profile 
  // ________________________________________________
  vector<double> myGeoProfile;
  getline(base,s);
  instream.str(s);
  copy(istream_iterator<double>(instream),
       istream_iterator<double>(),
       back_inserter(myGeoProfile));
  instream.clear();
	
  // Read in the base
  // ________________________________________________
  int i=0;
  while (getline(base,s)){
    myGeoData.push_back(vector<double>());
    instream.str(s);
    copy(istream_iterator<double>(instream),
	 istream_iterator<double>(),
	 back_inserter(myGeoData[i]));
    instream.clear();
    i++;
  }

  // Read in the offset 
  // ________________________________________________
  i=0;
  while (getline(offset,s)){
    myGeoOffset.push_back(vector<double>());
    instream.str(s);
    copy(istream_iterator<double>(instream),
	 istream_iterator<double>(),
	 back_inserter(myGeoOffset[i]));
    instream.clear();
    i++;
  }

  // Fill in GEO data
  // ________________________________________________
  for(int r=0; r<myGeoData.size(); r++){
    for(int c=0; c<myGeoData[1].size(); c++){
      myGeoData[r][c] = myGeoData[r][c]+ myGeoProfile[c]+1024;
    }
  }
}

// Statics problem
// _____________________________________________________________________________
double geo_eval( const vector<double> &params )
{
  int    dim=params.size();
  int    depth=myGeoData.size();
  vector<vector<double> > geoTemp;
  
  // Read in data?
  if( bNeedInit == true ){
      loadGeoData();
      bNeedInit = false;
  }

  // Create Temp vector that represents the point
  geoTemp.resize(depth);
  for(int r=0; r<depth; r++){
    geoTemp[r].resize(dim);
    for(int c=0; c<dim; c++){
      geoTemp[r][c] = myGeoData[r][c] - params[c];
		}
  }
  double power = 0.0;
  double sum=0.0;
  // Calculate fitness
  for(int c1=0; c1<(dim-1); c1++){
	for(int c2=(c1+1); c2<dim; c2++){
		for(int r1=0; r1 <depth; r1++){
 			for(int r2=0; r2<depth; r2++){
 				if(geoTemp[r1][c1] <= geoTemp[r2][c2]){
 					power = (geoTemp[r1][c1]+myGeoOffset[r1][c1]) - 
							(geoTemp[r2][c2]-myGeoOffset[r2][c2]);
 				}
				 else if(geoTemp[r2][c2] < geoTemp[r1][c1]){
 					power = (geoTemp[r2][c2]+myGeoOffset[r2][c2]) - 
							(geoTemp[r1][c1]-myGeoOffset[r1][c1]);
 				}
 			if(power>0)
 				sum=sum+power;
			}
 		}
 	}
  }
  
  return -1*sum;
}

// RANA 
// _____________________________________________________________________________
double rana_eval(const vector<double> &params){
  double p1=params[0];
  double p2=params[1];
  double sum=0.0;
  int dim = params.size();
  for(int i=0; i<dim-1; i++){
      p1=params[i];
      p2=params[i+1];
   
      sum+=(p1* sin(sqrt(fabs(p2+1.0-p1)))* cos(sqrt(fabs(p1+p2+1.0))) +
	    (p2+1.0)* cos(sqrt(fabs(p2+1.0-p1)))* sin(sqrt(fabs(p1+p2+1.0))));
    }
  return sum;
}

// GRIEWANGK 
// _____________________________________________________________________________
double griewangk_eval(const vector<double> &params){
  // the number of input parameters is arbitrary - the function
  // is defined for any non-zero dimension.
  assert(params.size()>0);
  double sum=0.0;
  double prod=1.0;

  for(int i=0;i<int(params.size());i++) {
    double xi=params[i];
    sum+=xi*xi/4000.0;
    prod*=cos(xi/sqrt(double(i+1)));
  }
  return 1.0+sum-prod;
}

// ROSENBROCK 
// _____________________________________________________________________________
double rosenbrock_eval(const vector<double> &params) 
{
  double x1=params[0];
  double x2=params[1];
  double sum=0.0;
  int dim = params.size();
  for(int i=0; i<dim-1; i++){
    x1=params[i];
    x2=params[i+1];
    double sq_x1=x1*x1;       
    double diff_x1=1.0-x1;
    sum=sum+(100.0*((sq_x1-x2)*(sq_x1-x2)))+(diff_x1*diff_x1);
  }
  return sum;
}

// SCHWEFEL 
// _____________________________________________________________________________
double schwefel_eval(const vector<double> &params ){
  assert(params.size()>0);
  double result=0.0;

  for(int i=0;i<int(params.size());i++) {
    double xi=params[i];
    result+=(-xi)*sin(sqrt(fabs(xi)));
  }
  return result;
}

// F101 
// _____________________________________________________________________________
double f101_eval(const vector<double> &params){
  double x=params[0];
  double y=params[1];
  double sum=0.0;
  int dim = params.size();
  for(int i=0; i<dim-1; i++){
      x=params[i];
      y=params[i+1];
      sum+= (-x*sin(sqrt(fabs(x-(y+47))))-(y+47)*sin(sqrt(fabs(y+47+(x/2)))));
  }
  return sum; 
}

// F8F2 
// _____________________________________________________________________________
double f8f2_eval( const vector<double> &params ){
  double p1=params[0];
  double p2=params[1];
  double sum = 0.0;

  int dim = params.size();
  for( int i=0; i < dim-1; i++ ){
      p1 = params[i];
      p2 = params[i+1];

      vector<double> f2_args;
      f2_args.push_back( p1 );
      f2_args.push_back( p2 );

      double f2_res = rosenbrock_eval( f2_args );

      // Shift/Scale range of F2 to be the domain of F8
      // Range of F2 ~ [0, 3900]
      f2_res /= 3900;
      f2_res *= 1024;
      f2_res -= 512;

      vector<double> f8_args;
      f8_args.push_back( f2_res );

      sum += griewangk_eval( f8_args );
    }
  return sum;
}

// Original Griewangk
// _____________________________________________________________________________
double g0_eval( const vector<double>& params ){
  double sum = 0.0;
  double prod = 1.0;

  for( int i = 0; i < params.size(); i++ ){
      sum += params[i] * params[i];
      prod *= cos( params[i] / sqrt((double)(i+1)) );
  }

  sum /= 4000.0;
  return (1.0 + sum - prod);
}

// Mod 1
// _____________________________________________________________________________
double g1_eval( const vector<double>& params ){
  double sum = 0.0;
  double prod = 1.0;

  for( int i = 0; i < params.size(); i++ ){
      sum += params[i] * params[i];
      prod *= cos( params[i] ) + 0.1;
  }

  sum /= (4000.0 * params.size());
  prod = log( prod + 1.0 );
  return (sum - prod);
}

// Mod 2
// _____________________________________________________________________________
double g2_eval( const vector<double>& params ){
  double sum = 0.0;
  double prod = 1.0;

  for( int i = 0; i < params.size(); i++ ){
      sum += params[i] * params[i];
      prod *= sqrt(cos( params[i] / params.size() + (i+1) ) + 1.0) * 1.5;
  }

  sum /= (4000.0 * params.size());
  prod = pow( prod, 0.25 );
  return (sum - prod);
}

// Mod 3
// _____________________________________________________________________________
double g3_eval( const vector<double>& params ){
  int N = params.size();
  double sum = params[0]*params[0];
  double prod = cos( pow( 1.1, N-1 ) * params[0] );
  for( int i = 1; i < params.size(); i++ ){
      sum += params[i] * params[i];
      prod *= cos( pow( 1.1, N-i-1 )  * params[i] / pow( 10.0, i-1 ) );
  }

  sum /= (4000.0 * N);
  return (1.0 + sum - prod);
}

// Rotated Functions
// _____________________________________________________________________________
double rot_rana_eval( const vector<double>& args )
{ return rana_eval( coord_transform(args, pdouble( -512.0, 512.0 )) ); }

double rot_griewangk_eval( const vector<double>& args )
{ return griewangk_eval( coord_transform(args, pdouble( -512.0, 512.0)) ); }

double rot_rosenbrock_eval( const vector<double>& args )
{ return rosenbrock_eval( coord_transform(args, pdouble( -2.048, 2.048 )) ); }

double rot_schwefel_eval( const vector<double>& args )
{ return schwefel_eval( coord_transform(args, pdouble( -512.0, 512.0 )) ); }

double rot_f101_eval( const vector<double>& args )
{ return f101_eval( coord_transform(args, pdouble( -512.0, 512.0 )) ); }

double rot_f8f2_eval( const vector<double>& args )
{ return f8f2_eval( coord_transform(args, pdouble( -2.048, 2.048 )) ); }

// Read in Rotation data
// _____________________________________________________________________________
vector<vector<double> > ReadRotMatrix( int nDims ){
  // Check the cache first
  if( rotMatCached.size() == nDims ) return rotMatCached;

  ifstream infile;
  switch( nDims )
    {
    case  2: infile.open("rotation.2");  break;
    case  5: infile.open("rotation.5");  break;
    case 10: infile.open("rotation.10"); break;
    case 20: infile.open("rotation.20"); break;
    default: assert(false);   break;
    }

  vector<vector<double> > resMatrix;
  double temp;
  resMatrix.resize(nDims);
  for( int r = 0; r < nDims; r++ )
    {
      resMatrix[r].resize(nDims);
      for(int c = 0; c < nDims; c++)
	{
	  infile >> temp;
	  resMatrix[r][c] = temp;
	}
    }
  infile.close();

  // Cache
  rotMatCached = resMatrix;

  return resMatrix;
}

// Transform the coordinates
// _____________________________________________________________________________
vector<double> coord_transform( vector<double> args, pair<double,double> range ){
  vector<double> params;
  params.resize( args.size() );

  // Count the number of dimensions
  int nDims = args.size();

  // Load the rotation matrix
  vector<vector<double> > rotMat = ReadRotMatrix( nDims );

  // Apply rotation
  for( int d = 0; d < args.size(); d++ )
    {
      double sum = 0.0;
      for( int i = 0; i < args.size(); i++ )
	sum += rotMat[i][d] * args[i];
      params[d] = sum;
    }

  // Apply translation and scaling
  double length = range.second - range.first;
  for( int d = 0; d < args.size(); d++ )
    params[d] = (params[d] + 0.05*length) * 1.05;

  return params;
}
