/*
 * vectorizer.c: this file is part of the Vectorizer project.
 *
 * Vectorizer, a vectorization module.
 *
 * Copyright (C) 2010 Louis-Noel Pouchet
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 3
 * of the License, or (at your option) any later version.
 *
 * The complete GNU Lesser General Public Licence Notice can be found
 * as the `COPYING.LESSER' file in the root directory.
 *
 * Author:
 * Louis-Noel Pouchet <pouchet@cse.ohio-state.edu>
 */
#if HAVE_CONFIG_H
# include <vectorizer/config.h>
#endif

#include <vectorizer/common.h>

# include <vectorizer/vectorizer.h>

# ifndef CANDL_SUPPORTS_SCOPLIB
#  define CANDL_SUPPORTS_SCOPLIB
# endif
# include <candl/program.h>
# include <candl/dependence.h>
# include <candl/options.h>
# include <scoptools/clast2scop.h>
# include <vectorizer/translator.h>


/**
 * Entry point for the vectorizer.
 *
 */
int
vectorizer (scoplib_scop_p scop, struct clast_stmt* root,
	    s_vectorizer_options_t* options)
{
  printf ("[Vectorizer] Starting vectorization\n");

  // 1- Build a scop according to the new clast.
  scoplib_scop_p newscop;
  if (options->rose_statement_preprocessor)
    newscop = scoptools_clast2scop_rose (root, scop);
  else
    newscop = scoptools_clast2scop (root, scop);

  // scoplib_scop_print (stdout, newscop);

  // 2- Compute the dependences
  CandlOptions* coptions = candl_options_malloc ();
  CandlProgram* cprogram = candl_program_convert_scop (newscop, NULL);
  CandlDependence* deps = candl_dependence (cprogram, coptions);
  // candl_dependence_pprint (stdout, deps);
  candl_options_free (coptions);

  // 3- Translate parallel for nodes to parfor
  if (options->mark_par_loops || options->vectorize_loops)
    vectorizer_translator_parallel (root, newscop, deps);

  // 4- Sink the appropriate parallel loops
  if (options->vectorize_loops)
    vectorizer_transform_sinkparallel (root, newscop, deps, options);

  // 5- Translate parallel inner loop into vectorfor
  if ((options->mark_vect_loops && options->mark_par_loops) ||
      options->vectorize_loops)
    vectorizer_translator_vectorfor (root, newscop, deps);

  // Be clean.
  candl_program_free (cprogram);
  candl_dependence_free (deps);

  printf ("[Vectorizer] Vectorization done\n");

  return 0;
}
