/*
 * explorer.c: this file is part of the LetSee project.
 *
 * LetSee, the LEgal Transformation SpacE Explorator.
 *
 * Copyright (C) 2006,2007,2008 Louis-Noel Pouchet
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * The complete GNU General Public Licence Notice can be found as the
 * `COPYING' file in the root directory.
 *
 * Author:
 * Louis-Noel Pouchet <Louis-Noel.Pouchet@inria.fr>
 *
 */
#if HAVE_CONFIG_H
# include <letsee/config.h>
#endif

#include <letsee/explorer.h>

// FIXME: Create explorer_t initializer and destroyer.



/**
 * Execute a command line (supplied by args) and return a long long
 * int read from its standard output (supposed to be the metric, eg,
 * cycles).
 *
 * Dedicated to execute a transformation.
 *
 */
cycles_t
ls_explorer_execprog (char** args)
{
  pid_t pid;
  int rv;
  int commpipe[2];
  char buf[1024];
  int i;

  if (pipe (commpipe))
    {
      fprintf (stderr, "Pipe error.\n");
      exit (1);
    }

  if((pid = fork ()) == -1)
    {
      fprintf (stderr, "Fork error.\n");
      exit (1);
    }

  if (pid)
    {
      // Parent.
      dup2 (commpipe[0], 0);
      close (commpipe[1]);
      for (i = 0; i < 1024; ++i)
	buf[i] = '\0';
      while (read (0, buf, 1024))
	;
      wait (&rv);
      if (rv != 0)
	{
	  printf ("exit status: %d\n", rv);
	  printf ("output: %s\n", buf);
	}
      close (commpipe[0]);
    }
  else
    {
      // Child.
      dup2 (commpipe[1], 1);
      close (commpipe[0]);
      if (execvp (args[0], args) == -1)
	{
	  fprintf (stderr, "execv Error.\n");
	  exit (1);
	}
      close (commpipe[1]);
    }

  return strtoll (buf, NULL, 10);
}

#ifdef HAVE_LIBPOCC_UTILS
cycles_t
ls_explorer_pocc_exectrans (CandlProgram* program, s_fm_vector_t** draw,
			    s_ls_options_t* options, unsigned count)
{
  if (options->verbose)
    fprintf (options->out_file, "[LetSee] Evaluating candidate %d\n", count);
  s_pocc_utils_options_t* puoptions = options->pocc_utils;
  if (puoptions->output_file_name)
    XFREE(puoptions->output_file_name);
  puoptions->output_file_name = XMALLOC(char, 8192);
  strcpy (puoptions->output_file_name, puoptions->input_file_name);
  int pos = strlen (puoptions->output_file_name);
  while (puoptions->output_file_name[--pos] != '.')
    ;
  puoptions->output_file_name[pos] = '\0';
  sprintf (puoptions->output_file_name, "%s.pocc-sched.%d.c",
	   puoptions->output_file_name, count);
  puoptions->point_idx = count;
  // Generate the cloog file.
  ls_transformations_generate_multi (draw, program, options, count);
  // Copy the transfo.
  puoptions->transfo_matrices = draw;
  // Generate and execute the transformation.
  puoptions->pocc_codegen (puoptions);
  cycles_t cycles;
  if (puoptions->program_exec_result)
    {
      // Print the timing.
      if (options->verbose)
	{
	  fprintf (options->out_file, "[LetSee] Version %d: %s", count,
		   puoptions->program_exec_result);
	  fflush (options->out_file);
	}
      // Print the timing in the iterative.dat file.
      fprintf (puoptions->data_file, "%d %f\n", count,
	       strtod(puoptions->program_exec_result, NULL));
      fflush (puoptions->data_file);
      // Store the best one.
      double value = strtod(puoptions->program_exec_result, NULL);
      value *= 10000;
      cycles = (cycles_t) value;
    }

  return cycles;
}
#endif


/**
 * Wrapper to the ls_explorer_executeprog () function. Transformation
 * is given as a NULL-terminated array of s_fm_vector_t.
 *
 * Dedicated to format the arguments of the script which launch a
 * single test.
 *
 */
cycles_t
ls_explorer_exectrans (CandlProgram* program,
		       s_ls_options_t* options,
		       s_fm_vector_t** draw,
		       unsigned count)
{
#ifdef HAVE_LIBPOCC_UTILS
  if (options->pocc_utils != NULL)
    return ls_explorer_pocc_exectrans (program, draw, options, count);
#endif

  if (! options->create_schedfiles || options->compile_line == NULL)
    return 0;

  char** args = XMALLOC(char*, 6);
  char str[100];
  char buf[100];
  cycles_t ret = 0;

  strcpy(str, options->transfo_dir);
  strcat(str, "/transformation_");
  sprintf(buf, "%09d", count);
  strcat(str, buf);
  strcat(str, ".cloog");

  if (! strcmp (options->transfo_dir, "transformations"))
    args[0] = strdup("src/scripts/onetest.sh");
  else
    {
      // FIXME: remove 2048.
      args[0] = XMALLOC(char, 2048);
      strcpy (args[0], options->transfo_dir);
      strcat (args[0], "/../src/scripts/onetest.sh");
    }
  args[1] = str;
  args[2] = options->compile_line;
  args[3] = options->input_file;
  args[4] = options->transfo_dir;
  args[5] = NULL;

  if (draw[1] != NULL)
    ls_transformations_generate_multi (draw, program, options, count);
  else
    ls_transformations_generate (draw[0], program, options, count);

  // FIXME: Temporarily inhibited
  //  while (ret == 0 || ret == 9223372036854775807)
    ret = ls_explorer_execprog (args);

  // Be clean.
  XFREE(args[0]);
  XFREE(args);

  return ret;
}



/**
 * Wrapper to the ls_explorer_executeprog () function.
 *
 * Dedicated to format the arguments of the script which launch
 * the original code.
 *
 */
cycles_t
ls_explorer_execorig (s_ls_options_t* options)
{
  if (! options->create_schedfiles || options->compile_line == NULL)
    return 0;

  char** args = XMALLOC(char*, 5);
  cycles_t ret = 0;

  args[0] = strdup("src/scripts/origtest.sh");
  args[1] = options->transfo_dir;
  args[2] = options->compile_line;
  args[3] = options->input_file;
  args[4] = NULL;

  while (ret == 0) //  || ret == 9223372036854775807
    ret = ls_explorer_execprog (args);

  // Be clean.
  XFREE(args[0]);
  XFREE(args);

  return ret;
}


/**
 * Cache hit check for executed transformations.
 *
 */
int
ls_explorer_transfo_exists (s_ls_explorer_t* data,
			    s_fm_vector_t* draw)
{
  int i;

  for (i = 0; i < data->count; ++i)
    if (fm_vector_equal (draw, data->draws[i]))
      return 1;

  return 0;
}



/**
 * Filter the results stored in the data structure, provided thresold
 * option.
 *
 */
void
ls_explorer_heuristic_filter (s_ls_options_t* options,
			      s_ls_explorer_t* data)
{
  int i, j;
  int tmpi;
  s_fm_vector_t* tmpv;
  cycles_t best, tmpc;

  if (data->count == 0)
    return;

  // Sort the transformations.
  for (i = 0; i < data->count; ++i)
    for (j = i; j < data->count; ++j)
      if (data->cycles[j] < data->cycles[i])
	{
	  tmpc = data->cycles[i];
	  data->cycles[i] = data->cycles[j];
	  data->cycles[j] = tmpc;
	  tmpv = data->draws[i];
	  data->draws[i] = data->draws[j];
	  data->draws[j] = tmpv;
	  tmpi = data->ids[i];
	  data->ids[i] = data->ids[j];
	  data->ids[j] = tmpi;
	}

  // Filter the transformations.
  tmpc = data->cycles[0] * (1 + (float)(options->thresold) / 100);
  for (i = 0; i < data->count; ++i)
    if (data->cycles[i] > tmpc)
      break;

  tmpi = i;
  for (; i < data->count; ++i)
    fm_vector_free (data->draws[i]);

  if (options->verbose)
    fprintf (options->out_file, "[LetSee] Filtered schedules: %d\n",
	     data->count - tmpi);

  data->count = tmpi;
}


/**
 * Entry point for the exploration phase. Call the heuristic given in
 * argument.
 *
 *
 */
void
ls_explorer (s_ls_space_t* space,
	     CandlProgram* program,
	     s_ls_options_t* options)
{
  if (options->verbose)
    fprintf (options->out_file, "[LetSee] Legal space polytope exploration\n");
  if (options->create_schedfiles)
    {
      // Create transformation directory, if it does not exist.
      char* args[4];
      args[3] = NULL;
      args[0] = "mkdir";
      args[1] = "-p";
      args[2] = options->transfo_dir;
      ls_explorer_execprog (args);
    }

  if (options->heuristic == LS_HEURISTIC_EXHAUST)
    {
      if (options->type == LS_TYPE_FS)
	ls_heuristic_plutocc (space, program, options);
      else
	ls_heuristic_exhaust (space, program, options);
    }
  else if (options->heuristic == LS_HEURISTIC_RANDOM)
    {
      if (options->type == LS_TYPE_FS)
	ls_heuristic_plutocc (space, program, options);
      else
	ls_heuristic_random_r1m (space, program, options);
    }
  else if (options->heuristic == LS_HEURISTIC_R1)
    ls_heuristic_random_r1 (space, program, options);
  else if (options->heuristic == LS_HEURISTIC_H1 ||
	   options->heuristic == LS_HEURISTIC_DH)
    // FIXME: call here the pseudo-multidimensional version, seems to
    // have a problem with multidimensional schedules, though.
    ls_heuristic_h1m (space, program, options);
  // Multidimensional heuristics.
  else if (options->heuristic == LS_HEURISTIC_R1M)
    ls_heuristic_random_r1m (space, program, options);
  else if (options->heuristic == LS_HEURISTIC_M1)
    ls_heuristic_m1 (space, program, options);
  // Pluto dedicated heuristic.
  else if (options->heuristic == LS_HEURISTIC_PLUTO)
    ls_heuristic_pluto (space, program, options);
  else if (options->heuristic == LS_HEURISTIC_PLUTOCC)
    ls_heuristic_plutocc (space, program, options);
  else if (options->heuristic == LS_HEURISTIC_PLUTOM ||
	   options->heuristic == LS_HEURISTIC_PLUTOM1 ||
	   options->heuristic == LS_HEURISTIC_PLUTOM2)
    ls_heuristic_plutom (space, program, options, options->heuristic);
  else if (options->heuristic == LS_HEURISTIC_PLUTOV2)
    ls_heuristic_plutov2 (space, program, options, options->heuristic);

  if (options->verbose)
    {
      if (space != NULL)
	fprintf (options->out_file,
		 "[LetSee] => Number of generated files: %Ld\n",
		 space->size);
      if (options->create_schedfiles)
	fprintf (options->out_file, "[LetSee] Files generated in %s/\n",
		 options->transfo_dir);
    }
}
