/*
 * farkas.c: this file is part of the LetSee project.
 *
 * LetSee, the LEgal Transformation SpacE Explorator.
 *
 * Copyright (C) 2007,2008 Louis-Noel Pouchet
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * The complete GNU General Public Licence Notice can be found as the
 * `COPYING' file in the root directory.
 *
 * Author:
 * Louis-Noel Pouchet <Louis-Noel.Pouchet@inria.fr>
 *
 */
#if HAVE_CONFIG_H
# include <letsee/config.h>
#endif

#include <letsee/farkas.h>
#include <letsee/lspace.h>

#define echo(s) { printf(#s); printf("\n"); }


/**
 * \brief Build and project the system of constraints for a given
 * dependence.
 *
 * Outputs a polytope with affine constraints for \Delta_dep\Theta \ge
 * 'type'
 *
 */
s_fm_solution_t*
ls_farkas_project_multipliers (CandlProgram* program,
			       s_ls_options_t* options,
			       CandlDependence* dep,
			       int type)
{
  s_fm_system_t* to_solve = NULL;
  s_fm_compsol_t* cs_tosolve;
  s_fm_solution_t* solution;
  int dimension;

  to_solve = ls_farkas_build_system (program, options, dep,
				     &dimension, type);
  // Preliminary consistency check.
  if (! fm_piptools_check_int (to_solve))
    {
      //fm_system_print (stdout, to_solve);
      fm_system_free (to_solve);
      return NULL;
    }

  if (options->maxscale_solver)
    {
      s_fm_system_t* ts = to_solve;
      solution = fm_solver
	(ts, FM_SOLVER_FAST | FM_SOLVER_REDREC_IRIGOIN | 
	 FM_SOLVER_AUTO_SIMPLIFY);
      fm_solution_cut (solution, dimension);
      fm_system_free (ts);
    }
  else
    {
      cs_tosolve = fm_compsol_init_sys (to_solve);
      fm_system_free (to_solve);
      to_solve = fm_solution_to_system (cs_tosolve->poly);
      fm_solution_free (cs_tosolve->poly);
      if (options->noredundancy_solver)
	  cs_tosolve->poly = fm_solver
	    (to_solve, FM_SOLVER_FAST | FM_SOLVER_REDREC_IRIGOIN | 
	     FM_SOLVER_VERBOSE);
      else
	cs_tosolve->poly = fm_solver
	  (to_solve, FM_SOLVER_FAST | FM_SOLVER_AUTO_SIMPLIFY | 
	   FM_SOLVER_REDREC_IRIGOIN | FM_SOLVER_VERBOSE);
      solution = fm_compsol_expand (cs_tosolve);
      fm_solution_cut (solution, dimension);
      fm_compsol_free (cs_tosolve);
    }

  return solution;
}


/**
 * \brief Initialize the solution polytope with predefined bounds
 * provided in `options' parameter.
 *
 * The solution polytope is formatted w.r.t. the [ i1 i2 p1 p2 c1 c2 ]
 * organization.
 *
 * input: a program (to extract the number of iterators for each statement)
 *
 * output: a polytope (bounding box of the bounds for each
 * transformation coefficients.)
 *
 */
s_fm_solution_t*
ls_farkas_initialize_solution (CandlProgram* program,
			       s_ls_options_t* options)
{
  int i, j;

  // Compute the size of the polytope.
  int sz = 0;
  for (i = 0; i < program->nb_statements; ++i)
    sz += program->statement[i]->domain->NbColumns - 1;
  int nb_param = program->statement[0]->domain->NbColumns -
    program->statement[0]->depth - 2;

  // Allocate the solution polytope.
  s_fm_solution_t* P = fm_solution_alloc (sz);

  // Fill the solution polytope with the initial bounds.
  int count = 1;
  s_fm_vector_t* tmp;
  z_type_t one; Z_INIT(one); Z_ASSIGN_SI(one, 1);
  z_type_t mone; Z_INIT(mone); Z_ASSIGN_SI(mone, -1);
  z_type_t mlb; Z_INIT(mlb); Z_ASSIGN_SI(mlb, options->lb); Z_OPP(mlb, mlb);

  // Deal with iterator coefficients.
  for (i = 0; i < program->nb_statements; ++i)
    for (j = 0; j < program->statement[i]->depth; ++j)
      {
	// Add iUb.
	tmp = fm_vector_alloc (count + 2);
	fm_vector_assign_int_idx (tmp, mone, count);
	fm_vector_assign_int_idx (tmp, options->Ub, count + 1);
	fm_vector_set_ineq (tmp);
	fm_solution_add_line_at (P, tmp, count);
	// Add ilb.
	tmp = fm_vector_alloc (count + 2);
	fm_vector_assign_int_idx (tmp, one, count);
	fm_vector_assign_int_idx (tmp, mlb, count + 1);
	fm_vector_set_ineq (tmp);
	fm_solution_add_line_at (P, tmp, count);
	++count;
      }

  // Deal with parameter coefficients.
  Z_ASSIGN_SI(mlb, options->plb); Z_OPP(mlb, mlb);
  for (i = 0; i < program->nb_statements; ++i)
    for (j = 0; j < nb_param; ++j)
      {
	// Add pUb.
	tmp = fm_vector_alloc (count + 2);
	fm_vector_assign_int_idx (tmp, mone, count);
	fm_vector_assign_int_idx (tmp, options->pUb, count + 1);
	fm_vector_set_ineq (tmp);
	fm_solution_add_line_at (P, tmp, count);
	// Add plb.
	tmp = fm_vector_alloc (count + 2);
	fm_vector_assign_int_idx (tmp, one, count);
	fm_vector_assign_int_idx (tmp, mlb, count + 1);
	fm_vector_set_ineq (tmp);
	fm_solution_add_line_at (P, tmp, count);
	++count;
      }

  // Deal with constant coefficients.
  Z_ASSIGN_SI(mlb, options->clb); Z_OPP(mlb, mlb);
  for (i = 0; i < program->nb_statements; ++i)
    {
      // Add cUb.
      tmp = fm_vector_alloc (count + 2);
      fm_vector_assign_int_idx (tmp, mone, count);
      fm_vector_assign_int_idx (tmp, options->cUb, count + 1);
      fm_vector_set_ineq (tmp);
      fm_solution_add_line_at (P, tmp, count);
      // Add clb.
      tmp = fm_vector_alloc (count + 2);
      fm_vector_assign_int_idx (tmp, one, count);
      fm_vector_assign_int_idx (tmp, mlb, count + 1);
      fm_vector_set_ineq (tmp);
      fm_solution_add_line_at (P, tmp, count);
      ++count;
    }

  Z_CLEAR(one);
  Z_CLEAR(mone);
  Z_CLEAR(mlb);

  return P;
}


/**
 *
 * \brief Compute the set of affine constraints compatible with FCO property.
 *
 * Outputs a polytope where, for all dependences in 'graph',
 * \Delta\Theta \ge 0.
 *
 */
s_fm_solution_t*
ls_farkas_build_fco_constraints (s_graph_t* graph,
				 CandlProgram* program,
				 s_ls_options_t* options)
{
  s_vertex_t* v;
  s_fm_solution_t* solution;
  CandlDependence* dep;
  s_fm_solution_t* P = ls_farkas_initialize_solution (program, options);

  if (options->verbose)
    fprintf (options->out_file, ".... Initialize solution\n");
  for (v = graph->root; v; v = v->next)
    {
      dep = v->data;
      {
	// Project Farkas multipliers on schedule coefficients.
	// \Delta\Theta \ge 0.
	solution = ls_farkas_project_multipliers
	  (program, options, dep, LS_FARKAS_GEZERO);
	// Add the obtained solution set to the solution polytope.
	if (solution)
	  ls_farkas_intersect (program, options, dep, P, solution);
	// Be clean.
	fm_solution_free (solution);
	if (options->verbose)
	  {
	    fprintf (options->out_file, "O");
	    fflush (options->out_file);
	  }
      }
    }
  if (options->verbose)
    fprintf (options->out_file, "\n");

  return P;
}



/**
 *
 * \brief Compute the set of affine constraints compatible with TH property.
 *
 * Outputs a polytope where, for all dependences in 'graph',
 * \Delta\Theta \ge 0.
 *
 */
s_fm_solution_t*
ls_farkas_build_th_constraints (s_graph_t* graph,
				CandlProgram* program,
				s_ls_options_t* options)
{
  s_vertex_t* v;
  s_fm_solution_t* solution;
  CandlDependence* dep;

  // Backup options.
  z_type_t ilb; Z_INIT(ilb); Z_ASSIGN(ilb, options->lb);
  z_type_t iUb; Z_INIT(iUb); Z_ASSIGN(iUb, options->Ub);
  z_type_t plb; Z_INIT(plb); Z_ASSIGN(plb, options->plb);
  z_type_t pUb; Z_INIT(pUb); Z_ASSIGN(pUb, options->pUb);
  z_type_t clb; Z_INIT(clb); Z_ASSIGN(clb, options->clb);
  z_type_t cUb; Z_INIT(cUb); Z_ASSIGN(cUb, options->cUb);
  // Set bounds.
  Z_ASSIGN_SI(options->lb, 0);
  Z_ASSIGN_SI(options->Ub, 1000);
  Z_ASSIGN_SI(options->plb, 0);
  Z_ASSIGN_SI(options->pUb, 0);
  Z_ASSIGN_SI(options->clb, -1000);
  Z_ASSIGN_SI(options->cUb, 1000);

  s_fm_solution_t* P = ls_farkas_initialize_solution (program, options);

  if (options->verbose)
    fprintf (options->out_file, ".... Initialize solution\n");
  for (v = graph->root; v; v = v->next)
    {
      dep = v->data;
      {
	// Project Farkas multipliers on schedule coefficients.
	// \Delta\Theta \ge 0.
	solution = ls_farkas_project_multipliers
	  (program, options, dep, LS_FARKAS_GEZERO);
	// Add the obtained solution set to the solution polytope.
	if (solution)
	  ls_farkas_intersect (program, options, dep, P, solution);
	// Be clean.
	fm_solution_free (solution);
	if (options->verbose)
	  {
	    fprintf (options->out_file, "O");
	    fflush (options->out_file);
	  }
      }
    }
  if (options->verbose)
    fprintf (options->out_file, "\n");

  // Restore options.
  Z_ASSIGN(options->lb, ilb); Z_CLEAR(ilb);
  Z_ASSIGN(options->Ub, iUb); Z_CLEAR(iUb);
  Z_ASSIGN(options->plb, plb); Z_CLEAR(plb);
  Z_ASSIGN(options->pUb, pUb); Z_CLEAR(pUb);
  Z_ASSIGN(options->clb, clb); Z_CLEAR(clb);
  Z_ASSIGN(options->cUb, cUb); Z_CLEAR(cUb);

  return P;
}


/**
 * \brief Compute the system to solve, for a given dependence and a
 * pair of statements. Outputs the number of schedule coefficients in
 * the system (int* dimension).
 *
 * The function computes the set of Farkas multipliers for the dependence
 * polyhedron (previously computed with Candl), and then equate them
 * with the schedule coefficients of the statement(s). There is
 * exactly one equality per column in the dependence polyhedron. The
 * set of inequality corresponds to the positivity constraints on the
 * Farkas multipliers.
 *
 * The system is of the form [ i1 i2 p1 p2 c1 c2 lambda ].
 *
 * input: a program, letsee options, a dependence (candl format), a type.
 *   The 'type' argument sets the relation between schedule
 *   coefficients in the constructed system. Possible values for type:
 *	LS_FARKAS_EQZERO: Theta_S - Theta_T = 0
 *	LS_FARKAS_GEZERO: Theta_S - Theta_T >= 0
 *	LS_FARKAS_LEZERO: Theta_S - Theta_T <= 0
 *	LS_FARKAS_GEONE: Theta_S - Theta_T >= 1
 *	LS_FARKAS_TH: Theta_S - Theta_T >= 0, i >= 0, \sum i > 0, p = 0
 *
 * output: A matrix of constraints.
 *
 * @fixme The coefficient constraints are not mandatory, but seem to
 * help the solver to be faster.
 *
 */
s_fm_system_t*
ls_farkas_build_system (CandlProgram* program,
			s_ls_options_t* options,
			CandlDependence* tmp,
			int* dimension,
			int type)
{
  int i, j, k;

  // Compute the number of Farkas multipliers for the dependence (plus
  // lambda 0).
  int sz_lambda = 1;
  for (i = 0; i < tmp->domain->NbRows; ++i, ++sz_lambda)
    if (Z_CMP_SI(tmp->domain->p[i][0], ==, 0))
      ++sz_lambda;

  // Compute the number of schedule coefficients related to the dependence.
  *dimension = tmp->source->domain->NbColumns - 1;
  if (tmp->source !=  tmp->target)
    *dimension += tmp->target->domain->NbColumns - 1;

  // Allocate the system.
  s_fm_system_t* system;
  int nb_lines = sz_lambda + tmp->domain->NbColumns - 1 + 2 * *dimension;
  int nb_cols = sz_lambda + *dimension + 2;
  if (type == LS_FARKAS_EQZERO)
    nb_lines += tmp->domain->NbColumns - 1;
  else if (type == LS_FARKAS_TH)
    nb_lines += tmp->source == tmp->target ? 1 : 2;
  system = fm_system_alloc (nb_lines, nb_cols);

  // System is: t_Target - t_Source = Lambda.
  //
  // Affect the "first" statement to st1, the "second" to st2.
  CandlStatement* st1;
  CandlStatement* st2;
  CandlStatement* stmp;
  z_type_t val; Z_INIT(val);
  z_type_t modifier; Z_INIT(modifier);
  if (tmp->source->label <= tmp->target->label)
    {
      st1 = tmp->source;
      st2 = tmp->target;
      Z_ASSIGN_SI(modifier, 1);
    }
  else
    {
      st2 = tmp->source;
      st1 = tmp->target;
      Z_ASSIGN_SI(modifier, -1);
    }

  // If we look for \Delta\Theta <= 0, we just revert src and dest of
  // the dependence
  if (type == LS_FARKAS_LEZERO)
    {
      stmp = st1;
      st1 = st2;
      st2 = stmp;
      Z_ASSIGN_SI(val, -1);
      Z_MUL(val, val, modifier);
    }

  int count = 0;
  int lambda_count = 0;
  int l_offs = 0;
  if (Z_CMP_SI(modifier, ==, -1))
    l_offs = st2->depth;
  // Fill the st1 iterator part.
  Z_ASSIGN_SI(val, -1);
  Z_MUL(val, val, modifier);
  for (i = 1; i <= st1->depth; ++i)
    {
      fm_vector_assign_int_idx (system->lines[count], val, i);
      lambda_count = *dimension + 2;
      for (j = 0; j < tmp->domain->NbRows; ++j)
	{
	  fm_vector_assign_int_idx (system->lines[count],
				    tmp->domain->p[j][i + l_offs],
				    lambda_count);
	  Z_OPP(system->lines[count]->vector[lambda_count].num,
		system->lines[count]->vector[lambda_count].num);
	  if (Z_CMP_SI(tmp->domain->p[j][0], ==, 0))
	    fm_vector_assign_int_idx (system->lines[count],
				      tmp->domain->p[j][i + l_offs],
				      ++lambda_count);
	  ++lambda_count;
	}
      ++count;
    }

  // Fill the st2 iterator part.
  Z_OPP(val, val);
  if (Z_CMP_SI(modifier, ==, -1))
    l_offs = st1->depth;
  for (; i <= st1->depth + st2->depth; ++i)
    {
      if (st1 != st2)
	fm_vector_assign_int_idx (system->lines[count], val, i);
      else
	fm_vector_assign_int_idx (system->lines[count], val, i - st1->depth);
      lambda_count = *dimension + 2;
      for (j = 0; j < tmp->domain->NbRows; ++j)
	{
	  fm_vector_assign_int_idx (system->lines[count],
				    tmp->domain->p[j][i - l_offs],
				    lambda_count);
	  Z_OPP(system->lines[count]->vector[lambda_count].num,
		system->lines[count]->vector[lambda_count].num);
	  if (Z_CMP_SI(tmp->domain->p[j][0], ==, 0))
	    fm_vector_assign_int_idx (system->lines[count],
				      tmp->domain->p[j][i - l_offs],
				      ++lambda_count);
	  ++lambda_count;
	}
      ++count;
    }

  // Fill the st1 and st2 parameter part.
  Z_OPP(val, val);
  int sz_param = st1->domain->NbColumns - st1->depth - 2;
  for (; i <= st1->depth + st2->depth + sz_param; ++i)
    {
      if (st1 != st2)
	{
	  fm_vector_assign_int_idx (system->lines[count], val, i);
	  Z_OPP(val, val);
	  fm_vector_assign_int_idx (system->lines[count], val, i + sz_param);
	  Z_OPP(val, val);
	}
      lambda_count = *dimension + 2;
      for (j = 0; j < tmp->domain->NbRows; ++j)
	{
	  fm_vector_assign_int_idx (system->lines[count],
				    tmp->domain->p[j][i],
				    lambda_count);
	  Z_OPP(system->lines[count]->vector[lambda_count].num,
		system->lines[count]->vector[lambda_count].num);
	  if (Z_CMP_SI(tmp->domain->p[j][0], ==, 0))
	      fm_vector_assign_int_idx (system->lines[count],
					tmp->domain->p[j][i],
					++lambda_count);
	  ++lambda_count;
	}
      ++count;
    }

  // Fill the st1 and st2 constant part.
  z_type_t one; Z_INIT(one); Z_ASSIGN_SI(one, -1);
  // Add -lambda_0, and -1.
  if (type == LS_FARKAS_GEONE)
    fm_vector_assign_int_idx (system->lines[count], one, system->nb_cols - 1);
  fm_vector_assign_int_idx (system->lines[count], one, *dimension + 1);
  if (st1 != st2)
    {
      fm_vector_assign_int_idx (system->lines[count], val, i + sz_param);
      Z_OPP(val, val);
      fm_vector_assign_int_idx (system->lines[count], val, i + sz_param + 1);
      Z_OPP(val, val);
    }
  lambda_count = *dimension + 2;
  for (j = 0; j < tmp->domain->NbRows; ++j)
    {
      fm_vector_assign_int_idx (system->lines[count],
				tmp->domain->p[j][i],
				lambda_count);
      Z_OPP(system->lines[count]->vector[lambda_count].num,
	    system->lines[count]->vector[lambda_count].num);
      if (Z_CMP_SI(tmp->domain->p[j][0], ==, 0))
	fm_vector_assign_int_idx (system->lines[count],
				  tmp->domain->p[j][i],
				  ++lambda_count);
      ++lambda_count;
    }
  ++count;

  // Deal with EQZERO: Add \Theta_S - \Theta_R - \Lambda_d \ge 0.
  if (type == LS_FARKAS_EQZERO)
    {
      int nb_eq = count;
      for (i = 0; i < nb_eq; ++i)
       {
         fm_vector_assign (system->lines[count++], system->lines[i]);
         // Oppose the schedule coefficients.
         for (j = 1; j <= *dimension; ++j)
           fm_rational_opp (&(system->lines[count - 1]->vector[j]),
                            &(system->lines[count - 1]->vector[j]));
       }
    }

  // Add positivity constraints for the Farkas multipliers.
  Z_OPP(one, one);
  for (i = *dimension + 1; i < system->nb_cols - 1; ++i)
    {
      fm_vector_set_ineq (system->lines[count]);
      fm_vector_assign_int_idx (system->lines[count++], one, i);
    }

  // Add coefficients constraints.
  // FIXME: This is not necessary.
  int bound;
  z_type_t lb; Z_INIT(lb); Z_ASSIGN(lb, options->lb); Z_OPP(lb, lb);
  z_type_t Ub; Z_INIT(Ub); Z_ASSIGN(Ub, options->Ub);

  // Copy iterator constraints.
  if (st1 != st2)
    bound = st1->depth + st2->depth;
  else
    bound = st1->depth;
  for (i = 1; i <= bound; ++i)
    {
      fm_vector_set_ineq (system->lines[count]);
      fm_vector_assign_int_idx (system->lines[count], 1, i);
      fm_vector_assign_int_idx (system->lines[count], lb,
				system->lines[count]->size - 1);
      count++;
      fm_vector_set_ineq (system->lines[count]);
      fm_vector_assign_int_idx (system->lines[count], -1, i);
      fm_vector_assign_int_idx (system->lines[count], Ub,
				system->lines[count]->size - 1);
      count++;
    }

  // Copy parameter constraints.
  Z_ASSIGN(lb, options->plb); Z_OPP(lb, lb);
  Z_ASSIGN(Ub, options->pUb);
  if (st1 != st2)
    bound = st1->depth + st2->depth + 2 * sz_param;
  else
    bound = st1->depth + sz_param;
  for (; i <= bound; ++i)
    {
      fm_vector_set_ineq (system->lines[count]);
      fm_vector_assign_int_idx (system->lines[count], 1, i);
      fm_vector_assign_int_idx (system->lines[count], lb,
				system->lines[count]->size - 1);
      count++;
      fm_vector_set_ineq (system->lines[count]);
      fm_vector_assign_int_idx (system->lines[count], -1, i);
      fm_vector_assign_int_idx (system->lines[count], Ub,
				system->lines[count]->size - 1);
      count++;
    }

  // Copy constant constraints.
  Z_ASSIGN(lb, options->clb); Z_OPP(lb, lb);
  Z_ASSIGN(Ub, options->cUb);
  if (st1 != st2)
    bound += 2;
  else
    bound += 1;
  for (; i <= bound; ++i)
    {
      fm_vector_set_ineq (system->lines[count]);
      fm_vector_assign_int_idx (system->lines[count], 1, i);
      fm_vector_assign_int_idx (system->lines[count], lb,
				system->lines[count]->size - 1);
      count++;
      fm_vector_set_ineq (system->lines[count]);
      fm_vector_assign_int_idx (system->lines[count], -1, i);
      fm_vector_assign_int_idx (system->lines[count], Ub,
				system->lines[count]->size - 1);
      count++;
    }

  // Add Tiling hyperplane constraints.
  if (type == LS_FARKAS_TH && 0)
    {
      // Add \sum i > 0.
      for (i = 0; i < st1->depth; ++i)
	fm_vector_assign_int_idx (system->lines[count], 1, i + 1);
      fm_vector_assign_int_idx (system->lines[count], -1,
				system->nb_cols - 1);
      fm_vector_set_ineq (system->lines[count]);
      ++count;
      if (st1 != st2)
	{
	  // Add \sum i > 0.
	  for (; i < st1->depth + st2->depth; ++i)
	    fm_vector_assign_int_idx (system->lines[count], 1, i + 1);
	  fm_vector_assign_int_idx (system->lines[count], -1,
				    system->nb_cols - 1);
	  fm_vector_set_ineq (system->lines[count]);
	}
    }

  // Be clean.
  Z_CLEAR(modifier);
  Z_CLEAR(val);
  Z_CLEAR(one);

  return system;
}




/**
 * \brief Intersect a (compacted) polytope of legal schedule
 * coefficients 'cs' computed for dependence 'dep' with the solution
 * polytope for the whole program 's'. Try to preserve FM-property as
 * much as possible.
 *
 * @fixme: This code is long and ugly, but mandatory to keep the [i1
 * i2 p1 p2 c1 c2] organization.
 *
 */
void
ls_farkas_intersect_cs (CandlProgram* program,
			s_ls_options_t* options,
			CandlDependence* dep,
			s_fm_compsol_t* cs,
			s_fm_solution_t* s)
{
  s_fm_list_t* tmp;
  CandlStatement* s1;
  CandlStatement* s2;
  CandlStatement* stmp;
  s_fm_vector_t* v;
  s_fm_vector_t* c;
  int i, j;
  int bound, P_idx;
  int i_start;


  // Affect the "first" statement to st1, the "second" to st2.
  if (dep->source->label <= dep->target->label)
    {
      s1 = dep->source;
      s2 = dep->target;
    }
  else
    {
      s2 = dep->source;
      s1 = dep->target;
    }

  // Copy the s1 iterator part.
  //
  // Compute i1 offset.
  int i1_offset = 0;
  for (i = 0; i < s1->label; ++i)
    i1_offset += program->statement[i]->depth;

  // Copy, for all i of s1.
  bound = s1->depth;
  for (i = 0; i < bound; ++i)
    {
      for (tmp = s->solution[i].positive; tmp != NULL; tmp = tmp->next)
	{
	  v = tmp->data;
	  c = fm_vector_alloc (i1_offset + i + 3);
	  P_idx = i1_offset;
	  for (j = 1; j < v->size - 1; ++j)
	    fm_vector_assign_idx (c, &(v->vector[j]), P_idx + j);
	  fm_vector_assign_idx (c, &(v->vector[v->size - 1]), c->size - 1);
	  fm_vector_set_ineq (c);
	  fm_compsol_add_unique (cs, c);
	  // fm_solution_add_unique_line_at (P, c, P_idx + j - 1);
	}
      for (tmp = s->solution[i].negative; tmp != NULL; tmp = tmp->next)
	{
	  v = tmp->data;
	  c = fm_vector_alloc (i1_offset + i + 3);
	  P_idx = i1_offset;
	  for (j = 1; j < v->size - 1; ++j)
	    fm_vector_assign_idx (c, &(v->vector[j]), P_idx + j);
	  fm_vector_assign_idx (c, &(v->vector[v->size - 1]), c->size - 1);
	  fm_vector_set_ineq (c);
	  fm_compsol_add_unique (cs, c);
	  // fm_solution_add_unique_line_at (P, c, P_idx + j - 1);
	}
    }

  // Copy the s2 iterator part.
  //
  // Compute i2 offset.
  int i2_offset = 0;
  for (j = 0; j < s2->label; ++j)
    i2_offset += program->statement[j]->depth;

  if (s1 != s2)
    {
      bound += s2->depth;
      i_start = i;
      for (; i < bound; ++i)
	{
	  for (tmp = s->solution[i].positive; tmp != NULL; tmp = tmp->next)
	    {
	      v = tmp->data;
	      c = fm_vector_alloc (i2_offset + i + 3 - i_start);
	      // Copy the i1 part.
	      for (j = 1; j <= s1->depth; ++j)
		fm_vector_assign_idx (c, &(v->vector[j]), i1_offset + j);
	      // Copy the i2 part.
	      P_idx = i2_offset - s1->depth;
	      for (; j < v->size - 1; ++j)
		fm_vector_assign_idx (c, &(v->vector[j]), P_idx + j);
	      fm_vector_assign_idx (c, &(v->vector[v->size - 1]), c->size - 1);
	      fm_vector_set_ineq (c);
	      fm_compsol_add_unique (cs, c);
	      // fm_solution_add_unique_line_at (P, c, P_idx + j - 1);
	    }
	  for (tmp = s->solution[i].negative; tmp != NULL; tmp = tmp->next)
	    {
	      v = tmp->data;
	      c = fm_vector_alloc (i2_offset + i + 3 - i_start);
	      // Copy the i1 part.
	      for (j = 1; j <= s1->depth; ++j)
		fm_vector_assign_idx (c, &(v->vector[j]), i1_offset + j);
	      // Copy the i2 part.
	      P_idx = i2_offset - s1->depth;
	      for (; j < v->size - 1; ++j)
		fm_vector_assign_idx (c, &(v->vector[j]), P_idx + j);
	      fm_vector_assign_idx (c, &(v->vector[v->size - 1]), c->size - 1);
	      fm_vector_set_ineq (c);
	      fm_compsol_add_unique (cs, c);
	      // fm_solution_add_unique_line_at (P, c, P_idx + j - 1);
	    }
	}
    }

  // Copy the s1 parameter part.
  //
  // Compute the number of parameters, and the p1 offset.
  int nb_param = program->statement[0]->domain->NbColumns - 2
    - program->statement[0]->depth;
  int p1_offset = 0;
  for (j = 0; j < program->nb_statements; ++j)
    p1_offset += program->statement[j]->depth;
  for (j = 0; j < s1->label; ++j)
    p1_offset += nb_param;

  bound += nb_param;
  i_start = i;
  for (; i < bound; ++i)
    {
      for (tmp = s->solution[i].positive; tmp != NULL; tmp = tmp->next)
	{
	  v = tmp->data;
	  c = fm_vector_alloc (p1_offset + i + 3 - i_start);
	  // Copy the i1 part.
	  for (j = 1; j <= s1->depth; ++j)
	    fm_vector_assign_idx (c, &(v->vector[j]), i1_offset + j);
	  // Copy the i2 part.
	  if (s1 != s2)
	    for (; j <= s1->depth + s2->depth; ++j)
	      fm_vector_assign_idx (c, &(v->vector[j]),
				    i2_offset + j - s1->depth);
	  // Copy the p1 part.
	  if (s1 != s2)
	    P_idx = p1_offset - s1->depth - s2->depth;
	  else
	    P_idx = p1_offset - s1->depth;
	  for (; j < v->size - 1; ++j)
	    fm_vector_assign_idx (c, &(v->vector[j]), P_idx + j);
	  fm_vector_assign_idx (c, &(v->vector[v->size - 1]), c->size - 1);
	  fm_vector_set_ineq (c);
	  fm_compsol_add_unique (cs, c);
	  // fm_solution_add_unique_line_at (P, c, P_idx + j - 1);
	}
      for (tmp = s->solution[i].negative; tmp != NULL; tmp = tmp->next)
	{
	  v = tmp->data;
	  c = fm_vector_alloc (p1_offset + i + 3 - i_start);
	  // Copy the i1 part.
	  for (j = 1; j <= s1->depth; ++j)
	    fm_vector_assign_idx (c, &(v->vector[j]), i1_offset + j);
	  // Copy the i2 part.
	  if (s1 != s2)
	    for (; j <= s1->depth + s2->depth; ++j)
	      fm_vector_assign_idx (c, &(v->vector[j]),
				    i2_offset + j - s1->depth);
	  // Copy the p1 part.
	  if (s1 != s2)
	    P_idx = p1_offset - s1->depth - s2->depth;
	  else
	    P_idx = p1_offset - s1->depth;
	  for (; j < v->size - 1; ++j)
	    fm_vector_assign_idx (c, &(v->vector[j]), P_idx + j);
	  fm_vector_assign_idx (c, &(v->vector[v->size - 1]), c->size - 1);
	  fm_vector_set_ineq (c);
	  fm_compsol_add_unique (cs, c);
	  // fm_solution_add_unique_line_at (P, c, P_idx + j - 1);
	}
    }

  // Copy the s2 parameter part.
  //
  // Compute the p2 offset.
  int p2_offset = 0;
  for (j = 0; j < program->nb_statements; ++j)
    p2_offset += program->statement[j]->depth;
  for (j = 0; j < s2->label; ++j)
    p2_offset += nb_param;

  if (s1 != s2)
    {
      bound += nb_param;
      i_start = i;
      for (; i < bound; ++i)
	{
	  for (tmp = s->solution[i].positive; tmp != NULL; tmp = tmp->next)
	    {
	      v = tmp->data;
	      c = fm_vector_alloc (p2_offset + i + 3 - i_start);
	      // Copy the i1 part.
	      for (j = 1; j <= s1->depth; ++j)
		fm_vector_assign_idx (c, &(v->vector[j]), i1_offset + j);
	      // Copy the i2 part.
	      for (; j <= s1->depth + s2->depth; ++j)
		fm_vector_assign_idx (c, &(v->vector[j]),
				      i2_offset + j - s1->depth);
	      // Copy the p1 part.
	      for (; j <= bound - nb_param; ++j)
		fm_vector_assign_idx (c, &(v->vector[j]),
				      p1_offset + j - s1->depth - s2->depth);
	      // Copy the p2 part.
	      P_idx = p2_offset - s1->depth - s2->depth - nb_param;
	      for (; j < v->size - 1; ++j)
		fm_vector_assign_idx (c, &(v->vector[j]), P_idx + j);
	      fm_vector_assign_idx (c, &(v->vector[v->size - 1]), c->size - 1);
	      fm_vector_set_ineq (c);
	      fm_compsol_add_unique (cs, c);
	      // fm_solution_add_unique_line_at (P, c, P_idx + j - 1);
	    }
	  for (tmp = s->solution[i].negative; tmp != NULL; tmp = tmp->next)
	    {
	      v = tmp->data;
	      c = fm_vector_alloc (p2_offset + i + 3 - i_start);
	      // Copy the i1 part.
	      for (j = 1; j <= s1->depth; ++j)
		fm_vector_assign_idx (c, &(v->vector[j]), i1_offset + j);
	      // Copy the i2 part.
	      for (; j <= s1->depth + s2->depth; ++j)
		fm_vector_assign_idx (c, &(v->vector[j]),
				      i2_offset + j - s1->depth);
	      // Copy the p1 part.
	      for (; j <= bound - nb_param; ++j)
		fm_vector_assign_idx (c, &(v->vector[j]),
				      p1_offset + j - s1->depth - s2->depth);
	      // Copy the p2 part.
	      P_idx = p2_offset - s1->depth - s2->depth - nb_param;
	      for (; j < v->size - 1; ++j)
		fm_vector_assign_idx (c, &(v->vector[j]), P_idx + j);
	      fm_vector_assign_idx (c, &(v->vector[v->size - 1]), c->size - 1);
	      fm_vector_set_ineq (c);
	      fm_compsol_add_unique (cs, c);
	      // fm_solution_add_unique_line_at (P, c, P_idx + j - 1);
	    }
	}
    }

  // Copy the s1 constant part.
  //
  // Compute the c1 offset.
  int c1_offset = 0;
  for (j = 0; j < program->nb_statements; ++j)
    c1_offset += program->statement[j]->depth;
  c1_offset += nb_param * program->nb_statements;
  c1_offset += s1->label;

  bound += 1;
  i_start = i;
  for (; i < bound; ++i)
    {
      for (tmp = s->solution[i].positive; tmp != NULL; tmp = tmp->next)
	{
	  v = tmp->data;
	  c = fm_vector_alloc (c1_offset + i + 3 - i_start);
	  // Copy the i1 part.
	  for (j = 1; j <= s1->depth; ++j)
	    fm_vector_assign_idx (c, &(v->vector[j]), i1_offset + j);
	  // Copy the i2 part.
	  if (s1 != s2)
	    for (; j <= s1->depth + s2->depth; ++j)
	      fm_vector_assign_idx (c, &(v->vector[j]),
				    i2_offset + j - s1->depth);
	  // Copy the p1 part.
	  if (s1 != s2)
	    for (; j <= s1->depth + s2->depth + nb_param; ++j)
	      fm_vector_assign_idx (c, &(v->vector[j]),
				    p1_offset + j - s1->depth - s2->depth);
	  else
	    for (; j <= s1->depth + nb_param; ++j)
	      fm_vector_assign_idx (c, &(v->vector[j]),
				    p1_offset + j - s1->depth);
	  // Copy the p2 part.
	  P_idx = p2_offset - s1->depth - s2->depth - nb_param;
	  if (s1 != s2)
	    for (; j <= s1->depth + s2->depth + 2 * nb_param; ++j)
	      fm_vector_assign_idx (c, &(v->vector[j]), P_idx + j);
	  // Copy the c1 part.
	  if (s1 != s2)
	    P_idx = c1_offset - s1->depth - s2->depth - 2 * nb_param;
	  else
	    P_idx = c1_offset - s1->depth - nb_param;
	  for (; j < v->size - 1; ++j)
	    fm_vector_assign_idx (c, &(v->vector[j]), P_idx + j);
	  fm_vector_assign_idx (c, &(v->vector[v->size - 1]), c->size - 1);
	  fm_vector_set_ineq (c);
	  fm_compsol_add_unique (cs, c);
	  // fm_solution_add_unique_line_at (P, c, P_idx + j - 1);
	}
      for (tmp = s->solution[i].negative; tmp != NULL; tmp = tmp->next)
	{
	  v = tmp->data;
	  c = fm_vector_alloc (c1_offset + i + 3 - i_start);
	  // Copy the i1 part.
	  for (j = 1; j <= s1->depth; ++j)
	    fm_vector_assign_idx (c, &(v->vector[j]), i1_offset + j);
	  // Copy the i2 part.
	  if (s1 != s2)
	    for (; j <= s1->depth + s2->depth; ++j)
	      fm_vector_assign_idx (c, &(v->vector[j]),
				    i2_offset + j - s1->depth);
	  // Copy the p1 part.
	  if (s1 != s2)
	    for (; j <= s1->depth + s2->depth + nb_param; ++j)
	      fm_vector_assign_idx (c, &(v->vector[j]),
				    p1_offset + j - s1->depth - s2->depth);
	  else
	    for (; j <= s1->depth + nb_param; ++j)
	      fm_vector_assign_idx (c, &(v->vector[j]),
				    p1_offset + j - s1->depth);
	  // Copy the p2 part.
	  P_idx = p2_offset - s1->depth - s2->depth - nb_param;
	  if (s1 != s2)
	    for (; j <= s1->depth + s2->depth + 2 * nb_param; ++j)
	      fm_vector_assign_idx (c, &(v->vector[j]), P_idx + j);
	  // Copy the c1 part.
	  if (s1 != s2)
	    P_idx = c1_offset - s1->depth - s2->depth - 2 * nb_param;
	  else
	    P_idx = c1_offset - s1->depth - nb_param;
	  for (; j < v->size - 1; ++j)
	    fm_vector_assign_idx (c, &(v->vector[j]), P_idx + j);
	  fm_vector_assign_idx (c, &(v->vector[v->size - 1]), c->size - 1);
	  fm_vector_set_ineq (c);
	  fm_compsol_add_unique (cs, c);
	  // fm_solution_add_unique_line_at (P, c, P_idx + j - 1);
	}
    }

  // Copy the s2 constant part.
  //
  // Compute the c2 offset.
  int c2_offset = 0;
  for (j = 0; j < program->nb_statements; ++j)
    c2_offset += program->statement[j]->depth;
  c2_offset += nb_param * program->nb_statements;
  c2_offset += s2->label;

  if (s1 != s2)
    {
      bound += 1;
      i_start = i;
      for (; i < bound; ++i)
	{
	  for (tmp = s->solution[i].positive; tmp != NULL; tmp = tmp->next)
	    {
	      v = tmp->data;
	      c = fm_vector_alloc (c2_offset + i + 3 - i_start);
	      // Copy the i1 part.
	      for (j = 1; j <= s1->depth; ++j)
		fm_vector_assign_idx (c, &(v->vector[j]), i1_offset + j);
	      // Copy the i2 part.
	      for (; j <= s1->depth + s2->depth; ++j)
		fm_vector_assign_idx (c, &(v->vector[j]),
				      i2_offset + j - s1->depth);
	      // Copy the p1 part.
	      for (; j <= s1->depth + s2->depth + nb_param; ++j)
		fm_vector_assign_idx (c, &(v->vector[j]),
				      p1_offset + j - s1->depth - s2->depth);
	      // Copy the p2 part.
	      P_idx = p2_offset - s1->depth - s2->depth - nb_param;
	      for (; j <= s1->depth + s2->depth + 2 * nb_param; ++j)
		fm_vector_assign_idx (c, &(v->vector[j]), P_idx + j);
	      // Copy the c1 part.
	      P_idx = c1_offset - s1->depth - s2->depth - 2 * nb_param;
	      for (; j <= s1->depth + s2->depth + 2 * nb_param + 1; ++j)
		fm_vector_assign_idx (c, &(v->vector[j]), P_idx + j);
	      // Copy the c2 part.
	      P_idx = c2_offset - s1->depth - s2->depth - 2 * nb_param - 1;
	      for (; j < v->size - 1; ++j)
		fm_vector_assign_idx (c, &(v->vector[j]), P_idx + j);
	      fm_vector_assign_idx (c, &(v->vector[v->size - 1]), c->size - 1);
	      fm_vector_set_ineq (c);
	      fm_compsol_add_unique (cs, c);
	      // fm_solution_add_unique_line_at (P, c, P_idx + j - 1);
	    }
	  for (tmp = s->solution[i].negative; tmp != NULL; tmp = tmp->next)
	    {
	      v = tmp->data;
	      c = fm_vector_alloc (c2_offset + i + 3 - i_start);
	      // Copy the i1 part.
	      for (j = 1; j <= s1->depth; ++j)
		fm_vector_assign_idx (c, &(v->vector[j]), i1_offset + j);
	      // Copy the i2 part.
	      for (; j <= s1->depth + s2->depth; ++j)
		fm_vector_assign_idx (c, &(v->vector[j]),
				      i2_offset + j - s1->depth);
	      // Copy the p1 part.
	      for (; j <= s1->depth + s2->depth + nb_param; ++j)
		fm_vector_assign_idx (c, &(v->vector[j]),
				      p1_offset + j - s1->depth - s2->depth);
	      // Copy the p2 part.
	      P_idx = p2_offset - s1->depth - s2->depth - nb_param;
	      for (; j <= s1->depth + s2->depth + 2 * nb_param; ++j)
		fm_vector_assign_idx (c, &(v->vector[j]), P_idx + j);
	      // Copy the c1 part.
	      P_idx = c1_offset - s1->depth - s2->depth - 2 * nb_param;
	      for (; j <= s1->depth + s2->depth + 2 * nb_param + 1; ++j)
		fm_vector_assign_idx (c, &(v->vector[j]), P_idx + j);
	      // Copy the c2 part.
	      P_idx = c2_offset - s1->depth - s2->depth - 2 * nb_param - 1;
	      for (; j < v->size - 1; ++j)
		fm_vector_assign_idx (c, &(v->vector[j]), P_idx + j);
	      fm_vector_assign_idx (c, &(v->vector[v->size - 1]), c->size - 1);
	      fm_vector_set_ineq (c);
	      fm_compsol_add_unique (cs, c);
	      // fm_solution_add_unique_line_at (P, c, P_idx + j - 1);
	    }
	}
    }
}




/**
 * \brief Intersect a polytope of legal schedule coefficients 'P'
 * computed for dependence 'dep' with the solution polytope for the
 * whole program 's'. Try to preserve FM-property as much as possible.
 *
 * @fixme: This code is long and ugly, but mandatory to keep the [i1
 * i2 p1 p2 c1 c2] organization.
 *
 * @fixme: This should be merged with the previous version somehow.
 *
 */
void
ls_farkas_intersect (CandlProgram* program,
		     s_ls_options_t* options,
		     CandlDependence* dep,
		     s_fm_solution_t* P,
		     s_fm_solution_t* s)
{
  s_fm_list_t* tmp;
  CandlStatement* s1;
  CandlStatement* s2;
  CandlStatement* stmp;
  s_fm_vector_t* v;
  s_fm_vector_t* c;
  int i, j;
  int bound, P_idx;
  int i_start;

  // Affect the "first" statement to st1, the "second" to st2.
  if (dep->source->label <= dep->target->label)
    {
      s1 = dep->source;
      s2 = dep->target;
    }
  else
    {
      s2 = dep->source;
      s1 = dep->target;
    }

  // Copy the s1 iterator part.
  //
  // Compute i1 offset.
  int i1_offset = 0;
  for (i = 0; i < s1->label; ++i)
    i1_offset += program->statement[i]->depth;

  // Copy, for all i of s1.
  bound = s1->depth;
  for (i = 0; i < bound; ++i)
    {
      for (tmp = s->solution[i].positive; tmp != NULL; tmp = tmp->next)
	{
	  v = tmp->data;
	  c = fm_vector_alloc (i1_offset + i + 3);
	  P_idx = i1_offset;
	  for (j = 1; j < v->size - 1; ++j)
	    fm_vector_assign_idx (c, &(v->vector[j]), P_idx + j);
	  fm_vector_assign_idx (c, &(v->vector[v->size - 1]), c->size - 1);
	  fm_vector_set_ineq (c);
	  fm_solution_add_unique_line_at (P, c, P_idx + j - 1);
	}
      for (tmp = s->solution[i].negative; tmp != NULL; tmp = tmp->next)
	{
	  v = tmp->data;
	  c = fm_vector_alloc (i1_offset + i + 3);
	  P_idx = i1_offset;
	  for (j = 1; j < v->size - 1; ++j)
	    fm_vector_assign_idx (c, &(v->vector[j]), P_idx + j);
	  fm_vector_assign_idx (c, &(v->vector[v->size - 1]), c->size - 1);
	  fm_vector_set_ineq (c);
	  fm_solution_add_unique_line_at (P, c, P_idx + j - 1);
	}
    }


  // Copy the s2 iterator part.
  //
  // Compute i2 offset.
  int i2_offset = 0;
  for (j = 0; j < s2->label; ++j)
    i2_offset += program->statement[j]->depth;

  if (s1 != s2)
    {
      bound += s2->depth;
      i_start = i;
      for (; i < bound; ++i)
	{
	  for (tmp = s->solution[i].positive; tmp != NULL; tmp = tmp->next)
	    {
	      v = tmp->data;
	      c = fm_vector_alloc (i2_offset + i + 3 - i_start);
	      // Copy the i1 part.
	      for (j = 1; j <= s1->depth; ++j)
		fm_vector_assign_idx (c, &(v->vector[j]), i1_offset + j);
	      // Copy the i2 part.
	      P_idx = i2_offset - s1->depth;
	      for (; j < v->size - 1; ++j)
		fm_vector_assign_idx (c, &(v->vector[j]), P_idx + j);
	      fm_vector_assign_idx (c, &(v->vector[v->size - 1]), c->size - 1);
	      fm_vector_set_ineq (c);
	      fm_solution_add_unique_line_at (P, c, P_idx + j - 1);
	    }
	  for (tmp = s->solution[i].negative; tmp != NULL; tmp = tmp->next)
	    {
	      v = tmp->data;
	      c = fm_vector_alloc (i2_offset + i + 3 - i_start);
	      // Copy the i1 part.
	      for (j = 1; j <= s1->depth; ++j)
		fm_vector_assign_idx (c, &(v->vector[j]), i1_offset + j);
	      // Copy the i2 part.
	      P_idx = i2_offset - s1->depth;
	      for (; j < v->size - 1; ++j)
		fm_vector_assign_idx (c, &(v->vector[j]), P_idx + j);
	      fm_vector_assign_idx (c, &(v->vector[v->size - 1]), c->size - 1);
	      fm_vector_set_ineq (c);
	      fm_solution_add_unique_line_at (P, c, P_idx + j - 1);
	    }
	}
    }

  // Copy the s1 parameter part.
  //
  // Compute the number of parameters, and the p1 offset.
  int nb_param = program->statement[0]->domain->NbColumns - 2
    - program->statement[0]->depth;
  int p1_offset = 0;
  for (j = 0; j < program->nb_statements; ++j)
    p1_offset += program->statement[j]->depth;
  for (j = 0; j < s1->label; ++j)
    p1_offset += nb_param;

  bound += nb_param;
  i_start = i;
  for (; i < bound; ++i)
    {
      for (tmp = s->solution[i].positive; tmp != NULL; tmp = tmp->next)
	{
	  v = tmp->data;
	  c = fm_vector_alloc (p1_offset + i + 3 - i_start);
	  // Copy the i1 part.
	  for (j = 1; j <= s1->depth; ++j)
	    fm_vector_assign_idx (c, &(v->vector[j]), i1_offset + j);
	  // Copy the i2 part.
	  if (s1 != s2)
	    for (; j <= s1->depth + s2->depth; ++j)
	      fm_vector_assign_idx (c, &(v->vector[j]),
				    i2_offset + j - s1->depth);
	  // Copy the p1 part.
	  if (s1 != s2)
	    P_idx = p1_offset - s1->depth - s2->depth;
	  else
	    P_idx = p1_offset - s1->depth;
	  for (; j < v->size - 1; ++j)
	    fm_vector_assign_idx (c, &(v->vector[j]), P_idx + j);
	  fm_vector_assign_idx (c, &(v->vector[v->size - 1]), c->size - 1);
	  fm_vector_set_ineq (c);
	  fm_solution_add_unique_line_at (P, c, P_idx + j - 1);
	}
      for (tmp = s->solution[i].negative; tmp != NULL; tmp = tmp->next)
	{
	  v = tmp->data;
	  c = fm_vector_alloc (p1_offset + i + 3 - i_start);
	  // Copy the i1 part.
	  for (j = 1; j <= s1->depth; ++j)
	    fm_vector_assign_idx (c, &(v->vector[j]), i1_offset + j);
	  // Copy the i2 part.
	  if (s1 != s2)
	    for (; j <= s1->depth + s2->depth; ++j)
	      fm_vector_assign_idx (c, &(v->vector[j]),
				    i2_offset + j - s1->depth);
	  // Copy the p1 part.
	  if (s1 != s2)
	    P_idx = p1_offset - s1->depth - s2->depth;
	  else
	    P_idx = p1_offset - s1->depth;
	  for (; j < v->size - 1; ++j)
	    fm_vector_assign_idx (c, &(v->vector[j]), P_idx + j);
	  fm_vector_assign_idx (c, &(v->vector[v->size - 1]), c->size - 1);
	  fm_vector_set_ineq (c);
	  fm_solution_add_unique_line_at (P, c, P_idx + j - 1);
	}
    }

  // Copy the s2 parameter part.
  //
  // Compute the p2 offset.
  int p2_offset = 0;
  for (j = 0; j < program->nb_statements; ++j)
    p2_offset += program->statement[j]->depth;
  for (j = 0; j < s2->label; ++j)
    p2_offset += nb_param;

  if (s1 != s2)
    {
      bound += nb_param;
      i_start = i;
      for (; i < bound; ++i)
	{
	  for (tmp = s->solution[i].positive; tmp != NULL; tmp = tmp->next)
	    {
	      v = tmp->data;
	      c = fm_vector_alloc (p2_offset + i + 3 - i_start);
	      // Copy the i1 part.
	      for (j = 1; j <= s1->depth; ++j)
		fm_vector_assign_idx (c, &(v->vector[j]), i1_offset + j);
	      // Copy the i2 part.
	      for (; j <= s1->depth + s2->depth; ++j)
		fm_vector_assign_idx (c, &(v->vector[j]),
				      i2_offset + j - s1->depth);
	      // Copy the p1 part.
	      for (; j <= bound - nb_param; ++j)
		fm_vector_assign_idx (c, &(v->vector[j]),
				      p1_offset + j - s1->depth - s2->depth);
	      // Copy the p2 part.
	      P_idx = p2_offset - s1->depth - s2->depth - nb_param;
	      for (; j < v->size - 1; ++j)
		fm_vector_assign_idx (c, &(v->vector[j]), P_idx + j);
	      fm_vector_assign_idx (c, &(v->vector[v->size - 1]), c->size - 1);
	      fm_vector_set_ineq (c);
	      fm_solution_add_unique_line_at (P, c, P_idx + j - 1);
	    }
	  for (tmp = s->solution[i].negative; tmp != NULL; tmp = tmp->next)
	    {
	      v = tmp->data;
	      c = fm_vector_alloc (p2_offset + i + 3 - i_start);
	      // Copy the i1 part.
	      for (j = 1; j <= s1->depth; ++j)
		fm_vector_assign_idx (c, &(v->vector[j]), i1_offset + j);
	      // Copy the i2 part.
	      for (; j <= s1->depth + s2->depth; ++j)
		fm_vector_assign_idx (c, &(v->vector[j]),
				      i2_offset + j - s1->depth);
	      // Copy the p1 part.
	      for (; j <= bound - nb_param; ++j)
		fm_vector_assign_idx (c, &(v->vector[j]),
				      p1_offset + j - s1->depth - s2->depth);
	      // Copy the p2 part.
	      P_idx = p2_offset - s1->depth - s2->depth - nb_param;
	      for (; j < v->size - 1; ++j)
		fm_vector_assign_idx (c, &(v->vector[j]), P_idx + j);
	      fm_vector_assign_idx (c, &(v->vector[v->size - 1]), c->size - 1);
	      fm_vector_set_ineq (c);
	      fm_solution_add_unique_line_at (P, c, P_idx + j - 1);
	    }
	}
    }

  // Copy the s1 constant part.
  //
  // Compute the c1 offset.
  int c1_offset = 0;
  for (j = 0; j < program->nb_statements; ++j)
    c1_offset += program->statement[j]->depth;
  c1_offset += nb_param * program->nb_statements;
  c1_offset += s1->label;

  bound += 1;
  i_start = i;
  for (; i < bound; ++i)
    {
      for (tmp = s->solution[i].positive; tmp != NULL; tmp = tmp->next)
	{
	  v = tmp->data;
	  c = fm_vector_alloc (c1_offset + i + 3 - i_start);
	  // Copy the i1 part.
	  for (j = 1; j <= s1->depth; ++j)
	    fm_vector_assign_idx (c, &(v->vector[j]), i1_offset + j);
	  // Copy the i2 part.
	  if (s1 != s2)
	    for (; j <= s1->depth + s2->depth; ++j)
	      fm_vector_assign_idx (c, &(v->vector[j]),
				    i2_offset + j - s1->depth);
	  // Copy the p1 part.
	  if (s1 != s2)
	    for (; j <= s1->depth + s2->depth + nb_param; ++j)
	      fm_vector_assign_idx (c, &(v->vector[j]),
				    p1_offset + j - s1->depth - s2->depth);
	  else
	    for (; j <= s1->depth + nb_param; ++j)
	      fm_vector_assign_idx (c, &(v->vector[j]),
				    p1_offset + j - s1->depth);
	  // Copy the p2 part.
	  P_idx = p2_offset - s1->depth - s2->depth - nb_param;
	  if (s1 != s2)
	    for (; j <= s1->depth + s2->depth + 2 * nb_param; ++j)
	      fm_vector_assign_idx (c, &(v->vector[j]), P_idx + j);
	  // Copy the c1 part.
	  if (s1 != s2)
	    P_idx = c1_offset - s1->depth - s2->depth - 2 * nb_param;
	  else
	    P_idx = c1_offset - s1->depth - nb_param;
	  for (; j < v->size - 1; ++j)
	    fm_vector_assign_idx (c, &(v->vector[j]), P_idx + j);
	  fm_vector_assign_idx (c, &(v->vector[v->size - 1]), c->size - 1);
	  fm_vector_set_ineq (c);
	  fm_solution_add_unique_line_at (P, c, P_idx + j - 1);
	}
      for (tmp = s->solution[i].negative; tmp != NULL; tmp = tmp->next)
	{
	  v = tmp->data;
	  c = fm_vector_alloc (c1_offset + i + 3 - i_start);
	  // Copy the i1 part.
	  for (j = 1; j <= s1->depth; ++j)
	    fm_vector_assign_idx (c, &(v->vector[j]), i1_offset + j);
	  // Copy the i2 part.
	  if (s1 != s2)
	    for (; j <= s1->depth + s2->depth; ++j)
	      fm_vector_assign_idx (c, &(v->vector[j]),
				    i2_offset + j - s1->depth);
	  // Copy the p1 part.
	  if (s1 != s2)
	    for (; j <= s1->depth + s2->depth + nb_param; ++j)
	      fm_vector_assign_idx (c, &(v->vector[j]),
				    p1_offset + j - s1->depth - s2->depth);
	  else
	    for (; j <= s1->depth + nb_param; ++j)
	      fm_vector_assign_idx (c, &(v->vector[j]),
				    p1_offset + j - s1->depth);
	  // Copy the p2 part.
	  P_idx = p2_offset - s1->depth - s2->depth - nb_param;
	  if (s1 != s2)
	    for (; j <= s1->depth + s2->depth + 2 * nb_param; ++j)
	      fm_vector_assign_idx (c, &(v->vector[j]), P_idx + j);
	  // Copy the c1 part.
	  if (s1 != s2)
	    P_idx = c1_offset - s1->depth - s2->depth - 2 * nb_param;
	  else
	    P_idx = c1_offset - s1->depth - nb_param;
	  for (; j < v->size - 1; ++j)
	    fm_vector_assign_idx (c, &(v->vector[j]), P_idx + j);
	  fm_vector_assign_idx (c, &(v->vector[v->size - 1]), c->size - 1);
	  fm_vector_set_ineq (c);
	  fm_solution_add_unique_line_at (P, c, P_idx + j - 1);
	}
    }

  // Copy the s2 constant part.
  //
  // Compute the c2 offset.
  int c2_offset = 0;
  for (j = 0; j < program->nb_statements; ++j)
    c2_offset += program->statement[j]->depth;
  c2_offset += nb_param * program->nb_statements;
  c2_offset += s2->label;

  if (s1 != s2)
    {
      bound += 1;
      i_start = i;
      for (; i < bound; ++i)
	{
	  for (tmp = s->solution[i].positive; tmp != NULL; tmp = tmp->next)
	    {
	      v = tmp->data;
	      c = fm_vector_alloc (c2_offset + i + 3 - i_start);
	      // Copy the i1 part.
	      for (j = 1; j <= s1->depth; ++j)
		fm_vector_assign_idx (c, &(v->vector[j]), i1_offset + j);
	      // Copy the i2 part.
	      for (; j <= s1->depth + s2->depth; ++j)
		fm_vector_assign_idx (c, &(v->vector[j]),
				      i2_offset + j - s1->depth);
	      // Copy the p1 part.
	      for (; j <= s1->depth + s2->depth + nb_param; ++j)
		fm_vector_assign_idx (c, &(v->vector[j]),
				      p1_offset + j - s1->depth - s2->depth);
	      // Copy the p2 part.
	      P_idx = p2_offset - s1->depth - s2->depth - nb_param;
	      for (; j <= s1->depth + s2->depth + 2 * nb_param; ++j)
		fm_vector_assign_idx (c, &(v->vector[j]), P_idx + j);
	      // Copy the c1 part.
	      P_idx = c1_offset - s1->depth - s2->depth - 2 * nb_param;
	      for (; j <= s1->depth + s2->depth + 2 * nb_param + 1; ++j)
		fm_vector_assign_idx (c, &(v->vector[j]), P_idx + j);
	      // Copy the c2 part.
	      P_idx = c2_offset - s1->depth - s2->depth - 2 * nb_param - 1;
	      for (; j < v->size - 1; ++j)
		fm_vector_assign_idx (c, &(v->vector[j]), P_idx + j);
	      fm_vector_assign_idx (c, &(v->vector[v->size - 1]), c->size - 1);
	      fm_vector_set_ineq (c);
	      fm_solution_add_unique_line_at (P, c, P_idx + j - 1);
	    }
	  for (tmp = s->solution[i].negative; tmp != NULL; tmp = tmp->next)
	    {
	      v = tmp->data;
	      c = fm_vector_alloc (c2_offset + i + 3 - i_start);
	      // Copy the i1 part.
	      for (j = 1; j <= s1->depth; ++j)
		fm_vector_assign_idx (c, &(v->vector[j]), i1_offset + j);
	      // Copy the i2 part.
	      for (; j <= s1->depth + s2->depth; ++j)
		fm_vector_assign_idx (c, &(v->vector[j]),
				      i2_offset + j - s1->depth);
	      // Copy the p1 part.
	      for (; j <= s1->depth + s2->depth + nb_param; ++j)
		fm_vector_assign_idx (c, &(v->vector[j]),
				      p1_offset + j - s1->depth - s2->depth);
	      // Copy the p2 part.
	      P_idx = p2_offset - s1->depth - s2->depth - nb_param;
	      for (; j <= s1->depth + s2->depth + 2 * nb_param; ++j)
		fm_vector_assign_idx (c, &(v->vector[j]), P_idx + j);
	      // Copy the c1 part.
	      P_idx = c1_offset - s1->depth - s2->depth - 2 * nb_param;
	      for (; j <= s1->depth + s2->depth + 2 * nb_param + 1; ++j)
		fm_vector_assign_idx (c, &(v->vector[j]), P_idx + j);
	      // Copy the c2 part.
	      P_idx = c2_offset - s1->depth - s2->depth - 2 * nb_param - 1;
	      for (; j < v->size - 1; ++j)
		fm_vector_assign_idx (c, &(v->vector[j]), P_idx + j);
	      fm_vector_assign_idx (c, &(v->vector[v->size - 1]), c->size - 1);
	      fm_vector_set_ineq (c);
	      fm_solution_add_unique_line_at (P, c, P_idx + j - 1);
	    }
	}
    }
}

/**
 * \brief Check if it is possible to find a legal one-dimensional
 * schedule for this program and the two given dependences.
 *
 * input: the program, the two dependences to check, and options.
 *
 * output: a boolean, 1 if both dependences can be solved at the same
 * time dimension, 0 otherwise.
 *
 */
static
int
ls_farkas_check_program (CandlProgram* prog,
			 CandlDependence* d1,
			 CandlDependence* d2,
			 s_ls_options_t* options)
{
  s_fm_solution_t* P;
  s_fm_compsol_t* cs;
  s_fm_system_t* to_solve;
  s_fm_system_t* system;
  s_fm_solution_t* solution;
  int dimension;
  int ret;
  s_fm_compsol_t* cs_tosolve;

  // Build a legal space for the program.
  P = ls_farkas_initialize_solution (prog, options);
  // dependence 1.
  to_solve = ls_farkas_build_system (prog, options, d1,
				     &dimension, LS_FARKAS_GEONE);

  // FIXME: Experimental. Use Gauss on small systems.
  cs_tosolve = fm_compsol_init_sys (to_solve);
  fm_system_free (to_solve);
  to_solve = fm_solution_to_system (cs_tosolve->poly);
  fm_solution_free (cs_tosolve->poly);

  cs_tosolve->poly = fm_solver_solution_to
    (to_solve, FM_SOLVER_FAST | FM_SOLVER_AUTO_SIMPLIFY, dimension);
  solution = fm_compsol_expand (cs_tosolve);
  fm_solution_cut (solution, dimension);
  fm_compsol_free (cs_tosolve);

  ls_farkas_intersect (prog, options, d1, P, solution);
  fm_system_free (to_solve);
  fm_solution_free (solution);

  // dependence 2.
  to_solve = ls_farkas_build_system (prog, options, d2,
				     &dimension, LS_FARKAS_GEONE);

  // FIXME: Experimental. Use Gauss on small systems.
  cs_tosolve = fm_compsol_init_sys (to_solve);
  fm_system_free (to_solve);
  to_solve = fm_solution_to_system (cs_tosolve->poly);
  fm_solution_free (cs_tosolve->poly);
  cs_tosolve->poly = fm_solver_solution_to
    (to_solve, FM_SOLVER_FAST | FM_SOLVER_AUTO_SIMPLIFY, dimension);
  solution = fm_compsol_expand (cs_tosolve);
  fm_solution_cut (solution, dimension);
  fm_compsol_free (cs_tosolve);

  ls_farkas_intersect (prog, options, d2, P, solution);
  fm_system_free (to_solve);
  fm_solution_free (solution);

  // Check emptiness from compacted solution.
  ret = fm_piptools_check_sol (P, FM_PIPTOOLS_INT);

  /*   ret = cs->empty || fm_piptools_check_sol (cs->poly); */
  /* 	  // Check emptiness from full solution. (useless security...) */
  /* 	  if (has) */
  /* 	    has = fm_piptools_check_sol (P); */

  fm_solution_free (P);

  return ret;
}


/**
 * \brief Check if two dependences can be strongly satisfied in the
 * same time dimension.
 *
 * Dependences are represented in a conflict graph: a graph with a
 * node per dependence, and an edge between two nodes if the 2
 * dependences cannot be strongly satisfied at the same time
 * dimension.
 *
 * The principle is to build a fake program with the involved
 * statements in the two dependences to check, and check if it is
 * possible to find a legal one-dimensional schedule for this
 * pseudo-program. If there is a conflict, an edge is added in the
 * conflict graph 'g'.
 *
 * input: The original program, the conflict graph, two of its nodes,
 * options.
 *
 * output: a boolean, 1 if both dependences can be solved at the same
 * time dimension, 0 otherwise.
 *
 *
 * fixme: can be optimized!!
 *
 */
static
int
ls_farkas_check_conflict (CandlProgram* program,
			  s_graph_t* g,
			  s_vertex_t* d1,
			  s_vertex_t* d2,
			  s_ls_options_t* options)
{
  CandlProgram* prog;
  CandlDependence* tmp = d1->data;
  CandlDependence* tmp2 = d2->data;
  int old_labels[4];
  int i;
  int check_program;
  int ret;
  int count = 0;

  // Build a program with the 1-4 statements involved in
  // tmp and tmp2.
  prog = candl_program_malloc ();
  prog->context = program->context;
  prog->nb_statements += 1 + (tmp->source != tmp->target);
  if (tmp2->source != tmp->source &&
      tmp2->source != tmp->target)
    prog->nb_statements++;
  if (tmp2->target != tmp2->source)
    if (tmp2->target != tmp->source &&
	tmp2->target != tmp->target)
      prog->nb_statements++;
  prog->statement = XMALLOC(CandlStatement*, prog->nb_statements);
  prog->statement[count++] = tmp->source;
  if (tmp->target != tmp->source)
    prog->statement[count++] = tmp->target;
  if (tmp2->source != tmp->source &&
      tmp2->source != tmp->target)
    prog->statement[count++] = tmp2->source;
  if (tmp2->target != tmp2->source)
    if (tmp2->target != tmp->source &&
	tmp2->target != tmp->target)
      prog->statement[count++] = tmp2->target;

  check_program = 1;
  // If there is 4 statements, then the dependences act on
  // a disjoint set, and it is impossible for them to conflict.
  if (count == 4)
    check_program = 0;

  if (check_program)
    {
      // Backup the original statement labels, and set new ones.
      for (i = 0; i < count; ++i)
	{
	  old_labels[i] = prog->statement[i]->label;
	  prog->statement[i]->label = i;
	}
      ret = ls_farkas_check_program (prog, tmp, tmp2, options);

/*       if (tmp->source != tmp2->source || tmp->target != tmp2->target) */
/* 	if (ret == 0) */
/* 	    assert (! "criterion 1 is false"); */

      if (count == 2 && tmp->source == tmp2->source &&
	  tmp->target == tmp2->target)
	if (ret == 0)
	  assert (! "criterion is false");

      // Restore the original statement labels.
      for (i = 0; i < count; ++i)
	prog->statement[i]->label = old_labels[i];
    }

  // Be clean.
  XFREE(prog->statement);
  XFREE(prog);

  return (ret == 0);
}



/**
 * \brief Compute the dependence conflict graph.
 *
 * Compute a graph where:
 *  - each node is a dependence
 *  - each edge is a conflict relation between two dependences (they cannot
 *    be strongly satisfied at the same time dimension).
 *
 * input: a program, a list of dependences, options
 *
 * output: a graph of dependences.
 *
 */
s_graph_t*
ls_farkas_build_conflicts (CandlProgram* program,
			   CandlDependence* dep,
			   s_ls_options_t* options)
{
  s_graph_t* g = ls_graph_alloc ();
  s_edge_t* e;
  s_vertex_t* src;
  s_vertex_t* dst;
  CandlDependence* tmp;
  CandlDependence* tmp2;

  if (options->verbose)
    fprintf (options->out_file,
	     ".... Compute internal dependence graph\n");

  // Compute a graph_t representation of the dependence graph,
  // to enable computation algorithms (typically SCC).
  s_graph_t* dg = ls_graph_alloc ();
  for (tmp = dep; tmp; tmp = tmp->next)
    {
      // Reset the 'solved' bit.
      CANDL_DEP_TAG(tmp, solved) = 0;
      src = ls_graph_create_vertex (g, NULL, tmp->source->label);
      dst = ls_graph_create_vertex (g, NULL, tmp->target->label);
      e = ls_graph_create_edge (g, src, dst, tmp);
    }

  // Compute SCCs.
  int nb_scc = ls_graph_compute_scc (g);
  printf ("nb_scc: %d\n", nb_scc);
  ls_graph_print_scc (g, nb_scc);

  // Report SCC information on dependences.
  s_vertex_t* v;
  s_fm_list_t* l;
  int* scc_ids;
  int* scc_id_1;
  int* scc_id_2;
  for (v = g->root; v; v = v->next)
    {
      for (l = v->in; l; l = l->next)
	{
	  e = l->data;
	  tmp = e->data;
	  scc_ids = (int*) malloc (2 * sizeof(int));
	  scc_ids[0] = e->src->scc;
	  if (e->src->scc != e->dst->scc)
	    scc_ids[1] = e->dst->scc;
	  else
	    scc_ids[1] = -1;
	  CANDL_DEP_TAG(tmp, tag)  = scc_ids;
	}
    }

  // Be clean.
  ls_graph_empty (g);

  if (options->verbose)
    fprintf (options->out_file,
	     ".... Compute dependence pairwise conflict graph\n");

  // Initialize the graph with one vertex per dependence, and store
  // the dependence information.
  for (tmp = dep; tmp; tmp = tmp->next)
    src = ls_graph_create_vertex (g, tmp, CANDL_DEP_TAG(tmp, id));

  // Update the SCC properties of dependences.
  // Criterion:

  CandlDependence* tmp_src;
  CandlDependence* tmp_dst;

  // Explore all the dependence combinations.
  for (tmp = dep; tmp; tmp = tmp->next)
    {
      if (options->verbose)
	fprintf (options->out_file,
		 "..... Check conflicts for dependence %d\n",
		 CANDL_DEP_TAG(tmp, id));
      // src exists in the graph. Retrieve it.
      src = ls_graph_create_vertex (g, NULL, CANDL_DEP_TAG(tmp, id));
      for (tmp2 = tmp->next; tmp2; tmp2 = tmp2->next)
	{
	  // dst exists in the graph. Retrieve it.
	  dst = ls_graph_create_vertex (g, NULL, CANDL_DEP_TAG(tmp2, id));
	  scc_id_1 = CANDL_DEP_TAG(tmp, tag);
	  scc_id_2 = CANDL_DEP_TAG(tmp2, tag);
	  if (scc_id_1[0] == scc_id_2[0])
	    {
	      if (ls_farkas_check_conflict (program, g, src, dst, options))
		{
		  tmp_src = src->data;
		  tmp_dst = dst->data;

		  if (tmp_src->domain->NbRows < tmp_dst->domain->NbRows)
		    e = ls_graph_create_edge (g, src, dst, NULL);
		  else
		    e = ls_graph_create_edge (g, dst, src, NULL);
		}
	    }
	}
    }


  if (options->verbose)
    fprintf (options->out_file,
	     ".... Conflict graph computed (%d conflicts)\n",
	     g->edge_count);

  ls_graph_topological_apply (g);

  ls_graph_print (stdout, g);


  return g;
}
