/*
 * fdspace.c: this file is part of the LetSee project.
 *
 * LetSee, the LEgal Transformation SpacE Explorator.
 *
 * Copyright (C) 2008 Louis-Noel Pouchet
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * The complete GNU General Public Licence Notice can be found as the
 * `COPYING' file in the root directory.
 *
 * Author:
 * Louis-Noel Pouchet <Louis-Noel.Pouchet@inria.fr>
 */
#if HAVE_CONFIG_H
# include <letsee/config.h>
#endif

#include <letsee/fdspace.h>
#include <letsee/lspace.h>


#undef echo
#define echo(S) { printf(#S); printf("\n"); }


static
void
ls_fdspace_put_param_first (s_fm_system_t* s,
			    CandlProgram* program)
{
  int i, j;
  int nb_param = 0;
  int iter_p = 0;

  for (i = 0; i < program->nb_statements; ++i)
    {
      iter_p += program->statement[i]->depth;
      nb_param += program->context->NbColumns - 2;
    }
  for (i = 0; i < nb_param; ++i)
    for (j = iter_p + 1 + i; j > 1 + i; --j)
      fm_system_swap_column (s, j, j - 1);
}


static
void
ls_fdspace_put_constant_first (s_fm_system_t* s,
			       CandlProgram* program)
{
  int i, j;
  int nb_param = 0;
  int iter_p = 0;

  for (i = 0; i < program->nb_statements; ++i)
    {
      iter_p += program->statement[i]->depth;
      nb_param += program->context->NbColumns - 2;
    }
  for (i = 0; i < program->nb_statements; ++i)
    for (j = iter_p + nb_param + 1 + i; j > 1 + i; --j)
      fm_system_swap_column (s, j, j - 1);
}

static
int
ls_fdspace_compute_fdconstraints (CandlStatement* s1,
				  CandlStatement* s2,
				  s_graph_t* g,
				  int depth,
				  CandlProgram* origprog,
				  s_ls_options_t* options)
{
  // Build a pseudo program for the 2 statements.
  CandlProgram* program = candl_program_malloc ();
  program->nb_statements = 2;
  program->context = origprog->context;
  program->statement = XMALLOC(CandlStatement*, 2);
  program->statement[0] = s1;
  program->statement[1] = s2;
  // Backup original statement labels.
  int bls1 = s1->label;
  int bls2 = s2->label;
  s1->label = 0;
  s2->label = 1;
  // Backup verbosity.
  int verb = options->verbose;
  options->verbose = 0;

  // Relax the bounds on coefficients.
  Z_ASSIGN_SI(options->lb, 0);
  Z_ASSIGN_SI(options->Ub, 0);
  Z_ASSIGN_SI(options->plb, 0);
  Z_ASSIGN_SI(options->pUb, 0);
  Z_ASSIGN_SI(options->clb, 0);
  Z_ASSIGN_SI(options->cUb, program->nb_statements);

  // Compute the FCO constraint system for the 2 statements.
  z_type_t zero; Z_INIT(zero); Z_ASSIGN_SI(zero, 0);
  z_type_t one; Z_INIT(one); Z_ASSIGN_SI(one, 1);
  z_type_t mone; Z_INIT(mone); Z_ASSIGN_SI(mone, -1);
  s_fm_solution_t* P = ls_farkas_build_fco_constraints (g, program, options);
  s_fm_system_t* sys = fm_solution_to_system (P);
  ls_fdspace_put_constant_first (sys, program);
  s_fm_solution_t* sol = fm_solver
    (sys, FM_SOLVER_FAST | FM_SOLVER_REDREC_IRIGOIN | FM_SOLVER_SIMPLIFY);
  fm_system_free (sys);
  fm_solution_cut (sol, program->nb_statements);

  // Check fusion/distribution possibility.
  int ret;
  s_fm_vector_t* p = fm_vector_alloc (3);
  fm_vector_assign_int_idx (p, zero, 1);
  fm_vector_assign_int_idx (p, one, 2);
  if (! fm_solution_point_included (sol, p))
    ret = LS_FDSPACE_EQ;
  else
    {
      Z_ASSIGN_SI(options->lb, 0);

      Z_ASSIGN_SI(options->Ub, 1000);
      Z_ASSIGN_SI(options->cUb, 1000);
      fm_solution_free (P);
      P = ls_farkas_build_fco_constraints (g, program, options);
      sys = fm_solution_to_system (P);
      int i;
      s_fm_vector_t* c1 = fm_vector_alloc (sys->nb_cols);
      fm_vector_set_ineq (c1);
      for (i = 0; i < s1->depth; ++i)
	fm_vector_assign_int_idx (c1, one, i + 1);
      fm_vector_assign_int_idx (c1, mone, c1->size - 1);
      fm_system_add_line (sys, c1);
      c1 = fm_vector_alloc (sys->nb_cols);
      fm_vector_set_ineq (c1);
      for (i = 0; i < s2->depth; ++i)
	fm_vector_assign_int_idx (c1, one, i + s1->depth + 1);
      fm_vector_assign_int_idx (c1, mone, c1->size - 1);
      fm_system_add_line (sys, c1);
      if (! fm_piptools_check_int (sys))
	ret = LS_FDSPACE_GT;
      else
	ret = LS_FDSPACE_GE;
      fm_system_free (sys);

    }
  fm_vector_free (p);

  // Restore original statement labels.
  s1->label = bls1;
  s2->label = bls2;
  // Restore verbosity.
  options->verbose = verb;

  // Be clean.
  Z_CLEAR(zero);
  Z_CLEAR(one);
  Z_CLEAR(mone);
  XFREE(program->statement);
  XFREE(program);
  fm_solution_free (sol);
  fm_solution_free (P);

  return ret;
}


static
int
ls_fdspace_check_pair (CandlStatement* s1,
		       CandlStatement* s2,
		       int s1_id,
		       int s2_id,
		       int depth,
		       s_fm_solution_t* solution,
		       s_graph_t* graph,
		       CandlProgram* program,
		       s_ls_options_t* options)
{
  s_graph_t* g = ls_graph_alloc ();
  s_vertex_t* v;
  CandlDependence* dep;
  int uid = 0;
  int ret;
  s_fm_vector_t* vect;
  z_type_t one; Z_INIT(one); Z_ASSIGN_SI(one, 1);
  z_type_t mone; Z_INIT(mone); Z_ASSIGN_SI(mone, -1);

  // Build the dependence graph of s1 and s2.
  for (v = graph->root; v; v = v->next)
    {
      dep = v->data;
      if ((dep->source == s1 || dep->source == s2) &&
	  (dep->target == s1 || dep->target == s2))
	{
	  if (dep->depth <= depth + 1)
	    ls_graph_create_vertex (g, dep, uid++);
	}
    }
  // If there's no dependence, no additional constraint.
  if (g->root == NULL)
    ret = LS_FDSPACE_NO;
  else
    {
      vect = fm_vector_alloc (s2_id + 3);
      fm_vector_assign_int_idx (vect, mone, s1_id + 1);
      fm_vector_assign_int_idx (vect, one, s2_id + 1);
      switch (ls_fdspace_compute_fdconstraints
	      (s1, s2, g, depth, program, options))
	{
	case LS_FDSPACE_GT:
	  fm_vector_set_ineq (vect);
	  fm_vector_assign_int_idx (vect, mone, vect->size - 1);
	  break;
	case LS_FDSPACE_GE:
	  fm_vector_set_ineq (vect);
	  break;
	case LS_FDSPACE_EQ:
	  break;
	default:
	  assert (0);
	  break;
	}
      fm_solution_add_unique_line_at (solution, vect, s2_id + 1);
    }

  Z_CLEAR(one);
  Z_CLEAR(mone);
  ls_graph_free (g);

  return ret;
}



s_fm_compsol_t*
ls_fdspace_compute_one (s_graph_t* graph,
			CandlProgram* program,
			s_ls_options_t* options,
			int cur_depth)
{
  int i, j, k;
  CandlStatement* s1;
  CandlStatement* s2;
  s_fm_solution_t* solution;

  solution = fm_solution_alloc (program->nb_statements);
  // Iterate on all pairs of statements of depth >= i.
  for (j = 0; j < program->nb_statements - 1; ++j)
    {
      if (program->statement[j]->depth < cur_depth)
	continue;
      for (k = j + 1; k < program->nb_statements; ++k)
	{
	  if (program->statement[k]->depth < cur_depth)
	    continue;
	  s1 = program->statement[j];
	  s2 = program->statement[k];
	  if (options->verbose)
	    {
	      fprintf (options->out_file, ".");
	      fflush (options->out_file);
	    }
	  ls_fdspace_check_pair
	    (s1, s2, j, k, i, solution, graph, program, options);
	}
    }

  // Optimize the space.
  if (options->verbose)
    fprintf (options->out_file, "\n.... Optimize the space\n");
  s_fm_compsol_t* cs = fm_compsol_init_sol (solution);
  cs->poly = fm_solution_simplify
    (cs->poly, FM_SOLVER_REDREC_IRIGOIN | FM_SOLVER_VERBOSE);
  s_fm_system_t* scs = fm_solution_to_system (cs->poly);
  fm_solution_free (cs->poly);
  cs->poly = fm_solver (scs, FM_SOLVER_FAST | FM_SOLVER_REDREC_IRIGOIN |
			FM_SOLVER_SIMPLIFY);
  fm_system_free (scs);

  return cs;
}


static
s_ls_space_t*
ls_fdspace_compute_solution (s_graph_t* graph,
			     CandlProgram* program,
			     s_ls_options_t* options)
{
  int i, j, k;
  CandlStatement* s1;
  CandlStatement* s2;
  s_fm_solution_t* solution;

  // Compute the maximum depth.
  int max_dim;
  for (i = 0, max_dim = 0; i < program->nb_statements; ++i)
    max_dim = max_dim < program->statement[i]->depth ?
      program->statement[i]->depth : max_dim;

  // Allocate the solution spaces.
  s_fm_compsol_t** CF = XMALLOC(s_fm_compsol_t*, max_dim + 1);
  s_fm_solution_t** EF = XMALLOC(s_fm_solution_t*, max_dim + 1);
  CF[max_dim] = NULL;
  EF[max_dim] = NULL;

  // Iterate on all dimension.
/*   for (i = 0; i < max_dim; ++i) */
  for (i = 0; i < 1; ++i)
    {
      if (options->verbose)
	fprintf (options->out_file,
		 ".... Compute the space for dimension %d\n", i);
      CF[i] = ls_fdspace_compute_one (graph, program, options, i);
      EF[i] = fm_compsol_expand (CF[i]);
    }
  if (options->verbose)
    fprintf (options->out_file, "\n");

  for (i = 1; i < max_dim; ++i)
    {
      CF[i] = fm_compsol_dup (CF[i - 1]);
      EF[i] = fm_compsol_expand (CF[i]);
    }


  // Allocate and fill the output.
  s_ls_space_t* space = ls_space_alloc ();
  space->polyhedron = NULL;
  space->u_polyhedron = EF;
  space->u_compacted = CF;
  space->dimension = max_dim;

  // Normalize the space.
  ls_fusionspace_normalize_space (program, options, space, 0,
				  program->nb_statements - 1);

  return space;
}


s_fm_solution_t*
ls_fdspace_add_normalization_constraints_2 (CandlProgram* program,
					    s_ls_options_t* options,
					    s_fm_solution_t* sol)
{
  s_fm_system_t* res;
  s_fm_solution_t* final;
  s_fm_system_t* base = fm_solution_to_system (sol);
  int nbvar = sol->size;
  int i, j, k;
  s_fm_vector_t* v;
  z_type_t one; Z_INIT(one); Z_ASSIGN_SI(one, 1);
  z_type_t mone; Z_INIT(mone); Z_ASSIGN_SI(mone, -1);
  z_type_t valmax; Z_INIT(valmax); Z_ASSIGN_SI(valmax, sol->size - 1);

  // Compute the number of decision variables
  int nbdecvar = nbvar * nbvar;
  int countdecvar = nbvar + 1;

  // Allocate the new system.
  res = fm_system_alloc (base->nb_lines, base->nb_cols + nbdecvar);

  letsee_debug (enter 2);

  // Fill it with the old one.
  for (i = 0; i < base->nb_lines; ++i)
    {
      for (j = 0; j < base->nb_cols - 1; ++j)
	fm_rational_copy (&(res->lines[i]->vector[j]),
			  &(base->lines[i]->vector[j]));
      fm_rational_copy (&(res->lines[i]->vector[res->nb_cols - 1]),
			&(base->lines[i]->vector[j]));
      fm_vector_compute_key (&(res->lines[i]->key), res->lines[i]);
    }

  letsee_debug (create x_i);

  // Create the x_i - x_k <= l_1 - l_2 constraints
  for (i = 0; i < nbvar; ++i)
    {
      for (j = 0; j < nbvar; ++j)
	{
	  if (i == j)
	    continue;
	  v = fm_vector_alloc (res->nb_cols);
	  fm_vector_assign_int_idx (v, mone, i + 1);
	  fm_vector_assign_int_idx (v, one, j + 1);
	  fm_vector_assign_int_idx (v, one, countdecvar);
	  fm_vector_assign_int_idx (v, mone, countdecvar + 1);
	  fm_vector_set_ineq (v);
	  fm_system_add_line (res, v);
	  ++countdecvar;
	}
      ++countdecvar;
    }

  letsee_debug (create cst);
  countdecvar = nbvar + 1;
  // Create the l_1 + l_3 <= max_i constraints
  for (i = 0; i < nbvar; ++i)
    {
      for (j = 0; j < nbvar - 1; ++j)
	{
	  if (i == j)
	    continue;
	  v = fm_vector_alloc (res->nb_cols);
	  fm_vector_assign_int_idx (v, mone, countdecvar);
	  fm_vector_assign_int_idx (v, mone, countdecvar + 2);
	  fm_vector_assign_int_idx (v, valmax, v->size - 1);
	  fm_vector_set_ineq (v);
	  fm_system_add_line (res, v);

/* 	  v = fm_vector_alloc (res->nb_cols); */
/* 	  fm_vector_assign_int_idx (v, one, countdecvar); */
/* 	  fm_vector_assign_int_idx (v, one, countdecvar + 2); */
/* 	  fm_vector_assign_int_idx (v, -valmax, v->size - 1); */
/* 	  fm_vector_set_ineq (v); */
/* 	  fm_system_add_line (res, v); */

	  ++countdecvar;
	}
    }

  // Add positivity constraints on the decision variables.
  countdecvar = nbvar + 1;
  for (i = 0; i < nbdecvar; ++i)
    {
      v = fm_vector_alloc (res->nb_cols);
      fm_vector_assign_int_idx (v, one, countdecvar);
      fm_vector_set_ineq (v);
      fm_system_add_line (res, v);
/*       v = fm_vector_alloc (res->nb_cols); */
/*       fm_vector_assign_int_idx (v, mone, countdecvar); */
/*       fm_vector_assign_int_idx (v, 2* valmax - 1, v->size - 1); */
/*       fm_vector_set_ineq (v); */
/*       fm_system_add_line (res, v); */
      ++countdecvar;
    }

  // Be clean.
  Z_CLEAR(valmax);
  Z_CLEAR(one);
  Z_CLEAR(mone);

/*   fm_system_print (stdout, res); */

  final = fm_system_to_solution (res);
  fm_solution_free (sol);
  fm_system_free (res);

  return final;
}

/**
 *
 *
 */
void
ls_fdspace_add_normalization_constraints (CandlProgram* program,
					  s_ls_options_t* options,
					  s_fm_solution_t* sol)
{

  Z_ASSIGN_SI(options->plb, 0);
  Z_ASSIGN_SI(options->pUb, program->nb_statements - 1);
  s_fm_vector_t* v = fm_vector_alloc (sol->size + 2);
  int i, j;
  z_type_t one; Z_INIT(one); Z_ASSIGN_SI(one, 1);
  z_type_t mone; Z_INIT(mone); Z_ASSIGN_SI(mone, -1);
  int sumnbst;
  for (sumnbst = 0, i = 0; i < program->nb_statements; sumnbst += i, ++i)
    ;
/*   printf ("bound: %d\n", sumnbst); */
  z_type_t nbst; Z_INIT(nbst); Z_ASSIGN_SI(nbst, sumnbst);


  // Add \Sum x_i \lt \Sum i
  fm_vector_set_ineq (v);
  for (i = 1; i < v->size - 1; ++i)
    fm_vector_assign_int_idx (v, mone, i);
  fm_vector_assign_int_idx (v, nbst, i);
/*   printf ("=> added: "); */
/*   fm_vector_print (stdout, v); printf ("\n"); */


  fm_solution_add_line_at (sol, v, v->size - 2);

  // k = argmax k . x_i \lt \Sum i
  int k;
  int sumik;
  for (k = 0; k * program->nb_statements < sumnbst; ++k)
    ;
/*   --k; */
  for (sumik = 0, i = 2; i <= k; sumik += i, ++i)
    ;
  sumik *= -1;
  z_type_t sumk; Z_INIT(sumk); Z_ASSIGN_SI(sumk, sumik);
/*   printf ("========== k: %d\n", sumik); */
  // Add \Sum x_i \gt \Sum (k - 1)
  v = fm_vector_alloc (sol->size + 2);
  fm_vector_set_ineq (v);
  for (i = 1; i < v->size - 1; ++i)
    fm_vector_assign_int_idx (v, one, i);
  fm_vector_assign_int_idx (v, sumk, i);

/*   fm_vector_assign_int_idx (v, -5, i); */

/*   printf ("=> added: "); */
/*   fm_vector_print (stdout, v); printf ("\n"); */
  fm_solution_add_line_at (sol, v, v->size - 2);

  // Remove extremal vertices.
  for (j = 0; j < sol->size; ++j)
    {
      v = fm_vector_alloc (sol->size + 2);
      fm_vector_set_ineq (v);
      for (i = 1; i < v->size; ++i)
	fm_vector_assign_int_idx (v, one, i);
      fm_vector_assign_int_idx (v, mone, j + 1);
      fm_solution_add_line_at (sol, v, v->size - 2);
    }



/*   for (j = 0; j < sol->size; ++j) */
/*     { */
/*       v = fm_vector_alloc (sol->size + 2); */
/*       fm_vector_set_ineq (v); */
/*       for (i = 1; i < v->size - 1; ++i) */
/* 	{ */
/* 	  fm_vector_assign_int_idx (v, 1, i); */
/* 	  v->vector[i].denum = 1; */
/* 	} */
/*       fm_vector_assign_int_idx (v, 1, i); */
/*       v->vector[i].denum = 1; */
/*       fm_vector_assign_int_idx (v, mone, j + 1); */
/* /\*       fm_vector_assign_int_idx (v, mone, j + 1); *\/ */
/* /\* 	  v->vector[j+1].denum = 2; *\/ */
/*       fm_vector_normalize_idx (v, v, v->size - 2); */
/*       fm_solution_add_line_at (sol, v, v->size - 2); */
/*     } */


/*   for (j = 0; j < sol->size; ++j) */
/*     { */
/*       v = fm_vector_alloc (sol->size + 2); */
/*       fm_vector_set_ineq (v); */
/*       for (i = 1; i < v->size - 1; ++i) */
/* 	{ */
/* 	  fm_vector_assign_int_idx (v, -1, i); */
/* 	  v->vector[i].denum = 1; */
/* 	} */
/*       fm_vector_assign_int_idx (v, 6, i); */
/*       v->vector[i].denum = 1; */
/*       fm_vector_assign_int_idx (v, one, j + 1); */
/* /\*       fm_vector_assign_int_idx (v, mone, j + 1); *\/ */
/* /\* 	  v->vector[j+1].denum = 2; *\/ */
/*       fm_vector_normalize_idx (v, v, v->size - 2); */
/*       fm_solution_add_line_at (sol, v, v->size - 2); */
/*     } */



  Z_CLEAR(one);
  Z_CLEAR(mone);
  Z_CLEAR(nbst);

/*   printf ("NEW Solution:\n"); */
/*   fm_solution_print (stdout, sol); */
}


/**
 * \brief Build the set of legal fusions.
 *
 *
 */
s_ls_space_t*
ls_fdspace_build (CandlProgram* program,
		  CandlDependence* dependences,
		  s_ls_options_t* options)
{
  int i, j, k;
  s_ls_space_t* space = NULL;
  s_graph_t* graph;
  CandlDependence* tmp;

  if (options->verbose)
    fprintf(options->out_file, "... Build and solve local systems\n");

  // Preprocess the dependence graph.
  CandlDependence* depgraph = // dependences;
    ls_schedspace_depgraph_preprocess (program, dependences);
  candl_dependence_pprint (stdout, depgraph);

  // Build the dependence graph from Candl's representation.
  graph = ls_graph_alloc ();
  int uid = 0;
  for (tmp = depgraph; tmp; tmp = tmp->next)
    {
      // Reset the 'solved' bit.
      CANDL_DEP_TAG(tmp, solved) = 0;
      ls_graph_create_vertex (graph, tmp, uid++);
    }

  // Compute the solution polytope.
  space = ls_fdspace_compute_solution (graph, program, options);

  if (options->verbose)
    fprintf(options->out_file, "... All systems solved\n");

  // Initial PIP check.
  for (i = 0; i < space->size; ++i)
    if (! fm_piptools_check_sol (space->u_polyhedron[i], FM_PIPTOOLS_INT))
      {
	if (options->verbose)
	  fprintf(options->out_file,
		  "=> Ending computation: no point at dimension %d\n",
		  i + 1);
	exit (1);
      }


  // Internal conflict/dependence graph is no longer needed.
  ls_graph_free (graph);

  // Copy space for compatibility with former heuritics.
  space->polyhedron = space->u_polyhedron[0];

  // Store the dependence graph in the space structure.
  space->dependences = depgraph;

  // Store the Candl program in the space structure.
  space->program = program;

  return space;
}
