/*
 * heuristic-m1.c: this file is part of the LetSee project.
 *
 * LetSee, the LEgal Transformation SpacE Explorator.
 *
 * Copyright (C) 2006,2007,2008 Louis-Noel Pouchet
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * The complete GNU General Public Licence Notice can be found as the
 * `COPYING' file in the root directory.
 *
 * Author:
 * Louis-Noel Pouchet <Louis-Noel.Pouchet@inria.fr>
 *
 */
#if HAVE_CONFIG_H
# include <letsee/config.h>
#endif

#include <letsee/heuristic-m1.h>



static int S_LIMIT = 1000;

static cycles_t G_BEST = 0;
static cycles_t G_ORIG = 0;
static int G_BEST_ID = 0;
static int GCOUNT = 0;


/**
 * Add a (prefix of a) vector to a solution polytope.
 *
 */
static
s_fm_solution_t*
ls_heuristic_m1_redeq (s_fm_vector_t* v, int max)
{
  int i;
  s_fm_vector_t* tmp;
  s_fm_solution_t* ret = fm_solution_alloc (v->size - 1);
  z_type_t mone; Z_INIT(mone); Z_ASSIGN_SI(mone, -1);

  for (i = 1; i < max; ++i)
    {
      tmp = fm_vector_alloc (i + 1);
      fm_vector_assign_int_idx (tmp, v->vector[i].num,
				tmp->size - 1);
      Z_OPP(tmp->vector[tmp->size - 1].num,
	    tmp->vector[tmp->size - 1].num);
      fm_list_add_head (&(ret->solution[i - 1].positive), tmp);
    }

  Z_CLEAR(mone);

  return ret;
}



/**
 * Completion heuristic (aka completion algorithm).
 *
 *
 */
static
int
ls_heuristic_m1_complete (s_ls_space_t* space,
			  CandlProgram* program,
			  s_ls_options_t* options,
			  s_fm_vector_t** schedule,
			  int dim,
			  int start_coef)
{
  int i;
  int ii = 0;
  int bad_val, last_bad, last_bad2;

  s_fm_rational_t* min = fm_rational_alloc ();
  s_fm_rational_t* max = fm_rational_alloc ();
  z_type_t zero; Z_INIT(zero); Z_ASSIGN_SI(zero, 0);

  s_fm_system_t* syst_tmp = NULL;
  s_fm_system_t* syst = NULL;
  s_fm_solution_t* seq = NULL;
  s_fm_compsol_t* cs = NULL;
  s_fm_vector_t* draw = NULL;
  s_fm_solution_t* P = NULL;
  s_fm_solution_t* Pe = NULL;

  bad_val = last_bad = -1;

  syst_tmp = fm_solution_to_system (space->u_polyhedron[dim]);
  seq = ls_heuristic_m1_redeq (schedule[dim], start_coef);
  syst = fm_system_reduce (syst_tmp, seq);
  fm_system_free (syst_tmp);
  fm_solution_free (seq);

  // Debug. This is never supposed to occur.
  if (syst == NULL)
    {
      printf ("Gros probleme... valeur impossible pour les iterateurs\n");
      return 1;
    }

  cs = fm_compsol_init_sys (syst);

  // Debug. This is never supposed to occur.
  if (cs->empty)
    {
      printf ("No integer point with this prefix\n");
      return 1;
    }

  Pe = fm_compsol_expand (cs);
  fm_compsol_free (cs);
  fm_system_free (syst);
  draw = fm_vector_alloc (Pe->size + 2);

  for (i = start_coef; i < schedule[dim]->size; ++i)
    {
      fm_solver_compute_min
	(&min,
	 Pe->solution[i - start_coef].positive,
	 draw, i - start_coef, FM_MINLEXICO_INT);
      fm_solver_compute_max
	(&max,
	 Pe->solution[i - start_coef].negative,
	 draw, i - start_coef, FM_MINLEXICO_INT);
      ls_heuristics_narrow_intbounds (min, max, Pe, i - start_coef, draw);

      if (Z_CMP(min->num, >, max->num) && i == start_coef)
	{
	  printf ("Schedule can't be completed\n");
	  return 1;
	}
      // FIXME: Debug.
      if (Z_CMP(min->num, >, max->num))
	{
	  bad_val = i - 1;
	  i = start_coef - 1;
	  continue;
	}
      if (i == bad_val)
	{
	  if (last_bad == bad_val && last_bad2 == bad_val)
	    {
	      fm_rational_free (min);
	      fm_rational_free (max);
	      fm_vector_free (draw);
	      fm_solution_free (Pe);
	      printf ("We're HERE!\n");
	      exit (42);
	      return 1;
	    }
	  if (last_bad == -1)
	    {
	      fm_vector_assign_int_idx (schedule[dim], min->num, i);
	      fm_vector_assign_int_idx (draw, min->num, i - start_coef + 1);
	    }
	  else
	    {
	      fm_vector_assign_int_idx (schedule[dim], max->num, i);
	      fm_vector_assign_int_idx (draw, max->num, i - start_coef + 1);
	      last_bad2 = bad_val;
	    }
	  last_bad = bad_val;
	  bad_val = -1;
	}
      else
	{
	  if (i > last_bad)
	    {
	      last_bad = -1;
	      last_bad2 = -1;
	    }
	  if (Z_CMP_SI(min->num, <=, 0))
	    {
	      if (Z_CMP_SI(max->num, >=, 0))
		{
		  fm_vector_assign_int_idx (schedule[dim], 0, i);
		  fm_vector_assign_int_idx (draw, 0, i - start_coef + 1);
		}
	      else
		{
		  fm_vector_assign_int_idx (schedule[dim], max->num, i);
		  fm_vector_assign_int_idx (draw, max->num,
					    i - start_coef + 1);
		}
	    }
	  else
	    {
	      fm_vector_assign_int_idx (schedule[dim], min->num, i);
	      fm_vector_assign_int_idx (draw, min->num, i - start_coef + 1);
	    }
	}
    }
  fm_rational_free (min);
  fm_rational_free (max);
  fm_vector_free (draw);
  fm_solution_free (Pe);
  Z_CLEAR(zero);
  return 0;
}



/**
 * Helper to compute the scheme (that is, the classes of coefficient
 * traversed for each schedule dimension).
 *
 */
static
int
get_bounds (int nb_coef_iter, int nb_coef_param, int nb_coef_cst, int scheme)
{
  switch (scheme)
    {
    case LS_HEURISTIC_M1_SCHEME_ITER:
      return nb_coef_iter;
    case LS_HEURISTIC_M1_SCHEME_ITERPARAM:
      return nb_coef_iter + nb_coef_param;
    case LS_HEURISTIC_M1_SCHEME_FULL:
      return nb_coef_iter + nb_coef_param + nb_coef_cst;
    case LS_HEURISTIC_M1_SCHEME_NONE:
      return 0;
    }
}


/**
 * Initialize with default scheme.
 *
 *
 */
static
void
ls_heuristic_m1_initialize_scheme (s_ls_options_t* options, int dim)
{
  int i;

  if (options->scheme_m1 == NULL)
    options->scheme_m1 = XMALLOC(int, dim + 1);
  options->scheme_m1[dim] = -1;

  for (i = 0; i < dim; ++i)
    options->scheme_m1[i] = LS_HEURISTIC_M1_SCHEME_NONE;
  for (i = 0; i < 2 && i <= dim; ++i)
    options->scheme_m1[i] = LS_HEURISTIC_M1_SCHEME_ITER;
}



/**
 * Exhaustive polytope explorer heuristic for multidimensionnal schedules..
 *
 *
 */
static
void
ls_heuristic_m1_exhaust (CandlProgram* program,
			 s_ls_options_t* options,
			 s_ls_space_t* space,
			 int count,
			 s_fm_vector_t** prototype)
{
  s_fm_solution_t* sol = space->u_polyhedron[count];
  s_fm_rational_t* lb = NULL;
  s_fm_rational_t* Ub = NULL;
  z_type_t* i = XMALLOC(z_type_t, sol->size);
  z_type_t* min = XMALLOC(z_type_t, sol->size);
  z_type_t* max = XMALLOC(z_type_t, sol->size);
  int ii;
  unsigned idx = 0;
  s_fm_vector_t* draw = prototype[count];
  cycles_t best = 0;
  cycles_t orig = 0, tmp;
  int id = 0;
  unsigned long long int badpath = 0;
  int transfo_count = 0;
  int t_count = 0;
  int ret;
  int ctr = 10;
  unsigned long int violation = 0;
  int bound;

  // Compute the index of the first coefficient to complete.
  int nb_coef_iter, nb_coef_param;
  for (ii = 0, nb_coef_iter = 0; ii < program->nb_statements; ++ii)
    nb_coef_iter += program->statement[ii]->depth;
  for (ii = 0, nb_coef_param = 0; ii < program->nb_statements; ++ii)
    nb_coef_param += program->context->NbColumns - 2;

  // Compute the bound for completion, depending on the current
  // explored dimension. See options->scheme_m1.
  // NOTE: The cooking is done here, change these parameters when needed!
  bound = get_bounds (nb_coef_iter, nb_coef_param, program->nb_statements,
		      options->scheme_m1[count]);

  for (ii = 0; ii < sol->size; ++ii)
    {
      Z_INIT(i[ii]);
      Z_INIT(min[ii]);
      Z_INIT(max[ii]);
    }

  fm_solver_compute_min (&lb, sol->solution[idx].positive,
			 draw, idx, FM_MINLEXICO_INT);
  fm_solver_compute_max (&Ub, sol->solution[idx].negative,
			 draw, idx, FM_MINLEXICO_INT);
/*   ls_heuristics_narrow_intbounds (lb, Ub, sol, idx, draw); */
  min[idx] = lb->num;
  max[idx] = Ub->num;

  for (Z_ASSIGN(i[idx], min[idx]); Z_CMP(i[idx], <=, max[idx]);
       Z_INC(i[idx], i[idx]))
    {
      if (space->size > S_LIMIT)
	break;
      fm_vector_assign_int_idx (draw, i[idx], idx + 1);
      if (idx + 1 >= bound)
	{
	  if (bound != prototype[0]->size - 1)
	    ret = ls_heuristic_m1_complete (space, program,
					    options, prototype,
					    count, bound + 1);
	  else
	    ret = 0;
	  if (ret)
	    {
	      // We're facing an integer hole.
	      if (options->scheme_m1[count] == LS_HEURISTIC_M1_SCHEME_NONE)
		goto clean;
	      else
		goto hack_continue;
	    }
	  if (count + 1 == space->dimension)
	    {
	      if (options->create_schedfiles)
		{
		  if (transfo_count++ > S_LIMIT)
		      break;
		  if (options->compile_line == NULL)
		    ret = ls_transformations_generate_multi
		      (prototype, program, options, space->size);
		  else
		    {
		      tmp = ls_explorer_exectrans (program, options,
						   prototype, GCOUNT);
		      if (tmp < G_BEST || G_BEST == 0)
			{
			  G_BEST = tmp;
			  G_BEST_ID = GCOUNT;
			}
		      if (tmp)
			printf ("%d %Ld\n", ++GCOUNT, tmp);

		      // dummy.
		      ret = 0;
		    }
		  if (! ret)
		    {
		      space->size++;
		      if (options->verbose)
			{
			  fprintf (options->out_file, ".");
			  fflush (options->out_file);
			}
		    }
		  else
		    {
		      transfo_count--;
		      violation++;
		    }
		  if (options->scheme_m1[count] == LS_HEURISTIC_M1_SCHEME_NONE)
		    goto clean;
		  // FIXME: this new line is very important to avoid
		  // doing multiple times the same completion.
		  if (bound == 0)
		    goto clean;
		}
	    }
	  else
	    {
	      ls_heuristic_m1_exhaust (program, options, space,
				       count + 1, prototype);
	      if (bound == 0 ||
		  options->scheme_m1[count] == LS_HEURISTIC_M1_SCHEME_NONE)
               goto clean;
	    }
	hack_continue:
	    while (idx > 0 && (Z_CMP(i[idx], ==, max[idx]) ||
			     Z_CMP(min[idx], >, max[idx])))
	    --idx;
	}
      else
	{
	  ++idx;
	  fm_solver_compute_min (&lb, sol->solution[idx].positive,
				 draw, idx, FM_MINLEXICO_INT);
	  fm_solver_compute_max (&Ub, sol->solution[idx].negative,
				 draw, idx, FM_MINLEXICO_INT);
	  ls_heuristics_narrow_intbounds (lb, Ub, sol, idx, draw);
	  Z_ASSIGN(min[idx], lb->num);
	  Z_ASSIGN(max[idx], Ub->num);
	  Z_ASSIGN(i[idx], min[idx]);
	  Z_DEC(i[idx], i[idx]);
	  while (idx > 0 && (Z_CMP(min[idx], >, max[idx]) ||
			     Z_CMP(i[idx], ==, max[idx])))
	    {
	      ++badpath;
	      --idx;
	    }
	}
    }

  if (options->verbose)
    {
      if (orig != 0 && 1 < 0)
	{
	  fprintf (options->out_file, "\n.... Original cycles: %lld\n", orig);
	  fprintf (options->out_file,
		   ".... Best transformation cycles: %lld\n", best);
	  fprintf (options->out_file,
		   ".... Best transformation Id: %d\n", id);
	}
    }

 clean:
  // Be clean.
  for (ii = 0; ii < sol->size; ++ii)
    {
      Z_CLEAR(i[ii]);
      Z_CLEAR(min[ii]);
      Z_CLEAR(max[ii]);
    }

  free (i);
  free (min);
  free (max);
  fm_rational_free (lb);
  fm_rational_free (Ub);
}




/**
 * M1 heuristic.
 *
 * Principle: iterate on classes coefficients, values must be -1,0,1,
 * schedule is completed with arbitrary values as close as possible to
 * 0.
 *
 */
void
ls_heuristic_m1 (s_ls_space_t* space,
		 CandlProgram* program,
		 s_ls_options_t* options)
{
  int i;
  int j;
  s_fm_vector_t* prototype[space->dimension + 1];
  cycles_t orig = 0;

  // Set the static limit.
  if (options->create_schedfiles && options->compile_line != NULL)
    S_LIMIT = options->rtries;

  if (options->verbose)
    {
      fprintf (options->out_file, "... Using M1 heuristic\n");
      fprintf (options->out_file, "... Number of schedule dimensions: %d\n",
	       space->dimension);
      fprintf (options->out_file, ".... Compile and run the original code\n");
    }
  // Execute the original code.
  orig = ls_explorer_execorig (options);
  if (options->verbose)
    fprintf (options->out_file, ".... Original cycles: %lld\n", orig);

  // Allocate the prototype schedule. Filled with a dummy value to
  // ease debugging.
  for (i = 0; i < space->dimension; ++i)
    {
      prototype[i] = fm_vector_alloc (space->u_polyhedron[0]->size + 1);
      for (j = 1; j <  space->u_polyhedron[0]->size + 1; ++j)
	fm_vector_assign_int_idx (prototype[i], 142, j);
    }
  prototype[space->dimension] = NULL;

  // Initialize the default scheme, if none was provided before.
  if (options->scheme_m1 == NULL)
    ls_heuristic_m1_initialize_scheme (options, space->dimension);


  // The core of the heuristic.
  ls_heuristic_m1_exhaust (program, options, space, 0, prototype);


  fprintf (options->out_file, "\n.... Original cycles: %lld\n", orig);
  fprintf (options->out_file,
	   ".... Best transformation cycles: %lld\n", G_BEST);
  fprintf (options->out_file,
	   ".... Best transformation Id: %d\n", G_BEST_ID);
  if (options->verbose)
    fprintf (options->out_file, "... Heuristic terminated\n");

  // Be clean.
  for (i = 0; i < space->dimension; ++i)
    fm_vector_free (prototype[i]);

}
