/*
 * heuristic-pluto.c: this file is part of the LetSee project.
 *
 * LetSee, the LEgal Transformation SpacE Explorator.
 *
 * Copyright (C) 2008 Louis-Noel Pouchet
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * The complete GNU General Public Licence Notice can be found as the
 * `COPYING' file in the root directory.
 *
 * Author:
 * Louis-Noel Pouchet <Louis-Noel.Pouchet@inria.fr>
 *
 */
#if HAVE_CONFIG_H
# include <letsee/config.h>
#endif

#include <letsee/common.h>
#include <letsee/heuristic-pluto.h>

static s_fm_system_t* G_SOLS;



int
ls_heuristic_pluto_tcb (CandlProgram* program,
			s_fm_solution_t* sol,
			s_ls_options_t* options)
{
  return sol->size;
}


int
ls_heuristic_pluto_tgp (s_fm_vector_t* draw,
			CandlProgram* program,
			s_ls_options_t* options)
{
  if (options->create_schedfiles)
    fm_system_add_line (G_SOLS, fm_vector_dup (draw));

  return LS_ENUMERATE_CONTINUE;
}


void
ls_heuristic_pluto_create_fst (s_fm_vector_t* v,
			       CandlProgram* program,
			       int count,
			       s_ls_options_t* options)
{
  int i, j, k;
  z_type_t tmp; Z_INIT(tmp);
  int nb_comp = 0;
  int nb_stmt;
  z_type_t* comps = XMALLOC(z_type_t, v->size - 1);
  for (i = 0; i < v->size - 1; ++i)
    Z_INIT(comps[i]);

  // Count the number of different components.
  for (i = 1, nb_comp = 0; i < v->size; ++i)
    {
      for (k = 0;
	   k < nb_comp && Z_CMP(v->vector[i].num, !=, comps[k]); ++k)
	;
      if (k == nb_comp)
	Z_ASSIGN(comps[nb_comp++], v->vector[i].num);
    }
  // Order the components.
  for (j = 0; j < nb_comp; ++j)
    for (k = j + 1; k < nb_comp; ++k)
      {
	if (Z_CMP(comps[j], >, comps[k]))
	  {
	    Z_ASSIGN(tmp, comps[j]);
	    Z_ASSIGN(comps[j], comps[k]);
	    Z_ASSIGN(comps[k], tmp);
	  }
      }
  Z_ASSIGN_SI(tmp, 0);

  // Dump the file.
  char str[256];
  char buff[16];
  strcpy(str, options->transfo_dir);
  strcat(str, "/transformation_");
  sprintf(buff, "%d", count);
  strcat(str, buff);
  strcat(str, ".fst");
  FILE* out = fopen (str, "w+");

  // Output the number of components.
  fprintf (out, "%d\n", nb_comp);
  for (i = 0; i < nb_comp; ++i)
    {
      int is_1d = 1;
      // Count the number of statements.
      for (j = 1, nb_stmt = 0; j < v->size; ++j)
	if (Z_CMP(v->vector[j].num, ==, tmp))
	  nb_stmt++;
      // Output the number of statements.
      fprintf (out, "%d\n", nb_stmt);
      // Output the statements of the component.
      for (j = 1, nb_stmt = 0; j < v->size; ++j)
	if (Z_CMP(v->vector[j].num, ==, tmp))
	  {
	    fprintf (out, "%d ", j - 1);
	    if (program->statement[j - 1]->depth != 1)
	      is_1d = 0;
	  }
      fprintf (out, "\n");
      Z_INC(tmp, tmp);
      // Output the tile size. If all statements of the component 
      // are 1d, do not tile.
      if (is_1d)
	fprintf (out, "%d\n", 0);
      else
	fprintf (out, "%d\n", options->tile_factor);
    }

  // Be clean.
  Z_CLEAR(tmp);
  for (i = 0; i < v->size - 1; ++i)
    Z_CLEAR(comps[i]);
  XFREE(comps);
}



int
ls_heuristic_pluto_generate_structures (s_fm_system_t* s,
					CandlProgram* program,
					s_ls_options_t* options)
{
  int i, j, k;
  z_type_t val; Z_INIT(val);
  z_type_t tmp; Z_INIT(tmp);
  s_fm_vector_t* v;
  s_fm_system_t* canon = fm_system_alloc (0, s->nb_cols);
  int nb_comp;
  z_type_t* comps = XMALLOC(z_type_t, s->nb_cols - 1);
  for (i = 0; i < s->nb_cols - 1; ++i)
    Z_INIT(comps[i]);

  for (i = 0; i < s->nb_lines; ++i)
    {
      v = s->lines[i];
      // Rule 1: cst <- cst - min (cst_i)
      Z_ASSIGN(val, v->vector[1].num);
      for (j = 2; j < s->nb_cols; ++j)
	if (Z_CMP(val, >, v->vector[j].num))
	  Z_ASSIGN(val, v->vector[j].num);
      for (j = 1; j < s->nb_cols; ++j)
	Z_SUB(v->vector[j].num, v->vector[j].num, val);

      // Rule 2: Normalization.
      // Count the number of different components.
      for (j = 1, nb_comp = 0; j < s->nb_cols; ++j)
	{
	  for (k = 0;
	       k < nb_comp && Z_CMP(v->vector[j].num, !=, comps[k]); ++k)
	    ;
	  if (k == nb_comp)
	    Z_ASSIGN(comps[nb_comp++], v->vector[j].num);
	}
      // Order the components.
      for (j = 0; j < nb_comp; ++j)
	for (k = j + 1; k < nb_comp; ++k)
	  {
	    if (Z_CMP(comps[j], >, comps[k]))
	      {
		Z_ASSIGN(tmp, comps[j]);
		Z_ASSIGN(comps[j], comps[k]);
		Z_ASSIGN(comps[k], tmp);
	      }
	  }
      // Normalize the vector.
      for (k = 0, Z_ASSIGN_SI(tmp, 0); k < nb_comp; ++k, Z_INC(tmp, tmp))
	for (j = 1; j < s->nb_cols; ++j)
	  if (Z_CMP(v->vector[j].num, ==, comps[k]))
	    Z_ASSIGN(v->vector[j].num, tmp);

      // Recompute the vector key (useful for remove_duplicates).
      fm_vector_compute_key (&(v->key), v);
    }

  // Remove duplicate lines in the system.
  fm_system_remove_duplicate (s);

  // Build PLuTo files for each possible fusion structure.
  for (i = 0; i < s->nb_lines; ++i)
    ls_heuristic_pluto_create_fst (s->lines[i], program, i, options);

  int size = s->nb_lines;

  // Be clean.
  fm_system_free (s);
  Z_CLEAR(val);
  Z_CLEAR(tmp);
  for (i = 0; i < s->nb_cols - 1; ++i)
    Z_CLEAR(comps[i]);
  XFREE(comps);

  return size;
}


void
ls_heuristic_pluto (s_ls_space_t* space,
		    CandlProgram* program,
		    s_ls_options_t* options)
{
  int size = 0;
  int fs;
  options->tile_factor = 256;
  G_SOLS = fm_system_alloc (0, space->u_polyhedron[0]->size + 1);
  if (options->verbose)
    fprintf (options->out_file, "... Using PLuTo heuristic\n");
    fprintf (options->out_file,
	     ".... Exploring legal fusion schedules\n", size);

  // Enumerate all structures, and store them in a temporary system.
  ls_heuristics_enumerate (program, options, space->u_polyhedron[0],
			   &size, ls_heuristic_pluto_tcb,
			   ls_heuristic_pluto_tgp);

  if (options->verbose)
    fprintf (options->out_file,
	     ".... Building legal fusion structures\n", size);

  // Build the unique structures.
  fs = ls_heuristic_pluto_generate_structures (G_SOLS, program, options);

  if (options->verbose)
    fprintf (options->out_file,
	     ".... Generated %d fusion structures from %d schedules\n",
	     fs, size);

  // Be compliant with LeTSeE heuristic output.
  space->size = fs;
}
