/*
 * heuristic-plutocc.c: this file is part of the LetSee project.
 *
 * LetSee, the LEgal Transformation SpacE Explorator.
 *
 * Copyright (C) 2008 Louis-Noel Pouchet
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * The complete GNU General Public Licence Notice can be found as the
 * `COPYING' file in the root directory.
 *
 * Author:
 * Louis-Noel Pouchet <Louis-Noel.Pouchet@inria.fr>
 *
 */
#if HAVE_CONFIG_H
# include <letsee/config.h>
#endif

#include <letsee/common.h>
#include <letsee/heuristic-plutocc.h>

static s_fm_system_t* G_SOLS;



int
ls_heuristic_plutocc_tcb (CandlProgram* program,
			  s_fm_solution_t* sol,
			  s_ls_options_t* options)
{
  return sol->size;
}


int
ls_heuristic_plutocc_tgp (s_fm_vector_t* draw,
			  CandlProgram* program,
			  s_ls_options_t* options)
{
  if (options->create_schedfiles)
      fm_system_add_line (G_SOLS, fm_vector_dup (draw));

  if (options->verbose)
    {
      fprintf (options->out_file, ".");
      fflush (options->out_file);
    }

  return LS_ENUMERATE_CONTINUE;
}


int
ls_heuristic_plutocc_random_tgp (s_fm_vector_t* draw,
				 CandlProgram* program,
				 s_ls_options_t* options)
{
  if (options->create_schedfiles)
    {
      int i;
      for (i = 0; i < G_SOLS->nb_lines; ++i)
	if (fm_vector_equal (draw, G_SOLS->lines[i]))
	  {
	    if (options->verbose)
	      {
		fprintf (options->out_file, "X");
		fflush (options->out_file);
	      }
	    return LS_ENUMERATE_DUPLICATE;
	  }
      fm_system_add_line (G_SOLS, fm_vector_dup (draw));
    }

  if (options->verbose)
    {
      fprintf (options->out_file, ".");
      fflush (options->out_file);
    }

  return LS_ENUMERATE_CONTINUE;
}


void
ls_heuristic_plutocc_create_fst (s_fm_vector_t* v,
				 CandlProgram* program,
				 int count,
				 s_ls_options_t* options,
				 double* min_value,
				 unsigned int* best_id)
{
  int i, j, k;
  z_type_t tmp; Z_INIT(tmp);
  int nb_comp = 0;
  int nb_stmt;
  z_type_t* comps = XMALLOC(z_type_t, v->size - 1);
  for (i = 0; i < v->size - 1; ++i)
    Z_INIT(comps[i]);

  // Count the number of different components.
  for (i = 1, nb_comp = 0; i < v->size; ++i)
    {
      for (k = 0;
	   k < nb_comp && Z_CMP(v->vector[i].num, !=, comps[k]); ++k)
	;
      if (k == nb_comp)
	Z_ASSIGN(comps[nb_comp++], v->vector[i].num);
    }
  // Order the components.
  for (j = 0; j < nb_comp; ++j)
    for (k = j + 1; k < nb_comp; ++k)
      {
	if (Z_CMP(comps[j], >, comps[k]))
	  {
	    Z_ASSIGN(tmp, comps[j]);
	    Z_ASSIGN(comps[j], comps[k]);
	    Z_ASSIGN(comps[k], tmp);
	  }
      }
  Z_ASSIGN_SI(tmp, 0);

  // Dump the file.
  char str[512];
  char buff[16];
  strcpy(str, options->transfo_dir);
  strcat(str, "/transformation_");
  sprintf(buff, "%09d", count);
  strcat(str, buff);
  strcat(str, ".fst");
  FILE* out = fopen (str, "w+");

  // Output the number of components.
  fprintf (out, "%d\n", nb_comp);
  for (i = 0; i < nb_comp; ++i)
    {
      int is_1d = 1;
      // Count the number of statements.
      for (j = 1, nb_stmt = 0; j < v->size; ++j)
	if (Z_CMP(v->vector[j].num, ==, tmp))
	  nb_stmt++;
      // Output the number of statements.
      fprintf (out, "%d\n", nb_stmt);
      // Output the statements of the component.
      for (j = 1, nb_stmt = 0; j < v->size; ++j)
	if (Z_CMP(v->vector[j].num, ==, tmp))
	  {
	    fprintf (out, "%d ", j - 1);
	    if (program->statement[j - 1]->depth != 1)
	      is_1d = 0;
	  }
      fprintf (out, "\n");
      Z_INC(tmp, tmp);
      // Output the tile size. If all statements of the component
      // are 1d, do not tile.
      if (is_1d)
	fprintf (out, "%d\n", 0);
      else
	fprintf (out, "%d\n", options->tile_factor);
    }
  // Be clean.
  Z_CLEAR(tmp);
  for (i = 0; i < v->size - 1; ++i)
    Z_CLEAR(comps[i]);
  XFREE(comps);
  fclose(out);
#ifdef HAVE_LIBPOCC_UTILS
  if (options->pocc_utils != NULL)
    {
      if (options->verbose)
	{
	  fprintf (options->out_file, "[LetSee] Evaluating candidate %d\n",
		   count);
	  fprintf (options->out_file, "[LetSee] Fusion structure is: ");
	  fm_vector_print (options->out_file, v);
	  fprintf (options->out_file, "\n");
	}
      // Copy transformation to .fst in base directory.
      char* args[5];
      args[4] = NULL;
      args[0] = "cp";
      args[1] = "-f";
      args[2] = str;
      args[3] = ".fst";
      ls_explorer_execprog (args);
      // Compile and run.
      s_pocc_utils_options_t* puoptions = options->pocc_utils;
      if (puoptions->output_file_name)
	XFREE(puoptions->output_file_name);
      puoptions->output_file_name = XMALLOC(char, 8192);
      strcpy (puoptions->output_file_name, puoptions->input_file_name);
      int pos = strlen (puoptions->output_file_name);
      while (puoptions->output_file_name[--pos] != '.')
	;
      puoptions->output_file_name[pos] = '\0';
      sprintf (puoptions->output_file_name, "%s.pocc-precut.%d.c",
	       puoptions->output_file_name, count);
      puoptions->pocc_codegen (puoptions);
      if (puoptions->program_exec_result)
	{
	  // Print the timing.
	  fprintf (options->out_file, "%d %s", count,
		   puoptions->program_exec_result);
	  // Print the timing in the iterative.dat file.
	  fprintf (puoptions->data_file, "%d %s", count,
		   puoptions->program_exec_result);
	  // Store the best one.
	  double value = strtod(puoptions->program_exec_result, NULL);
	  if ((count == 0 || value < *min_value) && value > 0.0)
	    {
	      *min_value = value;
	      *best_id = count;
	    }
	}
    }
#endif
}



int
ls_heuristic_plutocc_generate_structures (s_fm_system_t* s,
					  CandlProgram* program,
					  s_ls_options_t* options)
{
  int i;
  s_fm_vector_t* v;

  s_fm_system_t* syst = fm_system_alloc (s->nb_lines,
					 program->nb_statements + 1);
  // Convert each point to an ordering.
  for (i = 0; i < s->nb_lines; ++i)
    {
      v = ls_oset_convert_ordering (s->lines[i], program->nb_statements);
      if (v)
	{
	  fm_vector_assign (syst->lines[i], v);
	  fm_vector_free (v);
	}
    }

  // Build PLuTo files for each possible fusion structure.
  double min_value = 0;
  unsigned int best_id = 0;
  for (i = 0; i < syst->nb_lines; ++i)
    ls_heuristic_plutocc_create_fst (syst->lines[i], program, i, options,
				     &min_value, &best_id);
  if (min_value != 0)
    {
      fprintf (options->out_file,
	       "[LetSee] Best found version: %d\n", best_id);
#ifdef HAVE_LIBPOCC_UTILS
      if (options->pocc_utils != NULL)
	{
	  s_pocc_utils_options_t* puoptions = options->pocc_utils;
	  char final[8192];
	  char tmp[8192];
	  strcpy (tmp, puoptions->input_file_name);
	  int pos = strlen (tmp);
	  while (tmp[--pos] != '.')
	    ;
	  tmp[pos] = '\0';
	  strcpy (final, tmp);
	  strcat (final, ".pocc.c");
	  sprintf (tmp, "%s.pocc-precut.%d.c", tmp, best_id);
	  fprintf (options->out_file,
		   "[LetSee] Best version is %s\n", tmp);
	  char* args[5];
	  args[4] = NULL;
	  args[0] = "cp";
	  args[1] = "-f";
	  args[2] = tmp;
	  args[3] = final;
	  ls_explorer_execprog (args);
	  fprintf (options->out_file, "[PoCC] Output file is %s\n", final);
	}
    }
#endif
  int size = syst->nb_lines;

  // Be clean.
  fm_system_free (s);
  fm_system_free (syst);

  return size;
}


void
ls_heuristic_plutocc (s_ls_space_t* space,
		      CandlProgram* program,
		      s_ls_options_t* options)
{
  int size = 0;
  int fs;
  options->tile_factor = 256;
  G_SOLS = fm_system_alloc (0, space->u_polyhedron[0]->size + 1);
  if (options->verbose)
    {
      fprintf (options->out_file,
	       "[LetSee] Using PLuToCC exhaustive heuristic\n");
      fprintf (options->out_file,
	       "[LetSee] Exploring legal fusion schedules\n", size);
    }

  // Enumerate all structures, and store them in a temporary system.
  if (options->heuristic == LS_HEURISTIC_RANDOM)
    {
      if (options->verbose)
	fprintf (options->out_file,
		 "[LetSee] Generating %d random distinct precuts "
		 "(.: ok, X: duplicate)\n", options->rtries);
      ls_heuristics_random_enumerate (program, options, space->u_polyhedron[0],
				      &size, ls_heuristic_plutocc_tcb,
				      ls_heuristic_plutocc_random_tgp,
				      options->rtries);
      if (options->verbose)
	fprintf (options->out_file, "\n");
    }
  else
    {
      ls_heuristics_enumerate (program, options, space->u_polyhedron[0],
			       &size, ls_heuristic_plutocc_tcb,
			       ls_heuristic_plutocc_tgp);
/*       if (options->verbose) */
/* 	fprintf (options->out_file, "\n"); */
    }
  if (options->verbose)
    fprintf (options->out_file,
	     "[LetSee] %d precuts were generated\n", size);

  // Build the unique structures.
  fs = ls_heuristic_plutocc_generate_structures (G_SOLS, program, options);

  if (options->verbose)
    fprintf (options->out_file,
	     "[LetSee] Generated %d fusion structures from %d schedules\n",
	     fs, size);

  // Be compliant with LeTSeE heuristic output.
  space->size = fs;
}
