/*
 * heuristics.c: this file is part of the LetSee project.
 *
 * LetSee, the LEgal Transformation SpacE Explorator.
 *
 * Copyright (C) 2006,2007,2008 Louis-Noel Pouchet
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * The complete GNU General Public Licence Notice can be found as the
 * `COPYING' file in the root directory.
 *
 * Author:
 * Louis-Noel Pouchet <Louis-Noel.Pouchet@inria.fr>
 *
 */
#if HAVE_CONFIG_H
# include <letsee/config.h>
#endif


#include <letsee/heuristics.h>


void
ls_explorer_transfo_print (s_fm_vector_t** transfo)
{
  int i;

  for (i = 0; transfo[i]; ++i)
    {
      fm_vector_print (stdout, transfo[i]);
      printf ("\n");
    }
}



void
ls_explorer_transfo_copy (s_fm_vector_t*** dest, s_fm_vector_t** transfo)
{
  int i;

  for (i = 0; transfo[i]; ++i)
    ;
  *dest = XMALLOC(s_fm_vector_t*, i + 1);
  for (i = 0; transfo[i]; ++i)
    (*dest)[i] = fm_vector_dup (transfo[i]);
  (*dest)[i] = NULL;
}



int
ls_explorer_transfo_exists_multi (s_ls_explorer_t* data,
				  s_fm_vector_t** draw)
{
  int i, j, bool;
  for (i = 0; i < data->count; ++i)
    {
      bool = 1;
      for (j = 0; j < data->dimension; ++j)
	bool = bool && fm_vector_equal (draw[j], data->u_draws[i][j]);
      if (bool)
	return 1;
    }

  return 0;
}



void
ls_explorer_heuristic_filter_multi (s_ls_options_t* options,
				    s_ls_explorer_t* data)
{
  int i, j;
  int tmpi;
  s_fm_vector_t** tmpv;
  cycles_t best, tmpc;

  if (data->count == 0)
    return;

  // Sort the transformations.
  for (i = 0; i < data->count; ++i)
    for (j = i; j < data->count; ++j)
      if (data->cycles[j] < data->cycles[i])
	{
	  tmpc = data->cycles[i];
	  data->cycles[i] = data->cycles[j];
	  data->cycles[j] = tmpc;
	  tmpv = data->u_draws[i];
	  data->u_draws[i] = data->u_draws[j];
	  data->u_draws[j] = tmpv;
	  tmpi = data->ids[i];
	  data->ids[i] = data->ids[j];
	  data->ids[j] = tmpi;
	}

  // Filter the transformations.
  tmpc = data->cycles[0] * (1 + (float)(options->thresold) / 100);
  for (i = 0; i < data->count; ++i)
    if (data->cycles[i] > tmpc)
      break;

  tmpi = i;
  for (; i < data->count; ++i)
    for (j = 0; j < data->dimension; ++j)
      fm_vector_free (data->u_draws[i][j]);

  if (options->verbose)
    fprintf (options->out_file, "[LetSee] Filtered schedules: %d\n",
	     data->count - tmpi);

  data->count = tmpi;
}


/**
 * Modify bounds if necessary to ensure the existence of an integer
 * solution.
 *
 */
int
ls_heuristics_narrow_intbounds (s_fm_rational_t* lb,
				s_fm_rational_t* ub,
				s_fm_solution_t* polyhedron,
				int idx,
				s_fm_vector_t* p)
{
  if (idx == 0)
    {
/*       if (Z_CMP(lb->num, ==, ub->num)) */
/* 	{ */
	  return 0;
/* 	} */
/*       else */
/* 	fprintf (stderr, "val base: [%d,%d]\n", lb->num, ub->num); */
    }
/*   if (idx == 0 && Z_CMP(lb->num, ==, ub->num)) */
/*     return 0; */

  int ret = 0;
  int i;
  s_fm_vector_t* tmp;
  s_fm_vector_t* tmp2;
  s_fm_solution_t* sol = fm_solution_dup (polyhedron);
  // Build the input solution system.
  // Set the vector content in the solution.
  for (i = 0; i < idx; ++i)
    {
      tmp = fm_vector_alloc (i + 3);
      fm_vector_set_ineq (tmp);
      fm_vector_assign_int_idx (tmp, -1, i + 1);
      fm_vector_assign_int_idx (tmp, p->vector[i + 1].num, tmp->size - 1);
      fm_solution_add_line_at (sol, tmp, i + 1);
      tmp = fm_vector_alloc (i + 3);
      fm_vector_set_ineq (tmp);
      fm_vector_assign_int_idx (tmp, 1, i + 1);
      fm_vector_assign_int_idx (tmp, - p->vector[i + 1].num, tmp->size - 1);
      fm_solution_add_line_at (sol, tmp, i + 1);
    }
  // Prepare the check for the current variable.
  tmp = fm_vector_alloc (i + 3);
  fm_vector_set_ineq (tmp);
  fm_vector_assign_int_idx (tmp, -1, i + 1);
  fm_solution_add_line_at (sol, tmp, i + 1);
  tmp2 = fm_vector_alloc (i + 3);
  fm_vector_set_ineq (tmp2);
  fm_vector_assign_int_idx (tmp2, 1, i + 1);
  fm_solution_add_line_at (sol, tmp2, i + 1);

  // Check extremal bound values, and compute the integer hull of them.
  int has_intsol = 0;
  int k;
  z_type_t ii; Z_INIT(ii);
  s_fm_rational_t** minlex;

  for (Z_ASSIGN(ii, lb->num), k = 0; k < 2;
       Z_ASSIGN(ii, ub->num), ++k, has_intsol = 0)
    while (! has_intsol && (Z_CMP(lb->num, <=, ub->num)))
      {
	fm_vector_assign_int_idx (tmp, ii, tmp->size - 1);
	Z_OPP(ii, ii);
	fm_vector_assign_int_idx (tmp2, ii, tmp2->size - 1);
	Z_OPP(ii, ii);

	minlex = fm_solver_minlexico (sol, -42, FM_MINLEXICO_INT);
	has_intsol = (minlex != NULL);
	for (i = 0; minlex != NULL && minlex[i] != NULL; ++i)
	  fm_rational_free (minlex[i]);
	XFREE(minlex);

/* 	s_fm_system_t* st = fm_solution_to_system (sol); */
/* 	int pr = fm_piptools_check_int (st); */
/* 	fm_system_free (st); */

/* 	/\* 	if (pr != has_intsol) *\/ */
/* 	if (! has_intsol); */
/* 	  { */
/* 	    if (idx == 0) */
/* 	      { */
/* 		fprintf (stderr, "%d %d\n", ub->num, lb->num); */
/* 	      } */
/* 	    assert (! "Pb !!!!!"); */
/* 	  } */

	if (! has_intsol)
	  {
	    if (! k)
	      {
		Z_INC(lb->num, lb->num);
		Z_INC(ii, ii);
	      }
	    else
	      {
		Z_DEC(ub->num, ub->num);
		Z_DEC(ii, ii);
	      }
	    ret = 1;
	    if (idx == 0)
	      fprintf (stderr, "val modif: [%d,%d]\n", lb->num, ub->num);
	  }
      }
  fm_solution_free (sol);

  return ret;
}


/**
 * Output the coefficient type, in basis 0 (the first coefficient is
 * the coefficient #0).
 *
 *
 */
e_coeftype_t
ls_heuristics_coeftype (s_ls_space_t* space, int idx)
{
  if (space == NULL || space->program == NULL ||
      space->program->nb_statements == 0)
    {
      assert (! "No / bad input program");
      exit (1);
    }
  int i;
  int val = 0;
  int nb_params = space->program->statement[0]->domain->NbColumns -
    space->program->statement[0]->depth - 2;
  for (i = 0; i < space->program->nb_statements && val < idx; ++i)
    val += space->program->statement[i]->depth;
  if (i != space->program->nb_statements)
    return LS_CT_ITER;
  for (i = 0; i < space->program->nb_statements && val < idx; ++i)
    val += nb_params;
  if (i != space->program->nb_statements)
    return LS_CT_PARAM;
  for (i = 0; i < space->program->nb_statements && val < idx; ++i)
    ++val;
  if (i != space->program->nb_statements)
    return LS_CT_CONST;
  assert (! "Bad coefficient index");
  exit (1);
}


/**
 * Compute the degree of correlation of variables, for a given
 * dimension.  The higher the degree, the less 'constrained' the
 * dimension (beware it is the converse of the usual meaning of
 * 'correlation').
 *
 * degree \in [0,1]
 *
 */
double
ls_heuristics_correlationdegree (s_ls_space_t* space,
				 int dim)
{
  s_fm_system_t* sys = fm_solution_to_system (space->u_polyhedron[dim]);
  int** classes = fm_system_getconnected (sys);
  int nb_classes;
  int i;
  double res;
  // Count the number of classes.
  for (nb_classes = 0; classes[nb_classes] != NULL; ++nb_classes)
    ;
  res = (double) nb_classes / ((double) sys->nb_cols - 2.);

  // Be clean.
  fm_system_free (sys);
  for (i = 0; classes[i] != NULL; ++i)
    XFREE(classes[i]);
  XFREE(classes);

  return res;
}



/**
 * Fix and/or complete a schedule, with minimal corrections.
 * Return a 'correction distance'
 *
 */
int
ls_heuristics_fixcomplete (s_ls_options_t* options,
			   s_ls_space_t* space,
			   s_fm_vector_t** schedule,
			   int dim)
{
  s_fm_solution_t* polyhedron = space->u_polyhedron[dim];
  s_fm_vector_t* s = schedule[dim];
  int i;
  s_fm_rational_t* lb = NULL;
  s_fm_rational_t* ub = NULL;
  int distance = 0;

  // Iterate on all coefficients.
  for (i = 0; i < polyhedron->size; ++i)
    {
      // Compute the lower bound for x_i, given x_{1..i-1}
      fm_solver_compute_min (&lb,
			     polyhedron->solution[i].positive,
			     s, i, FM_MINLEXICO_INT);
      // Compute the upper bound for x_i, given x_{1..i-1}
      fm_solver_compute_max (&ub,
			     polyhedron->solution[i].negative,
			     s, i, FM_MINLEXICO_INT);
      // Ensure there's an integer point existing in the space for the
      // two possible bounds (and hence for each value in between).
      ls_heuristics_narrow_intbounds (lb, ub, polyhedron, i, s);

      // Fix the vector value, if needed.
      if (Z_CMP(s->vector[i + 1].num, <, lb->num))
	{
	  fm_vector_assign_int_idx (s, lb->num, i + 1);
	  ++distance;
	}
      else if (Z_CMP(s->vector[i + 1].num, >, ub->num))
	{
	  fm_vector_assign_int_idx (s, ub->num, i + 1);
	  ++distance;
	}
    }
  return distance;
}


/**
 * Exhaustive polytope enumerator heuristic for a single polytope.
 *
 *
 */
void
ls_heuristics_enumerate (CandlProgram* program,
			 s_ls_options_t* options,
			 s_fm_solution_t* sol,
			 int* size,
			 t_compute_bound tcb,
			 t_gen_point tgp)
{
  s_fm_rational_t* lb = NULL;
  s_fm_rational_t* Ub = NULL;
  z_type_t* i = XMALLOC(z_type_t, sol->size);
  z_type_t* min = XMALLOC(z_type_t, sol->size);
  z_type_t* max = XMALLOC(z_type_t, sol->size);
  int ii;
  unsigned idx = 0;
  s_fm_vector_t* draw = fm_vector_alloc (sol->size + 1);
/*   fm_solution_print (stdout, sol); */
  unsigned long long int badpath = 0;

  int bound_limit;

  // Compute the index of the first coefficient to complete. The
  // traversal goes from coef #0 to bound_limit. If bound limit is
  // equal to the number of variables in the system, then it counts
  // all points in the polytope.
  bound_limit = tcb (program, sol, options);
  for (ii = 0; ii < sol->size; ++ii)
    {
      Z_INIT(i[ii]);
      Z_INIT(min[ii]);
      Z_INIT(max[ii]);
    }

  for (ii = 0; ii < sol->size; ++ii)
    {
      Z_INIT(i[ii]);
      Z_INIT(min[ii]);
      Z_INIT(max[ii]);
    }

  fm_solver_compute_min (&lb, sol->solution[idx].positive,
			 draw, idx, FM_MINLEXICO_INT);
  fm_solver_compute_max (&Ub, sol->solution[idx].negative,
			 draw, idx, FM_MINLEXICO_INT);
  min[idx] = lb->num;
  max[idx] = Ub->num;

  if (options->verbose)
    fprintf (options->out_file, "[LetSee] Scan solution polyhedron\n");

  // Loop on each point.
  for (Z_ASSIGN(i[idx], min[idx]); Z_CMP(i[idx], <=, max[idx]);
       Z_INC(i[idx], i[idx]))
    {
/*       if (idx == 1) */
/* 	fm_vector_init (draw, draw->size); */
/*       printf ("picked value at %d: %d\n", idx, i[idx]); */
      fm_vector_assign_int_idx (draw, i[idx], idx + 1);
/*       printf ("draw (at %d): ", idx); fm_vector_print (stdout, draw); */
/*       printf ("\n"); */
      //0 1 0 0 0 -1

/*       0 0 0 0 0 0 */
/*       0 0 0 0 0 1 */
/*       0 0 0 1 0 0 */
/*       0 0 0 0 1 1 */
      if (idx + 1 == bound_limit)
	{
	  if (tgp (draw, program, options) == LS_ENUMERATE_BREAK)
	    break;
	  (*size)++;
	  while (idx > 0 && (Z_CMP(i[idx], ==, max[idx]) ||
			     Z_CMP(min[idx], >, max[idx])))
	    --idx;
	}
      else
	{
	  ++idx;
	  fm_solver_compute_min (&lb, sol->solution[idx].positive,
				 draw, idx, FM_MINLEXICO_INT);
	  fm_solver_compute_max (&Ub, sol->solution[idx].negative,
				 draw, idx, FM_MINLEXICO_INT);
	  Z_ASSIGN(min[idx], lb->num);
	  Z_ASSIGN(max[idx], Ub->num);
	  Z_ASSIGN(i[idx], min[idx]);
	  Z_DEC(i[idx], i[idx]);
	  int has_hole = 0;
	  if (Z_CMP(min[idx], >, max[idx]))
	    {
/* 	      printf ("has hole at %d (%d - %d)\n", idx, min[idx], max[idx]); */
/* 	      exit (42); */
	    }
	  int memidx = idx;
	  while (idx > 0 && (Z_CMP(min[idx], >, max[idx]) ||
			     Z_CMP(i[idx], ==, max[idx])))
	    {
	      // Integer hole.
	      if (Z_CMP(min[idx], >, max[idx]))
		{
		  has_hole = 1;
/* 		  if (idx != memidx) */
/* 		    printf ("AND has hole at %d (%d - %d)\n", idx, min[idx], max[idx]); */
		}
	      --idx;
	    }
	  if (has_hole)
	    {
	      ++badpath;
/* 	      fm_vector_print (stdout, draw); printf ("\n"); */
	    }
	}
    }

  if (options->verbose)
    fprintf (options->out_file, "\n[LetSee] Scan completed (%d holes)\n", 
	     badpath);

  // Be clean.
  for (ii = 0; ii < sol->size; ++ii)
    {
      Z_CLEAR(i[ii]);
      Z_CLEAR(min[ii]);
      Z_CLEAR(max[ii]);
    }

  free (i);
  free (min);
  free (max);
  fm_rational_free (lb);
  fm_rational_free (Ub);
  fm_vector_free (draw);
}


/**
 * Random polytope enumerator heuristic for a single polytope.
 *
 *
 */
void
ls_heuristics_random_enumerate (CandlProgram* program,
				s_ls_options_t* options,
				s_fm_solution_t* sol,
				int* size,
				t_compute_bound tcb,
				t_gen_point tgp,
				int random_tries)
{
  s_fm_rational_t* lb = NULL;
  s_fm_rational_t* Ub = NULL;
  z_type_t* i = XMALLOC(z_type_t, sol->size);
  z_type_t* min = XMALLOC(z_type_t, sol->size);
  z_type_t* max = XMALLOC(z_type_t, sol->size);
  int ii;
  unsigned idx = 0;
  s_fm_vector_t* draw = fm_vector_alloc (sol->size + 1);

  unsigned long long int badpath = 0;

  int bound_limit;

  // Compute the index of the first coefficient to complete. The
  // traversal goes from coef #0 to bound_limit. If bound limit is
  // equal to the number of variables in the system, then it counts
  // all points in the polytope.
  bound_limit = tcb (program, sol, options);
  for (ii = 0; ii < sol->size; ++ii)
    {
      Z_INIT(i[ii]);
      Z_INIT(min[ii]);
      Z_INIT(max[ii]);
    }

  for (ii = 0; ii < sol->size; ++ii)
    {
      Z_INIT(i[ii]);
      Z_INIT(min[ii]);
      Z_INIT(max[ii]);
    }

  fm_solver_compute_min (&lb, sol->solution[idx].positive,
			 draw, idx, FM_MINLEXICO_INT);
  fm_solver_compute_max (&Ub, sol->solution[idx].negative,
			 draw, idx, FM_MINLEXICO_INT);
  min[idx] = lb->num;
  max[idx] = Ub->num;

  if (options->verbose)
    fprintf (options->out_file,
	     "[LetSee] Random scan of the polyhedron\n");


  // Initialize the random number generator.
  srand (time (NULL));
  z_type_t value; Z_INIT(value);
  int ret, count;
  int stop = 0;

  // Loop on the number of tries.
  for (count = 0; count < random_tries && ! stop; ++count)
    {
      while (! stop)
	{
	  if (max[idx] - min[idx] == 0)
	    value = min[idx];
	  else
	    value = rand() % (max[idx] - min[idx] + 1) + min[idx];
	  fm_vector_assign_int_idx (draw, value, idx + 1);
	  if (idx + 1 == bound_limit)
	    {
	      ret = tgp (draw, program, options);
	      if (ret == LS_ENUMERATE_BREAK)
		stop = 1;
	      if (ret == LS_ENUMERATE_CONTINUE)
		(*size)++;
	      else
		{
		  if (random_tries <= 100001)
		    random_tries++;
		}
	      idx = 0;
	      break;
	    }
	  else
	    {
	      ++idx;
	      if (idx >= sol->size)
		exit (45);
	      fm_solver_compute_min (&lb, sol->solution[idx].positive,
				     draw, idx, FM_MINLEXICO_INT);
	      fm_solver_compute_max (&Ub, sol->solution[idx].negative,
				     draw, idx, FM_MINLEXICO_INT);
	      Z_ASSIGN(min[idx], lb->num);
	      Z_ASSIGN(max[idx], Ub->num);
	      if (Z_CMP(min[idx], >, max[idx]))
		{
		  //printf ("badpath: %d\n", badpath);
		  ++badpath;
		  idx = 0;
		}
	    }
	}
    }

  // Be clean.
  for (ii = 0; ii < sol->size; ++ii)
    {
      Z_CLEAR(i[ii]);
      Z_CLEAR(min[ii]);
      Z_CLEAR(max[ii]);
    }

  free (i);
  free (min);
  free (max);
  fm_rational_free (lb);
  fm_rational_free (Ub);
  fm_vector_free (draw);
  Z_CLEAR(value);
}
