/*
 * farkas.h: this file is part of the LetSee project.
 *
 * LetSee, the LEgal Transformation SpacE Explorator.
 *
 * Copyright (C) 2007,2008 Louis-Noel Pouchet
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * The complete GNU General Public Licence Notice can be found as the
 * `COPYING' file in the root directory.
 *
 * Author:
 * Louis-Noel Pouchet <Louis-Noel.Pouchet@inria.fr>
 *
 */
#ifndef LETSEE_FARKAS_H
# define LETSEE_FARKAS_H

# include <letsee/common.h>
# include <letsee/options.h>
# include <letsee/space.h>
# include <letsee/transformations.h>
# include <letsee/graph.h>
# include <letsee/dependence.h>
# include <fm/solution.h>
# include <fm/system.h>
# include <fm/solver.h>
# include <fm/compsol.h>
# include <fm/list.h>
# include <fm/piptools.h>
# include <candl/candl.h>


#define LS_FARKAS_GEZERO 0
#define LS_FARKAS_GEONE  1
#define LS_FARKAS_EQZERO 2
#define LS_FARKAS_LEZERO 3
#define LS_FARKAS_TH	 4
#define LS_FARKAS_D1_STRONG 1
#define LS_FARKAS_D2_STRONG 2


BEGIN_C_DECLS

/**
 * \brief Build and project the system of constraints for a given
 * dependence.
 *
 * Outputs a polytope with affine constraints for \Delta_dep\Theta \ge
 * 'type'
 *
 */
extern
s_fm_solution_t*
ls_farkas_project_multipliers (CandlProgram* program,
			       s_ls_options_t* options,
			       CandlDependence* dep,
			       int type);

/**
 * \brief Initialize the solution polytope with predefined bounds
 * provided in `options' parameter.
 *
 * The solution polytope is formatted w.r.t. the [ i1 i2 p1 p2 c1 c2 ]
 * organization.
 *
 * \input: a program (to extract the number of iterators for each statement)
 *
 * \output: a polytope (bounding box of the bounds for each
 * transformation coefficients.)
 *
 */
extern
s_fm_solution_t*
ls_farkas_initialize_solution (CandlProgram* program,
			       s_ls_options_t* options);

/**
 *
 * \brief Compute the set of affine constraints compatible with FCO property.
 *
 * Outputs a polytope where, for all dependences in 'graph',
 * \Delta\Theta \ge 0.
 *
 */
extern
s_fm_solution_t*
ls_farkas_build_fco_constraints (s_graph_t* graph,
				 CandlProgram* program,
				 s_ls_options_t* options);

/**
 *
 * \brief Compute the set of affine constraints with Tiling Hyperplane
 * property.
 *
 * Outputs a polytope where, for all dependences in 'graph',
 * \Delta\Theta \ge 0.
 *
 */
extern
s_fm_solution_t*
ls_farkas_build_th_constraints (s_graph_t* graph,
				CandlProgram* program,
				s_ls_options_t* options);

/**
 * \brief Compute the system to solve, for a given dependence and a
 * pair of statements. Outputs the number of schedule coefficients in
 * the system (int* dimension).
 *
 * The function computes the set of Farkas multipliers for the dependence
 * polyhedron (previously computed with Candl), and then equate them
 * with the schedule coefficients of the statement(s). There is
 * exactly one equality per column in the dependence polyhedron. The
 * set of inequality corresponds to the positivity constraints on the
 * Farkas multipliers.
 *
 * The system is of the form [ i1 i2 p1 p2 c1 c2 lambda ].
 *
 * \input: a program, letsee options, a dependence (candl format), a type.
 *   The 'type' argument sets the relation between schedule
 *   coefficients in the constructed system. Possible values for type:
 *	LS_FARKAS_EQZERO: Theta_S - Theta_T = 0
 *	LS_FARKAS_GEZERO: Theta_S - Theta_T >= 0
 *	LS_FARKAS_GEONE: Theta_S - Theta_T >= 1
 *	LS_FARKAS_TH: Theta_S - Theta_T >= 0, i >= 0, \sum i > 0, p = 0
 *
 * \output: A matrix of constraints.
 *
 * @fixme The coefficient constraints are not mandatory, but seem to
 * help the solver to be faster.
 *
 */
extern
s_fm_system_t*
ls_farkas_build_system (CandlProgram* program,
			s_ls_options_t* options,
			CandlDependence* tmp,
			int* dimension,
			int type);

/**
 * \brief Intersect a (compacted) polytope of legal schedule
 * coefficients 'cs' computed for dependence 'dep' with the solution
 * polytope for the whole program 's'. Try to preserve FM-property as
 * much as possible.
 *
 * @fixme: This code is long and ugly, but mandatory to keep the [i1
 * i2 p1 p2 c1 c2] organization.
 *
 */
extern
void
ls_farkas_intersect_cs (CandlProgram* program,
			s_ls_options_t* options,
			CandlDependence* dep,
			s_fm_compsol_t* cs,
			s_fm_solution_t* s);

/**
 * \brief Intersect a polytope of legal schedule coefficients 'P'
 * computed for dependence 'dep' with the solution polytope for the
 * whole program 's'. Try to preserve FM-property as much as possible.
 *
 * @fixme: This code is long and ugly, but mandatory to keep the [i1
 * i2 p1 p2 c1 c2] organization.
 *
 * @fixme: This should be merged with the previous version somehow.
 *
 */
extern
void
ls_farkas_intersect (CandlProgram* program,
		     s_ls_options_t* options,
		     CandlDependence* dep,
		     s_fm_solution_t* P,
		     s_fm_solution_t* s);

/**
 * \brief Compute the dependence conflict graph.
 *
 * Compute a graph where:
 *  - each node is a dependence
 *  - each edge is a conflict relation between two dependences (they cannot
 *    be strongly satisfied at the same time dimension).
 *
 * @input: a program, a list of dependences, options
 *
 * @output: a graph of dependences.
 *
 */
extern
s_graph_t*
ls_farkas_build_conflicts (CandlProgram* program,
			   CandlDependence* dep,
			   s_ls_options_t* options);

END_C_DECLS


#endif // LETSEE_FARKAS_H
