/*
 * schedspace.c: this file is part of the LetSee project.
 *
 * LetSee, the LEgal Transformation SpacE Explorator.
 *
 * Copyright (C) 2007,2008 Louis-Noel Pouchet
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * The complete GNU General Public Licence Notice can be found as the
 * `COPYING' file in the root directory.
 *
 * Author:
 * Louis-Noel Pouchet <Louis-Noel.Pouchet@inria.fr>
 */
#if HAVE_CONFIG_H
# include <letsee/config.h>
#endif

#include <letsee/schedspace.h>
#include <letsee/lspace.h>

#define LSPACE_REDREC_DESCENDANT 1
#define LSPACE_REDREC_IRIGOIN    2

#define LS_BACKTRACK_NONE	1
#define LS_BACKTRACK_USELESS	2
#define LS_BACKTRACK_USEFULL	3


#undef echo
#define echo(S) { printf(#S); printf("\n"); }


/*
 * @FIXME: This unit is poorly documented. Will be improved.
 */




/////////////////////////////////////////////////////////////////////////////
////////////////////////////// EXPERIMENTAL!! ///////////////////////////////
/////////////////////////////////////////////////////////////////////////////

// This block is dedicated to ordering dependences for traversal
// w.r.t. rough traffic rank estimation.

static
int
ls_rank (CandlMatrix* m1, int ref)
{
  int nb_dim = 1;
  while (nb_dim + ref < m1->NbRows && m1->p[ref + nb_dim][0] == 0)
    ++nb_dim;
  s_fm_system_t* s = fm_system_alloc (nb_dim, m1->NbColumns + 1);
  int i, j;
  for (i = 0; i < nb_dim; ++i)
    for (j = 1; j < m1->NbColumns; ++j)
      fm_vector_assign_int_idx (s->lines[i], m1->p[ref + i][j], j);

  int ret;
  fm_solver_gauss (s);
  for (i = 0, ret = 0; i < s->nb_lines; ++i)
    if (! fm_vector_is_null (s->lines[i]))
      ++ret;

  fm_system_free (s);

  return ret;
}


static
void
ls_compute_ranks (int* stmts, CandlProgram* p)
{
  int i, j, k;
  CandlStatement* s;

  for (i = 0; i < p->nb_statements; ++i)
    {
      s = p->statement[i];
      // Count the number of arrays for the statement.
      int arr[s->read->NbRows + s->written->NbRows];
      int count = 0;
      for (j = 0; j < s->read->NbRows; ++j)
	{
	  if (s->read->p[j][0] != 0)
	    {
	      for (k = 0; k < count && arr[k] != s->read->p[j][0]; ++k)
		;
	      if (k == count)
		arr[count++] = s->read->p[j][0];
	    }
	}
      for (j = 0; j < s->written->NbRows; ++j)
	{
	  if (s->written->p[j][0] != 0)
	    {
	      for (k = 0; k < count && arr[k] != s->written->p[j][0]; ++k)
		;
	      if (k == count)
		arr[count++] = s->written->p[j][0];
	    }
	}

      // Create the matrix.
      int sz = s->read->NbColumns < s->written->NbColumns ?
	s->written->NbColumns : s->read->NbColumns;
      CandlMatrix* m =
	candl_matrix_malloc (s->read->NbRows + s->written->NbRows,
			     count + sz + 1);
      // Fill the matrix.
      int id;
      int line = 0;
      for (j = 0; j < s->read->NbRows; ++j)
	{
	  // Compute the index of the array.
	  if (s->read->p[j][0] != 0)
	    for (id = 0; id < count && arr[id] != s->read->p[j][0]; ++id)
	      ;
	  m->p[line][id + 1] = 1;
	  // Fill the remainder of the access.
	  for (k = 1; k < s->read->NbColumns; ++k)
	    m->p[line][count + k] = s->read->p[j][k];
	  ++line;
	}
      for (j = 0; j < s->written->NbRows; ++j)
	{
	  if (s->written->p[j][0] != 0)
	    for (id = 0; id < count && arr[id] != s->written->p[j][0]; ++id)
	      ;
	  m->p[line][id + 1] = 1;
	  // Fill the remainder of the access.
	  for (k = 1; k < s->written->NbColumns; ++k)
	    m->p[line][count + k] = s->written->p[j][k];
	  ++line;
	}
      stmts[i] = ls_rank (m, 0);
      candl_matrix_free (m);
    }

}


static
void
ls_schedspace_depgraph_order (s_graph_t* g, CandlProgram* p)
{
  int i, j;
  s_vertex_t* v;
  CandlStatement* source;
  CandlStatement* target;
  CandlDependence* dep;
  CandlMatrix* m1;
  CandlMatrix* m2;
  int r1;
  int r2;

  int stmts[p->nb_statements];
  for (i = 0; i < p->nb_statements; ++i)
    stmts[i] = 0;

  ls_compute_ranks (stmts, p);

  // Order the dependence graph w.r.t. statement's ranks
  CandlDependence* deplist[g->vertex_count];
  s_vertex_t* nodelist[g->vertex_count];
  int statmentsord[p->nb_statements];
  for (i = 0; i < p->nb_statements; ++i)
    statmentsord[i] = i;

  for (i = 0; i < p->nb_statements; ++i)
    for (j = i; j < p->nb_statements; ++j)
      // FIXME: should be > instead of <
      if (stmts[i] < stmts[j])
	{
	  r1 = stmts[j];
	  stmts[j] = stmts[i];
	  stmts[i] = r1;
	  r1 = statmentsord[j];
	  statmentsord[j] = statmentsord[i];
	  statmentsord[i] = r1;
	}
  int count = 0;
  for (i = 0; i < p->nb_statements; ++i)
    for (v = g->root; v; v = v->next)
      {
	dep = v->data;
	source = dep->source;
	target = dep->target;
	if (source->label == statmentsord[i] ||
	    target->label == statmentsord[i])
	  {
	    for (j = 0; j < count && nodelist[j] != v; ++j)
	      ;
	    if (j == count)
	      nodelist[count++] = v;
	  }
      }
  g->root = nodelist[0];
  for (i = 0; i < g->vertex_count - 1; ++i)
    nodelist[i]->next = nodelist[i + 1];
  nodelist[i]->next = NULL;
}
/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////


/**
 * Get the next graph color to use (helper).
 *
 */
int
ls_schedspace_get_color (s_graph_t* g)
{
  s_vertex_t* v;
  int colors[g->vertex_count];
  int i;
  int max = 0;
  int max_id = -1;
  CandlDependence* tmp;

  for (i = 0; i < g->vertex_count; ++i)
    colors[i] = 0;

  for (v = g->root; v; v = v->next)
    {
      tmp = v->data;
      if (v->color != -1 && CANDL_DEP_TAG(tmp, solved) != 1)
	colors[v->color]++;
    }

  for (i = 0; i < g->vertex_count; ++i)
    if (max < colors[i])
      {
	max = colors[i];
	max_id = i;
      }

  return max_id;

}



/**
 * \brief Compute the solution polytope, given a set of dependences, program
 * informations and options provided to LetSee.
 *
 * The computation depends on:
 * - the initalization of the solution polytope,
 * - the construction of the system to solve,
 * - the intersection of the obtained solution set with the solution
 *   polytope.
 *
 * Possible options include backtracking, FM algorithm type,
 * redundancy elimination, dependence ordering.
 *
 * Features include strong dependence satisfaction interference graph,
 * polytope compaction, ...
 *
 */
static
s_ls_space_t*
ls_schedspace_compute_solution (s_graph_t* graph,
				CandlProgram* program,
				s_ls_options_t* options)
{
  // Array of solution polytopes.
  s_fm_solution_t** PA = XMALLOC(s_fm_solution_t*, 10);
  // Array of compacted solution polytopes.
  s_fm_compsol_t** CA = XMALLOC(s_fm_compsol_t*, 10);

  // Temporaries.
  s_fm_solution_t* P;
  s_fm_solution_t* Ptmp;
  s_fm_system_t* to_solve;
  s_fm_solution_t* solution;
  int d_count = 0, dimension;
  CandlDependence* tmp;
  CandlDependence* dep;
  s_vertex_t* vertex;
  s_vertex_t* vit;
  int count = 0;
  int ii;
  int unresolved = 1;
  int first = 1;
  s_fm_compsol_t* cs;
  s_fm_compsol_t* cs_tmp;
  s_fm_compsol_t* cs_tosolve;
  int current_color;
  int nb_solved;
  int nb_tested;

  // Backup structures (used for backtracking).
  s_graph_t* b_graph = NULL;
  s_fm_compsol_t* b_cs = NULL;
  s_graph_t* t_graph = NULL;
  s_fm_compsol_t* t_cs = NULL;
  int b_nb_solved;
  int b_nb_tested;
  int backtrack = 0;
  int had_backtrack = LS_BACKTRACK_NONE;

  // Create new schedule dimension while unresolved dependence remains.
  while (unresolved)
    {
      // Initialize the solution polytope for the given dimension.
      P = ls_farkas_initialize_solution (program, options);
      unresolved = 0;

      if (options->verbose)
	{
	  fprintf (options->out_file,
		   ".... Initializing schedule dimension %d\n", count + 1);
	  fprintf (options->out_file,
		   "..... ");
	}
      d_count = 0;
      // Set all the (unresolved) dependences as >= 0.
      for (vertex = graph->root; vertex; vertex = vertex->next)
	{
	  ++d_count;
	  tmp = vertex->data;

	  if (options->verbose)
	    {
	      fprintf (options->out_file, "0");
	      fflush (options->out_file);
	    }

	  // Build the system to solve, for the given dependence.
	  to_solve = ls_farkas_build_system (program, options, tmp,
					     &dimension, LS_FARKAS_GEZERO);

	  // FIXME: Experimental. Use Gauss on small systems.
	  cs_tosolve = fm_compsol_init_sys (to_solve);
	  fm_system_free (to_solve);
	  to_solve = fm_solution_to_system (cs_tosolve->poly);
	  fm_solution_free (cs_tosolve->poly);
	  if (options->noredundancy_solver)
	    cs_tosolve->poly = fm_solver
	      (to_solve, FM_SOLVER_FAST | FM_SOLVER_REDREC_IRIGOIN);
	  else
	    cs_tosolve->poly = fm_solver
	      (to_solve, FM_SOLVER_FAST);
	  solution = fm_compsol_expand (cs_tosolve);
	  fm_solution_cut (solution, dimension);
	  fm_compsol_free (cs_tosolve);

	  // Add the obtained solution set to the solution polytope.
	  ls_farkas_intersect (program, options, tmp, P, solution);
	  // Be clean.
	  fm_system_free (to_solve);
	  fm_solution_free (solution);
	}

      if (options->verbose)
	fprintf (options->out_file, "\n");

      // Perform implicit equalities detection on the solution
      // polytope, and use a compacted form 'cs'.
      cs = fm_compsol_init_sol (P);
      fm_solution_free (P);

      if (options->verbose)
	fprintf (options->out_file,
		 ".... Compaction: using %d/%d dimensions\n",
		 cs->poly->size, cs->size);

      // Check the consistency of the dimension.
      fm_piptools_check_sol_msg (".... Check consistency: ",
				 options->out_file, cs->poly,
				 FM_PIPTOOLS_INT);
      if (cs->empty)
	{
	  printf (".... Contradiction in initial polyhedron\n");
	  exit (99);
	  assert (! ".... Contradiction in initial polyhedron");
	}

      if (first)
	first = 0;
      d_count = 0;
      // fm_solution_print (stdout, P);

      // Color the conflict graph.
      graph->tag = LS_GRAPH_TAG_UNDEFINED;
      int maxcol = ls_graph_color (graph, LS_GRAPH_COLOR_RESET);

      if (options->backtrack_mode)
	{
	  // Backup the current state.
	  b_graph = ls_graph_dup (graph);
	  b_cs = fm_compsol_dup (cs);
	}
      // Get the current color to solve.
      current_color = ls_graph_max_color (graph);

    backtrack_entry:
      nb_solved = 0;
      nb_tested = 0;
      d_count = 0;

      // Order the dependences w.r.t. traffic computation
      if (options->traffic_order)
	ls_schedspace_depgraph_order (graph, program);
      // Reinitialize the tried mark.
      for (vertex = graph->root; vertex; vertex = vertex->next)
	vertex->tried = 0;

      // Loop on each dependence, and try to set them as >= 1.
      for (vertex = graph->root; vertex; )
	{
	  ++d_count;
	  tmp = vertex->data;

	  // FIXME: inhibited for sake of debugging.
	  // Skip the dependence if it is not the current color.
	  if (graph->edge_count)
	    {
	      if (vertex->tried || CANDL_DEP_TAG(tmp, solved) != 0)
		{
		  vertex = vertex->next;
		  continue;
		}
	      if (! options->traffic_order)
		if (vertex->color != current_color)
		  {
		    vertex = vertex->next;
		    continue;
		  }
	    }

	  ++nb_tested;

	  if (options->verbose)
	    fprintf (options->out_file,
		     ".... Solving dependence %d (id: %d)\n",
		     d_count, CANDL_DEP_TAG(tmp, id), tmp->depth);

	  // Build the system to solve, for the given dependence.
	  to_solve = ls_farkas_build_system (program, options, tmp,
					     &dimension, LS_FARKAS_GEONE);

	  // FIXME: Experimental. Use Gauss on small systems.
	  cs_tosolve = fm_compsol_init_sys (to_solve);
	  fm_system_free (to_solve);
	  to_solve = fm_solution_to_system (cs_tosolve->poly);
	  fm_solution_free (cs_tosolve->poly);
	  if (options->noredundancy_solver)
	    cs_tosolve->poly = fm_solver
	      (to_solve, FM_SOLVER_FAST | FM_SOLVER_REDREC_IRIGOIN);
	  else
	    cs_tosolve->poly = fm_solver
	      (to_solve, FM_SOLVER_FAST);
	  solution = fm_compsol_expand (cs_tosolve);
	  fm_solution_cut (solution, dimension);
	  fm_compsol_free (cs_tosolve);

	  cs_tmp = fm_compsol_dup (cs);

	  // Add the obtained solution set to the solution polytope.
	  ls_farkas_intersect_cs (program, options, tmp, cs, solution);

	  // Be clean.
	  fm_system_free (to_solve);
	  fm_solution_free (solution);

	  if (options->verbose)
	    {
	      fprintf (options->out_file,
		       "..... Strong dependence satisfaction: ");
	      fflush (options->out_file);
	    }
	  if (cs->empty || ! fm_piptools_check_sol (cs->poly, FM_PIPTOOLS_INT))
	    {
	      // No point.
	      unresolved = 1;
	      fm_compsol_free (cs);
	      cs = cs_tmp;
	      CANDL_DEP_TAG(tmp, solved) = 0;
	      if (options->verbose)
		fprintf (options->out_file, "no\n");

	      if (graph->edge_count)
		{
		  // FIXME: Inhibited for sake of debugging.
		  // If there is no point, update the coloring.
		  // Reset colors for all unsolved dependences.
		  for (vit = graph->root; vit; vit = vit->next)
		    {
		      // Skip the node if it has already been tested
		      // (marked with vertex_count for its colors).
		      if (vit == vertex || vit->color == graph->vertex_count)
			continue;
		      dep = vit->data;
		      if (CANDL_DEP_TAG(dep, solved) == 0)
			vit->color = -1;
		      else
			// Create an edge between all solved
			// dependences and the current one.
			ls_graph_create_edge (graph, vit, vertex, NULL);
		    }

		  ls_graph_topological_apply (graph);
		  /* 	      ls_graph_topological_apply_depth (graph); */
		  /* 	      vertex->color = graph->vertex_count; */
		  vertex->color = ! vertex->color;
		  // Tag to enable the use of current_color for coloring
		  // of independent nodes.
		  /* 	      graph->tag = current_color; */
		  graph->tag = LS_GRAPH_TAG_UNDEFINED;
		  ls_graph_color (graph, LS_GRAPH_COLOR_UPDATE);

		  if (options->traffic_order)
		    ls_schedspace_depgraph_order (graph, program);

 		  // Reinitialize.
		  vertex->tried = 1;
		  vertex = graph->root;

		  d_count = 0;
		  continue;
		}
	    }
	  else
	    {
	      fm_compsol_free (cs_tmp);
	      CANDL_DEP_TAG(tmp, solved) = 1;
	      if (options->verbose)
		fprintf (options->out_file, "yes\n");
	      ++nb_solved;
	    }
	  vertex = vertex->next;
	}

      if (options->backtrack_mode)
	{
      	  if (backtrack == 0)
	    {
	      // Count independent variables in the original graph.
	      int nb_indep;
	      for (nb_indep = 0, vit = b_graph->root; vit; vit = vit->next)
		if (vit->d_in == 0 && vit->d_out == 0)
		  ++nb_indep;

	      // If an insufficient number of dependences have been solved,
	      // backtrack once and use the second most represented color.
	      if ((nb_solved - nb_indep) <  (nb_tested - nb_indep) / 2)
		{
		  // Restore initial state.
		  t_graph = ls_graph_dup (graph);
		  t_cs = fm_compsol_dup (cs);
		  graph = b_graph;
		  cs = b_cs;
		  // Color the graph.
		  graph->tag = LS_GRAPH_TAG_UNDEFINED;
		  ls_graph_color (graph, LS_GRAPH_COLOR_RESET);
		  // Pick the second most represented color.
		  // (useful to compute the backtrack color)
		  current_color = ls_graph_second_max_color (graph);
		  // Re-color the graph, with the new current_color for
		  // independent nodes.
		  graph->tag = current_color;
		  ls_graph_color (graph, LS_GRAPH_COLOR_RESET);
/* 		  ls_graph_print (stdout, graph); */
		  for (vit = graph->root; vit; vit = vit->next)
		    {
		      tmp = vit->data;
		      CANDL_DEP_TAG(tmp, solved) = 0;
		    }
		  // Bactrack.
		  b_nb_solved = nb_solved;
		  b_nb_tested = nb_tested;
		  backtrack = 1;
		  goto backtrack_entry;
		}
	      else
		had_backtrack = LS_BACKTRACK_NONE;
	    }
	  // Backtrak
	  else
	    {
	      // Backtrack phase was less efficient.
	      if (nb_solved <= b_nb_solved)
		{
		  ls_graph_free (graph);
		  fm_compsol_free (cs);
		  graph = t_graph;
		  cs = t_cs;
		  nb_solved = b_nb_solved;
		  nb_tested = b_nb_tested;
		  had_backtrack = LS_BACKTRACK_USELESS;
		}
	      else
		{
		  ls_graph_free (t_graph);
		  fm_compsol_free (t_cs);
		  had_backtrack = LS_BACKTRACK_USEFULL;
		  // assert (!"Backtrack was useful");
		}
	      backtrack = 0;
	    }
	}

      CA[count] = cs;
      PA[count++] = fm_compsol_expand (cs);

      // Remove all solved dependences from the dependence graph.
      for (vit = graph->root; vit; )
	{
	  tmp = vit->data;
	  vertex = vit->next;
	  if (CANDL_DEP_TAG(tmp, solved) == 1)
	    ls_graph_remove_vertex (graph, vit);
	  vit = vertex;
	}

      // Output the number of solved dependences.
      printf ("Solved %d/%d dependences ", nb_solved, nb_tested);
      switch (had_backtrack)
	{
	case LS_BACKTRACK_NONE:
	  printf ("(no backtrack)\n"); break;
	case LS_BACKTRACK_USEFULL:
	  printf ("(backtrack useful)\n"); break;
	case LS_BACKTRACK_USELESS:
	  printf ("(backtrack useless)\n"); break;
	}
      if (graph->root != NULL)
	unresolved = 1;
    }

  PA[count] = NULL;
  CA[count] = NULL;

  // Allocate and fill the output.
  s_ls_space_t* space = ls_space_alloc ();
  space->polyhedron = NULL;
  space->u_polyhedron = PA;
  space->u_compacted = CA;
  space->dimension = count;

  return space;
}


/**
 * \brief Compact the dependence graph.
 *	  criterion: if source_1 = source_2 and target_1 = target_2
 *			and domain_1 = domain_2 then remove 2.
 *
 */
static
void
ls_schedspace_depgraph_compact (CandlDependence* dep)
{
  if (dep == NULL)
    return;

  CandlDependence* tmp;
  CandlDependence* tmp2;
  CandlDependence* aux;
  int count = 0;
  int total = 0;

  // FIXME: stupid.
  for (tmp = dep; tmp; tmp = tmp->next, ++total)
    ;

  // Note: null lines have been removed by preprocess function.
  for (tmp = dep; tmp; tmp = tmp->next)
    for (tmp2 = tmp; tmp2->next;)
      {
	if (tmp->source == tmp2->next->source &&
	    tmp->target == tmp2->next->target &&
	    (fm_piptools_pipmatrix_equal (tmp->domain, tmp2->next->domain)))
	  // FIXME: should also deal with included dependences!
	  // ||  fm_piptools_pm_is_included (tmp->domain, tmp2->next->domain)))
	  {
	    aux = tmp2->next->next;
	    pip_matrix_free (tmp2->next->domain);
	    XFREE (tmp2->next);
	    tmp2->next = aux;
	    ++count;
	  }
	else
	  tmp2 = tmp2->next;
      }
  printf ("Removed %d/%d dependences\n", count, total);
}


/**
 * \brief Preprocess the dependence graph:
 * - order the graph in lexicographic order of the dependences
 * - compact the graph (remove identical dependence polyhedra)
 * - affect uid to each dependence
 *
 *
 */
CandlDependence*
ls_schedspace_depgraph_preprocess (CandlProgram* program,
				   CandlDependence* dependences)
{
  //return dependences;

  if (dependences == NULL)
    return NULL;

  int i, j, size = 0, min_i, min_j, k;
  CandlDependence* head = dependences;
  CandlDependence* tmp;

  // Compute the number of dependences.
  for (tmp = dependences; tmp != NULL; tmp = tmp->next, ++size)
    ;

  // Copy data into a temporary array.
  CandlDependence** tab = XMALLOC(CandlDependence*, size);
  for (i = 0; i < size; ++i, head = head->next)
    tab[i] = head;

  // Sort the graph.
  for (i = 0; i < size; ++i)
    for (j = i; j < size; ++j)
      {
	min_i = tab[i]->source->label > tab[i]->target->label ?
	  tab[i]->source->label : tab[i]->target->label;
	min_j = tab[j]->source->label > tab[j]->target->label ?
	  tab[j]->source->label : tab[j]->target->label;
	if (min_i < min_j)
	  {
	    tmp = tab[i];
	    tab[i] = tab[j];
	    tab[j] = tmp;
	  }
      }
  // Finalize the sorted list.
  head = tmp = tab[0];
  for (i = 0; i < size - 1; ++i, tmp->next = tab[i], tmp = tmp->next)
    ;
  tmp->next = NULL;

  // Be clean.
  XFREE(tab);


  // Remove all blank lines in dependence matrices.
  for (tmp = head; tmp != NULL; tmp = tmp->next)
    {
      for (i = 0; i < tmp->domain->NbRows; ++i)
	{
	  for (j = 1; j < tmp->domain->NbColumns; ++j)
	    if (tmp->domain->p[i][j] != 0)
	      break;
	  // Remove the line.
	  if (j == tmp->domain->NbColumns)
	    {
	      for (k = i + 1; k < tmp->domain->NbRows; ++k)
		for (j = 0; j < tmp->domain->NbColumns; ++j)
		  tmp->domain->p[k - 1][j] = tmp->domain->p[k][j];
	      (tmp->domain->NbRows)--;
	    }
	}
    }

/*   // Compact dependence graph. */
/*   ls_schedspace_depgraph_compact (head); */

  // Update dependence ID.
  for (tmp = head, i = 0; tmp; tmp = tmp->next)
    CANDL_DEP_TAG(tmp, id) = ++i;

  return head;
}



/**
 * \brief Optimize the solution space. Apply FM-compaction and LeFur
 * descending method on each polytope.
 *
 */
void
ls_schedspace_optimize_space (CandlProgram* program,
			      s_ls_options_t* options,
			      s_ls_space_t* space)
{
  int i;


  // DEBUG.
  for (i = 0; i < space->dimension; ++i)
    {
      // Dump the solution.
      char buf[1024];
      char buf2[16];
      strcpy (buf, "s_base_dimension_");
      sprintf (buf2, "%d", i);
      strcat (buf, buf2);
      strcat (buf, ".sol");
      FILE* fout = fopen (buf, "w");
      s_fm_system_t* sysout = fm_solution_to_system (space->u_polyhedron[i]);
      fm_system_print (fout, sysout);
      fclose (fout);
      fm_system_free (sysout);
    }

  for (i = 0; i < space->dimension; ++i)
    {
      if (options->verbose)
	fprintf(options->out_file, "... Normalize dimension %d\n", i + 1);
      // s_fm_compsol_t* cs = fm_compsol_init_sol (space->u_polyhedron[i]);
      s_fm_compsol_t* cs = space->u_compacted[i];
      // Look for inequalities.
      s_fm_solution_t* soltmp = fm_compsol_expand (cs);
      s_fm_compsol_t* cs_tmp = fm_compsol_init_sol (soltmp);
      fm_compsol_free (cs);
      space->u_compacted[i] = cs = cs_tmp;
      cs->poly = fm_solution_simplify
	(cs->poly, FM_SOLVER_REDREC_IRIGOIN | FM_SOLVER_VERBOSE);

      s_fm_system_t* scs = fm_solution_to_system (cs->poly);
      fm_solution_free (cs->poly);
      cs->poly = fm_solver (scs, FM_SOLVER_FAST | FM_SOLVER_REDREC_IRIGOIN |
			    FM_SOLVER_SIMPLIFY);
      space->u_polyhedron[i] = fm_compsol_expand (cs);
      fm_system_free (scs);
    }
  if (options->verbose)
    fprintf(options->out_file, "... Solution normalized\n");

  // DEBUG.
  for (i = 0; i < space->dimension; ++i)
    {
      // Dump the solution.
      char buf[1024];
      char buf2[16];
      strcpy (buf, "s_dimension_");
      sprintf (buf2, "%d", i);
      strcat (buf, buf2);
      strcat (buf, ".sol");
      FILE* fout = fopen (buf, "w");
      s_fm_system_t* sysout = fm_solution_to_system (space->u_polyhedron[i]);
      fm_system_print (fout, sysout);
      fclose (fout);
      fm_system_free (sysout);
    }
}


/**
 * \brief Build the legal scheduling space, for p-dimensional affine
 * schedules.
 *
 * The output solution is represented by the polytope array Pa,
 * organized as follows:
 * For s1 = [i1 p1 c1] the schedule of the first statement (i1 is the
 * "iterator" part of the schedule, p1 the parameter part and c1 is
 * the constant one), and respectivly s2 = [ i2 p2 c2 ] for the second
 * statement; the output polytope Pa[x] is of the form Pa[x] = [ i1 i2
 * p1 p2 c1 c2 ]. The process is the same for each dimension x of the
 * schedule.
 *
 * See the Master Thesis "Legal transformation space exploration in
 * the polyhedral model: the monodimensional affine scheduling case",
 * LN Pouchet, 2006.
 *
 */
s_ls_space_t*
ls_schedspace_build (CandlProgram* program,
		     CandlDependence* dependences,
		     s_ls_options_t* options)
{
  int i, k;
  s_ls_space_t* space;
  s_graph_t* graph;

  if (options->verbose)
    fprintf(options->out_file, "... Build and solve local systems\n");

  // Preprocess the dependence graph.
  CandlDependence* depgraph = // dependences;
    ls_schedspace_depgraph_preprocess (program, dependences);
  candl_dependence_pprint (stdout, depgraph);

  // Build the conflict list.
  graph = ls_farkas_build_conflicts (program, depgraph, options);

  // Compute the solution polytope.
  space = ls_schedspace_compute_solution (graph, program, options);

  if (options->verbose)
    fprintf(options->out_file, "... All systems solved\n");

  // Initial PIP check.
  for (i = 0; i < space->size; ++i)
    if (! fm_piptools_check_sol (space->u_polyhedron[i], FM_PIPTOOLS_INT))
      {
	if (options->verbose)
	  fprintf(options->out_file,
		  "=> Ending computation: no point at dimension %d\n",
		  i + 1);
	exit (1);
      }

  // Optimize the space for exploration.
  if (options->normalize_space)
    ls_schedspace_optimize_space (program, options, space);

  // Internal conflict/dependence graph is no longer needed.
  ls_graph_free (graph);

  // Copy space for compatibility with former heuritics.
  space->polyhedron = space->u_polyhedron[0];

  return space;
}
