/*
 * options.c: this file is part of the LetSee project.
 *
 * LetSee, the LEgal Transformation SpacE Explorator.
 *
 * Copyright (C) 2006,2007,2008 Louis-Noel Pouchet
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * The complete GNU General Public Licence Notice can be found as the
 * `COPYING' file in the root directory.
 *
 * Author:
 * Louis-Noel Pouchet <Louis-Noel.Pouchet@inria.fr>
 *
 */
#if HAVE_CONFIG_H
# include <config.h>
#endif

#include "options.h"


static const struct s_opt       opts[LS_NB_OPTS] =
{
  { 'a', "ilb", 1, "\tLower iterator coefficient bound [-1]" },
  { 'b', "iub", 1, "\tUpper iterator coefficient bound [ 1]" },
  { 'c', "plb", 1, "\tLower parameter coefficient bound [-1]" },
  { 'd', "pub", 1, "\tUpper parameter coefficient bound [ 1]" },
  { 'e', "clb", 1, "\tLower constant coefficient bound [-1]" },
  { 'f', "cub", 1, "\tUpper constant coefficient bound [ 1]" },
  { 'o', "output", 1, "Output file [stdout]" },
  { 'n', "no-files", 0, "Don't create CLooG files for drawn schedules" },
  { 'z', "transfo-dir", 1, "Directory for CLooG files [transformations]" },
  { 't', "type", 1, "Space computation algorithm ([ospace], multi, fusion, fd, fs)"},
  { 'w', "walk", 1, "Exploration heuristic ([exhaust], h1, r1, r1m, m1, \n\t\t\tpluto, plutocc)"},
  { 's', "spacenorm", 0, "Normalize legal space" },
  { 'x', "execute", 1, "Compilation line" },
  { 'r', "rate", 1, "Thresold for heuristic filter (in %) [5]" },
  { 'q', "rtries", 1, "Number of tries for heuristic [20]" },
  { 'l', "le-fur", 0, "Use Le Fur redundancy elimination" },
  { 'M', "maxscale", 0, "Use maximum scalability mode to solve systems" },
  { 'L', "load-space", 1, "Load space from file" },
  { 'j', "vector", 1, "Prefix vector for schedule" },
  { 'i', "vector2", 1, "Prefix vector2 for schedule" },
  { 'y', "backtrack", 0, "Enable backtrack mode" },
  { 'p', "dep-order", 0, "Enable dependence ordering w.r.t. traffic" },
  { 'S', "scheme-m1", 1, "Specify scheme (imply -w m1): \"i+p+c,i+p,i,0\"" },
  { 'P', "prune-oset", 0, "Prune OSet, for fs space mode" },
  { 'v', "verbose", 0, "Verbose output" },
  { 'h', "help", 0, "Print this help" }
};


static const struct s_option    option =
  {
    opts,
    {"Unknown argument: -", "unknown argument: --",
     "Expected argument for option: "},
    LS_NB_OPTS,
    1
  };


static void     print_help (void)
{
  int           i;

  printf("Options for letsee are: \n");
  for (i = 0; i < LS_NB_OPTS; ++i)
    printf("-%c\t--%s  \t%s\n",
           opts[i].short_opt,
           opts[i].long_opt,
           opts[i].description);
  exit (1);
}



void
ls_usage ()
{
  fprintf (stderr, "Usage: letsee [options] candl-file\n");
  exit (1);
}



int
letsee_getopts (s_ls_options_t* options, int argc, char** argv)
{
  char          **opt_tab;
  int		ret;
  unsigned	i;

  opt_tab = malloc (sizeof (char *) * LS_NB_OPTS);
  for (i = 0; i < LS_NB_OPTS; ++i)
    opt_tab[i] = NULL;
  ret = get_cmdline_opts (&option, 1, argc, argv, opt_tab);

  if (opt_tab[LS_OPT_HELP])
    print_help();
  if (ret)
    {
      options->in_file = fopen (argv[ret], "r");
      if (options->in_file == NULL)
	ls_usage ();
      options->input_file = strdup (argv[ret]);
      if (ret < argc)
	ret = get_cmdline_opts (&option, ret + 1, argc, argv, opt_tab);
    }

  if (opt_tab[LS_OPT_HELP])
    print_help();
  if (ret != argc)
    ls_usage ();

  if (opt_tab[LS_OPT_LB])
    Z_ASSIGN_SI(options->lb, atoi (opt_tab[LS_OPT_LB]));

  if (opt_tab[LS_OPT_UB])
    Z_ASSIGN_SI(options->Ub, atoi (opt_tab[LS_OPT_UB]));


  if (opt_tab[LS_OPT_PLB])
    Z_ASSIGN_SI(options->plb, atoi (opt_tab[LS_OPT_PLB]));

  if (opt_tab[LS_OPT_PUB])
    Z_ASSIGN_SI(options->pUb, atoi (opt_tab[LS_OPT_PUB]));

  if (opt_tab[LS_OPT_CLB])
    Z_ASSIGN_SI(options->clb, atoi (opt_tab[LS_OPT_CLB]));

  if (opt_tab[LS_OPT_CUB])
    Z_ASSIGN_SI(options->cUb, atoi (opt_tab[LS_OPT_CUB]));

  if (opt_tab[LS_OPT_OUTPUT])
    {
      options->out_file = fopen (opt_tab[LS_OPT_OUTPUT], "w");
      options->has_out_file = 1;
      if (options->out_file == NULL)
	ls_usage ();
    }
  else
    options->out_file = stdout;

  if (opt_tab[LS_OPT_CREATE_SCHEDFILES])
    options->create_schedfiles = 0;

  if (opt_tab[LS_OPT_TRANSFO_DIR])
    {
      XFREE (options->transfo_dir);
      options->transfo_dir = strdup (opt_tab[LS_OPT_TRANSFO_DIR]);
    }

  if (opt_tab[LS_OPT_LOADSPACE])
    {
      XFREE (options->load_file);
      options->load_file = strdup (opt_tab[LS_OPT_LOADSPACE]);
    }


  if (opt_tab[LS_OPT_TYPE])
    {
      if (! strcmp(opt_tab[LS_OPT_TYPE], "global"))
	options->type = LS_TYPE_GLOBAL;
      else if (! strcmp(opt_tab[LS_OPT_TYPE], "mixed"))
	options->type = LS_TYPE_MIXED;
      else if (! strcmp(opt_tab[LS_OPT_TYPE], "local"))
	options->type = LS_TYPE_LOCAL;
      else if (! strcmp(opt_tab[LS_OPT_TYPE], "ospace"))
	options->type = LS_TYPE_OSPACE;
      else if (! strcmp(opt_tab[LS_OPT_TYPE], "multi"))
	options->type = LS_TYPE_MULTI;
      else if (! strcmp(opt_tab[LS_OPT_TYPE], "fusion"))
	options->type = LS_TYPE_FUSION;
      else if (! strcmp(opt_tab[LS_OPT_TYPE], "fd"))
	options->type = LS_TYPE_FD;
      else if (! strcmp(opt_tab[LS_OPT_TYPE], "fs"))
	options->type = LS_TYPE_FS;
      else if (! strcmp(opt_tab[LS_OPT_TYPE], "fs2"))
	options->type = LS_TYPE_FS2;
      else
	print_help();
    }

  if (opt_tab[LS_OPT_VERBOSE])
    options->verbose = 1;

  if (opt_tab[LS_OPT_SPACENORM])
    options->normalize_space = 1;

  if (opt_tab[LS_OPT_HEURISTIC])
    {
      if (! strcmp(opt_tab[LS_OPT_HEURISTIC], "exhaust"))
	options->heuristic = LS_HEURISTIC_EXHAUST;
      else if (! strcmp(opt_tab[LS_OPT_HEURISTIC], "h1"))
	options->heuristic = LS_HEURISTIC_H1;
      else if (! strcmp(opt_tab[LS_OPT_HEURISTIC], "r1"))
	options->heuristic = LS_HEURISTIC_R1;
      else if (! strcmp(opt_tab[LS_OPT_HEURISTIC], "r2"))
	options->heuristic = LS_HEURISTIC_R2;
      else if (! strcmp(opt_tab[LS_OPT_HEURISTIC], "r3"))
	options->heuristic = LS_HEURISTIC_R3;
      else if (! strcmp(opt_tab[LS_OPT_HEURISTIC], "r1m"))
	options->heuristic = LS_HEURISTIC_R1M;
      else if (! strcmp(opt_tab[LS_OPT_HEURISTIC], "h1m"))
	options->heuristic = LS_HEURISTIC_H1M;
      else if (! strcmp(opt_tab[LS_OPT_HEURISTIC], "elast"))
	options->heuristic = LS_HEURISTIC_E1M;
      else if (! strcmp(opt_tab[LS_OPT_HEURISTIC], "m1"))
	options->heuristic = LS_HEURISTIC_M1;
      else if (! strcmp(opt_tab[LS_OPT_HEURISTIC], "skip"))
	options->heuristic = LS_HEURISTIC_SKIP;
      else if (! strcmp(opt_tab[LS_OPT_HEURISTIC], "pluto"))
	options->heuristic = LS_HEURISTIC_PLUTO;
      else if (! strcmp(opt_tab[LS_OPT_HEURISTIC], "plutocc"))
	options->heuristic = LS_HEURISTIC_PLUTOCC;
      else if (! strcmp(opt_tab[LS_OPT_HEURISTIC], "plutom"))
	options->heuristic = LS_HEURISTIC_PLUTOM;
      else if (! strcmp(opt_tab[LS_OPT_HEURISTIC], "plutom1"))
	options->heuristic = LS_HEURISTIC_PLUTOM1;
      else if (! strcmp(opt_tab[LS_OPT_HEURISTIC], "plutom2"))
	options->heuristic = LS_HEURISTIC_PLUTOM2;
      else if (! strcmp(opt_tab[LS_OPT_HEURISTIC], "plutov2"))
	options->heuristic = LS_HEURISTIC_PLUTOV2;
      else
	print_help();
    }

  if (opt_tab[LS_OPT_COMPILE_LINE])
    options->compile_line = strdup (opt_tab[LS_OPT_COMPILE_LINE]);

  if (opt_tab[LS_OPT_H1_THRESOLD])
    Z_ASSIGN_SI(options->thresold, atoi (opt_tab[LS_OPT_H1_THRESOLD]));

  if (opt_tab[LS_OPT_R_TRIES])
    Z_ASSIGN_SI(options->rtries, atoi (opt_tab[LS_OPT_R_TRIES]));

  if (opt_tab[LS_OPT_LEFUR])
    options->noredundancy_solver = 1;

  if (opt_tab[LS_OPT_MAXSCALE])
    options->maxscale_solver = 1;

  if (opt_tab[LS_OPT_VECTOR])
    options->vector = fm_vector_read_str (strdup (opt_tab[LS_OPT_VECTOR]));

  if (opt_tab[LS_OPT_VECTOR2])
    options->vector2 = fm_vector_read_str (strdup (opt_tab[LS_OPT_VECTOR2]));

  if (opt_tab[LS_OPT_BACKTRACK])
    options->backtrack_mode = LS_BACKTRACK_ENABLE;

  if (opt_tab[LS_OPT_TRAFFIC])
    options->traffic_order = 1;

  if (opt_tab[LS_OPT_SCHEME_M1])
    {
      char buff[1024];
      int i, j, dim;
      options->heuristic = LS_HEURISTIC_M1;
      options->scheme_m1 = XMALLOC(int, LS_HEURISTIC_MAX_SCHEME_SIZE);
      for (i = 0; i < LS_HEURISTIC_MAX_SCHEME_SIZE; ++i)
	options->scheme_m1[i] = LS_HEURISTIC_M1_SCHEME_NONE;
      for (dim = i = j = 0; opt_tab[LS_OPT_SCHEME_M1][i]; ++dim, j = 0)
	{
	  while (opt_tab[LS_OPT_SCHEME_M1][i] &&
		 opt_tab[LS_OPT_SCHEME_M1][i] != ',')
	    buff[j++] = opt_tab[LS_OPT_SCHEME_M1][i++];
	  buff[j] = '\0';
	  if (! strcmp (buff, "i"))
	    options->scheme_m1[dim] = LS_HEURISTIC_M1_SCHEME_ITER;
	  else if (! strcmp (buff, "i+p"))
	    options->scheme_m1[dim] = LS_HEURISTIC_M1_SCHEME_ITERPARAM;
	  else if (! strcmp (buff, "i+p+c"))
	    options->scheme_m1[dim] = LS_HEURISTIC_M1_SCHEME_FULL;
	  else if (! strcmp (buff, "0"))
	    options->scheme_m1[dim] = LS_HEURISTIC_M1_SCHEME_NONE;
	  else
	    print_help ();
	  if (opt_tab[LS_OPT_SCHEME_M1][i])
	    ++i;
	}
    }

  if (opt_tab[LS_OPT_PRUNE_OSET])
    options->prune_oset = 1;

  free (opt_tab);
  return 0;
}
