/*
 * pragmatize.c: this file is part of the Pragmatize project.
 *
 * Pragmatize, an automatic #pragma insertion mechanism for CLAST
 *
 * Copyright (C) 2009 Louis-Noel Pouchet
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 3
 * of the License, or (at your option) any later version.
 *
 * The complete GNU Lesser General Public Licence Notice can be found
 * as the `COPYING.LESSER' file in the root directory.
 *
 * Author:
 * Louis-Noel Pouchet <pouchet@cse.ohio-state.edu>
 */
#if HAVE_CONFIG_H
# include <pragmatize/config.h>
#endif

#include <pragmatize/common.h>

# define CLOOG_SUPPORTS_SCOPLIB
# include <cloog/cloog.h>
# include <cloog/clast.h>
# include <pragmatize/pragmatize.h>
# include <clasttools/clastext.h>


/**
 * Create a clast_pragma_omp_for node right before the for node given
 * in argument.
 *
 * In addition, creates two new nodes for upper and lower bound
 * computation, lb1 and Ub1, inserted before the pragma omp for node.
 *
 */
void
pragmatize_insert_pragma_omp_for_at_node(struct clast_for* f,
					 struct clast_stmt** last,
					 char* iterator, char* clause)
{
  // fprintf (stdout, "[Pragmatize] At %s: inserting %s\n", iterator, clause);

  // Creating all the required new nodes.
  struct clast_pragma_omp_for* pragma =
    new_clast_pragma_omp_for (strdup (clause));
  struct clast_expr* lb = f->LB;
  struct clast_expr* Ub = f->UB;
  cloog_int_t one; cloog_int_init (one); cloog_int_set_si (one, 1);
  struct clast_expr* newlb =
    &new_clast_term(one, &new_clast_name("lb1")->expr)->expr;
  struct clast_expr* newUb =
    &new_clast_term(one, &new_clast_name("ub1")->expr)->expr;
  struct clast_assignment* asslb =
    new_clast_assignment("lb1", lb);
  struct clast_assignment* assUb =
    new_clast_assignment("ub1", Ub);

  // Chaining them.
  *last = (struct clast_stmt*) asslb;
  ((struct clast_stmt*) asslb)->next = (struct clast_stmt*) assUb;
  ((struct clast_stmt*) assUb)->next = (struct clast_stmt*) pragma;
  ((struct clast_stmt*) pragma)->next = (struct clast_stmt*) f;

  // Updating the for conditions.
  f->LB = newlb;
  f->UB = newUb;

  // Clear.
  cloog_int_clear(one);
}


/**
 * Create a clast_pragma_vectorize node right before the for node given
 * in argument.
 *
 * In addition, creates two new nodes for upper and lower bound
 * computation, lb1 and Ub1, inserted before the pragma omp for node.
 *
 */
void
pragmatize_insert_pragma_vectorize_at_node(struct clast_for* f,
					   struct clast_stmt** last,
					   char* iterator, char* clause)
{
  // fprintf (stdout, "[Pragmatize] At %s: inserting %s\n", iterator, clause);

  // Creating all the required new nodes.
  struct clast_pragma_vectorize* pragma =
    new_clast_pragma_vectorize (strdup (clause));

  // Chaining them.
  *last = (struct clast_stmt*) pragma;
  ((struct clast_stmt*) pragma)->next = (struct clast_stmt*) f;
}


/**
 * Insert pragmas around for loops.
 *
 * For each iterator in interators (NULL-terminated array), find the
 * for loop using this iterator, and call
 * pragmatize_insert_pragma_{omp_for,vectorize}_at_node.
 *
 */
void
pragmatize_insert_pragma_in_clast (struct clast_stmt* s,
				   struct clast_stmt** last,
				   char** iterators,
				   char** clauses)
{
  // Traverse the clast.
  for ( ; s; last = &(s->next), s = s->next)
    {
      if (CLAST_STMT_IS_A(s, stmt_for) || CLAST_STMT_IS_A(s, stmt_parfor))
	{
	  struct clast_for* f = (struct clast_for *) s;
	  int i;
	  // If it's the for loop we are looking for, trigger the
	  // treatment.
	  for (i = 0; iterators[i]; ++i)
	    if (! strcmp (f->iterator, iterators[i]))
	      {
		if (! strcmp ("#pragma ivdep", clauses[i]))
		  pragmatize_insert_pragma_vectorize_at_node
		    (f, last, iterators[i], clauses[i]);
		else
		  pragmatize_insert_pragma_omp_for_at_node
		    (f, last, iterators[i], clauses[i]);
	      }
	  // Continue the traversal, to allow nested OpenMP pragmas.
	  pragmatize_insert_pragma_in_clast (f->body, &(f->body),
					     iterators, clauses);
	}
      else if (CLAST_STMT_IS_A(s, stmt_vectorfor))
	pragmatize_insert_pragma_vectorize_at_node
	  ((struct clast_for *) s, last, NULL,
	   "#pragma ivdep\n#pragma vector always");
      else if (CLAST_STMT_IS_A(s, stmt_guard))
	pragmatize_insert_pragma_in_clast (((struct clast_guard *)s)->then,
					   &(((struct clast_guard *)s)->then),
					   iterators, clauses);
      else if (CLAST_STMT_IS_A(s, stmt_block))
	pragmatize_insert_pragma_in_clast (((struct clast_block *)s)->body,
					   &(((struct clast_block *)s)->body),
					   iterators, clauses);
    }
}


/**
 * Create clast_pragma nodes directly from the extended CLAST tree.
 *
 */
int
pragmatize_clast (scoplib_scop_p scop, struct clast_stmt* root)
{
  char* iterators[1]; iterators[0] = NULL;
  char* clauses[1]; clauses[0] = NULL;

  pragmatize_insert_pragma_in_clast (root, NULL, iterators, clauses);
}


/**
 * Read .pragma and .vectorize files, as generated by pluto, and
 * create the corresponding clast nodes.
 *
 */
int
pragmatize (scoplib_scop_p scop, struct clast_stmt* root)
{
  char** iterators = XMALLOC(char*, 512);
  char** clauses = XMALLOC(char*, 512);
  char buffer[8192];
  int line = 0;
  int pos, start, max;
  int i, j;

  // 1- Retrieve the list of iterators of parallel loops.
  FILE* f = fopen (".pragmas", "r");
  if (f != NULL)
    {
      while ((max = fread (buffer, sizeof(char), 8192, f)))
      {
	for (i = 0; i < max; ++i)
	  {
	    pos = 0;
	    while (buffer[i + pos++] != ' ' && i + pos < max)
	      ;
	    iterators[line] = XMALLOC(char, pos);
	    for (j = 0; j < pos - 1; ++j)
	      iterators[line][j] = buffer[i + j];
	    iterators[line][j] = '\0';
	    i += pos;
	    for (pos = 0; buffer[i + pos] && buffer[i + pos] != '\n'; ++pos)
	      ;
	    clauses[line] = XMALLOC(char, pos + 1);
	    for (j = 0; j < pos; ++j)
	      clauses[line][j] = buffer[i + j];
	    clauses[line][j] = '\0';
	    ++line;
	    i += pos;
	  }
      }
      fclose (f);
    }

  // 2- Retrieve the list of vectorizable loop iterators.
  f = fopen (".vectorize", "r");
  if (f != NULL)
    {
      while ((max = fread (buffer, sizeof(char), 8192, f)))
	{
	  for (i = 0; i < max; ++i)
	    {
	      pos = 0;
	      for (pos = 0; buffer[i + pos] != '\n' && i + pos < max; ++pos)
		;
	      iterators[line] = XMALLOC(char, pos + 1);
	      for (j = 0; j < pos; ++j)
		iterators[line][j] = buffer[i + j];
	      iterators[line][j] = '\0';
	      i += pos;
	      clauses[line] = strdup ("#pragma ivdep");
	      ++line;
	    }
	}
      fclose (f);
    }
  clauses[line] = iterators[line] = NULL;

  // 2- Traverse the clast, and insert nodes.
  pragmatize_insert_pragma_in_clast (root, NULL, iterators, clauses);

  // 3- Be clean.
  for (i = 0; i < line; ++i)
    {
      XFREE(clauses[i]);
      XFREE(iterators[i]);
    }
  XFREE(clauses);
  XFREE(iterators);

  return 0;
}
