/*
 * rational.c: this file is part of the FM project.
 *
 * FM, a fast and optimized C implementation of Fourier-Motzkin
 * projection algorithm.
 *
 * Copyright (C) 2006-2008 Louis-Noel Pouchet
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 3
 * of the License, or (at your option) any later version.
 *
 * The complete GNU Lesser General Public Licence Notice can be found
 *  as the `COPYING.LESSER' file in the root directory.
 *
 * Author:
 * Louis-Noel Pouchet <Louis-Noel.Pouchet@inria.fr>
 *
 */
#if HAVE_CONFIG_H
# include <fm/config.h>
#endif

#include <fm/common.h>
#include <fm/rational.h>


static
z_type_t
fm_gcd_s(z_type_t x, z_type_t y)
{
  z_type_t z;
  Z_INIT(z);
  Z_ASSIGN_SI(z, 1);
  //   printf("GCD of %ld %ld :", x, y);

  while (Z_CMP_SI(z, !=, 0))
    {
      Z_MOD(z, x, y);
      Z_ASSIGN(x, y);
      Z_ASSIGN(y, z);
    }

  Z_CLEAR(z);

  //printf(" %ld\n", x);

  return x;
}

z_type_t
fm_z_gcd(z_type_t a, z_type_t b)
{
  z_type_t mone; Z_INIT(mone); Z_ASSIGN_SI(mone, -1);

  if (Z_CMP_SI(a, <, 0))
    Z_MUL(a, a, mone);
  if (Z_CMP_SI(b, <, 0))
    Z_MUL(b, b, mone);
  Z_CLEAR(mone);
  if (Z_CMP(a, <, b))
    return fm_gcd_s (a, b);
  else
    return fm_gcd_s (b, a);
}


z_type_t
fm_z_lcm(z_type_t a, z_type_t b)
{
  // FIXME: Not compliant to Z_xxx...
   return (b * a) / fm_z_gcd (a, b);
}


s_fm_rational_t*
fm_rational_alloc ()
{
  s_fm_rational_t* r = XMALLOC(s_fm_rational_t, 1);
  Z_ASSIGN(r->num, 0);
  Z_ASSIGN(r->denum, 1);

  return r;
}


void
fm_rational_init (s_fm_rational_t* r)
{
  Z_ASSIGN(r->num, 0);
  Z_ASSIGN(r->denum, 1);
}


void
fm_rational_print (FILE* stream, s_fm_rational_t* r)
{
  Z_PRINT(stream, r->num);
  if (Z_CMP_SI(r->denum, !=, 1))
    {
      fprintf(stream, "/");
      Z_PRINT(stream, r->denum);
    }
}


void
fm_rational_free (s_fm_rational_t* r)
{
  if (r != NULL)
    {
      Z_CLEAR(r->num);
      Z_CLEAR(r->denum);
    }
  XFREE(r);
}


int
fm_rational_assign_int (s_fm_rational_t* r, z_type_t num)
{
  Z_ASSIGN(r->num, num);
  Z_ASSIGN_SI(r->denum, 1);

  return FM_STATUS_OK;
}

int
fm_rational_copy (s_fm_rational_t* r, s_fm_rational_t* s)
{
  fm_rational_assign (r, s->num, s->denum);

  return FM_STATUS_OK;
}


int
fm_rational_assign (s_fm_rational_t* r, z_type_t num, z_type_t denum)
{
  /* A problem occurs when denominator is 0. */
  if (Z_CMP_SI(denum, ==, 0))
    {
      assert (! "Assigning 0 to denominator of a rational.");
      return FM_STATUS_ERROR;
    }

  /* Simple when numerator is 0. */
  if (Z_CMP_SI(num, ==, 0))
    {
      Z_ASSIGN_SI(r->num, 0);
      Z_ASSIGN_SI(r->denum, 1);
      return FM_STATUS_OK;
    }

  Z_ASSIGN(r->num, num);
  Z_ASSIGN(r->denum, denum);

  /* Deal with signs. */
  if (Z_CMP_SI(r->denum, <, 0))
    {
      Z_OPP(r->num, r->num);
      Z_OPP(r->denum, r->denum);
    }

  /* Compute gcd, and simplify if needed. */
  if (Z_CMP_SI(num, !=, 1) && Z_CMP_SI(denum, !=, 1))
    {
      z_type_t the_gcd;
      Z_INIT(the_gcd);
      Z_GCD(the_gcd, r->num, r->denum);
      if (Z_CMP_SI(the_gcd, !=, 1))
	{
	  Z_DIV(r->num, r->num, the_gcd);
	  Z_DIV(r->denum, r->denum, the_gcd);
	}

      Z_CLEAR(the_gcd);
    }

  return FM_STATUS_OK;
}


int
fm_rational_cmp (s_fm_rational_t* r1, s_fm_rational_t* r2)
{
  z_type_t v1;
  z_type_t v2;

  Z_MUL(v1, r1->num, r2->denum);
  Z_MUL(v2, r2->num, r1->denum);

  if (Z_CMP(v1, ==, v2))
    return 0;
  if (Z_CMP(v1, >, v2))
    return 1;
  return -1;
}





void
fm_rational_add (s_fm_rational_t*r,  s_fm_rational_t* r1, s_fm_rational_t* r2)
{
  z_type_t num;
  z_type_t denum;
  z_type_t tmp;

  Z_INIT(num);
  Z_INIT(denum);
  Z_INIT(tmp);

  Z_MUL(denum, r1->denum, r2->denum);
  Z_MUL(num, r1->num, r2->denum);
  Z_MUL(tmp, r2->num, r1->denum);
  Z_ADD(num, num, tmp);
  fm_rational_assign (r, num, denum);

  Z_CLEAR(num);
  Z_CLEAR(denum);
  Z_INIT(tmp);
}


void
fm_rational_sub (s_fm_rational_t*r,  s_fm_rational_t* r1, s_fm_rational_t* r2)
{
  z_type_t num;
  z_type_t denum;
  z_type_t tmp;

  Z_INIT(num);
  Z_INIT(denum);
  Z_INIT(tmp);

  Z_MUL(denum, r1->denum, r2->denum);
  Z_MUL(num, r1->num, r2->denum);
  Z_MUL(tmp, r2->num, r1->denum);
  Z_SUB(num, num, tmp);
  fm_rational_assign (r, num, denum);

  Z_CLEAR(num);
  Z_CLEAR(denum);
  Z_INIT(tmp);
}


void
fm_rational_mul (s_fm_rational_t*r,  s_fm_rational_t* r1, s_fm_rational_t* r2)
{
  z_type_t num;
  z_type_t denum;

  Z_INIT(num);
  Z_INIT(denum);

  Z_MUL(denum, r1->denum, r2->denum);
  Z_MUL(num, r1->num, r2->num);
  fm_rational_assign (r, num, denum);

  Z_CLEAR(num);
  Z_CLEAR(denum);
}


void
fm_rational_div (s_fm_rational_t*r,  s_fm_rational_t* r1, s_fm_rational_t* r2)
{
  z_type_t num;
  z_type_t denum;

  Z_INIT(num);
  Z_INIT(denum);

  Z_MUL(denum, r1->denum, r2->num);
  Z_MUL(num, r1->num, r2->denum);
  fm_rational_assign (r, num, denum);

  Z_CLEAR(num);
  Z_CLEAR(denum);
}


void
fm_rational_opp (s_fm_rational_t*r,  s_fm_rational_t* r1)
{
  z_type_t num;
  z_type_t denum;

  Z_INIT(num);
  Z_INIT(denum);

  Z_ASSIGN(denum, r1->denum);
  Z_ASSIGN(num, r1->num);
  Z_OPP(num, num);
  fm_rational_assign (r, num, denum);

  Z_CLEAR(num);
  Z_CLEAR(denum);
}



int
fm_rational_equal (s_fm_rational_t* r1, s_fm_rational_t* r2)
{
  if (Z_CMP(r1->num, !=, r2->num) || Z_CMP(r1->denum, !=, r2->denum))
    return 0;

  return 1;
}
