/*
 * fm_minlexico.c: this file is part of the FM project.
 *
 * FM, a fast and optimized C implementation of Fourier-Motzkin
 * projection algorithm.
 *
 * Copyright (C) 2006-2008 Louis-Noel Pouchet
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 3
 * of the License, or (at your option) any later version.
 *
 * The complete GNU Lesser General Public Licence Notice can be found
 *  as the `COPYING.LESSER' file in the root directory.
 *
 * Author:
 * Louis-Noel Pouchet <Louis-Noel.Pouchet@inria.fr>
 *
 */
#if HAVE_CONFIG_H
# include <config.h>
#endif

#include <fm/common.h>
#include <fm/system.h>
#include <fm/solution.h>
#include <fm/solver.h>


void
print_rational_array(FILE* stream, s_fm_rational_t** v)
{
  unsigned i = 0;

  if (v == NULL)
    fprintf (stream, "[(null)]");
  else if (v[i] == NULL)
    fprintf (stream, "[]");
  else
    {
      fprintf (stream, "[");
      for (i = 0; v[i + 1] != NULL; ++i)
	{
	  fm_rational_print (stream, v[i]);
	  fprintf (stream, " ");
	}
      fm_rational_print (stream, v[i]);
      fprintf (stream, "]");
    }
  fprintf (stream, "\n");
}


void
fms_usage ()
{
  fprintf (stderr, "Usage: fm-minlexico input-file value-type min-value [output-file]\n");
  fprintf (stderr, "value-type: 0 for rational values, 1 for integral ones\n");
  fprintf (stderr, "min-value: an integral value to be substituted to -inf\n");
  exit (1);
}


void
fms_error ()
{
  fprintf (stderr, "fm-solver: Bad input file.\n");
  exit (1);
}


int
main (int argc, char * const argv[])
{
  if (argc < 4 || argc > 5)
    fms_usage ();

  FILE* f = fopen (argv[1], "r");
  if (f == NULL)
    fms_error ();

  // Read the input system.
  s_fm_system_t* system = fm_system_read (f);
  fclose (f);

  if (f == NULL)
    fms_error ();

  // Solve the system.
  s_fm_solution_t* solution = fm_solver (system, FM_SOLVER_FAST);
  unsigned i;
  s_fm_rational_t** min_lexico;

  min_lexico = fm_solver_minlexico (solution, atoi(argv[3]), atoi(argv[2]));

  if (argc == 5)
    f = fopen (argv[4], "w");
  else
    f = fdopen(dup(fileno(stdout)), "w");
  if (f == NULL)
    fms_usage ();

  print_rational_array (f, min_lexico);


  for (i = 0; min_lexico != NULL && min_lexico[i] != NULL; ++i)
    fm_rational_free (min_lexico[i]);
  XFREE(min_lexico);

  // Be clean.
  fm_system_free (system);
  fm_solution_free (solution);

  return 0;
}
