/*
 * fm_compsol_test.c: this file is part of the FM project.
 *
 * FM, a fast and optimized C implementation of Fourier-Motzkin
 * projection algorithm.
 *
 * Copyright (C) 2006-2008 Louis-Noel Pouchet
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 3
 * of the License, or (at your option) any later version.
 *
 * The complete GNU Lesser General Public Licence Notice can be found
 *  as the `COPYING.LESSER' file in the root directory.
 *
 * Author:
 * Louis-Noel Pouchet <Louis-Noel.Pouchet@inria.fr>
 *
 */
#if HAVE_CONFIG_H
# include <fm/config.h>
#endif

#include <stdio.h>
#include <fm/common.h>
#include <fm/solver.h>
#include <fm/compsol.h>


void
print_rational_array(FILE* stream, s_fm_rational_t** v)
{
  unsigned i = 0;

  if (v == NULL)
    fprintf (stream, "[(null)]");
  else if (v[i] == NULL)
    fprintf (stream, "[]");
  else
    {
      fprintf (stream, "[");
      for (i = 0; v[i + 1] != NULL; ++i)
	{
	  fm_rational_print (stream, v[i]);
	  fprintf (stream, " ");
	}
      fm_rational_print (stream, v[i]);
      fprintf (stream, "]");
    }
  fprintf (stream, "\n");
}


int
main (int argc, char** argv)
{
  if (argc < 3)
    return 0;

  fprintf(stderr, "** Testing compacter for %s\n", argv[1]);

  s_fm_system_t* system;

  FILE* f = fopen(argv[1], "r");
  if (f == NULL)
    return 1;

  system = fm_system_read (f);
  fclose(f);

  s_fm_solution_t* solution;
  s_fm_rational_t** min_lexico;
  s_fm_rational_t** min_lexico_int;
  s_fm_compsol_t* compsol;
  s_fm_solution_t* tmpsol;
  s_fm_system_t* system2;
  unsigned i;

  // Testing compsol. Result should be identity, so keep the solver test
  // as is.
  compsol = fm_compsol_init_sys (system);
  fm_system_free (system);
  tmpsol = fm_compsol_expand (compsol);
  system = fm_solution_to_system (tmpsol);
  fm_solution_free (tmpsol);
  fm_compsol_free (compsol);

  // Solver test.
  solution = fm_solver (system, FM_SOLVER_FAST | FM_SOLVER_NORMALIZE_EQ);

  min_lexico = fm_solver_minlexico (solution, -4, FM_MINLEXICO_RAT);
  min_lexico_int = fm_solver_minlexico (solution, -4, FM_MINLEXICO_INT);

  f = fopen(argv[2], "w");
  fm_solution_print (f, solution);
  print_rational_array (f, min_lexico);
  print_rational_array (f, min_lexico_int);
  fclose(f);

  fm_system_free (system);
  fm_solution_free (solution);
  for (i = 0; min_lexico != NULL && min_lexico[i] != NULL; ++i)
    fm_rational_free (min_lexico[i]);
  for (i = 0; min_lexico_int != NULL && min_lexico_int[i] != NULL; ++i)
    fm_rational_free (min_lexico_int[i]);

  XFREE(min_lexico);
  XFREE(min_lexico_int);

  return 0;
}
