/*
 * fm_vector_test.c: this file is part of the FM project.
 *
 * FM, a fast and optimized C implementation of Fourier-Motzkin
 * projection algorithm.
 *
 * Copyright (C) 2006-2008 Louis-Noel Pouchet
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 3
 * of the License, or (at your option) any later version.
 *
 * The complete GNU Lesser General Public Licence Notice can be found
 *  as the `COPYING.LESSER' file in the root directory.
 *
 * Author:
 * Louis-Noel Pouchet <Louis-Noel.Pouchet@inria.fr>
 *
 */
#if HAVE_CONFIG_H
# include <fm/config.h>
#endif

#include <stdio.h>
#include <fm/common.h>
#include <fm/vector.h>


int
main ()
{
  fprintf(stderr, "=> Testing vector class\n");
  fprintf(stderr, "* Allocation tests\n");
  s_fm_vector_t* v1 = fm_vector_alloc (5);
  if (v1->size != 5 || v1->key != 0)
    return 1;

  fprintf(stderr, "* Assignation tests\n");

  s_fm_rational_t* r1 = fm_rational_alloc ();
  s_fm_rational_t* r2 = fm_rational_alloc ();
  fm_rational_assign (r1, -2, 4);
  fm_rational_assign (r2, -2, 3);
  fm_vector_assign_idx (v1, r1, 2);
  if (v1->vector[2].num != -1 || v1->vector[2].denum != 2)
    return 1;


  fprintf(stderr, "* Operators tests:\n");
  s_fm_vector_t* v2 = fm_vector_alloc (5);
  unsigned i;

  fprintf(stderr, "** Assign tests\n");
  fm_vector_assign (v2, v1);
  for (i = 0; i < v1->size; ++i)
    if (! fm_rational_equal(&(v1->vector[i]), &(v2->vector[i])))
      return 1;

  fprintf(stderr, "** EQ tests\n");
  if (! fm_vector_equal (v2, v1))
    return 1;

  fprintf(stderr, "** ADD and SUB tests\n");
  fm_vector_add (v2, v1, v2);
  fm_vector_sub (v2, v2, v1);
  if (! fm_vector_equal (v2, v1))
    return 1;

  fprintf(stderr, "** Normalize idx tests\n");
  s_fm_vector_t* v3 = fm_vector_alloc (5);
  fm_vector_assign_idx (v2, r1, 4);
  fm_vector_assign_idx (v2, r2, 3);
  fm_vector_normalize_idx (v2, v2, 2);
  fm_vector_assign (v3, v2);
  if (v2->vector[2].num != -1 || v2->vector[2].denum != 1)
    return 1;
  if (v2->vector[3].num != -4 || v2->vector[3].denum != 3)
    return 1;
  if (v2->vector[4].num != -1 || v2->vector[4].denum != 1)
    return 1;

  fprintf(stderr, "** to-z tests\n");
  fm_vector_to_z (v2, v2);
  if (v2->vector[2].num != -3 || v2->vector[2].denum != 1)
    return 1;
  if (v2->vector[3].num != -4 || v2->vector[3].denum != 1)
    return 1;
  if (v2->vector[4].num != -3 || v2->vector[4].denum != 1)
    return 1;

  fprintf(stderr, "** do-subsume tests\n");
  fm_vector_set_ineq (v3);
  fm_vector_assign (v2, v3);
  if (! fm_vector_do_subsume (v2, v3))
    return 1;
  if (! fm_vector_equal (v2, v3))
    return 1;
  fm_vector_assign_idx (v3, r2, 4);
  if (! fm_vector_do_subsume (v2, v3))
    return 1;
  if (fm_vector_equal (v2, v3))
    return 1;
  if (! fm_vector_do_subsume (v3, v2))
    return 1;
  if (! fm_vector_equal (v2, v3))
    return 1;
  if (v2->vector[4].num != -1 || v2->vector[4].denum != 1)
    return 1;

  fprintf(stderr, "== All done for vector class.\n");

  return 0;
}
