/**
 * LoopbackPortForwarding.java: this file is part of the LoopbackPortForwarding project.
 *
 * LoopbackPortForwarding, a tiny port forward utility between 127.0.0.1 and your public IP.
 *
 * Copyright (C) 2008 Louis-Noel Pouchet
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * You should have received a copy of the GNU General Public License
 * along with this software; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * Author:
 * Louis-Noel Pouchet <Louis-Noel.Pouchet@inria.fr>
 */

import java.awt.*;
import java.util.*;


public class LoopbackPortForwarding {
	
	// Global to set the maximum number of forwarded ports.
	static final int max_nb_ports = 1000;

	// Keep the list of mapped ports.
	static final int[] ports = new int[max_nb_ports];
	// Keep the list of existing threads.
	private static ListenThread[] threads = new ListenThread[max_nb_ports];
	// GUI elements accessible by other threads.
	static final java.awt.TextField fields[] = new java.awt.TextField[max_nb_ports];
	static final java.awt.TextField rule = new java.awt.TextField();
	private static Panel pane = new Panel();
	private static TheGUI the_gui = new TheGUI();

	// Class for the GUI (convenience).
	static final class TheGUI extends Frame
	{	
		// GUI elements with a listener.
		private static java.awt.Button set_button = new java.awt.Button();
		private static java.awt.MenuItem quit_item = new java.awt.MenuItem("Stop and Quit");

		// Action listeners.
		class SymAction implements java.awt.event.ActionListener
	    {
			public void actionPerformed(java.awt.event.ActionEvent event)
			{
				Object object = event.getSource();			
				if (object == set_button)
					initialize_rules();				
				if (object == quit_item)
				{
					// Clean all threads.
					for (int i = 0; i < max_nb_ports; ++i)
						if (threads[i] != null)
							threads[i].close_sockets_and_die();
					// Clean exit.
					System.exit(0);
				}
			}
	    }
		
		// Definition of all GUI elements in the constructor.
		public TheGUI()
		{
			// Set global config.
			setSize(400, 300);
			try { setTitle(ListenThread.getLocalHost().toString()); } catch (Exception e) { };
	    	setVisible(false);
	    	// Header panel.
	    	pane.setBackground(Color.gray);
	    	pane.setLayout(new GridLayout(max_nb_ports + 3, 1));
	    	// Label.
	    	Label label = new Label("Port forwarding list:");
	    	label.setBackground(Color.gray);
	    	pane.add(label);
	    	// Rule textfield.
	    	rule.setText("3000,3002,3005:3008,3010");
	    	rule.setBackground(Color.white);
	    	pane.add(rule);
	    	// Button.
	    	set_button.setLabel("Forward listed ports");
	    	SymAction lSymAction = new SymAction();
	    	set_button.addActionListener(lSymAction);	    	
	    	pane.add(set_button);
	    	// Panel filled, validate it.
	    	pane.validate();	    	
	    	// Put the panel in a scrollpane.
	    	ScrollPane scroll = new ScrollPane(ScrollPane.SCROLLBARS_AS_NEEDED);
	    	scroll.add(pane);
	    	scroll.validate();
	    	scroll.setBackground(Color.gray);
	    	add(scroll);			
	    	// Menu.
			MenuBar mbar = new MenuBar();
			Menu m = new Menu("Quit");
			quit_item.addActionListener(lSymAction);
			m.add(quit_item);
			mbar.add(m);
			setMenuBar(mbar);
		}
	}

	// Method to set the information about the port to filter, w.r.t the GUI.
	private static void initialize_rules()
	{	
		// Clean the current state.
		for (int i = 0; i < max_nb_ports; ++i)
		{
			if (fields[i] != null)
			{
				// Clean the GUI.
				pane.remove(fields[i]);
				// Shutdown all active threads.
				threads[i].close_sockets_and_die();
				// Reset arrays.
				fields[i] = null;
				threads[i] = null;
			}
		}
		int cur_idx = 0;
		Scanner s = new Scanner(rule.getText());
		s.useDelimiter(",");
		String tmp = s.next();
		while (tmp.length() > 0)
		{
			// Scan the textbox content. Rule is:
			// rule -> range[,range]
			// range -> port[:port]
			Scanner r = new Scanner(tmp);
			r.useDelimiter(":");
			String start = r.next();
			int pstart = Integer.parseInt(start);
			int pstop = pstart;
			if (r.hasNext())
			{
				String stop = r.next();
				pstop = Integer.parseInt(stop);
			}
			// For each port to forward in the range, initialize.
			for (int i = 0; i <= pstop - pstart; ++i)
			{
				// Ensure we don't go beyond the max_nb_ports.
				if (i + cur_idx >= max_nb_ports)
					break;
				// Initialize the port.
				ports[i + cur_idx] = pstart + i;
				// Create the textfield.
	    		fields[i + cur_idx] = new java.awt.TextField();
				pane.add(fields[i + cur_idx]);
				// Start the listener thread.
				threads[i + cur_idx] = new ListenThread(ports[i + cur_idx], fields[i + cur_idx]);
				threads[i + cur_idx].start();
			}
			cur_idx += pstop - pstart + 1;
			if (s.hasNext())
				tmp = s.next();
			else
				tmp = "";
		}
		// Refresh the view.
		pane.validate();
	}

		
	/**
	 * Entry point.
	 */
	public static void main(String[] args) {
		// Make the GUI visible.
		the_gui.setVisible(true);
		// Initialize with base rule.
		initialize_rules();
	}
}
