/*
 * ConvexHullFormer.hpp: This file is part of the Parametric Tiling project.
 * 
 * Parametric Tiling: A CLAST-to-CLAST parametric tiling software
 * 
 * Copyright (C) 2011 Sanket Tavargeri
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 3
 * of the License, or (at your option) any later version.
 * 
 * The complete GNU Lesser General Public Licence Notice can be found
 * as the `COPYING.LESSER' file in the root directory.
 * 
 * Author:
 * Sanket Tavargeri <sanket.tavargeri@gmail.com>
 * 
 */

#if HAVE_CONFIG_H
# include <ptile/config.h>
#endif

#include <ptile/ConvexHullFormer.hpp>

  void ConvexHullFormer::ScoplibStatementToCloogDomain(scoplib_statement_p scopStatement, CloogDomain** cloogDomain)
  {
    //printf("Statement 1\n");        
    scoplib_matrix_p scopDomain = scopStatement->domain->elt;
    //printf("Statement 2\n");        
    if (scopStatement->domain->next != NULL)
      {
	assert("The statement has more than one domain"); // The domain of a statement being union of domains. - Unhandled case
      }// if (scopStatement->domain->elt != NULL)
      
    int NumberOfParameters = scopDomain->NbColumns - scopStatement->nb_iterators - 2;      
    int NbRows = scopDomain->NbRows;   
    int NbColumns = scopDomain->NbColumns;
            
    //printf("Statement 3\n");     
    CloogMatrix *cloogMatrix = (CloogMatrix*) malloc(sizeof(CloogMatrix));
    cloogMatrix->NbRows = NbRows;
    cloogMatrix->NbColumns = NbColumns;
    cloogMatrix->p_Init = (cloog_int_t *) malloc(sizeof(cloog_int_t) * NbRows * NbColumns);
      
    if (cloogMatrix->p_Init == NULL)
      {
	assert("The memory allocation failed");
      }// if (cloogMatrix->p_Init == NULL)
      
    //printf("NbRows * NbColumns = %d\n", NbRows * NbColumns);
    //printf("Statement 4\n");      
    cloogMatrix->p = (cloog_int_t **) malloc(sizeof(cloog_int_t*) * NbRows);
    int i, j;
    for (i = 0; i < NbRows; i++)
      {
	(cloogMatrix->p)[i] = &((cloogMatrix->p_Init)[i * NbColumns]);
      }// for (i = 0; i < NbRows; i++)
      
       //printf("Statement 5\n");      
    for (i = 0; i < scopDomain->NbRows * NbColumns; i++)
      {
	cloog_int_init((cloogMatrix->p_Init)[i]);
	cloog_int_set_si((cloogMatrix->p_Init)[i], (scopDomain->p_Init)[i]);
	//printf("%d ", (scopDomain->p_Init)[i]);
	if ( ((i + 1) % NbColumns ) == 0)
	  {
	    //printf("\n");
	  }// if ( ((i + 1) % NbColumns ) == 0)
 
      }// for (i = 0; i < NbRows; i++)

    //printf("Statement 6\n"); 
    int offsetRows = scopDomain->NbRows;
    
    
    CloogState* cstate = cloog_state_malloc();
    CloogOptions* coptions = cloog_options_malloc(cstate);

    //printf("Calling cloog_domain_from_cloog_matrix()\n");
    //printf("Number of parameters = %d\n", NumberOfParameters);
    (*cloogDomain) = cloog_domain_from_cloog_matrix(coptions->state, cloogMatrix, NumberOfParameters);
    //printf("Returning from cloog_domain_from_cloog_matrix()\n");
    
    cloog_options_free (coptions);
    
//     FILE *scopFile = fopen("CloogDomainFile", "a");
//     cloog_domain_print_constraints(scopFile, *cloogDomain, 0);
//     fclose(scopFile);
      
    //printf("Returning from ScoplibToCloogDomain\n");
    return;
  }// void ScoplibToCloogDomain(struct scoplib_statement scopStatement, CloogDomain* cloogDomain

   void ConvexHullFormer::CreateDefaultCloogDomainForContext(scoplib_statement_p scopStatement, CloogDomain** cloogDomain)
  {
    scoplib_matrix_p scopDomain = scopStatement->domain->elt;
    if (scopStatement->domain->next != NULL)
      {
	assert("The statement has more than one domain"); // The domain of a statement being union of domains. - Unhandled case
      }// if (scopStatement->domain->elt != NULL)
                              
    unsigned NbColumns = scopDomain->NbColumns;
    int NumberOfParameters = NbColumns - scopStatement->nb_iterators - 2;
    unsigned NbRows = NumberOfParameters;   
      
    CloogMatrix *cloogMatrix = (CloogMatrix*) malloc(sizeof(CloogMatrix));
    cloogMatrix->NbRows = NbRows;
    cloogMatrix->NbColumns = NbColumns;      
    cloogMatrix->p_Init = (cloog_int_t *) malloc(sizeof(cloog_int_t) * NbRows * NbColumns);
    cloogMatrix->p = (cloog_int_t **) malloc(sizeof(cloog_int_t*) * NbRows);
    int i, j;
    for (i = 0; i < NbRows; i++)
      {
	(cloogMatrix->p)[i] = &((cloogMatrix->p_Init)[i * NbColumns]);
      }// for (i = 0; i < NbRows; i++)
         
    for (i = 0; i < NbRows * NbColumns; i++)
      {
	cloog_int_init((cloogMatrix->p_Init)[i]);
	cloog_int_set_si((cloogMatrix->p_Init)[i], 0);    
      }// for (i = 0; i < NbRows; i++)

    // If T, N are parameters introduce T >= 1 and N >= 1

    int ColumnPosition;
    for (i = 0; i < NumberOfParameters; i++)
      {
	ColumnPosition = NbColumns - 1 - (NumberOfParameters - i); // 1 for constant. 
	cloog_int_set_si((cloogMatrix->p_Init)[i * NbColumns + ColumnPosition], 1);   
	cloog_int_set_si((cloogMatrix->p_Init)[i * NbColumns + 0], 1);    // The first column for inequality
	cloog_int_set_si((cloogMatrix->p_Init)[i * NbColumns + NbColumns - 1], -1);    // The last column for constant 1
      }// for (i = 0; i < NumberOfParameters; i++)
      
    //printf("Calling cloog_domain_from_cloog_matrix()\n");
    //printf("Number of parameters = %d\n", NbColumns - scopStatement->nb_iterators - 2);

    CloogState* cstate = cloog_state_malloc ();
    *cloogDomain =   cloog_domain_from_cloog_matrix(cstate, cloogMatrix, NbColumns - scopStatement->nb_iterators - 2);
    cloog_state_free (cstate);
    //printf("Returning from cloog_domain_from_cloog_matrix()\n");
  
    FILE *scopFile = fopen("CloogDomainFile", "a");
    cloog_domain_print_constraints(scopFile, *cloogDomain, 0);      
    fclose(scopFile);
      
    return;    
  }// static  void CreateDefaultCloogDomainForContext(scoplib_statement_p scopStatement, CloogDomain** cloogDomain, CloogProgram *cp, CloogOptions* coptions)
    
  int ConvexHullFormer::ScoplibToCloogDomains(scoplib_scop_p inputScop, CloogDomain*** cloogDomains)
  {
    scoplib_statement_p scopStatement = inputScop->statement;
    scoplib_statement_p temp = scopStatement;
    int n = 0;
        
    // Count the number of statements
    while (temp != NULL)
      {
	temp = temp->next;
	n++;
      }//while (temp != NULL)
         
    //printf("Number of statements = %d\n", n);

    int i = 0;
    *cloogDomains = (CloogDomain**) malloc(sizeof (CloogDomain*) * n);
    temp = scopStatement;
                                
    while (temp != NULL)
      {
	//printf("Calling ScoplibStatementToCloogDomain()\n");
	ScoplibStatementToCloogDomain(temp, &((*cloogDomains)[i]));
	//printf("Returning from ScoplibStatementToCloogDomain()\n");
	temp = temp->next;
	i++;
      }// while (temp != NULL)
        
        
    return n;
  }//int ScoplibToCloogDomain(scoplib_scop_p inputScop, CloogDomain* cloogDomains)
    

  int* ConvexHullFormer::cloog_domain_to_ints(CloogDomain* convex_domain)
  {

    // 1. Write the CloogDomain to a file
    // 2. Read the file
	
    // 1. Write the CloogDomain to a file
    string tempFileName = "temp_ConvexHull";
    FILE *tempConvexHullFile = fopen(tempFileName.c_str(), "w+");
    if (tempConvexHullFile == NULL)
      {
	assert("temp_ConvexHull file could not be opened");
      }// if (tempConvexHullFile == NULL)
	
    //printf("Calling cloog_domain_print_constraints()\n");
    cloog_domain_print_constraints(tempConvexHullFile, convex_domain, 0);
    //printf("Returning from cloog_domain_print_constraints()\n");
	      
    //printf("Writing to the temporary file complete\n");
    // 2. Read the file
		
    // The file should have the domain with respect to a single convex domain
    char str[50] = {0};		
    fseek(tempConvexHullFile, 0L, SEEK_SET); // bring the file pointer to the beginning of the file

    if (tempConvexHullFile == NULL)
      {
	assert("Error reading temp_ConvexHull file");
      }// if (tempConvexHullFile == NULL)


    if (tempConvexHullFile == NULL)
      {
	assert("Error reading temp_ConvexHull file");
      }// if (tempConvexHullFile == NULL)

    fscanf(tempConvexHullFile, "%s", str);
    int rows = atoi(str);

    if (tempConvexHullFile == NULL)
      {
	assert("Error reading temp_ConvexHull file");
      }// if (tempConvexHullFile == NULL)

    fscanf(tempConvexHullFile, "%s", str);
    int columns = atoi(str);
	      
    // Read the domain into an array
    int* domain = (int*) malloc(sizeof(int) * (rows * columns + 2));
    domain[0] = rows;
    domain[1] = columns;

    //printf("rows = %d, columns = %d\n", rows, columns);
    int i;
    int offset = 2;
    for (i = 0; i < rows * columns; i++)
      {
	fscanf(tempConvexHullFile, "%s", str);
	domain[i + offset] = atoi(str);
      }// for (i = 0; i < rows * columns; i++)
		
    fclose(tempConvexHullFile);

		
    // Remove the temporary file
    remove(tempFileName.c_str());
    return domain;

  }// int* cloog_domain_to_ints(CloogDomain* convex_domain)

  int* ConvexHullFormer::ExtractIteratorInequalities(int *domain, scoplib_scop_p inputScop)
  {
    int NumberOfParameters = inputScop->nb_parameters;
    //printf("NumberOfParameters = %d\n", NumberOfParameters);
    int rows = domain[0];
    int columns = domain[1];
    int NumberOfIterators = columns - 2 - NumberOfParameters;	

    //Pass 1. Determine the number of inequalities to be retained
    int i, j;
    bool IteratorFound;
    int NumberOfInequalities = 0;
    int offset = 2;

    for (i = 0; i < rows; i++)
      {
	IteratorFound = false;
	for (j = 1; j < columns - NumberOfParameters - 1; j++)
	  {
	    if (domain[i * columns + j + offset] != 0)
	      {
		IteratorFound = true;
	      }// if (domain[i * columns + j + offset] != 0)
	  }// for (j = 0; j < columns; j++)
			
	if (IteratorFound == true)
	  {
	    NumberOfInequalities++;

	    if (domain[i * columns + offset] == 0)
	      {
		// If the equation is an equality, need to make it both a lower bound and an upper bound.
		// Hence increment.				
		NumberOfInequalities++; 
	      }// if (domain[i * columns + offset] == 0)
	  }// if (IteratorFound == true)
      } // for (i = 0; i < rows; i++)

    // Pass 2. Add the inequality to a new data structure		
    int* inequalities = (int*) malloc(sizeof(int) * ((NumberOfInequalities * columns) + 2));
    int current = 0;
    inequalities[0] = NumberOfInequalities;
    //printf("NumberOfInequalities = %d\n", inequalities[0]);
    inequalities[1] = columns;

    for (i = 0; i < rows; i++)
      {
	IteratorFound = false;
	for (j = 1; j < columns - NumberOfParameters - 1; j++)
	  {
	    if (domain[i * columns + j + offset] != 0)
	      {
		IteratorFound = true;
	      }// if (domain[i * columns + j] != 0)
	  }// for (j = 0; j < columns; j++)
			
	if (IteratorFound == true)
	  {
	    inequalities[current * columns + offset] = 1; // Always an inequality
	    for (j = 1; j < columns; j++)
	      {
		inequalities[current * columns + j + offset] = domain[i * columns + j + offset];
	      }// for (j = 0; j < columns; j++)

	    current++;
				
	    // If the current equation is an equality, need to make it both a lower bound and an upper bound
	    if (domain[i * columns + offset] == 0)
	      {
		inequalities[current * columns + offset] = 1; // Always an inequality
		for (j = 1; j < columns; j++)
		  {
		    inequalities[current * columns + j + offset] = -1 * (domain[i * columns + j + offset]);
		  }// for (j = 0; j < columns; j++)

		current++;
	      }// if (domain[i * columns + offset] == 0)

	  }// if (IteratorFound == true)
      } // for (i = 0; i < rows; i++)
						
    return inequalities;	
  }// int* ExtractIteratorInequalities(int *domain)

  int* ConvexHullFormer::ArrangeDomainInOrder(int *domain, scoplib_scop_p inputScop)
  {
    // Put the inequalities corresponding the outer loop iterators before the inequalities corresponding to the inner loop iterators
    // Put the lower bound before the upperbound

    int NumberOfParameters = inputScop->nb_parameters;
    int rows = domain[0];
    int columns = domain[1];
    int NumberOfIterators = columns - 2 - NumberOfParameters;
    int* domainInOrder = (int*) malloc(sizeof(int) * ((rows * columns) + 2));
    domainInOrder[0] = domain[0];
    domainInOrder[1] = domain[1];
		
    int i1, i2, j, rowIndex, columnIndex, currentIterator, cellValue;
		
    currentIterator = 0; // Column position
    int offset = 2;
    int expectedCellValue;
    for (i1 = 0; i1 < rows; i1++)
      {
	if ((i1 % 2) == 0)
	  {
	    expectedCellValue = 1;
	    currentIterator++;
	  }// if ((i1 % 2) == 0)
	else
	  {
	    expectedCellValue = -1;
	  }//else

	rowIndex = -1; 
	columnIndex = -1;
		    
	// Find the lower bound corresponding to the currentIterator
	for (i2 = 0; i2 < rows; i2++)
	  {
	    for (j = 1; j < columns - NumberOfParameters - 1; j++) // iterate over all the iterators
	      {
		if (domain[i2 * columns + j + offset] != 0) // Find the rightmost iterator
		  {					
		    columnIndex = j;
		    cellValue = domain[i2 * columns + j + offset];
		  }// if (domain[i * columns + j] != 0)
	      }// for (j = 0; j < columns; j++)

	    if ((columnIndex == currentIterator) && (cellValue == expectedCellValue))
	      {
		rowIndex = i2;
		break;
	      }// if ((columnIndex == currentIterator) && (cellValue == expectedCellValue))
	  }// for (i2 = 0; i2 < rows; i2++)

	if (rowIndex == -1)
	  {
	    assert("The lower/upper bound for an iterator not found");
	  }// if (rowIndex == -1)

	// Copy the row to the new data structure
	for (j = 0; j < columns; j++)
	  {
	    domainInOrder[i1 * columns + j + offset] = domain[rowIndex * columns + j + offset];
	  }// for (j = 0; j < columns; j++)
      }// for (i1 = 0; i < rows; i++)


    // TODO : Need to decide what is to be done when an iterator happens to assume more than one lower bound / upper bound.
    domainInOrder[0] = NumberOfIterators * 2;

    // assert(domainInOrder[0] == domain[0]); // The number of rows should be the same
    // assert(domainInOrder[1] == domain[1]); // The number of columns should be the same
		
    return domainInOrder;
  }//static int* ArrangeDomainInOrder(int *domain, scoplib_scop_p inputScop)

  scoplib_scop_p ConvexHullFormer::cloog_domain_to_scoplib(int* convex_domain,
						scoplib_scop_p inputScop)
  {

    //printf("Inside cloog_domain_to_scoplib() accepting integer input\n");
    int l1, l2;
    //printf("Rows = %d, Columns = %d\n", convex_domain[0], convex_domain[1]);
    for (l1 = 0; l1 < convex_domain[0]; l1++)
      {
	for (l2 = 0; l2 < convex_domain[1]; l2++)
	  {
	    //printf("%d ", convex_domain[l1 * convex_domain[1] + l2 + 2]);
	  }
		
	//printf("\n");
      }
	
    // Populate the parameter names, iterator names, lower and upper bounds for the iterators
    scoplib_scop_p outputScop = (scoplib_scop_p) malloc(sizeof(struct scoplib_scop));
    outputScop->nb_parameters = inputScop->nb_parameters;
    int i;
    outputScop->parameters = (char**) malloc(sizeof(char*) * inputScop->nb_parameters);		
    for (i = 0; i <  inputScop->nb_parameters; i++)
      {
	outputScop->parameters[i] = (char*) malloc(sizeof(char) * (strlen(inputScop->parameters[i]) + 1));
	strcpy(outputScop->parameters[i], inputScop->parameters[i]);
      }// for (i = 0; i <  inputScop->nb_parameters; i++)

    outputScop->statement = (scoplib_statement_p) malloc(sizeof(struct scoplib_statement));
    outputScop->statement->nb_iterators = inputScop->statement->nb_iterators;

    outputScop->statement->iterators = (char**) malloc(sizeof(char*) * inputScop->statement->nb_iterators);		

    for (i = 0; i < inputScop->statement->nb_iterators; i++)
      {
	outputScop->statement->iterators[i] = (char*) malloc(sizeof(char) * (strlen(inputScop->statement->iterators[i]) + 1) );
	strcpy(outputScop->statement->iterators[i], inputScop->statement->iterators[i]);
      }// for (i = 0; i < inputScop->statement->nb_iterators; i++)

    outputScop->statement->domain = (scoplib_matrix_list_p) malloc(sizeof(struct scoplib_matrix_list));
    outputScop->statement->domain->elt = (scoplib_matrix_p) malloc(sizeof(struct scoplib_matrix));
    outputScop->statement->domain->next = NULL;
    outputScop->statement->next = NULL;

    outputScop->statement->domain->elt->NbRows = convex_domain[0];
    outputScop->statement->domain->elt->NbColumns = convex_domain[1];
    int rows = convex_domain[0];
    int columns = convex_domain[1];
    outputScop->statement->domain->elt->p_Init = (scoplib_int_t*) malloc(sizeof(scoplib_int_t) * rows * columns);
    int offset = 2;
    for (i = 0; i < rows * columns; i++)
      {
	(outputScop->statement->domain->elt->p_Init)[i] = convex_domain[i + offset];
      }// for (i = 0; i < statement->domain->elt->NbRows * statement->domain->elt->NbColumns; i++)


    outputScop->statement->domain->elt->p = (scoplib_int_t **) malloc(sizeof(scoplib_int_t *) * rows);
    for (i = 0; i < rows; i++)
      {
	(outputScop->statement->domain->elt->p)[i] = &((outputScop->statement->domain->elt->p_Init)[i * columns]);
      }// for (i = 0; i < statement->domain->elt->NbRows; i++)

    //printf("From memory\n");
    for (l1 = 0; l1 < rows; l1++)
      {
	for (l2 = 0; l2 < columns; l2++)
	  {
	    //printf("%d ", (outputScop->statement->domain->elt->p_Init)[l1 * columns + l2]);
	  }
		
	//printf("\n");
      }

    return outputScop;
  }// static scoplib_scop_p cloog_domain_to_scoplib(int* convex_domain, scoplib_scop_p inputScop)
	
  scoplib_scop_p ConvexHullFormer::cloog_domain_to_scoplib(CloogDomain* convex_domain, scoplib_scop_p inputScop)
  {	
    //1. Convert the CloogDomain into array of integers
    int* domain = cloog_domain_to_ints(convex_domain);


    //printf("Inside cloog_domain_to_scoplib()\n");
    //printf("Rows = %d, Columns = %d\n", domain[0], domain[1]);
    int l1, l2;		
    for (l1 = 0; l1 < domain[0]; l1++)
      {
	for (l2 = 0; l2 < domain[1]; l2++)
	  {
	    //printf("%d ", domain[l1 * domain[1] + l2 + 2]);
	  }
		
	//printf("\n");
      }

    // 2. Convert the array of integers into scoplib structure
    // Retain inequalities concerning only the iterators
    int* iteratorInequalities = ExtractIteratorInequalities(domain, inputScop);
    int* iteratorInequalitiesInOrder = ArrangeDomainInOrder(iteratorInequalities, inputScop);

    //printf("Number of inequalities = %d, columns = %d\n", iteratorInequalitiesInOrder[0], iteratorInequalitiesInOrder[1]);
    scoplib_scop_p outputScop = cloog_domain_to_scoplib(iteratorInequalitiesInOrder, inputScop);		
    return outputScop;
  }// static scoplib_scop_p cloog_domain_to_scoplib(CloogDomain* convex_domain)

  scoplib_scop_p ConvexHullFormer::CreateConvexHull(scoplib_scop_p inputScop)
  {
    //1. Convert scoplib to CloogMatrix->CloogDomain.
    //2. Call cloog_loop_get_convex_hull
    //3. Convert the CloogDomain back to ScopLib
      
    //1. Convert scoplib to CloogMatrix->CloogDomain.
    CloogDomain** cloogDomains = NULL;
    //printf("Calling ScoplibToCloogDomains()\n");
    int NumberOfDomains = ScoplibToCloogDomains(inputScop, &cloogDomains);
    //printf("Returning from ScoplibToCloogDomains()\n");
      
    //2. Form the convex hull - union of domains
    CloogDomain* convex_domain = NULL;
    CloogDomain* temp_domain = NULL;
      
    //printf("NumberOfDomains = %d\n", NumberOfDomains);
    if (NumberOfDomains == 1)
      {  
	convex_domain = cloogDomains[0];
      }// if (NumberOfDomains == 1)
    else if (NumberOfDomains > 1)
      {
	//printf("Calling cloog_domain_union()\n");
	convex_domain = cloog_domain_union(cloogDomains[0], cloogDomains[1]);
	//printf("Returning from cloog_domain_union()...\n");
            
	int i;
	for (i = 2; i < NumberOfDomains; i++)
	  {
	    temp_domain = cloog_domain_union(convex_domain, cloogDomains[i]) ;
	    convex_domain = temp_domain;
	  }// for (i = 2; i < NumberOfDomains; i++)
      }// else
      

    temp_domain = cloog_domain_simple_convex(convex_domain);
    convex_domain = temp_domain;
    
//     FILE *scopFile = fopen("ConvexHull", "w");
//     cloog_domain_print_constraints(scopFile, convex_domain, 1);
//     fclose(scopFile);


    //3. Convert the CloogDomain back to ScopLib
    // Convert convex_domain to ScopLib

    //printf("Calling cloog_domain_to_scoplib\n");
    scoplib_scop_p convex_domain_scoplib = cloog_domain_to_scoplib(convex_domain, inputScop);	      
    //printf("Returning from cloog_domain_to_scoplib\n");
                
    return convex_domain_scoplib;

  }//scoplib_scop_p CreateConvexHull(scoplib_scop_p inputScop)

