/*
 * TileLoopInserter.cpp: This file is part of the Parametric Tiling project.
 *
 * Parametric Tiling: A CLAST-to-CLAST parametric tiling software
 *
 * Copyright (C) 2011 Sanket Tavargeri
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 3
 * of the License, or (at your option) any later version.
 *
 * The complete GNU Lesser General Public Licence Notice can be found
 * as the `COPYING.LESSER' file in the root directory.
 *
 * Author:
 * Sanket Tavargeri <sanket.tavargeri@gmail.com>
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include<string>
#include<iostream>
#include<vector>
#include <set>
#include<sstream>


# include <ptile/TileLoopInserter.hpp>
# include <ptile/FullTileMaker.hpp>

  s_past_node_t* TileLoopInserter::CombineExpressions(s_past_node_t* expr1, s_past_node_t* expr2,
				    int type)
  {
    // min = 0;
    // max = 1;

    cs_past_node_type_t* minmax;
    switch (type) {
    case 0:
      minmax = past_min;
      break;
    case 1:
      minmax = past_max;
      break;
    default:
      return NULL;
    }

    return &past_binary_create(minmax, expr1, expr2)->node;
  }// struct ext_clast_expr* CombineExpressions(struct ext_clast_expr * expr1, struct ext_clast_expr * expr2, int type)


void TileLoopInserter::CombineCorrespondingLoops (s_past_node_t* loop1, s_past_node_t* loops2)
{
	// If the iterator names match, combine lower bound and upper bound expressions from loop1 & loop2.

	// Iterate through all the loops. Whenever a 'for' loop encountered, search for the loop with the same
	// iterator name in 'w_body'. Combine the lower bound and upper bound expressions
	for ( ; loop1; loop1 = loop1->next)
	{
	  if (PAST_NODE_IS_A(loop1, past_for) || PAST_NODE_IS_A(loop1, past_parfor))
	  {
	
                  PAST_DECLARE_TYPED(for, tempfor1, loop1);
		  char* iteratorName = (char*) ((s_past_variable_t*) tempfor1->iterator)->symbol->data;
		  s_past_node_t* matchingLoop = ReturnForLoopCorrespondingToIterator (iteratorName, loops2, 0);

		  if (matchingLoop != NULL)
		  {
		    PAST_DECLARE_TYPED(for, tempfor2, matchingLoop);
		    s_past_node_t* LB1 =
		      ((s_past_binary_t*)tempfor1->init)->rhs;
		    s_past_node_t* LB2 =
		      ((s_past_binary_t*)tempfor2->init)->rhs;
		    //printf("Calling CombineExpressions()\n");
		    s_past_node_t* minmaxexpr =
		      CombineExpressions(LB1, LB2, 1);
		    minmaxexpr->parent = (((s_past_binary_t*)tempfor1->init)->rhs)->parent;
		    ((s_past_binary_t*)tempfor1->init)->rhs = minmaxexpr;
		    //past_deep_free(((s_past_binary_t*)tempfor2->init)->lhs);
		    //printf("Returned from CombineExpressions()\n");

		    s_past_node_t* UB1 =
		      ((s_past_binary_t*)tempfor1->test)->rhs;
		    s_past_node_t* UB2 =
		      ((s_past_binary_t*)tempfor2->test)->rhs;
		    minmaxexpr = CombineExpressions(UB1, UB2, 0);
		    //printf("Calling CombineExpressions()\n");
		    minmaxexpr->parent = (((s_past_binary_t*)tempfor1->test)->rhs)->parent;
		    ((s_past_binary_t*)tempfor1->test)->rhs = minmaxexpr;
		    //past_deep_free(((s_past_binary_t*)tempfor2->test)->lhs);

		    //printf("Returned from CombineExpressions()\n");

		    if (tempfor1->body != NULL && PAST_NODE_IS_A(tempfor1->body,
								 past_for))
		      {
			CombineCorrespondingLoops (tempfor1->body, loops2);
		      }// if (tempfor1->body != NULL && PAST_NODE_IS_A(tempfor1->body, past_for))
		  }// if (matchingLoop != NULL)
 	  }// if (PAST_NODE_IS_A(loops, past_for) || PAST_NODE_IS_A(loops, past_parfor))
	}// for ( ; loops; loops = loops->next)

}// void CombineCorrespondingLoops (s_past_node_t* loop1, s_past_t* loops2)

/*
This function is to be called if the tiled loop structure is to mirror the original loop structure.
In which case, the rsfme loops and the tile loops are combined in the order they appear in 'tileLoops'.
*/
  s_past_node_t* TileLoopInserter::CombineTileLoopsAndRSFMELoopsInOrder (s_past_node_t* rsfmeLoops, s_past_node_t* tileLoops)
  {

    s_past_node_t* combinedTileLoops = NULL;
    // Assumption: If tileLoops has n for loops then rsfmeLoops has n + 1 loops

    s_past_node_t* w_body = NULL;

    if (rsfmeLoops != NULL && PAST_NODE_IS_A(rsfmeLoops, past_for))
    {
	w_body = ((s_past_for_t*) rsfmeLoops)->body;
	((s_past_for_t*) rsfmeLoops)->body = tileLoops;
	combinedTileLoops = rsfmeLoops;

	// Combine the corresponding expressions here
  	// printf ("Calling CombineCorrespondingLoops\n");
	CombineCorrespondingLoops (tileLoops, w_body);
	// printf("Returned from CombineCorrespondingLoops\n");
    }// if (rsfmeLoops != NULL)
   else
    {
	// rsfmeLoops shouldn't have been NULL. assert (0)
	assert (0);
    }//else

    return combinedTileLoops;
  }// struct ext_clast_for* CombineTileLoopsAndRSFMELoops(struct TileLoops tileLoopsStruct)


  s_past_node_t* TileLoopInserter::CombineTileLoopsAndRSFMELoops(struct TileLoops tileLoopsStruct)
  {

    s_past_node_t* combinedTileLoops = NULL;
    // Assumption: If tileLoopsStruct.tileLoops has n for loops then tileLoopsStruct.rsfmeLoops has n + 1 loops

    if (tileLoopsStruct.rsfmeLoops != NULL)
      {
	combinedTileLoops = tileLoopsStruct.rsfmeLoops;
	// Modify rsmfeLoops
	PAST_DECLARE_TYPED(for, tlsr, tileLoopsStruct.rsfmeLoops);
	s_past_node_t* tempStmt = tlsr->body;
	if (tempStmt != NULL)
	  {
	    if (PAST_NODE_IS_A(tempStmt, past_for))
	      {
		s_past_for_t* tempfor1 = (s_past_for_t*) tempStmt;
		PAST_DECLARE_TYPED(for, tempfor2, tileLoopsStruct.tileLoops);

		// Iterate through all the for loops
		while (tempfor1 != NULL && tempfor2 != NULL)
		  {
		    s_past_node_t* LB1 =
		      ((s_past_binary_t*)tempfor1->init)->rhs;
		    s_past_node_t* LB2 =
		      ((s_past_binary_t*)tempfor2->init)->rhs;
		    //printf("Calling CombineExpressions()\n");
		    s_past_node_t* minmaxexpr =
		      CombineExpressions(LB1, LB2, 1);
		    minmaxexpr->parent = (((s_past_binary_t*)tempfor1->init)->rhs)->parent;
		    ((s_past_binary_t*)tempfor1->init)->rhs = minmaxexpr;
		    //past_deep_free(((s_past_binary_t*)tempfor2->init)->lhs);
		    //printf("Returned from CombineExpressions()\n");

		    s_past_node_t* UB1 =
		      ((s_past_binary_t*)tempfor1->test)->rhs;
		    s_past_node_t* UB2 =
		      ((s_past_binary_t*)tempfor2->test)->rhs;
		    minmaxexpr = CombineExpressions(UB1, UB2, 0);
		    //printf("Calling CombineExpressions()\n");
		    minmaxexpr->parent = (((s_past_binary_t*)tempfor1->test)->rhs)->parent;
		    ((s_past_binary_t*)tempfor1->test)->rhs = minmaxexpr;
		    //past_deep_free(((s_past_binary_t*)tempfor2->test)->lhs);

		    //printf("Returned from CombineExpressions()\n");

		    if (tempfor1->body != NULL && PAST_NODE_IS_A(tempfor1->body,
								 past_for))
		      {
			tempfor1 = (s_past_for_t*) tempfor1->body;
			tempfor2 = (s_past_for_t*) tempfor2->body;
		      }// if (EXT_CLAST_STMT_IS_A(tempfor1->body, ext_stmt_for))
		    else
		      {
			tempfor1 = NULL;
			tempfor2 = NULL;
		      }// else
		  }// while (tempfor1 != NULL && tempfor2 != NULL)

	      }// if (EXT_CLAST_STMT_IS_A(tempStmt, ext_stmt_for))
	  }// if (tempStmt != NULL)

      }// if (tileLoopsStruct.rsfmeLoops != NULL)

    return combinedTileLoops;
  }// struct ext_clast_for* CombineTileLoopsAndRSFMELoops(struct TileLoops tileLoopsStruct)


/* Number the for loops.
 for (i ) - 1
   for (j ) - 2
   for (j ) -3

The function assigns the parameter 'id' to the first loop encountered and a value > id to the for loops hence encountered.
*/



static
void metainfoprint (s_past_node_t* node, FILE* out)
{
  if (node->metainfo)
    fprintf (out, "// %d\n", *((int*) node->metainfo));
}

// Find the tile loop for the given iteratorName, make a copy of the for loop with bounds and no body and return.
// When 'clone = 1', a copy of the asked loop is made and returned, otherwise, a pointer to the existing for loop is returned
s_past_node_t* TileLoopInserter::ReturnForLoopCorrespondingToIterator (char *iteratorName, s_past_node_t* loops, int clone)
{
	for ( ; loops; loops = loops->next)
	{
	  if (PAST_NODE_IS_A(loops, past_for) || PAST_NODE_IS_A(loops, past_parfor))
	  {

	  s_past_variable_t* pv = (s_past_variable_t*) ((s_past_for_t*) loops)->iterator;
	  if (pv->symbol->is_char_data)
	    {
		if (strcmp (iteratorName, (char*) pv->symbol->data) == 0)
		{	

			if (clone == 1)
			{
			// If the strings matched, temporarily make the body and sibling of the for loop null and make a clone, restore the body and sibling and return the clone
			s_past_node_t* child = ((s_past_for_t*) loops)->body;
			s_past_node_t* sibling = loops->next;
	
			// Make the child and the sibling NULL
			((s_past_for_t*) loops)->body = NULL;
			loops->next = NULL;

				// Make a copy of the 'for' loop
				s_past_node_t* returnLoop = past_clone (loops);
			
				// Restore the child and the sibling
				((s_past_for_t*) loops)->body = child;
				loops->next = sibling;
		
				return returnLoop;
			}
			else
			{
				return loops;
			}// else
		}// if (strcmp (iteratorName, (char*) pv->symbol->data) == 0)
		else
		{	
			// Search in the inner 'for' loops
			s_past_node_t* returnValue = ReturnForLoopCorrespondingToIterator (iteratorName, ((s_past_for_t*) loops)->body, clone);
			if (returnValue != NULL)
			{
				return returnValue;
			}// if (returnValue != NULL)
		
		}//else
	    } //if (pv->symbol->is_char_data)
	  else
	    {
		// There is no way to compare the value of the iterator. Therefore assert (0) here
	    	assert (0);
	    }// else


	  }// if (PAST_NODE_IS_A(loops, past_for) || PAST_NODE_IS_A(loops, past_parfor))
	}// for ( ; loops; loops = loops->next)

	return NULL;
}// s_past_node_t* TileLoopInserter::ReturnForLoopCorrespondingToIterator (char *iteratorName, s_past_node_t* loops)

/* Build the tile loops mirroring the point loop structure
  1. Get the iterator names from visiting the point loops
  2. For each point loop visited, build the corresponding tile loop
*/
s_past_node_t* TileLoopInserter::BuildTileLoopsFromPointLoopStructure (s_past_node_t* pointLoops, s_past_node_t* tileLoops, s_ptile_options_t* ptileOptions)
{
   s_past_node_t* head = NULL;
    for ( ; pointLoops; pointLoops = pointLoops->next)
      {
	// pointLoops->metainfo != NULL condition is to check if the 'for' loop has been numbered. If it's not then it's a one-time-loop 
	// and do not generate a tile loop corresponding to otl.
	if ((PAST_NODE_IS_A(pointLoops, past_for) || PAST_NODE_IS_A(pointLoops, past_parfor)) && pointLoops->metainfo != NULL)
	  {

		PAST_DECLARE_TYPED(for, localPointLoops, pointLoops);
		char *iteratorName = NULL;

		  if (localPointLoops->iterator->symbol->is_char_data)
		  {
		    iteratorName = (char*) (localPointLoops->iterator->symbol->data);
		  }// if (localPointLoops->iterator->symbol->is_char_data)
		  else
		  {
			assert (0);
		  }//else

		// Form the tile iterator corresponding to the point iterator
		char *tileIteratorName = (char*) malloc (sizeof(char) * strlen (iteratorName) + (ExpressionLibrary::GetTileIteratorSuffix(1, 1)).length() + 1);
		strcpy (tileIteratorName, iteratorName);
		strcat (tileIteratorName, (ExpressionLibrary::GetTileIteratorSuffix(1, 1)).c_str());
		
		// Form a 'for' loop for the tile iterator
		// printf ("Calling ReturnForLoopCorrespondingToIterator()\n");
		s_past_node_t* forLoop = ReturnForLoopCorrespondingToIterator (tileIteratorName, tileLoops, 1);
		// printf ("Returned from ReturnForLoopCorrespondingToIterator()\n");

		// Let this tile loop also have the same loop number as its corresponding point loop
		forLoop->metainfo = pointLoops->metainfo;

		if (forLoop != NULL)
		{
			// Do the same for inner loops
			((s_past_for_t*) forLoop)->body = BuildTileLoopsFromPointLoopStructure (localPointLoops->body, tileLoops, ptileOptions);
		}// if (forLoop != NULL)
		else
		{
			printf ("[Ptile] Tile loop for %s not found\n", tileIteratorName);
			assert (0);
		}// else

		if (head == NULL)
		{
		  head = forLoop;
		}// if (head == NULL)
		else
		{			
		  (GeneralUtilityClass::GetLastNode(head))->next = forLoop;
		} //else
	    
		 
	  }// if (PAST_NODE_IS_A(pointLoops, past_for) || PAST_NODE_IS_A(pointLoops, past_parfor))
	 else if ((PAST_NODE_IS_A(pointLoops, past_for) || PAST_NODE_IS_A(pointLoops, past_parfor)) && pointLoops->metainfo == NULL)
	  {
		PAST_DECLARE_TYPED(for, localPointLoops, pointLoops);
		s_past_node_t* temp = BuildTileLoopsFromPointLoopStructure (localPointLoops->body, tileLoops, ptileOptions);

		if (head == NULL)
		{
		  head = temp;
		}// if (head == NULL)
		else
		{			
		  (GeneralUtilityClass::GetLastNode(head))->next = temp;
		} //else

	  }// else if ((PAST_NODE_IS_A(pointLoops, past_for) || PAST_NODE_IS_A(pointLoops, past_parfor)) && pointLoops->metainfo == NULL)
         else if (PAST_NODE_IS_A(pointLoops, past_affineguard))
	  {
	
		PAST_DECLARE_TYPED(affineguard, localPointLoops, pointLoops);
		s_past_node_t* temp = BuildTileLoopsFromPointLoopStructure (localPointLoops->then_clause, tileLoops, ptileOptions);

		if (head == NULL)
		{
		  head = temp;
		}// if (head == NULL)
		else
		{			
		  (GeneralUtilityClass::GetLastNode(head))->next = temp;
		} //else
	
	  }// else if (PAST_NODE_IS_A(pointLoops, past_affineguard))
      }// for ( ; pointLoops; pointLoops = pointLoops->next)


	return head;
}// void TileLoopInserter::ConvertPointLoopsToTileLoops (s_past_node_t* pointLoops, s_past_node_t* tileLoops)


// Form the task body with the loops in the given 'path' and the cloog statement and return the task body.
s_past_node_t* TileLoopInserter::GetTaskBodyForPath (s_past_node_t* pointLoops, string path, string IteratorsInPath)
{
    s_past_node_t* pointLoopsIterator = pointLoops;
    s_past_node_t* head = NULL;
    for ( ; pointLoopsIterator; pointLoopsIterator = pointLoopsIterator->next)
      {
	if (PAST_NODE_IS_A(pointLoopsIterator, past_for) || PAST_NODE_IS_A(pointLoopsIterator, past_parfor))
	  {

		// append the current 'for' loop number to the path
	     bool found = false;
	     bool otl = false;
             if (pointLoopsIterator->metainfo != NULL)
	      {
		string localPath = "," + GeneralUtilityClass::NumberToString (*((int*) pointLoopsIterator->metainfo)) + ",";
 		// cout <<"localPath: "<< localPath<<"path: "<<path<<"IteratorsInPath: "<<IteratorsInPath<<endl;

		// Search for the current 'for' loop number in the input 'path'. If present, set found to true
		if (path.find (localPath) != string::npos)
		{
			found = true;
		}// if (path.find (localPath) != string::npos)
	      }// if (pointLoopsIterator->metainfo != NULL)
	     else if (pointLoopsIterator->metainfo == NULL)
	      {
		// metainfo is NULL in the case of otls. 
		otl = true;

		// Search if the otl loop iterator has the corresponding tile iterator in the loop nest. If so, include this otl, else do not.

		// Form the tile iterator corresponding to the point iterator
		char *iteratorName = (char*) (((s_past_for_t*)pointLoopsIterator)->iterator->symbol->data);
		string iteratorNameString (iteratorName);
		iteratorNameString = iteratorNameString + ExpressionLibrary::GetTileIteratorSuffix(1, 1);
		iteratorNameString = "," + iteratorNameString + ",";	

		if (IteratorsInPath.find (iteratorNameString) != string::npos)
		{		
			found = true;
		}//if (IteratorsInPath.find (iteratorNameString) != string::npos)

		// Now set the metainfo to -1 so that this otl is not included in any subsequent task_bodies.
		int* negativeOne = (int*) malloc (sizeof (int));
		*negativeOne = -1;
		pointLoopsIterator->metainfo = negativeOne;			
	      }// else if (pointLoopsIterator->metainfo == NULL)
		
		if (found == true)
		{
			// This loop is to be included
			PAST_DECLARE_TYPED(for, currentLoop, pointLoopsIterator);

			// Backup
			s_past_node_t* body = currentLoop->body;
			s_past_node_t* next = ((s_past_node_t*) currentLoop)->next;
			
			// Clone					
			currentLoop->body = NULL;
			((s_past_node_t*) currentLoop)->next = NULL;
			s_past_node_t* currentLoopClone = past_clone ((s_past_node_t*) currentLoop);

			if (past_node_is_a (body, past_cloogstmt))
			{
// 				printf ("A statement is found in the body of the included loop\n");
				((s_past_for_t*) currentLoopClone)->body = body;
				// printf ("stmt_number = %d\n", ((struct past_cloogstmt_t*) body)->stmt_number);
			}// if (PAST_NODE_IS_A (body, cloogstmt))
			else
			{
				// Call recursively
				((s_past_for_t*) currentLoopClone)->body = GetTaskBodyForPath (body, path, IteratorsInPath);
			}// else

			// Restore
			currentLoop->body = body;
			((s_past_node_t*) currentLoop)->next = next;


			if (head == NULL)
			{
				head = currentLoopClone;
			}// if (head == NULL)
			else
			{
				(GeneralUtilityClass::GetLastNode(head))->next = currentLoopClone;
			}// else
		}// if (found == true)
		else if (found == false && otl == true)
		{
			// Look inside of otl s.
// 			printf ("Looking inside otls\n");
			s_past_node_t* t = NULL;

		    if (PAST_NODE_IS_A(((s_past_for_t*) pointLoopsIterator)->body, past_for) || PAST_NODE_IS_A(((s_past_for_t*) pointLoopsIterator)->body, past_parfor))
		    {		 
			t = GetTaskBodyForPath (((s_past_for_t*) pointLoopsIterator)->body, path, IteratorsInPath);
		    }// if (PAST_NODE_IS_A(((s_past_for_t*) pointLoopsIterator)->body, past_for) || PAST_NODE_IS_A(((s_past_for_t*) pointLoopsIterator)->body, past_parfor))
		    else if (past_node_is_a (((s_past_for_t*) pointLoopsIterator)->body, past_cloogstmt))
		    {
			t = ((s_past_for_t*) pointLoopsIterator)->body;
// 			printf ("stmt_number = %d\n", ((struct past_cloogstmt_t*) t)->stmt_number);
// 			printf ("A statement is found in the body of the excluded loop\n");
		    }// else if (past_node_is_a (((s_past_for_t*) pointLoopsIterator)->body, past_cloogstmt))
		
			if (t != NULL)
			{

				if (head == NULL)
				{
					head = t;
				}// if (head == NULL)
				else
				{
					(GeneralUtilityClass::GetLastNode(head))->next = t;
				}// else

			}// if (t != NULL)
		}// else if (found == false && otl == true)
		else if (pointLoopsIterator->metainfo != NULL && (*((int*) pointLoopsIterator->metainfo)) == -1)
		{
			// Look into the body of the 'for' loop
		   s_past_node_t* t = NULL;
		    if (PAST_NODE_IS_A(((s_past_for_t*) pointLoopsIterator)->body, past_for) || PAST_NODE_IS_A(((s_past_for_t*) pointLoopsIterator)->body, past_parfor))
		    {		 
			t = GetTaskBodyForPath (((s_past_for_t*) pointLoopsIterator)->body, path, IteratorsInPath);
		    }// if (PAST_NODE_IS_A(((s_past_for_t*) pointLoopsIterator)->body, past_for) || PAST_NODE_IS_A(((s_past_for_t*) pointLoopsIterator)->body, past_parfor))

			if (t != NULL)
			{

				if (head == NULL)
				{
					head = t;
				}// if (head == NULL)
				else
				{
					(GeneralUtilityClass::GetLastNode(head))->next = t;
				}// else

			}// if (t != NULL)

		}// else if (pointLoopsIterator->metainfo != NULL && (*((int*) pointLoopsIterator->metainfo)) == -1)
	  }// if (PAST_NODE_IS_A(pointLoopsIterator, past_for) || PAST_NODE_IS_A(pointLoopsIterator, past_parfor))
	/*
	else if (past_node_is_a (pointLoopsIterator, past_cloogstmt))
	{

		// If this is a cloogstmt, include this in the returned node.

			if (head == NULL)
			{
				head = pointLoopsIterator;
			}// if (head == NULL)
			else
			{
				(GeneralUtilityClass::GetLastNode(head))->next = pointLoopsIterator;
			}// else		

		// Fast forward to the end of the chain
		pointLoopsIterator = (GeneralUtilityClass::GetLastNode(pointLoopsIterator));
// 		printf ("A statement is found stand alone\n");
	}// else if (past_node_is_a (pointLoopsIterator, past_cloogstmt))
	*/
	else if (past_node_is_a (pointLoopsIterator, past_affineguard))
	{
		// Look inside if conditionals
		s_past_node_t* t = GetTaskBodyForPath (((s_past_affineguard_t*) pointLoopsIterator)->then_clause, path, IteratorsInPath);

			if (t != NULL)
			{
				// If something is present inside the if conditionals, include the if conditional also
				// Back-up
				s_past_node_t* body = ((s_past_affineguard_t*) pointLoopsIterator)->then_clause;
				s_past_node_t* next = pointLoopsIterator->next;				

				// Make a copy
				((s_past_affineguard_t*) pointLoopsIterator)->then_clause = NULL;
				pointLoopsIterator->next = NULL;
				s_past_node_t* ifConditionalClone = past_clone (pointLoopsIterator);
				((s_past_affineguard_t*) ifConditionalClone)->then_clause = t;

				// Restore
				((s_past_affineguard_t*) pointLoopsIterator)->then_clause = body;
				pointLoopsIterator->next = next;

				if (head == NULL)
				{
					head = ifConditionalClone;
				}// if (head == NULL)
				else
				{
					(GeneralUtilityClass::GetLastNode(head))->next = ifConditionalClone;
				}// else

			}// if (t != NULL)

	}// else if (past_node_is_a (pointLoopsIterator, past_affineguard))

	
      }// for ( ; pointLoopsIterator; pointLoopsIterator = pointLoopsIterator->next)

	
	return head;
}// s_past_node* GetTaskBodyForPath (s_past_node_t* pointLoops, string path)



/*
Input: The tile loops and point loop nests, each of whose 'for' loop is assigned a unique number and the 
	matching tile loops and point loops having the same number.

The function inserts the point loops at appropriate places in the tile loop nest.

*/
  void TileLoopInserter::InsertPointLoopsInsideTileLoops (s_past_node_t* tileLoops, s_past_node_t* pointLoops, string path, string IteratorsInPath, s_ptile_options_t* ptileOptions)
  {

	// Traverse the tile loops to get to the innermost loop. Along the way, note the 'path' string - sequence of numbers from outermost to innermost for that innermost task body.
    s_past_node_t* tileLoopsIterator = tileLoops;
    for ( ; tileLoopsIterator; tileLoopsIterator = tileLoopsIterator->next)
      {
	if (PAST_NODE_IS_A(tileLoopsIterator, past_for) || PAST_NODE_IS_A(tileLoopsIterator, past_parfor))
	  {

	   PAST_DECLARE_TYPED(for, tileLoopsInstance, tileLoopsIterator);		

	   string newPath = path;	
	   string newIteratorsInPath = IteratorsInPath;

	   // Check if metainfo is not NULL. It will be NULL for cases like wavefront loop
	    if (tileLoopsIterator->metainfo != NULL)
	    {
		// append the current 'for' loop number to the path		
		newPath.append ("," + GeneralUtilityClass::NumberToString (*((int*) tileLoopsIterator->metainfo)) + ",");
		string iteratorName ((char*) ((s_past_variable_t*) tileLoopsInstance->iterator)->symbol->data);
		newIteratorsInPath.append ("," + iteratorName + ",");	
	    }// if (tileLoopsIterator->metainfo != NULL)
		
		if (tileLoopsInstance->body != NULL)
		{				
			InsertPointLoopsInsideTileLoops (tileLoopsInstance->body, pointLoops, newPath, newIteratorsInPath, ptileOptions);		
		}// if (tileLoopsInstance->body != NULL)
		else
		{
			// We have reached the innermost loop. Need to insert the task-body here
//			cout<<"newPath: "<<newPath<<endl;
// 			cout <<"newIteratorsInPath: "<<newIteratorsInPath<<endl;

//			FILE* fp = fopen ("SearchTree.c", "a");
//			fprintf (fp, "newPath: %s, newIteratorsInPath: %s in\n", newPath.c_str(), newIteratorsInPath.c_str());
//			past_pprint_metainfo (fp, pointLoops, metainfoprint);
//			fclose (fp);

			s_past_node_t* taskBody = GetTaskBodyForPath (pointLoops, newPath, newIteratorsInPath);			
			

/*{
   FILE *fp = fopen ("FullTiles.c", "a");
   fprintf (fp, "\n****************************\n");
   past_pprint (fp, taskBody);
   fprintf (fp, "\n****************************\n");
   fclose (fp);
}*/

			// Perform Full tile separation if fullTileSeparation option is set
			s_past_node_t* fullTiles = NULL;
			// ptileOptions->fullTileSeparation = 0;
			if (ptileOptions->fullTileSeparation == 1)
			  {
			   // printf("Calling BuildFullTiles\n");
			    fullTiles = FullTileMaker::FullTileMakerDriver(taskBody, ptileOptions);

/*{
   FILE *fp = fopen ("FullTiles.c", "a");
   fprintf (fp, "\n****************************\n");
   past_pprint (fp, fullTiles);
   fprintf (fp, "\n****************************\n");
   fclose (fp);
}*/			    			    

			   // printf("Returning from BuildFullTiles\n");
			  }// if (ptileOptions->fullTileSeparation == 1)

			// Replace the bounds of the point loop with mins and maxes appropriately
			ModifyBoundsOfPointLoops (taskBody);
	
			if (ptileOptions->fullTileSeparation == 1 &&
			    fullTiles != NULL)
			  {
			    // Build the if then list
			    s_past_node_t* temp = fullTiles;
			    taskBody = FullTileMaker::BuildIfElseList (fullTiles, taskBody);
				/*{
					FILE *fp = fopen ("FullTiles.c", "a");
					fprintf (fp, "\n****************************\n");
					past_pprint (fp, taskBody);
					fprintf (fp, "\n****************************\n");
					fclose (fp);
				}*/

			  }// if (ptileOptions.fullTileSeparation == 1)

			tileLoopsInstance->body = taskBody;
		}// else
	   
	  }// if (PAST_NODE_IS_A(tileLoopsIterator, past_for) || PAST_NODE_IS_A(tileLoopsIterator, past_parfor))

      }// for ( ; tileLoopsIterator; tileLoopsIterator = tileLoopsIterator->next)

  }// void InsertPointLoopsInsideTileLoops (s_past_node_t* tileLoops, s_past_node_t* pointLoops, s_ptile_options_t* ptileOptions)

 void TileLoopInserter::InsertPointLoopsInsideTileLoops (s_past_node_t* tileLoops, s_past_node_t* pointLoops, s_ptile_options_t* ptileOptions)
  {
	return InsertPointLoopsInsideTileLoops (tileLoops, pointLoops, "", "", ptileOptions);
  }// void TileLoopInserter::InsertPointLoopsInsideTileLoops (s_past_node_t* tileLoops, s_past_node_t* pointLoops)


// Distributes the statements - doesn't fuse
  s_past_node_t*
  TileLoopInserter::InsertTileLoopsInOriginalLoopOrder(s_past_node_t* point_loops,
		  struct TileLoops tileLoopsStruct,
		  s_ptile_options_t* ptileOptions)
 {	
	int local_id = GeneralUtilityClass::NumberForLoopHierarchy(point_loops, 1, 1); // Assign id - 1 to the first loop encountered. 1 - skip otl loops


//	FILE *fp = fopen("ModifiedLoopsOnlyClone.c", "a");
//	fprintf (fp, "Point loops\n");
//	past_pprint_metainfo (fp, point_loops, metainfoprint);
//	fclose(fp);



	// Modify in place 'point_loops_clone' to tile looops using the bounds provided in tileLoopsStruct.tileLoops
	// printf ("Calling BuildTileLoopsFromPointLoopStructure()\n");
	s_past_node_t* tile_loop_structure =  BuildTileLoopsFromPointLoopStructure (point_loops, tileLoopsStruct.tileLoops, ptileOptions);
	// printf ("Returned from BuildTileLoopsFromPointLoopStructure()\n");


//	fp = fopen("ModifiedLoopsOnlyClone.c", "a");
//	fprintf (fp, "\ntile_loop_structure\n");
//	past_pprint_metainfo (fp, tile_loop_structure, metainfoprint);
//	fclose(fp);



	// Insert and combine RSFME loops if RSFME flag is set
	    s_past_node_t* consolidatedForLoop = NULL;
	    if (ptileOptions->RSFME == 1)
	      {
		// printf("Calling CombineTileLoopsAndRSFMELoopsInOrder()\n");
		consolidatedForLoop =
		  CombineTileLoopsAndRSFMELoopsInOrder (tileLoopsStruct.rsfmeLoops, tile_loop_structure);
		// printf("Returned from CombineTileLoopsAndRSFMELoopsInOrder()\n");
	      }// if (ptileOptions.RSFME == 1)
	     else
	      {
		 consolidatedForLoop = tile_loop_structure;
	      }// else

	// Insert the point loops at the appropriate places. Pass an empty path to 'path' argument
	InsertPointLoopsInsideTileLoops (consolidatedForLoop, point_loops, ptileOptions);

	// Erase loop numbering
	GeneralUtilityClass::DeNumberForLoopHierarchy (consolidatedForLoop);

//	fp = fopen("ModifiedLoopsOnlyClone.c", "a");
//	fprintf (fp, "\nconsolidatedForLoop\n");
//	past_pprint_metainfo (fp, consolidatedForLoop, metainfoprint);
//	fclose(fp);
	
	
	return consolidatedForLoop;
 }




  s_past_node_t*
  TileLoopInserter::InsertTileLoops(s_past_node_t* point_loops,
		  struct TileLoops tileLoopsStruct,
		  s_ptile_options_t* ptileOptions)
  {
    // 1. InnermostTiledForLoopNest.Body <- PointForLoop
    // 2. Replace the bounds of the point loop with mins and maxes appropriately

    s_past_node_t* tiledCode = NULL;
    // 1. InnermostTiledForLoopNest.Body <- PointForLoop
    if (point_loops != NULL)
      {
	s_past_node_t* task_body = NULL;

	s_past_node_t* fullTiles = NULL;
	if (ptileOptions->fullTileSeparation == 1)
	  {
	   // printf("Calling BuildFullTiles\n");
	    fullTiles = FullTileMaker::FullTileMakerDriver(point_loops, ptileOptions);
	   // printf("Returning from BuildFullTiles\n");
	  }// if (ptileOptions->fullTileSeparation == 1)


	// 2. Replace the bounds of the point loop with mins and maxes appropriately
	ModifyBoundsOfPointLoops(point_loops);

	if (ptileOptions->fullTileSeparation == 1 &&
	    fullTiles != NULL)
	  {
	    // Build the if then list
	    s_past_node_t* temp = fullTiles;
	    task_body = FullTileMaker::BuildIfElseList (fullTiles, point_loops);
		/*{
			FILE *fp = fopen ("TaskBody.c", "a");
			past_pprint (fp, task_body);
			fprintf (fp, "\n\n");
			fclose (fp);
		}*/

	  }// if (ptileOptions.fullTileSeparation == 1)
	else
	  {
	    task_body = point_loops;
	  }//else


	if (ptileOptions->parallel == 1)
	  {
	    //printf("Calling ProduceParallelTiledCode()\n");
	    /// LNP: Commented out for the moment.
	    assert(0);
// 	    tiledCode =
// 	      Parallelizer::ProduceParallelTiledCode(tileLoopsStruct.tileLoops,
// 						     task_body);
	    //printf("Returned from ProduceParallelTiledCode()\n");
	  }// if (ptileOptions.parallel == 1)
	else
	  {
	    s_past_node_t* consolidatedForLoop = NULL;
	    if (ptileOptions->RSFME == 1)
	      {
		//printf("Calling CombineTileLoopsAndRSFMELoops()\n");
		consolidatedForLoop =
		  CombineTileLoopsAndRSFMELoops(tileLoopsStruct);
		//printf("Returned from CombineTileLoopsAndRSFMELoops()\n");
	      }// if (ptileOptions.RSFME == 1)
	    else
	      {
		consolidatedForLoop = tileLoopsStruct.tileLoops;
	      }// else
	    s_past_for_t* innermostTileLoop = (s_past_for_t*)
	      GeneralUtilityClass::FindInnermostForLoopInLoopNest
	      (consolidatedForLoop);
	    innermostTileLoop->body = task_body;
	    tiledCode = consolidatedForLoop;
	  }
      }// if (point_loops != NULL)


	// Erase loop numbering
	GeneralUtilityClass::DeNumberForLoopHierarchy (tiledCode);


    return tiledCode;
  }//void InsertTileLoops(struct clast_stmt* root, struct ext_clast_for *tileLoops)



  void TileLoopInserter::ModifyBoundsOfPointLoops(s_past_node_t* PointLoops)
  {
    for ( ; PointLoops; PointLoops = PointLoops->next)
      {
	if (PAST_NODE_IS_A(PointLoops, past_for))
	  {
	    PAST_DECLARE_TYPED(for, LocalPointLoops, PointLoops);

	    s_past_node_t* LB = LocalPointLoops->init;
	    s_past_node_t* UB = LocalPointLoops->test;

	    /* 1. Extract the iterator name from expressions
	     * Assume that the expressions are of the form i = 0; i <= N. The LHS gives the iterator name
	     * 2. NewLB = (i = max(it1*T1i, LB)): Replace the RHS with the max of the original expression and it1*T1i
	     * 3. NewUB = (i <= min(it1*T1i + T1i - 1, N)) : Replace the RHS with the min of the original expression and it1*T1i + T1i - 1
	     */


	    const char * iterator =
	      (const char*)LocalPointLoops->iterator->symbol->data;

	    string tileIterator = string(iterator);
	    tileIterator += ExpressionLibrary::GetTileIteratorSuffix(1, 1);
	    string tileSize = string(iterator);
	    tileSize = ExpressionLibrary::GetTileSizePrefix(1, 1) + tileSize;

	    // Constructing the LB
	    s_symbol_t* tisymb = symbol_add_from_char(NULL,
						      tileIterator.c_str());
	    s_past_node_t* tileIteratorClastName =
	      &past_variable_create(tisymb)->node;
	    s_symbol_t* tssymb = symbol_add_from_char(NULL,
						      tileSize.c_str());

	    s_past_node_t* tileSizeClastName =
	      &past_variable_create(tssymb)->node;

	    s_past_node_t* newLB =
	      &past_binary_create(past_mul, tileIteratorClastName,
				  tileSizeClastName)->node;
	    PAST_DECLARE_TYPED(binary, pb, LocalPointLoops->init);

	    pb->rhs = past_node_binary_create(past_max, newLB, pb->rhs);

	    // Constructing the UB

	    tisymb = symbol_add_from_char(NULL, tileIterator.c_str());
	    tileIteratorClastName = &past_variable_create(tisymb)->node;
	    tssymb = symbol_add_from_char(NULL, tileSize.c_str());
	    tileSizeClastName = &past_variable_create(tssymb)->node;

	    s_past_node_t* newUB1 =
	      &past_binary_create(past_mul, tileIteratorClastName,
				  tileSizeClastName)->node;
	    s_past_node_t* minusOne = past_node_value_create_from_int(-1);

	    tssymb = symbol_add_from_char(NULL, tileSize.c_str());
	    tileSizeClastName = past_node_variable_create(tssymb);

	    s_past_node_t* newUB2 =
	      past_node_binary_create(past_add, tileSizeClastName, minusOne);

	    s_past_node_t* newUB =
	      &past_binary_create(past_add, newUB1, newUB2)->node;

	    PAST_DECLARE_TYPED(binary, pb2, LocalPointLoops->test);

	    pb2->rhs = past_node_binary_create(past_min, newUB, pb2->rhs);

	    ModifyBoundsOfPointLoops(LocalPointLoops->body);
	  }//if (CLAST_STMT_IS_A((struct clast_stmt*) tileLoops, stmt_for))
	 else if (PAST_NODE_IS_A(PointLoops,  past_affineguard))
	 {
	      PAST_DECLARE_TYPED(affineguard, IfConditionStmt, PointLoops);
              ModifyBoundsOfPointLoops(IfConditionStmt->then_clause);
	 }// else if (PAST_NODE_IS_A(PointLoops,  past_affineguard))
      }// for ( ; PointLoops; PointLoops = PointLoops->next)
  }// void ModifyBoundsOfPointLoops(struct ext_clast_for* PointLoops)

