/*
 * compsol.c: this file is part of the FM project.
 *
 * FM, a fast and optimized C implementation of Fourier-Motzkin
 * projection algorithm.
 *
 * Copyright (C) 2006,2007,2008 Louis-Noel Pouchet
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 3
 * of the License, or (at your option) any later version.
 *
 * The complete GNU Lesser General Public Licence Notice can be found
 *  as the `COPYING.LESSER' file in the root directory.
 *
 * Author:
 * Louis-Noel Pouchet <Louis-Noel.Pouchet@inria.fr>
 *
 */
#if HAVE_CONFIG_H
# include <fm/config.h>
#endif

#include <fm/common.h>
#include <fm/compsol.h>

// NOTE: valid without redfree at rev 8343.


s_fm_compsol_t*
fm_compsol_alloc (unsigned size)
{
  s_fm_compsol_t* ret = XMALLOC(s_fm_compsol_t, 1);

  ret->redeq = fm_solution_alloc (size);
  ret->redfree = NULL;
  ret->poly = NULL;
  ret->size = size;
  ret->nb_reduc = 0;
  ret->nb_free = 0;
  ret->empty = 0;

  return ret;
}


s_fm_compsol_t*
fm_compsol_dup (s_fm_compsol_t* s)
{
  s_fm_compsol_t* ret = XMALLOC(s_fm_compsol_t, 1);

  ret->redeq = fm_solution_dup (s->redeq);
  ret->redfree = fm_solution_dup (s->redfree);
  ret->poly = fm_solution_dup (s->poly);
  ret->size = s->size;
  ret->nb_reduc = s->nb_reduc;
  ret->nb_free = s->nb_free;
  ret->empty = s->empty;

  return ret;
}


void
fm_compsol_free (s_fm_compsol_t* s)
{
  fm_solution_free (s->redeq);
  fm_solution_free (s->redfree);
  fm_solution_free (s->poly);
  XFREE(s);
}


/**
 * Helper. Remove free variables for a system.
 *
 *
 */
static
s_fm_system_t*
fm_compsol_init_remove_free (s_fm_compsol_t* ret,
			     s_fm_solution_t* sfree,
			     s_fm_system_t* s_red)
{
  int i, j;
  int count;
  s_fm_system_t* s_redf = s_red;

  // If there are free variables, remove them.
  if (sfree)
    {
      // Use a mark vector.
      char mark[sfree->size];

      // Compute the number of free variables.
      for (i = 0, count = 0; i < sfree->size; ++i)
	{
	  if (sfree->solution[i].positive != NULL ||
	      sfree->solution[i].negative != NULL)
	    ++count;
	  // Check for conflict with reduction equations.
	  // FIXME: Not implemented yet, so assert if any.
	  if ((sfree->solution[i].positive != NULL ||
	       sfree->solution[i].negative != NULL) &&
	      ret->redeq->solution[i].positive != NULL)
	    assert (! "a variable is simultaneously free and reduced");
	  // Reset the marks.
	  mark[i] = 0;
	}
      if (count)
	{
	  ret->nb_free = count;
	  ret->redfree = sfree;
	  if (s_red->nb_cols - count > 2)
	    {
	      s_redf = fm_system_alloc (s_red->nb_lines,
					s_red->nb_cols - count);
	      // Fill the system.
	      for (i = 0; i < s_red->nb_lines; ++i)
		{
		  count = 0;
		  fm_rational_copy (&(s_redf->lines[i]->vector[count++]),
				    &(s_red->lines[i]->vector[0]));
		  for (j = 1; j < s_red->nb_cols - 1; ++j)
		    if (mark[j - 1])
		      fm_rational_copy (&(s_redf->lines[i]->vector[count++]),
					&(s_red->lines[i]->vector[j]));
		  fm_rational_copy (&(s_redf->lines[i]->vector[count++]),
				    &(s_red->lines[i]->vector[j]));
		}
	      // Remove useless lines.
	      // FIXME: No check for consistency. Must be implemented.
	      j = 0;
	      for (i = 0; i < s_redf->nb_lines; ++i)
		if (fm_vector_is_empty (s_redf->lines[i]))
		  {
		    fm_vector_free (s_redf->lines[i]);
		    for (j = i; j < s_redf->nb_lines - 1; ++j)
		      s_redf->lines[j] = s_redf->lines[j + 1];
		    s_redf->nb_lines--;
		  }
	      if (j)
		s_redf->lines =
		  XREALLOC(s_fm_vector_t*, s_redf->lines, s_redf->nb_lines);
	    }
	  else
	    s_redf = fm_system_alloc (0, s_red->nb_cols - count);

	  // Replace the system.
	  fm_system_free (s_red);
	}
    }

/*   printf ("CompFree: eliminated %d variables (%d remain)\n", */
/* 	  ret->size - (s_redf->nb_cols - 2), s_redf->nb_cols - 2); */

  return s_redf;
}


/**
 * Initialization of a compacted solution. Works in place on s_in.
 * Reduce the system with echelon-like form.
 *
 */
static
s_fm_compsol_t*
fm_compsol_init (s_fm_compsol_t* ret,
		 s_fm_system_t* s_in,
		 int nb_eq,
		 s_fm_solution_t* sfree)
{
  int i, j, k;
  s_fm_system_t* s_eq = NULL;
  s_fm_system_t* s_red = NULL;
  s_fm_system_t* s_redf;

  ret->redfree = sfree;

  // Extract the equalities from the system.
  s_eq = fm_system_alloc (nb_eq, s_in->nb_cols);
  for (i = 0; i < nb_eq; ++i)
    fm_vector_assign (s_eq->lines[i], s_in->lines[i]);
  // Extract replacement vectors from the input set of equalities.
  ret->redeq = fm_solver_linind (s_eq);
  fm_system_free (s_eq);
  // Use replacement vectors in the input system.
  s_red = fm_system_reduce (s_in, ret->redeq);

  // If the polyhedron is empty, stop the computation.
  if (s_red == NULL)
    {
      ret->empty = 1;
      fm_solution_free (ret->poly);
      ret->poly = NULL;
      // Be clean.
      fm_system_free (s_in);
      return ret;
    }

  // Remove useless equalities in the input system..
  for (i = 0; i < s_red->nb_lines &&
	 Z_CMP_SI(s_red->lines[i]->vector[0].num, ==, 0); ++i)
    fm_vector_free (s_red->lines[i]);
  for (k = 0; k < s_red->nb_lines - i; ++k)
    s_red->lines[k] = s_red->lines[k + i];
  s_red->nb_lines -= i;
  ret->nb_reduc = s_in->nb_cols - s_red->nb_cols;

  if (s_red->nb_lines != 0)
    {
      s_red->lines =
	XREALLOC(s_fm_vector_t*, s_red->lines, s_red->nb_lines);
      // Remove free variables, if any.
      if (sfree)
	s_red = fm_compsol_init_remove_free (ret, sfree, s_red);
    }

  // Create a (compact) solution from the reduced system.
  if (s_red->nb_lines != 0)
    ret->poly = fm_system_to_solution (s_red);
  else
    ret->poly = NULL;

/*   printf ("Total: eliminated %d variables (%d remain)\n", */
/* 	  ret->size - ret->poly->size, ret->poly->size); */

  fm_system_free (s_red);

  return ret;
}


/**
 * Initialize a compacted solution from a fm_solution_t,
 * and a set of free variables (bounds are given in a solution_t,
 * where each non-null variable is a free variable).
 * Note: A free variable must be bounded in at least one direction.
 *
 */
s_fm_compsol_t*
fm_compsol_init_sol_free (s_fm_solution_t* in, s_fm_solution_t* sfree)
{
  s_fm_solution_t* solin = fm_solution_dup (in);
  int nb_eq = fm_solution_equalities_find (solin);
  s_fm_compsol_t* ret = XMALLOC (s_fm_compsol_t, 1);
  s_fm_system_t* s_in;

  ret->redfree = NULL;
  ret->nb_free = 0;
  ret->nb_reduc = 0;

  // Polyhedron is empty.
  if (in == NULL)
    {
      ret->redeq = NULL;
      ret->poly = NULL;
      ret->empty = 1;
      ret->size = 0;
      return ret;
    }

  // No equations in the polyhedron.
  ret->size = in->size;
  ret->empty = 0;
  ret->redfree = fm_solution_dup (sfree);
  if (nb_eq == 0)
    {
      ret->redeq = fm_solution_alloc (in->size);
      if (sfree)
	{
	  s_in = fm_solution_to_system (solin);
	  fm_solution_free (solin);
	  // no need to free s_in here, done in init_remove.
	  s_in = fm_compsol_init_remove_free (ret, sfree, s_in);
	  ret->poly = fm_system_to_solution (s_in);
	  fm_system_free (s_in);
	}
      else
	ret->poly = solin;
      return ret;
    }
  // There are equations. Use classical initialization.
  s_in = fm_solution_to_system (solin);
  fm_system_equalities_sort (s_in);
  fm_compsol_init (ret, s_in, nb_eq, sfree);
  fm_system_free (s_in);
  fm_solution_free (solin);

  return ret;
}


/**
 * Initialize a compacted solution from a fm_solution_t.
 *
 *
 */
s_fm_compsol_t*
fm_compsol_init_sol (s_fm_solution_t* in)
{
  s_fm_solution_t* solin = fm_solution_dup (in);
  int nb_eq = fm_solution_equalities_find (solin);
  s_fm_compsol_t* ret = XMALLOC (s_fm_compsol_t, 1);

  ret->redfree = NULL;
  ret->nb_free = 0;
  ret->nb_reduc = 0;

  // Polyhedron is empty.
  if (in == NULL)
    {
      ret->redeq = NULL;
      ret->poly = NULL;
      ret->empty = 1;
      ret->size = 0;
      return ret;
    }

  // No equations in the polyhedron.
  ret->size = in->size;
  ret->empty = 0;
  if (nb_eq == 0)
    {
      ret->redeq = fm_solution_alloc (in->size);
      ret->poly = solin;
      return ret;
    }
  // There are equations. Use classical initialization.
  s_fm_system_t* s_in = fm_solution_to_system (solin);
  fm_system_equalities_sort (s_in);
  fm_compsol_init (ret, s_in, nb_eq, NULL);
  fm_system_free (s_in);
  fm_solution_free (solin);

  return ret;
}


/**
 * Initialize a compacted solution from a fm_system_t.
 *
 *
 */
s_fm_compsol_t*
fm_compsol_init_sys (s_fm_system_t* in)
{
  int i, k;
  int nb_eq;
  s_fm_system_t* s_in;
  s_fm_system_t* s_eq = NULL;
  s_fm_system_t* s_red = NULL;
  s_fm_compsol_t* ret = XMALLOC (s_fm_compsol_t, 1);

  ret->redfree = NULL;
  ret->nb_free = 0;
  ret->nb_reduc = 0;

  // Polyhedron is empty.
  if (in == NULL)
    {
      ret->redeq = NULL;
      ret->poly = NULL;
      ret->empty = 1;
      ret->size = 0;
      return ret;
    }

  // The algorithms work in place. Duplicate the input system.
  s_in = fm_system_dup (in);
  // Find the equalities.
  nb_eq = fm_system_equalities_find (s_in);
  ret->size = s_in->nb_cols - 2;
  ret->empty = 0;

  if (nb_eq == 0)
    {
      ret->redeq = fm_solution_alloc (in->nb_cols - 2);
      ret->poly = fm_system_to_solution (in);
    }
  else
    fm_compsol_init (ret, s_in, nb_eq, NULL);

  fm_system_free (s_in);

  return ret;
}


/**
 * Add a (full dimensional) vector to a compacted solution.
 *
 *
 */
void
fm_compsol_add_unique (s_fm_compsol_t* s, s_fm_vector_t* v)
{
  if (s == NULL)
    assert(! "s is null");

  int i;
  int idx;
  int size;
  s_fm_vector_t* tmp;
  s_fm_vector_t* tmp2;

  // Compute the size of the vector.
  for (size = 2, i = 1; i < v->size - 1; ++i)
    if (s->redeq->solution[i - 1].positive == NULL)
      ++size;
  // FIXME: Assume no conflict between free and redeq.
  if (s->redfree)
    size -= s->nb_free;

  tmp = fm_vector_alloc (size);

  // Fill the vector.
  fm_rational_copy (&(tmp->vector[0]), &(v->vector[0]));
  fm_rational_copy (&(tmp->vector[size - 1]),
		    &(v->vector[v->size - 1]));
  // Apply the reduction contained in redeq.
  for (idx = 1, i = 1; i < v->size - 1; ++i)
    {
      if (s->redeq->solution[i - 1].positive != NULL)
	fm_system_subst_in_vector (tmp, v,
				   s->redeq->solution[i - 1].positive->data,
				   s->redeq, i);
      else
	if ((! s->redfree) ||
	       s->redfree->solution[i - 1].positive == NULL ||
	       s->redfree->solution[i - 1].negative == NULL)
	fm_rational_copy (&(tmp->vector[idx++]),
			  &(v->vector[i]));
    }

  // Compute the last non-null column.
  for (i = tmp->size - 2; i > 0 && Z_CMP_SI(tmp->vector[i].num, ==, 0); --i)
    ;
  // We need to shrink the vector.
  if (i != tmp->size - 2)
    {
      tmp2 = fm_vector_alloc (i + 2);
      fm_vector_shrink (tmp2, tmp, i);
      fm_vector_free (tmp);
      tmp = tmp2;
    }

  // If there is no non-null variable coefficient, then it is a
  // redundant constraint.
  if (i == 0)
    {
      if (fm_vector_is_valid (tmp))
	fm_vector_free (tmp);
      else
	{
	  // assert (! "system infeasible");
	  // FIXME: Big Hack :) Add a contradictory vector to
	  // the solution.
	  tmp = fm_vector_alloc (3);
	  fm_vector_assign_int_idx (tmp, 1, 1);
	  fm_solution_add_unique_line_at (s->poly, tmp, 1);
	  tmp = fm_vector_alloc (3);
	  fm_vector_assign_int_idx (tmp, 1, 1);
	  fm_vector_assign_int_idx (tmp, 1, 2);
	  fm_solution_add_unique_line_at (s->poly, tmp, 1);
	  s->empty = 1;
	}
      return;
    }

  // Normalize the vector.
  fm_vector_normalize_idx (tmp, tmp, tmp->size - 2);

  // Add it in the solution.
  fm_solution_add_unique_line_at (s->poly, tmp, tmp->size - 2);
}


/**
 * Expand a compacted solution to a full polyhedron
 *
 *
 */
s_fm_solution_t*
fm_compsol_expand (s_fm_compsol_t* s)
{
  if (s == NULL)
    return NULL;

  int i, j, k;
  int idx;
  s_fm_solution_t* ret = fm_solution_alloc (s->size);
  s_fm_vector_t* tmp;
  s_fm_vector_t* v;
  s_fm_list_t* start;
  s_fm_list_t* l;
  z_type_t one; Z_INIT(one); Z_ASSIGN_SI(one, 1);

  // Fill the solution with the base polyhedron.
  for (i = 0; s->poly && i < s->poly->size; ++i)
    {
      for (start = s->poly->solution[i].positive, k = 0; k < 2;
	   ++k, start = s->poly->solution[i].negative)
	for (l = start; l != NULL; l = l->next)
	  {
	    v = l->data;
	    // Compute the size of the vector.
	    for (j = 1, idx = 0; j < v->size - 1 + idx; ++j)
	      if (s->redeq->solution[j - 1].positive != NULL)
		++idx;
	    tmp = fm_vector_alloc (v->size + idx);
	    // Fill the vector.
	    fm_rational_copy (&(tmp->vector[0]), &(v->vector[0]));
	    fm_rational_copy (&(tmp->vector[tmp->size - 1]),
			      &(v->vector[v->size - 1]));
	    for (j = 1, idx = 0; j < v->size - 1; ++j)
	      {
		while (s->redeq->solution[j - 1 + idx].positive != NULL)
		  ++idx;
		fm_rational_copy (&(tmp->vector[j + idx]), &(v->vector[j]));
	      }
	    fm_solution_add_unique_line_at (ret, tmp, j + idx - 1);
	  }
    }

  // Add reduction equalities to the solution.
  for (i = 0; s->redeq && i < s->redeq->size; ++i)
    if (s->redeq->solution[i].positive != NULL)
      {
	v = s->redeq->solution[i].positive->data;
	// Add the positive constraint.
	tmp = fm_vector_alloc (v->size + 1);
	for (k = 1; k < v->size - 1; ++k)
	  fm_rational_copy (&(tmp->vector[k]), &(v->vector[k]));
	fm_rational_copy (&(tmp->vector[k + 1]), &(v->vector[k]));
	fm_vector_set_ineq (tmp);
	fm_vector_assign_int_idx (tmp, one, i + 1);
	fm_solution_add_unique_line_at (ret, tmp, i + 1);
	// Add the negative constraint.
	tmp = fm_vector_alloc (v->size + 1);
	for (k = 1; k < v->size - 1; ++k)
	  fm_rational_copy (&(tmp->vector[k]), &(v->vector[k]));
	fm_rational_copy (&(tmp->vector[k + 1]), &(v->vector[k]));
	fm_vector_set_ineq (tmp);
	fm_vector_assign_int_idx (tmp, one, i + 1);
	fm_vector_opp (tmp, tmp);
	fm_solution_add_unique_line_at (ret, tmp, i + 1);
      }

  // Add free variables to the solution.
  if (s->redfree)
    {
      for (i = 0; i < s->redfree->size; ++i)
	{
	  if (s->redeq->solution[i].positive != NULL)
	    {
	      // Add the positive constraint.
	      v = fm_vector_dup (s->redeq->solution[i].positive->data);
	      fm_solution_add_unique_line_at (ret, tmp, i + 1);
	    }
	  if (s->redeq->solution[i].negative != NULL)
	    {
	      // Add the negative constraint.
	      v = fm_vector_dup (s->redeq->solution[i].negative->data);
	      fm_solution_add_unique_line_at (ret, tmp, i + 1);
	    }
	}
    }

  Z_CLEAR(one);

  return ret;
}


/**
 * Remove some redundant inequalities.
 *
 */
s_fm_solution_t*
fm_compsol_redundancy_elimination (s_fm_solution_t* s)
{
  s_fm_compsol_t* cs = fm_compsol_init_sol (s);
  s_fm_solution_t* res = fm_compsol_expand (cs);
  fm_compsol_free (cs);

  return res;
}



/**
 * Remove some redundant inequalities. Works in place.
 *
 *
 */
s_fm_system_t*
fm_compsol_redundancy_elimination_sys (s_fm_system_t* s)
{
  s_fm_compsol_t* cs = fm_compsol_init_sys (s);
  s_fm_solution_t* sol = fm_compsol_expand (cs);
  fm_compsol_free (cs);
  s_fm_system_t* ret = fm_solution_to_system (sol);
  fm_solution_free (sol);
  fm_system_free (s);
  
  return ret;
}
