/*
 * piptools.c: this file is part of the FM project.
 *
 * FM, a fast and optimized C implementation of Fourier-Motzkin
 * projection algorithm.
 *
 * Copyright (C) 2007,2008 Louis-Noel Pouchet
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 3
 * of the License, or (at your option) any later version.
 *
 * The complete GNU Lesser General Public Licence Notice can be found
 *  as the `COPYING.LESSER' file in the root directory.
 *
 * Author:
 * Louis-Noel Pouchet <Louis-Noel.Pouchet@inria.fr>
 *
 */
#if HAVE_CONFIG_H
# include <fm/config.h>
#endif

/* Must have piplib on the system. */
#ifdef HAVE_LIBPIPLIB

# include <fm/common.h>
# include <fm/piptools.h>


/**
 *
 * Check if the system has an integer solution (using PIP).
 *
 *
 **/
int
fm_piptools_check_int (s_fm_system_t* sys)
{
  return fm_piptools_check (sys, FM_PIPTOOLS_INT);
}


/**
 *
 * Check if the system has a rational solution (using PIP).
 *
 *
 **/
int
fm_piptools_check_rat (s_fm_system_t* sys)
{
  return fm_piptools_check (sys, FM_PIPTOOLS_RAT);
}


/**
 *
 * Check if the system has a solution.
 *
 * mode can be: FM_PIPTOOLS_RAT (rational solution expected) or
 * FM_PIPTOOLS_INT (integer solution expected)
 *
 *
 **/
int
fm_piptools_check (s_fm_system_t* sys, int mode)
{
  int i, j;
  int ret;
  PipOptions* pipoptions;
  PipQuast* solution;

  pipoptions = pip_options_init ();
  pipoptions->Simplify = 1;
  //pipoptions->Urs_parms = -1;
  pipoptions->Urs_unknowns = -1;
  if (mode == FM_PIPTOOLS_RAT)
    pipoptions->Nq = 0;
  else
    pipoptions->Nq = 1;

  PipMatrix* syst = fm_piptools_st_to_pipmatrix (sys);

  solution = pip_solve (syst, NULL, -1, pipoptions);

  pip_matrix_free (syst);
  pip_options_free (pipoptions);

  if ((solution != NULL) &&
      ((solution->list != NULL) || (solution->condition != NULL)))
    ret = 1;
  else
    ret = 0; //printf ("no point \n");

  pip_quast_free (solution);

  return ret;
}


/**
 *
 * Call PIP on the input system.
 *
 * mode can be: FM_PIPTOOLS_RAT (rational solution expected) or
 * FM_PIPTOOLS_INT (integer solution expected)
 *
 *
 **/
PipQuast*
fm_piptools_pip (s_fm_system_t* sys, s_fm_system_t* context, int mode)
{
  int i, j;
  PipOptions* pipoptions;
  PipQuast* solution;

  pipoptions = pip_options_init ();
  pipoptions->Simplify = 1;
  //pipoptions->Urs_parms = -1;
  pipoptions->Urs_unknowns = -1;
  if (mode == FM_PIPTOOLS_RAT)
    pipoptions->Nq = 0;
  else
    pipoptions->Nq = 1;

  PipMatrix* syst = fm_piptools_st_to_pipmatrix (sys);
  PipMatrix* contx = fm_piptools_st_to_pipmatrix (context);

  solution = pip_solve (syst, contx, -1, pipoptions);

  pip_matrix_free (syst);
  pip_options_free (pipoptions);

  return solution;
}


/**
 *
 * Check if the system (represented as a solution_t)  has a solution.
 *
 * mode can be: FM_PIPTOOLS_RAT (rational solution expected) or
 * FM_PIPTOOLS_INT (integer solution expected)
 *
 *
 **/
int
fm_piptools_check_sol (s_fm_solution_t* sol, int mode)
{
  s_fm_system_t* sys2 = fm_solution_to_system (sol);
  s_fm_system_t* sys = fm_system_to_z (sys2);
  fm_system_free (sys2);
  int ret = fm_piptools_check (sys, mode);
  fm_system_free (sys);

  return ret;
}


/**
 *
 * Check if the system (represented as a solution_t) has a solution,
 * and print a message once the check is done.
 *
 * mode can be: FM_PIPTOOLS_RAT (rational solution expected) or
 * FM_PIPTOOLS_INT (integer solution expected)
 *
 *
 **/
int
fm_piptools_check_sol_msg (char* msg,
			   FILE* stream,
			   s_fm_solution_t* sol,
			   int mode)
{
  fprintf (stream, msg);
  int ret = fm_piptools_check_sol (sol, mode);
  if (ret)
    fprintf (stream, "yes\n");
  else
    fprintf (stream, "no\n");

  return ret;
}


/**
 * Check if two PIP matrices are equal.
 *
 *
 **/
int
fm_piptools_pipmatrix_equal (PipMatrix* a,
			     PipMatrix* b)
{
  int i, j;

  if (a->NbColumns != b->NbColumns ||
      a->NbRows != b->NbRows)
    return 0;

  for (i = 0; i < a->NbRows; ++i)
    for (j = 0; j < a->NbColumns; ++j)
      if (Z_CMP(a->p[i][j], !=, b->p[i][j]))
	return 0;
  return 1;
}


/**
 * Convert a system_t to a PIP matrix.
 *
 *
 **/
PipMatrix*
fm_piptools_st_to_pipmatrix (s_fm_system_t* sys)
{
  if (sys == NULL)
    return NULL;

  PipMatrix* ret = pip_matrix_alloc (sys->nb_lines, sys->nb_cols);
  s_fm_system_t* sysz = fm_system_to_z (sys);
  int i, j;

  for (i = 0; i < sys->nb_lines; ++i)
    for (j = 0; j < sys->nb_cols; ++j)
      Z_ASSIGN(ret->p[i][j], sysz->lines[i]->vector[j].num);

  fm_system_free (sysz);

  return ret;
}


/**
 * Convert a PIP matrix to a system_t.
 *
 *
 **/
s_fm_system_t*
fm_piptools_pm_to_system (PipMatrix* m)
{
  if (m == NULL)
    return NULL;

  s_fm_system_t* ret = fm_system_alloc (m->NbRows, m->NbColumns);
  int i, j;

  for (i = 0; i < m->NbRows; ++i)
    for (j = 0; j < m->NbColumns; ++j)
      Z_ASSIGN(ret->lines[i]->vector[j].num, m->p[i][j]);

  return ret;
}


#endif // HAVE_LIBPIPLIB
