/*
 * solver.c: this file is part of the FM project.
 *
 * FM, a fast and optimized C implementation of Fourier-Motzkin
 * projection algorithm.
 *
 * Copyright (C) 2006,2007,2008 Louis-Noel Pouchet
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 3
 * of the License, or (at your option) any later version.
 *
 * The complete GNU Lesser General Public Licence Notice can be found
 *  as the `COPYING.LESSER' file in the root directory.
 *
 * Author:
 * Louis-Noel Pouchet <Louis-Noel.Pouchet@inria.fr>
 *
 */
#if HAVE_CONFIG_H
# include <fm/config.h>
#endif

#include <fm/common.h>
#include <fm/solver.h>

#define echo(S) { printf(#S); printf("\n"); }
//#define echo(S)


/**
 * Internal fonction. Helper for bounds computation.
 *
 */
static
void
fm_solver_compute_val (s_fm_rational_t** lb,
		       s_fm_list_t* l,
		       s_fm_vector_t* vect,
		       unsigned idx,
		       unsigned is_int,
		       unsigned is_min_computation)
{
  s_fm_rational_t* tmp = fm_rational_alloc ();
  s_fm_rational_t* val = fm_rational_alloc ();
  s_fm_vector_t* v;
  unsigned i;
  unsigned tst;

  for (; l != NULL; l = l->next)
    {
      v = l->data;

      fm_rational_assign (val, v->vector[v->size - 1].num,
			  v->vector[v->size - 1].denum);
      for (i = 1; i < idx + 1; ++i)
	{
	  fm_rational_mul (tmp, &(v->vector[i]), &(vect->vector[i]));
	  fm_rational_add (val, val, tmp);
	}

      if (is_min_computation)
	fm_rational_opp (val, val);

      if (*lb == NULL)
	{
	  *lb = fm_rational_alloc ();
	  z_type_t num;
	  z_type_t denum;
	  Z_INIT(num);
	  Z_INIT(denum);

	  if (is_int)
	    {
	      Z_DIV(num, val->num, val->denum);
	      Z_ASSIGN_SI(denum, 1);
	      if (Z_CMP_SI(val->denum, !=, 1))
		{
		  if (is_min_computation)
		  {
		    if (Z_CMP_SI(val->num, >, 0))
		      Z_INC(num, num);
/* 		    else */
/* 		      Z_DEC(num, num); */
		  }
		else
		  {
		    if (Z_CMP_SI(val->num, <, 0))
/* 		      Z_INC(num, num); */
/* 		    else */
		      Z_DEC(num, num);
		  }
		}
	    }
	  else
	    {
	      Z_ASSIGN(num, val->num);
	      Z_ASSIGN(denum, val->denum);
	    }

	  fm_rational_assign (*lb, num, denum);

	  Z_CLEAR(num);
	  Z_CLEAR(denum);
	}
      else
	{
	  if (is_min_computation)
	    tst = (fm_rational_cmp (*lb, val) < 0);
	  else
	    tst = (fm_rational_cmp (*lb, val) > 0);
	  if (tst)
	    {
	      if (is_int && Z_CMP_SI(val->denum, !=, 1))
		{
		  z_type_t num;
		  z_type_t one;
		  Z_INIT(one);
		  Z_INIT(num);
		  Z_ASSIGN_SI(one, 1);
		  Z_DIV(num, val->num, val->denum);
		  if (is_min_computation)
		    {
		      if (Z_CMP_SI(val->num, >, 0))
			Z_INC(num, num);
/* 		      else */
/* 			Z_DEC(num, num); */
		    }
		  else
		    {
		      if (Z_CMP_SI(val->num, <, 0))
/* 			Z_INC(num, num); */
/* 		      else */
			Z_DEC(num, num);
		    }
		  fm_rational_assign (*lb, num, one);
		  Z_CLEAR(one);
		  Z_CLEAR(num);
		}
	      else
		fm_rational_assign (*lb, val->num, val->denum);
	    }
	}
    }

  fm_rational_free (val);
  fm_rational_free (tmp);
}



static
void
fm_solver_add_unique (s_fm_vector_t*** lines,
		      unsigned* count, unsigned* sup,
		      s_fm_vector_t* v)
{
  unsigned k;

  for (k = 0; k < *count && !fm_vector_do_subsume ((*lines)[k], v); ++k)
    ;
  if (k < *count)
    fm_vector_free (v);
  else
    {
      if (*count == *sup - 1)
	{
	  *sup += FM_SIZE_BUF;
	  *lines = XREALLOC(s_fm_vector_t*, *lines, *sup);
	}
      (*lines)[(*count)++] = v;
    }
}



/**
 * Internal fonction. Helper to create the next system to operate on,
 * in Fourier-Motzkin elimination algorithm.
 *
 */
static
s_fm_system_t*
fm_solver_create_next_system (s_fm_system_t* system,
			      unsigned idx, unsigned n1, unsigned n2,
			      int solver_type,
			      unsigned* nn1, unsigned* nn2, unsigned* nn3)
{
  s_fm_vector_t** lines;
  s_fm_vector_t** lines_pos;
  s_fm_vector_t** lines_neg;
  s_fm_vector_t** lines_null;
  s_fm_vector_t* tmp;
  s_fm_vector_t* stmp;
  s_fm_system_t* res;
  int i, j, k, first_id;
  int* count;
  unsigned size = 0;
  int new_size = system->nb_lines - n1 + n1 * n2;

  unsigned sup_pos = FM_SIZE_BUF;
  unsigned sup_neg = FM_SIZE_BUF;
  unsigned sup_null = FM_SIZE_BUF;

  // Initialize bounds.
  *nn1 = *nn2 = *nn3 = 0;

  // No more inequality remains.
  if (new_size <= 0)
    return NULL;

  // Size is _at most_ new_size.
  lines_pos = XMALLOC(s_fm_vector_t*, FM_SIZE_BUF);
  lines_neg = XMALLOC(s_fm_vector_t*, FM_SIZE_BUF);
  lines_null = XMALLOC(s_fm_vector_t*, FM_SIZE_BUF);

  for (i = 0; i < n1; ++i)
    {
      for (j = n1; j < n1 + n2; ++j)
	{
	  tmp = fm_vector_alloc (system->nb_cols);
	  fm_vector_add (tmp, system->lines[i], system->lines[j]);
	  for (k = system->nb_cols - 2; k > 0; --k)
	    if (Z_CMP_SI(tmp->vector[k].num, !=, 0))
	      break;
	  if (k > 0)
	    fm_vector_normalize_idx (tmp, tmp, k);
	  for (k = 1; k < tmp->size - 1 &&
		 Z_CMP_SI(tmp->vector[k].num, ==, 0); ++k)
	    ;

	  if (k == tmp->size - 1)
	    {
	      if (Z_CMP_SI(tmp->vector[k].num, >=, 0))
		{
		  fm_vector_free (tmp);
		  continue;
		}
	      else
	       assert(!"Contradiction in an (in-)equality, system infeasible");
	    }
	  stmp = fm_vector_alloc (system->nb_cols - 1);
	  fm_vector_shrink (stmp, tmp, tmp->size - 3);
	  fm_vector_free (tmp);
	  first_id = stmp->size - 2;
	  if (Z_CMP_SI(stmp->vector[first_id].num, >, 0))
	    fm_solver_add_unique (&lines_pos, nn1, &sup_pos, stmp);
	  else if (Z_CMP_SI(stmp->vector[first_id].num, <, 0))
	    fm_solver_add_unique (&lines_neg, nn2, &sup_neg, stmp);
	  else
	    fm_solver_add_unique (&lines_null, nn3, &sup_null, stmp);
	}
    }

  for (i = n1 + n2; i < system->nb_lines && system->nb_cols > 3; ++i)
    {
      tmp = fm_vector_alloc (system->nb_cols - 1);
      fm_vector_shrink (tmp, system->lines[i], system->lines[i]->size - 3);
      fm_vector_normalize_idx (tmp, tmp, tmp->size - 2);
      if (Z_CMP_SI(tmp->vector[tmp->size - 2].num, >, 0))
	fm_solver_add_unique (&lines_pos, nn1, &sup_pos, tmp);
      else if (Z_CMP_SI(tmp->vector[tmp->size - 2].num, <, 0))
	fm_solver_add_unique (&lines_neg, nn2, &sup_neg, tmp);
      else
	fm_solver_add_unique (&lines_null, nn3, &sup_null, tmp);
    }

  size = *nn1 + *nn2 + *nn3;
  res = XMALLOC(s_fm_system_t, 1);
  res->nb_lines = size;
  res->lines = XMALLOC(s_fm_vector_t*, res->nb_lines);
  for (i = 0; i < res->nb_lines; ++i)
    res->lines[i] = NULL;
  res->nb_cols = system->nb_cols - 1;

  for (i = 0; i < *nn1; ++i)
    res->lines[i] = lines_pos[i];
  for (i = 0; i < *nn2; ++i)
    res->lines[i + *nn1] = lines_neg[i];
  for (i = 0; i < *nn3; ++i)
    res->lines[i + *nn1 + *nn2] = lines_null[i];

  XFREE(lines_pos);
  XFREE(lines_neg);
  XFREE(lines_null);

  // If the space size is too large, backup to simplify mode.
  if (((solver_type & FM_SOLVER_AUTO_SIMPLIFY) > 0 &&
       res->nb_lines > FM_SOLVER_AUTOSIMP_THRESOLD))
    {
      fm_system_free (res);
      return (s_fm_system_t*) FM_SOLVER_ERRPTR;
    }

  if ((solver_type & FM_SOLVER_SIMPLIFY) > 0)
    {
#ifdef HAVE_LIBPIPLIB
      res = fm_system_simplify (res, solver_type);
      // Recompute nn1 and nn2.
      for (i = 0, *nn1 = 0; i < res->nb_lines &&
	     Z_CMP_SI(res->lines[i]->vector[res->nb_cols - 2].num, ==, 1);
	   ++i, ++(*nn1))
	;
      for (*nn2 = 0; i < res->nb_lines &&
	     Z_CMP_SI(res->lines[i]->vector[res->nb_cols - 2].num, ==, -1);
	   ++i, ++(*nn2))
	;
#endif // ! HAVE_LIBPIPLIB
    }

  return res;
}


/**
 * Fourier-Motzkin projection algorithm.
 *
 * set last and to to -1 if not used.
 *
 */
static
s_fm_solution_t*
fm_solver_algo (s_fm_system_t* system, int solver_type, int last, int to)
{
  if (system == NULL)
    return NULL;

  s_fm_solution_t* solution;
  s_fm_system_t* cur_system[2];
  unsigned n1, n2, nn1, nn2, nn3;
  int i, idx;
  unsigned toggle = 0;

 start_label:
  // Check if valid parameters.
  if (to != -1 && last != -1)
    assert (! "Error in FM parameters: to and last cannot be set together");

  // Allocate the solution.
  int szsl = last == -1 ? (to == -1 ? system->nb_cols - 2 : to + 1) : last + 1;
  solution = fm_solution_alloc (szsl);

  // Duplicate the input system.
  cur_system[toggle] = fm_system_dup (system);

  // Step 0: normalize equations into inequations.
  // FIXME: This step should be useless.
  if ((solver_type & FM_SOLVER_NORMALIZE_EQ) > 0)
    fm_system_normalize_ineq (cur_system[toggle]);

  // Step 1: normalize the row by the given variable.
  idx = system->nb_cols - 2;
  for (i = 0; i < cur_system[toggle]->nb_lines; ++i)
    fm_vector_normalize_idx (cur_system[toggle]->lines[i],
			     cur_system[toggle]->lines[i],
			     idx);


  // Step 2: sort rows ((+) then (-) then (0) regarding the
  // coefficient of the given variable).
  fm_system_sort (cur_system[toggle], idx, &n1, &n2);

  for (idx = system->nb_cols - 2; idx > 0; --idx, toggle = toggle ? 0 : 1)
    {
      if ((solver_type & FM_SOLVER_VERBOSE) > 0)
      	printf ("at var %d: %d\n", idx, cur_system[toggle]->nb_lines);

      if (idx > to)
	{
	  // Step 3: simplify and rearrange the system, and produce a fresh one.
	  cur_system[toggle ? 0 : 1] = fm_solver_create_next_system
	    (cur_system[toggle], idx, n1, n2, solver_type, &nn1, &nn2, &nn3);
	  // Auto-mode: if the space is too large, use fm-simplify.
	  if (cur_system[toggle ? 0 : 1] == (s_fm_system_t*) FM_SOLVER_ERRPTR)
	    {
	      solver_type |= FM_SOLVER_SIMPLIFY | FM_SOLVER_REDREC_IRIGOIN;
	      solver_type -= FM_SOLVER_AUTO_SIMPLIFY;
	      fm_solution_free (solution);
	      fm_system_free (cur_system[toggle]);
	      goto start_label;
	    }
	}
      else
	cur_system[toggle ? 0 : 1] = NULL;

      if (last != -1 && to == -1)
	{
	  // Store the normalized interesting (in-)equalities in the solution
	  // list, and free others.
	  if (idx <= last)
	    {
	      for (i = 0; i < n1 + n2; ++i)
		if (! fm_solution_add_unique_line_at
		    (solution, cur_system[toggle]->lines[i], idx))
		  {
		    fm_vector_free (cur_system[toggle]->lines[i]);
		    assert (! "fm-solver-at: it was useful(can remove assert)");
		  }
	    }
	  else
	    i = 0;
	}
      else if (to != -1)
	{
	  i = 0;
	  if (idx == to)
	    {
	      fm_solution_free (solution);
	      solution = fm_system_to_solution (cur_system[toggle]);
	    }
	}
      else
	{
	  // Store the normalized interesting (in-)equalities in the solution
	  // list, and free others.
	  for (i = 0; i < n1 + n2; ++i)
	    if (! fm_solution_add_unique_line_at (solution,
						  cur_system[toggle]->lines[i],
						  idx))
	      fm_vector_free (cur_system[toggle]->lines[i]);
	}

      // Be clean.
      for (; i < cur_system[toggle]->nb_lines; ++i)
	fm_vector_free (cur_system[toggle]->lines[i]);
      XFREE(cur_system[toggle]->lines);
      XFREE(cur_system[toggle]);
      cur_system[toggle] = NULL;

      // Stop if there is no more inequality.
      if (cur_system[toggle ? 0 : 1] == NULL)
	break;

      n1 = nn1;
      n2 = nn2;
    }

  if (cur_system[toggle] != NULL)
    fm_system_free (cur_system[toggle]);

  return solution;
}





/**
 * Fourier-Motzkin projection algorithm.
 *
 * \brief Fourier-Motzkin projection algorithm, to solve rational and
 * integral systems of (in-)equalities.
 *
 * Possible options for solver_type:
 *  - FM_SOLVER_VERBOSE: Be verbose
 *  - FM_SOLVER_SIMPLIFY: Use Le Fur descending method at each step, to
 *    harness the number of constraints defining the polyhedron.
 *
 * input s_fm_system_t*		: A system to solve.
 * input int			: The option to provide.
 *
 * output s_fm_solution_t*	: The solution of the system, if it exists
 *				  (NULL otherwise).
 *
 * @see fm/system.h fm/solution.h
 *
 */
s_fm_solution_t*
fm_solver (s_fm_system_t* system, int solver_type)
{
  return fm_solver_algo (system, solver_type, -1, -1);
}



/**
 * Fourier-Motzkin projection algorithm.
 *
 * \brief Fourier-Motzkin projection algorithm, to solve rational and
 * integral systems of (in-)equalities.
 *
 * Possible options for solver_type:
 *  - FM_SOLVER_VERBOSE: Be verbose
 *  - FM_SOLVER_SIMPLIFY: Use Le Fur descending method at each step, to
 *    harness the number of constraints defining the polyhedron.
 *
 * input s_fm_system_t*		: A system to solve.
 * input int			: The option to provide.

 * input unsigned	        : The last variable index to store in the
 * 				  solution (full projection is
 * 				  performed, but we only store
 * 				  inequalities involving variable 1 to
 * 				  'last' in the solution.
 *
 * output s_fm_solution_t*	: The solution of the system, if it exists
 *				  (NULL otherwise).
 *
 * @see fm/system.h fm/solution.h
 *
 */
s_fm_solution_t*
fm_solver_solution_at (s_fm_system_t* system, int solver_type, unsigned last)
{
  return fm_solver_algo (system, solver_type, last, -1);
}



/**
 * Fourier-Motzkin projection algorithm.
 *
 * \brief Fourier-Motzkin projection algorithm. Projection done up to the 'to'
 *	  dimension.
 *
 * Possible options for solver_type:
 *  - FM_SOLVER_VERBOSE: Be verbose
 *  - FM_SOLVER_SIMPLIFY: Use Le Fur descending method at each step, to
 *    harness the number of constraints defining the polyhedron.
 *
 * input s_fm_system_t*	: A system to solve.
 * input int			: The option to provide.
 *
 * input unsigned	        : The variable index at which to stop the
 *				  projection (the projection is performed
 *				  from the last variable of the system to
 *				  the 'to' variable)
 *
 * output s_fm_solution_t*	: The solution of the system, if it exists
 *				  (NULL otherwise).
 *
 * @see fm/system.h fm/solution.h
 *
 */
s_fm_solution_t*
fm_solver_solution_to (s_fm_system_t* system, int solver_type, unsigned to)
{
  return fm_solver_algo (system, solver_type, -1, to);
}




/**
 * Return the lower bound for a given variable at index idx, NULL if
 * there is no bounds.
 *
 */
void
fm_solver_compute_min (s_fm_rational_t** lb,
		       s_fm_list_t* l,
		       s_fm_vector_t* vect,
		       unsigned idx,
		       int is_int)
{
  s_fm_rational_t* val = NULL;

  fm_solver_compute_val (&val, l, vect, idx, is_int, 1);

  if (*lb == NULL)
    *lb = val;
  else
    {
      fm_rational_assign (*lb, val->num, val->denum);
      fm_rational_free (val);
    }
}


/**
 * Return the Upper bound for a given variable at index idx, NULL if
 * there is no bounds.
 *
 */
void
fm_solver_compute_max (s_fm_rational_t** Ub,
		       s_fm_list_t* l,
		       s_fm_vector_t* vect,
		       unsigned idx,
		       int is_int)
{
  s_fm_rational_t* val = NULL;

  fm_solver_compute_val (&val, l, vect, idx, is_int, 0);

  if (*Ub == NULL)
    *Ub = val;
  else
    {
      fm_rational_assign (*Ub, val->num, val->denum);
      fm_rational_free (val);
    }
}



/**
 * Lexicographically smallest computation.
 *
 *
 * \brief Compute lexio-smallest solution. If lower bound is
 * -infinite, use argument 'min' instead. is_integral is a bit to
 * force computation of integral lexico-smallest vector.
 *
 * input s_fm_solution_t*   : input solution for the system.
 * input z_type_t	     : minimum value to use for -infinite bounds.
 * input int		     : set to 0 for rational values, to 1 for integral
 *			      values.
 *
 * output s_fm_rational_t** : A NULL-terminated array of rationals*, where
 *			       denominators are be '1' if is_integral is set.
 *
 */
s_fm_rational_t**
fm_solver_minlexico (s_fm_solution_t* sol, z_type_t min, int is_integral)
{
  unsigned i;
  s_fm_vector_t* vect = fm_vector_alloc (sol->size + 1);
  s_fm_rational_t* rlb = fm_rational_alloc ();
  s_fm_rational_t* lb = NULL;
  s_fm_rational_t* Ub = NULL;
  s_fm_rational_t** res;

  // Initialize the solution with min.
  z_type_t one;
  Z_INIT(one);
  Z_ASSIGN_SI(one, 1);
  fm_rational_assign (rlb, min, one);
  for (i = 1; i < sol->size + 1; ++i)
    fm_vector_assign_idx (vect, rlb, i);
  Z_CLEAR(one);

  // Compute the lower bound for each variable in the solution.
  for (i = 0; i < sol->size; ++i)
    {
      fm_solver_compute_min (&lb, sol->solution[i].positive,
			     vect, i, is_integral);
      fm_solver_compute_max (&Ub, sol->solution[i].negative,
			     vect, i, is_integral);

      if (sol->solution[i].positive != NULL &&
	  sol->solution[i].negative != NULL &&
	  fm_rational_cmp (lb, Ub) > 0)
	{
	  fm_vector_free (vect);
	  fm_rational_free (lb); lb = NULL;
	  fm_rational_free (Ub); Ub = NULL;
	  vect = NULL;
/* 	  fprintf (stderr, "FM: pb at idx %d\n", i); */
	  break;
	}

      if (sol->solution[i].positive != NULL)
	fm_vector_assign_idx (vect, lb, i + 1);
      fm_rational_free (lb); lb = NULL;
      fm_rational_free (Ub); Ub = NULL;
    }

  // No solution exists, output NULL.
  if (vect == NULL)
    return NULL;

  // Output a NULL-terminated array with the desired values.
  res = XMALLOC(s_fm_rational_t*, sol->size + 1);
  res[sol->size] = NULL;
  for (i = 0; i < sol->size; ++i)
    {
      res[i] = fm_rational_alloc ();
      fm_rational_assign (res[i], vect->vector[i + 1].num,
			  vect->vector[i + 1].denum);
    }
  fm_vector_free (vect);

  return res;
}


/**
 * Returns the minimal or maximal value of variable at column 'min' in the
 * 'domain' matrix.
 *
 */
static
z_type_t
fm_solver_bound(PipMatrix* domain, PipMatrix* context, int dim, int is_min)
{
  z_type_t ret;
  s_fm_system_t* s = fm_piptools_pm_to_system (domain);
  fm_system_swap_column (s, dim, 1);
  s_fm_solution_t* sol = fm_solver (s, FM_SOLVER_FAST);
  s_fm_rational_t** lexm;
  if (is_min) 
    lexm = fm_solver_minlexico (sol, -42, FM_PIPTOOLS_INT);
  else
    lexm = fm_solver_maxlexico (sol, 42, FM_PIPTOOLS_INT);
  Z_INIT(ret);
  Z_DIV(ret, lexm[0]->num, lexm[0]->denum);

  fm_system_free (s);
  fm_solution_free (sol);
  free (lexm);

  return ret;
}

/**
 * Returns the minimal value of variable at column 'min' in the
 * 'domain' matrix.
 *
 */
z_type_t
fm_solver_min_bound(PipMatrix* domain, PipMatrix* context, int dim)
{
  return fm_solver_bound(domain, context, dim, 1);
}


/**
 * Returns the maximal value of variable at column 'min' in the
 * 'domain' matrix.
 *
 */
z_type_t
fm_solver_max_bound(PipMatrix* domain, PipMatrix* context, int dim)
{
  return fm_solver_bound(domain, context, dim, 0);
}


/**
 * Lexicographically largest computation.
 *
 *
 * \brief Compute lexio-largest solution. If upper bound is
 * infinite, use argument 'max' instead. is_integral is a bit to
 * force computation of integral lexico-largest vector.
 *
 * input s_fm_solution_t*   : input solution for the system.
 * input z_type_t	     : maximum value to use for infinite bounds.
 * input int		     : set to 0 for rational values, to 1 for integral
 *			      values.
 *
 * output s_fm_rational_t** : A NULL-terminated array of rationals*, where
 *			       denominators are be '1' if is_integral is set.
 *
 */
s_fm_rational_t**
fm_solver_maxlexico(s_fm_solution_t* sol, z_type_t max, int is_integral)
{
  unsigned i;
  s_fm_vector_t* vect = fm_vector_alloc (sol->size + 1);
  s_fm_rational_t* rUb = fm_rational_alloc ();
  s_fm_rational_t* lb = NULL;
  s_fm_rational_t* Ub = NULL;
  s_fm_rational_t** res;


  // Initialize the solution with max.
  z_type_t one;
  Z_INIT(one);
  Z_ASSIGN_SI(one, 1);
  fm_rational_assign (rUb, max, one);
  for (i = 1; i < sol->size + 1; ++i)
    fm_vector_assign_idx (vect, rUb, i);
  Z_CLEAR(one);

  // Compute the lower bound for each variable in the solution.
  for (i = 0; i < sol->size; ++i)
    {
      fm_solver_compute_min (&lb, sol->solution[i].positive,
			     vect, i, is_integral);
      fm_solver_compute_max (&Ub, sol->solution[i].negative,
			     vect, i, is_integral);

      if (sol->solution[i].positive != NULL &&
	  sol->solution[i].negative != NULL &&
	  fm_rational_cmp (lb, Ub) > 0)
	{
	  fm_vector_free (vect);
	  fm_rational_free (lb); lb = NULL;
	  fm_rational_free (Ub); Ub = NULL;
	  vect = NULL;
	  break;
	}

      if (sol->solution[i].negative != NULL)
	fm_vector_assign_idx (vect, Ub, i + 1);
      fm_rational_free (lb); lb = NULL;
      fm_rational_free (Ub); Ub = NULL;
    }

  // No solution exists, output NULL.
  if (vect == NULL)
    return NULL;

  // Output a NULL-terminated array with the desired values.
  res = XMALLOC(s_fm_rational_t*, sol->size + 1);
  res[sol->size] = NULL;
  for (i = 0; i < sol->size; ++i)
    {
      res[i] = fm_rational_alloc ();
      fm_rational_assign (res[i], vect->vector[i + 1].num,
			  vect->vector[i + 1].denum);
    }
  fm_vector_free (vect);

  return res;
}



#define AGET(A, i, j) (&(A->lines[i]->vector[j]))
#define RABS(X) (Z_ABS_(((float)(((float)(X->num))/((float)(X->denum))))))



/**
 * Input: A system of linear equation
 * Output: A list of replacement vectors (on the `positive'
 * attribute), or NULL if there is a contradiction in the input system
 * format: x = 2 -> [ 0 1 2 ] in pip so output is [ 0 2 ]
 *     0x + 0y -2z + t = 1-> [ 0 0 0 -2 -1 ] (value for the current variable
 *                                            to replace is always 1).
 */
s_fm_solution_t*
fm_solver_linind (s_fm_system_t* A)
{
  int i;
  int j;
  int k,u;
  int max_ind;
  s_fm_rational_t* max_val = fm_rational_alloc ();
  s_fm_rational_t* val;
  s_fm_rational_t* rtmp = fm_rational_alloc ();
  s_fm_vector_t* tmp;

  // First step. Classical Gauss elimination, but starting form the
  // last column.
  /////////////////////////////////////////////////////////////////////////
  // i -> m
  // j -> n
  i = 0;
  j = A->nb_cols - 2;
  while (i < A->nb_lines && j > 0)
    {
      // #Find pivot in column j, starting in row i:
      fm_rational_copy (max_val, AGET(A,i,j));
      max_ind = i;
      for (k=i+1; k < A->nb_lines; ++k)
	{
	  val = AGET(A,k,j);
	  if (RABS(val) > RABS(max_val))
	    {
	      fm_rational_copy (max_val, val);
	      max_ind = k;
	    }
	}

      if (Z_CMP_SI(max_val->num, !=, 0))
	{
	  // switch rows i and max_ind #but remain the values of i
	  // and max_ind
	  tmp = A->lines[i];
	  A->lines[i] = A->lines[max_ind];
	  A->lines[max_ind] = tmp;

	  //#Now A[i, j] will contain the same value as max_val
	  // divide row i by max_val
	  for (k = 1; k < A->nb_cols; ++k)
	    fm_rational_div (&(A->lines[i]->vector[k]),
			     &(A->lines[i]->vector[k]),
			     max_val);

	  // #Now A[i, j] will have the value 1
	  for (u = i+1; u < A->nb_lines; ++u)
	    {
	      fm_rational_copy (max_val, &(A->lines[u]->vector[j]));
	      // subtract A[u, j] * row i from row u
	      for (k = 1; k < A->nb_cols; ++k)
		{
		  fm_rational_copy (rtmp, &(A->lines[i]->vector[k]));
		  fm_rational_mul (rtmp, rtmp, max_val);
		  fm_rational_sub (&(A->lines[u]->vector[k]),
				   &(A->lines[u]->vector[k]),
				   rtmp);
		}
	    }
	  ++i;
	}
      --j;
    }
  ///////////////////////////////////////////////////////////////////////////

  // Second step. Substitute lines.
  ///////////////////////////////////////////////////////////////////////////
  for (j = 1, i = A->nb_cols - 2; i > 0; --i)
    {
      while (j < A->nb_lines && Z_CMP_SI(A->lines[j]->vector[i].num, !=, 0))
	{
	  // Substract the j-line to all lines in the system.
	  for (k = 0; k < j; ++k)
	    {
	      tmp = fm_vector_dup (A->lines[j]);
	      for (u = 1; u < A->nb_cols; ++u)
		fm_rational_mul (&(tmp->vector[u]), &(tmp->vector[u]),
				 &(A->lines[k]->vector[i]));
	      fm_vector_sub (A->lines[k], A->lines[k], tmp);
	      fm_vector_free (tmp);
	    }
	  ++j;
	}

    }
  ///////////////////////////////////////////////////////////////////////////

  // Check for solution integrity.
  // FIXME: This is a weak check.
  for (i = 0; i < A->nb_lines; ++i)
    {
      for (j = 1; j < A->nb_cols - 1 &&
	     Z_CMP_SI(A->lines[i]->vector[j].num, ==, 0); ++j)
	;
      if (j == A->nb_cols - 1 && Z_CMP_SI(A->lines[i]->vector[j].num, !=, 0))
	{
	  assert (! "Contradiction in the system of equalities.");
	  return NULL;
	}
    }
  // Build the output solution.
  int count = 0;
  for (i = 0; i < A->nb_lines; ++i)
    if (! fm_vector_is_null (A->lines[i]))
      ++count;

  s_fm_solution_t* sol = fm_solution_alloc (A->nb_cols - 2);
  for (i = 0; i < count; ++i)
    {
      for (max_ind = A->nb_cols - 2;
	   Z_CMP_SI(A->lines[i]->vector[max_ind].num, ==, 0); --max_ind)
	;
      tmp = fm_vector_alloc (max_ind + 1);
      fm_vector_shrink (tmp, A->lines[i], max_ind - 1);
      fm_list_add_head (&(sol->solution[max_ind - 1].positive), tmp);
    }

  fm_rational_free (max_val);
  fm_rational_free (rtmp);

  return sol;
}






int
fm_solver_gauss (s_fm_system_t* A)
{
  int i=0;
  int j=1;
  int k,u;
  int max_ind;
  s_fm_rational_t* max_val = fm_rational_alloc ();
  s_fm_rational_t* val;
  s_fm_rational_t* rtmp = fm_rational_alloc ();
  s_fm_vector_t* tmp;

  ///////////////////////////////////////////////////////////////////////////
  // i -> m
  // j -> n
  while (i < A->nb_lines && j < A->nb_cols - 1)
    {
      // #Find pivot in column j, starting in row i:
      fm_rational_copy (max_val, AGET(A,i,j));
      max_ind = i;
      for (k=i+1; k < A->nb_lines; ++k)
	{
	  val = AGET(A,k,j);
	  if (RABS(val) > RABS(max_val))
	    {
	      fm_rational_copy (max_val, val);
	      max_ind = k;
	    }
	}

      if (Z_CMP_SI(max_val->num, !=, 0))
	{
	  // switch rows i and max_ind #but remain the values of i
	  // and max_ind
	  tmp = A->lines[i];
	  A->lines[i] = A->lines[max_ind];
	  A->lines[max_ind] = tmp;

	  //#Now A[i, j] will contain the same value as max_val
	  // divide row i by max_val
	  for (k = 1; k < A->nb_cols; ++k)
	    fm_rational_div (&(A->lines[i]->vector[k]),
			     &(A->lines[i]->vector[k]),
			     max_val);

	  // #Now A[i, j] will have the value 1
	  for (u = i+1; u < A->nb_lines; ++u)
	    {
	      fm_rational_copy (max_val, &(A->lines[u]->vector[j]));
	      // subtract A[u, j] * row i from row u
	      for (k = 1; k < A->nb_cols; ++k)
		{
		  fm_rational_copy (rtmp, &(A->lines[i]->vector[k]));
		  fm_rational_mul (rtmp, rtmp, max_val);
		  fm_rational_sub (&(A->lines[u]->vector[k]),
				   &(A->lines[u]->vector[k]),
				   rtmp);
		}
	    }
	  ++i;
	}
      ++j;
    }
  if (i == A->nb_lines)
    --i;
  if (j == A->nb_cols - 1)
    --j;
  ///////////////////////////////////////////////////////////////////////////

  // Backward substitution.
  while (i >= 0 && j >= 1)
    {
      //#Find pivot in column j, starting in row i:
      fm_rational_copy (max_val, AGET(A,i,j));
      max_ind = i;
      for (k = i - 1; k >= 0; --k)
	{
	  val = AGET(A,k,j);
	  if (RABS(val) > RABS(max_val))
	    {
	      fm_rational_copy (max_val, val);
	      max_ind = k;
	    }
	}
      if (Z_CMP_SI(max_val->num, !=, 0))
	{
	  // switch rows i and max_ind #but remain the values of i
	  // and max_ind
	  tmp = A->lines[i];
	  A->lines[i] = A->lines[max_ind];
	  A->lines[max_ind] = tmp;

	  //#Now A[i, j] will contain the same value as max_val
	  // divide row i by max_val
	  for (k = 1; k < A->nb_cols; ++k)
	    fm_rational_div (&(A->lines[i]->vector[k]),
			     &(A->lines[i]->vector[k]),
			     max_val);

	  // #Now A[i, j] will have the value 1
	  for (u = i - 1; u >= 0; --u)
	    {
	      fm_rational_copy (max_val, &(A->lines[u]->vector[j]));
	      // subtract A[u, j] * row i from row u
	      for (k = 1; k < A->nb_cols; ++k)
		{
		  fm_rational_copy (rtmp, &(A->lines[i]->vector[k]));
		  fm_rational_mul (rtmp, rtmp, max_val);
		  fm_rational_sub (&(A->lines[u]->vector[k]),
				   &(A->lines[u]->vector[k]),
				   rtmp);
		}
	    }
	  --i;
	}
      --j;
    }

  fm_rational_free (max_val);
  fm_rational_free (rtmp);
}
