/*
 * vector.c: this file is part of the FM project.
 *
 * FM, a fast and optimized C implementation of Fourier-Motzkin
 * projection algorithm.
 *
 * Copyright (C) 2006,2007,2008 Louis-Noel Pouchet
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 3
 * of the License, or (at your option) any later version.
 *
 * The complete GNU Lesser General Public Licence Notice can be found
 *  as the `COPYING.LESSER' file in the root directory.
 *
 * Author:
 * Louis-Noel Pouchet <Louis-Noel.Pouchet@inria.fr>
 *
 */
#if HAVE_CONFIG_H
# include <fm/config.h>
#endif

#include <fm/common.h>
#include <fm/vector.h>


/**
 *
 * The key must work on 'size - 1' since its value is used in
 * subsumption test.
 *
 */
void
fm_vector_compute_key (z_type_t* key, s_fm_vector_t* v)
{
  unsigned i = v->size - 1;
  unsigned j;

  *key = 0;
  for (i = 0; i < v->size - 1; ++i)
    {
      *key += i * v->vector[i].num * v->vector[i].denum;
      *key <<= 1;
    }
}


/**
 *
 * Beware to consider appropriate length to deal with 'eq/ineq' bit.
 *
 */
s_fm_vector_t*
fm_vector_alloc (size_t size)
{
  if (size < 0)
    {
      assert (! "Size is less than 0.");
      return NULL;
    }

  unsigned i;
  s_fm_vector_t* v = XMALLOC(s_fm_vector_t, 1);
  v->size = size;
  v->vector = XMALLOC(s_fm_rational_t, size);

  for (i = 0; i < size; ++i)
    fm_rational_init (&(v->vector[i]));

  fm_vector_compute_key(&(v->key), v);

  return v;
}

s_fm_vector_t*
fm_vector_dup (s_fm_vector_t* v1)
{
  if (v1 == NULL)
    return NULL;

  s_fm_vector_t* v = XMALLOC(s_fm_vector_t, 1);
  v->size = v1->size;
  v->vector = XMALLOC(s_fm_rational_t, v1->size);

  fm_vector_assign (v, v1);

  return v;
}



int
fm_vector_init (s_fm_vector_t* v, size_t size)
{
  if (size < 0)
    {
      assert ("Size is less than 0.");
      return FM_STATUS_ERROR;
    }

  unsigned i;
  if (v->vector == NULL)
    {
      v->vector = XMALLOC(s_fm_rational_t, size);
      v->size = size;
    }

  for (i = 0; i < v->size; ++i)
    fm_rational_init (&(v->vector[i]));

  fm_vector_compute_key(&(v->key), v);

  return FM_STATUS_OK;
}


void
fm_vector_free (s_fm_vector_t* v)
{
  unsigned i;

  if (v != NULL)
    XFREE(v->vector);
  XFREE(v);
}


/**
 * Allowed separators for vector are ',' '\t' ' '
 */
static
int
fm_issep (char c)
{
  if (c == ' ' ||
      c == '\t' ||
      c == ',')
    return 1;
  return 0;
}


void
fm_vector_read (FILE* stream, s_fm_vector_t* v, unsigned size)
{
  unsigned i;
  char buffer[2048];
  char* buf = buffer;
  char ratbuf[128];
  char intbuf[128];
  z_type_t num;
  z_type_t denum;

  Z_INIT(num);
  Z_INIT(denum);

  if (v->vector == NULL)
    {
      v->vector = XMALLOC(s_fm_rational_t, size);
      v->size = size;
    }

  /* Skip blank lines. */
  while (fgets(buffer, 2048, stream) == 0)
    ;

  /* Skip commented lines. */
  while ((*buf == '#' || *buf == '\n'))
    fgets(buffer, 2048, stream);

  buf = buffer;
  while ((isspace (*buf) || *buf == '[') && *buf != '\n')
    buf++;

  /* Initialize the patterns. */
  strcpy (ratbuf, Z_STRING_MODIFIER);
  strcat (ratbuf, "/");
  strcat (ratbuf, Z_STRING_MODIFIER);
  strcat (ratbuf, "");
  strcpy (intbuf, Z_STRING_MODIFIER);
  strcat (intbuf, "");

  /* Scan the input stream. */
  for (i = 0; i < size; ++i)
    {
      Z_ASSIGN_SI(denum, 1);
      if (sscanf(buf, ratbuf,
		 &num, &denum) < 2)
	sscanf(buf, intbuf, &num);
      while (!fm_issep (*buf) && *buf != '\n')
	buf++;
      while (fm_issep (*buf) && *buf != '\n')
	buf++;
      if (*buf == '\n' && i < size - 1)
	{
	  fprintf (stderr, "Not enough columns\n");
	  exit (1);
	}
      fm_rational_assign (&(v->vector[i]), num, denum);
    }

  Z_CLEAR(num);
  Z_CLEAR(denum);
}

#define FM_MAX_VECT_SIZE 4096

s_fm_vector_t*
fm_vector_read_str (char* stream)
{
  unsigned i;
  char buffer[2048];
  char* buf = buffer;
  char ratbuf[128];
  char intbuf[128];
  z_type_t num;
  z_type_t denum;
  s_fm_vector_t* v;
  s_fm_vector_t* ret;
  int size = 0;

  Z_INIT(num);
  Z_INIT(denum);

  v = fm_vector_alloc (FM_MAX_VECT_SIZE);

  strcpy (buffer, stream);

  buf = buffer;
  while ((isspace (*buf) || *buf == '[') && *buf != '\n' && *buf != '\0')
    buf++;
  /* Initialize the patterns. */
  strcpy (ratbuf, Z_STRING_MODIFIER);
  strcat (ratbuf, "/");
  strcat (ratbuf, Z_STRING_MODIFIER);
  strcat (ratbuf, "");
  strcpy (intbuf, Z_STRING_MODIFIER);
  strcat (intbuf, "");
  /* Scan the input stream. */
  for (i = 0; i < FM_MAX_VECT_SIZE; ++i)
    {
      Z_ASSIGN_SI(denum, 1);
      if (sscanf(buf, ratbuf,
		 &num, &denum) < 2)
	sscanf(buf, intbuf, &num);
      while (!fm_issep (*buf) && *buf != '\n' && *buf != '\0')
	buf++;
      while (fm_issep (*buf) && *buf != '\n' && *buf != '\0')
	buf++;
      fm_rational_assign (&(v->vector[i]), num, denum);
      ++size;
      if (*buf == '\n' || *buf == '\0')
	break;
    }
  /* shrink the vector. */
  ret = fm_vector_alloc (size);
  for (i = 0; i < size; ++i)
    fm_rational_assign (&(ret->vector[i]),
			v->vector[i].num, v->vector[i].denum);

  fm_vector_free (v);
  Z_CLEAR(num);
  Z_CLEAR(denum);
  XFREE(stream);

  return ret;
}



void
fm_vector_print (FILE* stream, s_fm_vector_t* v)
{
  unsigned i;

  if (v == NULL)
    {
      fprintf (stream, "[(null)]");
      return;
    }

  fprintf (stream, "[");
  for (i = 0; i < v->size - 1; ++i)
    {
      fm_rational_print (stream, &(v->vector[i]));
      fprintf (stream, " ");
    }
  fm_rational_print (stream, &(v->vector[i]));
  fprintf (stream, "]");
}


int
fm_vector_assign (s_fm_vector_t* v, s_fm_vector_t* v1)
{
  if ((v == NULL || v1 == NULL))
    {
      assert (! "One of the parameters is NULL.");
      return FM_STATUS_ERROR;
    }

  if (v->size != v1->size)
    {
      assert (! "Size are differents.");
      return FM_STATUS_ERROR;
    }

  unsigned i;

  for (i = 0; i < v->size; ++i)
    {
      Z_ASSIGN(v->vector[i].num, v1->vector[i].num);
      Z_ASSIGN(v->vector[i].denum, v1->vector[i].denum);
    }

  Z_ASSIGN(v->key, v1->key);

  return FM_STATUS_OK;
}


int
fm_vector_assign_at (s_fm_vector_t* v, s_fm_vector_t* v1, unsigned idx)
{
 if ((v == NULL || v1 == NULL))
    {
      assert (! "One of the parameters is NULL.");
      return FM_STATUS_ERROR;
    }

  if (idx + v1->size - 1 > v->size)
    {
      assert (! "Wrong index given.");
      return FM_STATUS_ERROR;
    }

  unsigned i;

  for (i = 1; i < v1->size; ++i)
    {
      Z_ASSIGN(v->vector[idx + i].num, v1->vector[i].num);
      Z_ASSIGN(v->vector[idx + i].denum, v1->vector[i].denum);
    }

  Z_ASSIGN_SI(v->vector[0].num, v1->vector[0].num);

  fm_vector_compute_key (&(v->key), v);

  return FM_STATUS_OK;
}


void
fm_vector_set_ineq (s_fm_vector_t* v)
{
  if (v == NULL)
    return;

  Z_ASSIGN_SI(v->vector[0].num, 1);
}


void
fm_vector_set_eq (s_fm_vector_t* v)
{
  if (v == NULL)
    return;

  Z_ASSIGN_SI(v->vector[0].num, 0);
}


int
fm_vector_assign_idx (s_fm_vector_t* v, s_fm_rational_t* r, unsigned idx)
{
  if ((v == NULL || r == NULL))
    {
      assert (! "One of the parameters is NULL.");
      return FM_STATUS_ERROR;
    }

  if (idx < 1 || idx > v->size - 1)
    {
      assert (! "Wrong index given.");
      return FM_STATUS_ERROR;
    }

  fm_rational_assign (&(v->vector[idx]), r->num, r->denum);

  fm_vector_compute_key (&(v->key), v);

  return FM_STATUS_OK;
}


int
fm_vector_expand (s_fm_vector_t* v, s_fm_vector_t* v1)
{
 if ((v == NULL || v1 == NULL))
    {
      assert (! "One of the parameters is NULL.");
      return FM_STATUS_ERROR;
    }

  if (v1->size > v->size)
    {
      assert (! "Wrong size.");
      return FM_STATUS_ERROR;
    }

  unsigned i;

  for (i = 1; i < v1->size - 1; ++i)
    fm_rational_assign (&(v->vector[i]),
			v1->vector[i].num, v1->vector[i].denum);

  fm_rational_assign (&(v->vector[v->size - 1]),
		      v1->vector[i].num,
		      v1->vector[i].denum);

  Z_ASSIGN_SI(v->vector[0].num, v1->vector[0].num);

  fm_vector_compute_key (&(v->key), v);

  return FM_STATUS_OK;
}


int
fm_vector_expand_at (s_fm_vector_t* v, s_fm_vector_t* v1, unsigned idx)
{
 if ((v == NULL || v1 == NULL))
    {
      assert (! "One of the parameters is NULL.");
      return FM_STATUS_ERROR;
    }

  if (v1->size + idx - 1 > v->size)
    {
      assert (! "Wrong size or idx.");
      return FM_STATUS_ERROR;
    }

  unsigned i;

  for (i = 1; i < v1->size - 1; ++i)
    fm_rational_assign (&(v->vector[i + idx]),
			v1->vector[i].num, v1->vector[i].denum);

  fm_rational_assign (&(v->vector[v->size - 1]),
		      v1->vector[i].num,
		      v1->vector[i].denum);

  Z_ASSIGN_SI(v->vector[0].num, v1->vector[0].num);

  fm_vector_compute_key (&(v->key), v);

  return FM_STATUS_OK;
}


int
fm_vector_shrink (s_fm_vector_t* v, s_fm_vector_t* v1, unsigned idx)
{
 if ((v == NULL || v1 == NULL))
    {
      assert (! "One of the parameters is NULL.");
      return FM_STATUS_ERROR;
    }

  if (idx < 0 || idx != v->size - 2)
    {
      assert (! "Wrong index given.");
      return FM_STATUS_ERROR;
    }

  unsigned i;

  for (i = 1; i < idx + 1; ++i)
    fm_rational_assign (&(v->vector[i]),
			v1->vector[i].num, v1->vector[i].denum);

  fm_rational_assign (&(v->vector[i]),
		      v1->vector[v1->size - 1].num,
		      v1->vector[v1->size - 1].denum);

  Z_ASSIGN_SI(v->vector[0].num, v1->vector[0].num);

  fm_vector_compute_key (&(v->key), v);

  return FM_STATUS_OK;
}


int
fm_vector_assign_int_idx (s_fm_vector_t* v, z_type_t i, unsigned idx)
{
  if (v == NULL)
    {
      assert (! "One of the parameters is NULL.");
      return FM_STATUS_ERROR;
    }

  if (idx < 1 || idx > v->size - 1)
    {
      fprintf (stderr, "idx: %d, v->size: %d\n", idx, v->size);
      fm_vector_print (stderr, v); fprintf (stderr, "\n");
      assert (! "Wrong index given.");
      return FM_STATUS_ERROR;
    }

  Z_ASSIGN_SI(v->vector[idx].denum, 1);
  Z_ASSIGN(v->vector[idx].num, i);

  fm_vector_compute_key (&(v->key), v);

  return FM_STATUS_OK;
}


int
fm_vector_is_null (s_fm_vector_t* v)
{
  if (v == NULL)
    {
      assert (! "One of the parameters is NULL.");
      return FM_STATUS_ERROR;
    }

  unsigned i;

  for (i = 1; i < v->size; ++i)
    if (Z_CMP_SI(v->vector[i].num, !=, 0))
      return 0;

  return 1;
}



int
fm_vector_is_empty (s_fm_vector_t* v)
{
  if (v == NULL)
    {
      assert (! "One of the parameters is NULL.");
      return FM_STATUS_ERROR;
    }

  unsigned i;

  for (i = 1; i < v->size - 1; ++i)
    if (Z_CMP_SI(v->vector[i].num, !=, 0))
      return 0;

  return 1;
}


int
fm_vector_is_valid (s_fm_vector_t* v)
{
  if (v == NULL)
    {
      assert (! "One of the parameters is NULL.");
      return FM_STATUS_ERROR;
    }

  unsigned i;

  if (! fm_vector_is_empty (v))
    return 1;
  if (Z_CMP_SI (v->vector[0].num, ==, 0))
    return Z_CMP_SI (v->vector[v->size - 1].num, ==, 0);
  return Z_CMP_SI (v->vector[v->size - 1].num, >=, 0);
}


int
fm_vector_is_scalar_cst (s_fm_vector_t* v)
{
  if (v == NULL)
    {
      assert (! "One of the parameters is NULL.");
      return FM_STATUS_ERROR;
    }

  unsigned i;
  unsigned val = 0;

  for (i = 1; i < v->size - 1; ++i)
    if (Z_CMP_SI(v->vector[i].num, !=, 0))
      val++;

  return val == 1;
}


int
fm_vector_opp (s_fm_vector_t* v, s_fm_vector_t* v1)
{
  if (v == NULL || v1 == NULL)
    {
      assert (! "One of the parameters is NULL.");
      return FM_STATUS_ERROR;
    }

  if (v->size != v1->size)
    {
      assert (! "Size are differents.");
      return FM_STATUS_ERROR;
    }

  unsigned i;

  for (i = 1; i < v->size; ++i)
    fm_rational_opp (&(v->vector[i]), &(v1->vector[i]));

  Z_ASSIGN_SI(v->vector[0].num, v1->vector[0].num);

  fm_vector_compute_key (&(v->key), v);

  return FM_STATUS_OK;
}


int
fm_vector_add (s_fm_vector_t* v, s_fm_vector_t* v1, s_fm_vector_t* v2)
{
  if ((v == NULL || v1 == NULL || v2 == NULL))
    {
      assert (! "One of the parameters is NULL.");
      return FM_STATUS_ERROR;
    }

  if (v->size != v1->size || v1->size != v2->size)
    {
      assert (! "Size are differents.");
      return FM_STATUS_ERROR;
    }

  unsigned i;

  //   printf("enter for %d\n", v->size);
  for (i = 1; i < v->size; ++i) {
    fm_rational_add (&(v->vector[i]), &(v1->vector[i]), &(v2->vector[i]));
    //    printf("%d ", i);
    //    fflush(stdout);
  }
  //  printf("\ndone\n");


  Z_ADD(v->vector[0].num, v1->vector[0].num, v2->vector[0].num);
  if (Z_CMP_SI(v->vector[0].num, >, 1))
    Z_ASSIGN_SI(v->vector[0].num, 1);

  //  printf("end\n");

  fm_vector_compute_key (&(v->key), v);

  return FM_STATUS_OK;
}


int
fm_vector_sub (s_fm_vector_t* v, s_fm_vector_t* v1, s_fm_vector_t* v2)
{
  if ((v == NULL || v1 == NULL || v2 == NULL))
    {
      assert (! "One of the parameters is NULL.");
      return FM_STATUS_ERROR;
    }

  if (v->size != v1->size || v1->size != v2->size)
    {
      assert (! "Size are differents.");
      return FM_STATUS_ERROR;
    }

  unsigned i;

  for (i = 1; i < v->size; ++i)
    fm_rational_sub (&(v->vector[i]), &(v1->vector[i]), &(v2->vector[i]));

  Z_ADD(v->vector[0].num, v1->vector[0].num, v2->vector[0].num);
  if (Z_CMP_SI(v->vector[0].num, >, 1))
    Z_ASSIGN_SI(v->vector[0].num, 1);

  fm_vector_compute_key (&(v->key), v);

  return FM_STATUS_OK;
}


int
fm_vector_normalize_idx (s_fm_vector_t* v, s_fm_vector_t* v1, unsigned idx)
{
  if ((v == NULL || v1 == NULL))
    {
      assert (! "One of the parameters is NULL.");
      return FM_STATUS_ERROR;
    }

  if (v->size != v1->size)
    {
      assert (! "Size are differents.");
      return FM_STATUS_ERROR;
    }

  // NOTE: Enabling the full condition inteferes with subsume_check.
  // FIXME: Remove this bug !
  // NOTE: The bug doesn't seem to occur anymore.
  if (Z_CMP_SI(v1->vector[idx].num, ==, 0) ||
      (Z_CMP_SI(v1->vector[idx].denum, ==, 1) &&
       (Z_CMP_SI(v1->vector[idx].num, ==, 1) ||
	Z_CMP_SI(v1->vector[idx].num, ==, -1))))
    return FM_STATUS_OK;


  s_fm_rational_t* normalizer = fm_rational_alloc ();
  z_type_t anum;
  unsigned i;

  Z_INIT(anum);
  Z_ASSIGN(anum, v1->vector[idx].num);
  //if (Z_CMP_SI(anum, <, 0))
  //  Z_OPP(anum, anum);
  Z_ABS(anum, anum);

  fm_rational_assign (normalizer, v1->vector[idx].denum, anum);

  Z_CLEAR(anum);

  for (i = 1; i < v->size; ++i)
    fm_rational_mul (&(v->vector[i]), &(v1->vector[i]), normalizer);

  fm_rational_free (normalizer);
  Z_ASSIGN(v->vector[0].num, v1->vector[0].num);

  fm_vector_compute_key (&(v->key), v);

  return FM_STATUS_OK;
}


int
fm_vector_to_z (s_fm_vector_t* v, s_fm_vector_t* v1)
{
  if ((v == NULL || v1 == NULL))
    {
      assert (! "One of the parameters is NULL.");
      return FM_STATUS_ERROR;
    }

  if (v->size != v1->size || v1->size < 2)
    {
      assert (! "Error in size(s).");
      return FM_STATUS_ERROR;
    }

  unsigned i;
  z_type_t row_lcm;
  Z_INIT(row_lcm);

  Z_ASSIGN(row_lcm, v1->vector[1].denum);

  for (i = 2; i < v1->size; ++i)
    Z_LCM(row_lcm, row_lcm, v1->vector[i].denum);

  if (Z_CMP_SI(row_lcm, !=, 1))
    for (i = 1; i < v1->size; ++i)
      {
	Z_MUL(v->vector[i].num, v1->vector[i].num, row_lcm);
	if (Z_CMP_SI(v1->vector[i].denum, !=, 1))
	  Z_DIV(v->vector[i].num, v->vector[i].num, v1->vector[i].denum);
	Z_ASSIGN_SI(v->vector[i].denum, 1);
      }
  else
    for (i = 1; i < v1->size; ++i)
      {
	Z_ASSIGN(v->vector[i].num, v1->vector[i].num);
	Z_ASSIGN(v->vector[i].denum, v1->vector[i].denum);
      }

  Z_ASSIGN(v->vector[0].num, v1->vector[0].num);
  Z_CLEAR(row_lcm);

  fm_vector_compute_key (&(v->key), v);

  return FM_STATUS_OK;
}


int
fm_vector_resize (s_fm_vector_t* v, s_fm_vector_t* v1)
{
  assert(! "FIXME: Code this function.");
}


int
fm_vector_equal (s_fm_vector_t* v1, s_fm_vector_t* v2)
{
  if ((v1 == NULL || v2 == NULL))
    return 0;

  if (v1->size != v2->size)
    {
      assert (! "Size are differents.");
      return FM_STATUS_ERROR;
    }

  if (v1->key != v2->key)
    return 0;

  if (Z_CMP(v1->vector[0].num, !=, v2->vector[0].num))
    return 0;

  unsigned i;
  for (i = 1; i < v1->size; ++i)
    if (Z_CMP(v1->vector[i].num, !=, v2->vector[i].num) ||
	Z_CMP(v1->vector[i].denum, !=, v2->vector[i].denum))
      return 0;

  return 1;
}


int
fm_vector_do_subsume (s_fm_vector_t* v1, s_fm_vector_t* v2)
{
  if ((v1 == NULL || v2 == NULL))
    {
      assert (! "One of the parameters is NULL.");
      return FM_STATUS_ERROR;
    }

  if (v1->size != v2->size)
    {
      assert (! "Size are differents.");
      return FM_STATUS_ERROR;
    }

  if (Z_CMP(v1->vector[0].num, !=, v2->vector[0].num))
    /// LNP
/*  || */
/*       Z_CMP(v1->vector[0].num, ==, 0)) */
    return 0;

  if (v1->key != v2->key)
    return 0;

  unsigned i;
  for (i = 1; i < v1->size - 1; ++i)
    if (Z_CMP(v1->vector[i].num, !=, v2->vector[i].num) ||
	Z_CMP(v1->vector[i].denum, !=, v2->vector[i].denum))
      return 0;

  if (Z_CMP(v1->vector[i].num, ==, v2->vector[i].num) &&
      Z_CMP(v1->vector[i].denum, ==, v2->vector[i].denum))
    return 1;

  z_type_t num1;
  z_type_t num2;
  z_type_t the_lcm;
  Z_INIT(num1);
  Z_INIT(num2);
  Z_INIT(the_lcm);

  Z_LCM(the_lcm, v1->vector[i].denum, v2->vector[i].denum);
  if (Z_CMP_SI(the_lcm, !=, 1))
    {
      Z_MUL(num1, v1->vector[i].num, the_lcm);
      Z_DIV(num1, num1, v1->vector[i].denum);
      Z_MUL(num2, v2->vector[i].num, the_lcm);
      Z_DIV(num2, num2, v2->vector[i].denum);
    }
  else
    {
      Z_ASSIGN(num1, v1->vector[i].num);
      Z_ASSIGN(num2, v2->vector[i].num);
    }

  if (num2 < num1)
    {
      Z_ASSIGN(v1->vector[i].num, v2->vector[i].num);
      Z_ASSIGN(v1->vector[i].denum, v2->vector[i].denum);
    }

  Z_CLEAR(num1);
  Z_CLEAR(num2);
  Z_CLEAR(the_lcm);

  return 1;
}
