/*
 * fm_getconnected.c: this file is part of the FM project.
 *
 * FM, a fast and optimized C implementation of Fourier-Motzkin
 * projection algorithm.
 *
 * Copyright (C) 2006-2008 Louis-Noel Pouchet
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 3
 * of the License, or (at your option) any later version.
 *
 * The complete GNU Lesser General Public Licence Notice can be found
 *  as the `COPYING.LESSER' file in the root directory.
 *
 * Author:
 * Louis-Noel Pouchet <Louis-Noel.Pouchet@inria.fr>
 *
 */
#if HAVE_CONFIG_H
# include <config.h>
#endif

#include <fm/common.h>
#include <fm/system.h>
#include <fm/solution.h>
#include <fm/solver.h>
#include <fm/compsol.h>


void
fms_usage ()
{
  fprintf (stderr, "Usage: fm-getconnected input-file [input-file]\n");
  exit (1);
}

void
fms_error ()
{
  fprintf (stderr, "fm-getconnected: Bad input file.\n");
  exit (1);
}

#define FM_MAX_INPUT_SYSTEMS 512


int
main (int argc, char * const argv[])
{
  if (argc < 2)
    fms_usage ();

  FILE* f;
  s_fm_system_t* systems[FM_MAX_INPUT_SYSTEMS];
  int i, j;
  int size_sys = 0;
  int nb_sys;
  int ci;
  int** classes;
  int offset;

  // Read the input systems.
  for (nb_sys = 0; nb_sys < argc - 1; ++nb_sys)
    {
      f = fopen (argv[nb_sys + 1], "r");
      if (f == NULL)
	fms_error ();
      systems[nb_sys] = fm_system_read (f);
      fclose (f);
      if (size_sys == 0)
	size_sys = systems[nb_sys]->nb_cols - 2;
      // Different system sizes is not allowed.
      else if (size_sys != systems[nb_sys]->nb_cols - 2)
	fms_error ();
    }

  // Compute the connected variables (classes), for each system.
  for (i = 0, offset = 0; i < nb_sys; ++i, offset += size_sys)
    {
      printf ("Dealing with dimension %d\n", i + 1);

      classes = fm_system_getconnected (systems[i]);
      for (ci = 0; classes[ci]; ++ci)
	{
	  printf ("class: ");
	  for (j = 0; classes[ci][j] != -1; ++j)
	    printf ("%d ", classes[ci][j] + 1 + offset);
	  printf ("\n");
	  XFREE(classes[ci]);
	}
      fm_system_free (systems[i]);
      XFREE(classes);
    }

  return 0;
}

