/*
 * fsspace2.c: this file is part of the LetSee project.
 *
 * LetSee, the LEgal Transformation SpacE Explorator.
 *
 * Copyright (C) 2008 Louis-Noel Pouchet
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * The complete GNU General Public Licence Notice can be found as the
 * `COPYING' file in the root directory.
 *
 * Author:
 * Louis-Noel Pouchet <Louis-Noel.Pouchet@inria.fr>
 */
#if HAVE_CONFIG_H
# include <letsee/config.h>
#endif

#include <letsee/fsspace2.h>


#undef echo
#define echo(S) { printf(#S); printf("\n"); }



static
s_vertex_t**
compute_posnegclos (s_graph_t* graph, int start, int stop)
{
  int i, j, k, l;
  int count = 0;
  int sz = graph->vertex_count;
  s_vertex_t** res = XMALLOC(s_vertex_t*, sz + 1);
  s_vertex_t* v;
  s_edge_t* e;
  // It's a null-terminated array.
  for (i = 0; i <= sz; ++i)
    res[i] = NULL;

  // Create the (partial) adjacency matrix.
  int** matrix = XMALLOC(int*, sz);
  for (i = 0; i < sz; ++i)
    {
      matrix[i] = XMALLOC(int, sz);
      for (j = 0; j < sz; ++j)
	matrix[i][j] = 0;
    }

  // Fill the adjacency matrix.
  for (e = graph->first_edge; e; e = e->next)
    if (start == 0)
      {
	if (e->src->id <= stop && e->dst->id <= stop)
	  {
	    matrix[e->src->id][e->dst->id] = 1;
	    matrix[e->dst->id][e->src->id] = 1;
	  }
      }
    else
      if (e->src->id >= start && e->dst->id >= start)
	{
	  matrix[e->src->id][e->dst->id] = 1;
	  matrix[e->dst->id][e->src->id] = 1;
	}

  // Compute the transitive closure with Warshall algorithm.
  for( i = 0; i < sz; i++)
    for (j = 0; j < sz; j++)
      if (matrix[i][j] == 1)
	for (k = 0; k < sz; k++)
	  if (matrix[j][k] == 1)
	    matrix[i][k] = 1;

  // Reset start to the id of F.
  if (start == 0)
    start = stop;
  // F is in the transitive closure.
  matrix[start][start] = 1;
  // Compute F+/-.
  for (i = 0, count = 0, v = graph->root; i < sz; ++i, v = v->next)
    if (matrix[start][i] != 0)
      res[count++] = v;

  // Be clean.
  for (i = 0; i < sz; ++i)
      XFREE(matrix[i]);
  XFREE(matrix);

  return res;
}


static
s_vertex_t**
compute_posclos (s_graph_t* deps, s_vertex_t* r)
{
  return compute_posnegclos (deps, r->id, deps->vertex_count);
}


static
s_vertex_t**
compute_negclos (s_graph_t* deps, s_vertex_t* r)
{
  return compute_posnegclos (deps, 0, r->id);
}


static
int
must_skew (s_fm_solution_t* t, int i, int j,
	   CandlProgram* program)
{
  if (t == NULL)
    // Should never occur.
    return 0;

  int k;
  s_fm_vector_t* v;
  CandlStatement* st1 = program->statement[i];
  CandlStatement* st2 = program->statement[j];
  s_fm_system_t* s = fm_solution_to_system (t);

  // Add \sum i_R > 0.
  v = fm_vector_alloc (s->nb_cols);
  for (k = 0; k < st1->depth; ++k)
    fm_vector_assign_int_idx (v, 1, k + 1);
  fm_vector_assign_int_idx (v, -1, v->size - 1);
  fm_vector_set_ineq (v);
  fm_system_add_line (s, v);

  // Add \sum i_S > 0.
  v = fm_vector_alloc (s->nb_cols);
  for (; k < st1->depth + st2->depth; ++k)
    fm_vector_assign_int_idx (v, 1, k + 1);
  fm_vector_assign_int_idx (v, -1, v->size - 1);
  fm_vector_set_ineq (v);
  fm_system_add_line (s, v);

  // Check for the lexmin.
  PipQuast* quast = fm_piptools_pip (s, NULL, FM_PIPTOOLS_INT);
  fm_system_free (s);
  if (quast == NULL || quast->list == NULL)
    // Should never occur.
    return 0;
  else
    {
      PipList* l = quast->list;
      v = fm_vector_alloc (t->size + 1);
      for (k = 1; l; l = l->next, ++k)
	fm_vector_assign_int_idx (v, l->vector->the_vector[0], k);
    }
  pip_quast_free (quast);
  int l;
  int ret = 1;
  int sz = 0;
  for (l = 1; l < st1->depth; ++l)
    if (v->vector[l].num != 0)
      ++sz;
  if (sz == 1)
    for (sz = 0, ++l; l < st1->depth + st2->depth; ++l)
      if (v->vector[l].num != 0)
	++sz;
  fm_vector_free (v);

  return sz > 1;
}


static
int
must_distribute (s_fm_solution_t* t, int ii, int jj,
		 CandlProgram* program)
{
  CandlStatement* st1 = program->statement[ii];
  CandlStatement* st2 = program->statement[jj];
  s_fm_vector_t* v1 = fm_vector_alloc (t->size + 2);
  s_fm_vector_t* v2 = fm_vector_alloc (t->size + 2);
  int i;

  // Add \sum i > 0.
  for (i = 0; i < st1->depth; ++i)
    fm_vector_assign_int_idx (v1, 1, i + 1);
  fm_vector_assign_int_idx (v1, -1, v1->size - 1);
  fm_vector_set_ineq (v1);

  // Add \sum i > 0.
  for (; i < st1->depth + st2->depth; ++i)
    fm_vector_assign_int_idx (v2, 1, i + 1);
  fm_vector_assign_int_idx (v2, -1, v2->size - 1);
  fm_vector_set_ineq (v2);

  s_fm_solution_t* sol = fm_solution_dup (t);
  fm_solution_add_line_at (sol, v1, st1->depth);
  fm_solution_add_line_at (sol, v2, i);

  int ret = fm_piptools_check_sol (sol, FM_PIPTOOLS_INT);
  fm_solution_free (sol);
/*   if (!ret) */
/*     printf ("MUST DISTRIBUTE %d and %d\n", ii + 1, jj + 1); */
  return !ret;
}

static
int
must_fuse (s_fm_solution_t* t, int i, int j,
	   CandlProgram* program)
{
  s_fm_vector_t* v = fm_vector_alloc (t->size + 1);
  fm_vector_assign_int_idx (v, 1, v->size - 1);
  int ret = fm_solution_point_included (t, v);
  fm_vector_free (v);

/*   if (!ret) */
/*     printf ("MUST FUSE %d and %d\n", i + 1, j + 1); */

  return !ret;
}

static
int
is_legal_perm (s_fm_solution_t* t, int i, int j,
	       int ii, int jj,
	       CandlProgram* program)
{
  if (t == NULL)
    return 0;

  int k;
  s_fm_vector_t* v;
  CandlStatement* st1 = program->statement[i];
  CandlStatement* st2 = program->statement[j];
  s_fm_system_t* s = fm_solution_to_system (t);
  // Put the two iterator variables as the 'first' variable of their
  // own iterator variable segment.
  fm_system_swap_column (s, ii + 1, st1->depth);
  fm_system_swap_column (s, st1->depth + jj + 1, st1->depth + st2->depth);
  // Add \sum i_R > 0.
  v = fm_vector_alloc (s->nb_cols);
  for (k = 0; k < st1->depth; ++k)
    fm_vector_assign_int_idx (v, 1, k + 1);
  fm_vector_assign_int_idx (v, -1, v->size - 1);
  fm_vector_set_ineq (v);
  fm_system_add_line (s, v);

  // Add \sum i_S > 0.
  v = fm_vector_alloc (s->nb_cols);
  for (; k < st1->depth + st2->depth; ++k)
    fm_vector_assign_int_idx (v, 1, k + 1);
  fm_vector_assign_int_idx (v, -1, v->size - 1);
  fm_vector_set_ineq (v);
  fm_system_add_line (s, v);

  // Check for the lexmin.
  PipQuast* quast = fm_piptools_pip (s, NULL, FM_PIPTOOLS_INT);
  fm_system_free (s);
  if (quast == NULL || quast->list == NULL)
    return 0;
  else
    {
      PipList* l = quast->list;
      v = fm_vector_alloc (t->size + 1);
      for (k = 1; l; l = l->next, ++k)
	fm_vector_assign_int_idx (v, l->vector->the_vector[0], k);
    }
  pip_quast_free (quast);
  int l;
  int ret = 0;
  if (v->vector[st1->depth].num != 0 &&
      v->vector[st1->depth + st2->depth].num != 0)
    {
      for (l = 1; l < st1->depth && v->vector[l].num == 0;
	   ++l)
	;
      if (l == st1->depth)
	{
	  for (++l; l < st1->depth + st2->depth && v->vector[l].num == 0; ++l)
	    ;
	  if (l == st1->depth + st2->depth)
	    ret = 1;
	}
    }
/*   if (ret) */
/*     printf ("LEGAL PERM %c%d,%c%d\n", 'i' + ii, i + 1, 'i' + jj, j + 1); */
  fm_vector_free (v);
  return ret;
}

static
s_fm_solution_t*
build_th_constraints (s_graph_t* depgraph,
		      CandlProgram* program,
		      s_ls_options_t* options,
		      int i,
		      int j)
{
  // Create the temporary prgram containing i and j.
  CandlProgram* p = candl_program_malloc (2);
  CandlStatement* stmts[2];
  stmts[0] = program->statement[i];
  stmts[1] = program->statement[j];
  p->statement = stmts;
  p->nb_statements = 2;
  p->context = program->context;

  // Backup originial statement labels.
  int id1 = stmts[0]->label;
  int id2 = stmts[1]->label;

  // Create the temporary dependence graph.
  s_graph_t* g = ls_graph_alloc ();
  s_vertex_t* v;
  CandlDependence* dep;
  int uid = 0;
  for (v = depgraph->root; v; v = v->next)
    {
      dep = v->data;
      if (dep->source->label == id1 || dep->source->label == id2)
	if (dep->target->label == id1 || dep->target->label == id2)
	  ls_graph_create_vertex (g, dep, uid++);
    }

  // Set the new temporary labels.
  stmts[0]->label = 0;
  stmts[1]->label = 1;

  // Compute the legal space.
  int verb = options->verbose;
  int ms = options->maxscale_solver;
  options->verbose = 0;
  options->maxscale_solver = 1;
  s_fm_solution_t* sol = ls_farkas_build_th_constraints
    (g, p, options);
  options->verbose = verb;
  options->maxscale_solver = ms;
  // Restore original label.
  stmts[0]->label = id1;
  stmts[1]->label = id2;

  // Be clean.
  ls_graph_free (g);
  XFREE(p);

  return sol;
}

static
void
dotty_print_fuse (int* screen, CandlProgram* program)
{
  int i, j;

  FILE* f = fopen("/tmp/fuse.dot", "w");
  fprintf (f, "digraph G {\n");
  for (i = 0; i < program->nb_statements - 1; ++i)
    for (j = i + 1; j < program->nb_statements; ++j)
      {
	if (screen[ls_oset_idx_t (program->nb_statements, i + 1, j + 1) - 1]
	    == 1)
	  fprintf (f, "S%d -> S%d\n", i, j);
      }
  fprintf (f, "}\n");
  fclose (f);
}

static
void
update_transclos_fuse (int* screen, CandlProgram* program,
		       int ii, int jj)
{
  int i, j, k;
  int sz = program->nb_statements;
  int matrix[sz][sz];

  for (i = 0; i < sz - 1; ++i)
    for (j = i + 1; j < sz; ++j)
      {
	if (screen[ls_oset_idx_t (sz, i + 1, j + 1) - 1]
	    == 1)
	  {
	    matrix[i][j] = 1;
	    matrix[j][i] = 1;
	  }
      }

  for( i = 0; i < sz; i++)
    for (j = 0; j < sz; j++)
      if (matrix[i][j] == 1)
	for (k = 0; k < sz; k++)
	  if (matrix[j][k] == 1)
	    matrix[i][k] = 1;

  int iid, jid;
  for (i = 0; i < sz; ++i)
    {
      if (matrix[ii][i] != 0)
	{
	  iid = ii < i ? ii : i;
	  jid = iid == i ? ii : i;
	  screen[ls_oset_idx_p (sz, iid + 1, jid + 1) - 1] = 0;
	  screen[ls_oset_idx_t (sz, iid + 1, jid + 1) - 1] = 1;
	}
      if (matrix[jj][i] != 0)
	{
	  iid = jj < i ? jj : i;
	  jid = iid == i ? jj : i;
	  screen[ls_oset_idx_p (sz, iid + 1, jid + 1) - 1] = 0;
	  screen[ls_oset_idx_t (sz, iid + 1, jid + 1) - 1] = 1;
	}
    }
}

s_graph_t*
ls_fsspace2_build_fusiongraph (s_graph_t* deps,
			      CandlProgram* program,
			      CandlDependence* dependences,
			      s_ls_options_t* options,
			      int* screen)
{
  int i, j;
  int ii, jj;
  s_graph_t* F = ls_graph_alloc();
  s_vertex_t* v;
  s_vertex_t* stmts[program->nb_statements];
  s_vertex_t* subst[program->nb_statements];
  s_fm_solution_t* t;
  CandlDependence* tmp;
  s_vertex_t** negclos;
  s_vertex_t** posclos;
  s_edge_t* e;

  for (i = 0, v = deps->root; i < program->nb_statements; ++i, v = v->next)
    stmts[i] = subst[i] = v;

  for (i = 0; i < program->nb_statements - 1; ++i)
    for (j = i + 1; j < program->nb_statements; ++j)
      {
	// Create local dependence graph.
	s_graph_t* depgraph = ls_graph_alloc ();
	int uid = 0;
	for (tmp = dependences; tmp; tmp = tmp->next)
	  {
	    if (tmp->source->label == i || tmp->target->label == i)
	      if (tmp->source->label == j || tmp->target->label == j ||
		  tmp->source->label == i && tmp->target->label == i ||
		  tmp->source->label == j && tmp->target->label == j)
		ls_graph_create_vertex (depgraph, tmp, uid++);
	  }
	if (depgraph->vertex_count == 0)
	  continue;

	s_vertex_t* st_i = ls_graph_create_vertex
	  (F, NULL, ((CandlStatement*)(stmts[i]->data))->label);
	s_vertex_t* st_j = ls_graph_create_vertex
	  (F, NULL, ((CandlStatement*)(stmts[j]->data))->label);

	// There is a dependence. Variable t_ij can be eliminated.
	negclos = compute_negclos (deps, st_i);
	posclos = compute_posclos (deps, st_j);
	for (ii = 0; negclos[ii] != NULL; ++ii)
	  for (jj = 0; posclos[jj] != NULL; ++jj)
	    {
	      if (screen[ls_oset_idx_t (program->nb_statements,
					negclos[ii]->id + 1,
					posclos[jj]->id + 1) - 1]
		  == LS_OSET_UNSET)
		screen[ls_oset_idx_t (program->nb_statements,
				      negclos[ii]->id + 1,
				      posclos[jj]->id + 1) - 1] = LS_OSET_ELIM;
	    }
	XFREE(negclos);
	XFREE(posclos);

	// Compute T_RS.
	if (options->verbose)
	  {
	    fprintf (options->out_file, ".");
	    fflush (options->out_file);
	  }
/* 	printf ("space for (%d,%d):\n", i + 1, j + 1); */
	t = build_th_constraints (depgraph, program, options, i, j);

	// Check for distribution.
	if (must_distribute (t, i, j, program))
	  {
	    // Apply prop. 3.
	    negclos = compute_negclos (deps, st_i);
	    posclos = compute_posclos (deps, st_j);
	    for (ii = 0; negclos[ii] != NULL; ++ii)
	      for (jj = 0; posclos[jj] != NULL; ++jj)
		{
		  screen[ls_oset_idx_p (program->nb_statements,
					negclos[ii]->id + 1,
					posclos[jj]->id + 1) - 1] = 1;
		  screen[ls_oset_idx_t (program->nb_statements,
					negclos[ii]->id + 1,
					posclos[jj]->id + 1) - 1] = 0;
		}
	    for (e = F->first_edge; e; e = e->next)
	      {
		if (e->src->id == i && e->dst->id == j ||
		    e->dst->id == i && e->src->id == j)
		  ls_graph_remove_edge (F, e);
	      }
	    continue;
	  }
	// Check for fusion.
	if (must_fuse (t, i, j, program))
	  {
	    screen[ls_oset_idx_p (program->nb_statements, i + 1, j + 1)- 1] = 0;
	    screen[ls_oset_idx_t (program->nb_statements, i + 1, j + 1)- 1] = 1;
	    update_transclos_fuse (screen, program, i, j);
	  }
	// Check for skew.
	if (must_skew (t, i, j, program))
	  {
	    int* val = XMALLOC(int, 2);
	    val[0] = -1;
	    val[1] = -1;
	    ls_graph_create_edge (F, st_i, st_j, val);
	    continue;
	  }
	// Check for each (ir,is).
	for (ii = 0; ii < ((CandlStatement*)(stmts[i]->data))->depth; ++ii)
	  for (jj = 0; jj < ((CandlStatement*)(stmts[j]->data))->depth; ++jj)
	    {
	      if (is_legal_perm (t, i, j, ii, jj, program))
		{
		  int* val = XMALLOC(int, 2);
		  val[0] = ii;
		  val[1] = jj;
		  ls_graph_create_edge (F, st_i, st_j, val);
		}
	    }

	// Some cleaning.
	fm_solution_free (t);
	ls_graph_free (depgraph);
      }

  if (options->verbose)
    fprintf (options->out_file, "\n");

  return F;
}


static
int
check_fusible (s_graph_t* deps, s_graph_t* fgraph, s_vertex_t** path,
	       int b, int e)
{


  return 1;
}


static
void
build_comp (s_graph_t* g, s_vertex_t* v)
{
  int i;
  s_vertex_t** path = g->usr;
  for (i = 0; i < g->vertex_count && path[i] != NULL; ++i)
    ;
  path[i] = v;
}



s_fm_system_t*
ls_fsspace2_build_fset (s_graph_t* deps,
		       CandlProgram* program,
		       s_ls_options_t* options,
		       int* screen,
		       s_graph_t* fgraph)
{
  int i, j;
  int b;
  int e;
  int length;
  s_fm_system_t* s;
  s_vertex_t* v;
  s_fm_vector_t* cst;
  s_vertex_t** path = XMALLOC(s_vertex_t*, fgraph->vertex_count);
  s_vertex_t** negclos;
  s_vertex_t** posclos;

  int sz = 0;
  int nb_vars = program->nb_statements * (program->nb_statements - 1);
  for (i = 0; i < nb_vars; ++i)
    if (screen[i] != -1)
      ++sz;

/*   dotty_print_fuse (screen, program); */

  if (options->verbose)
    fprintf (options->out_file,
	     ".... Oset: %d fixed (out of %d)\n", sz, nb_vars);

  s = ls_oset_build_system (program->nb_statements, screen);

  fgraph->usr = path;
  for (v = fgraph->root; v; v = v->next)
    {
      if (v->walk == 0)
	continue;
      // Compute the current component.
      for (i = 0; i < fgraph->vertex_count; ++i)
	path[i] = NULL;
      ls_graph_dfs_from (fgraph, v, build_comp, NULL, NULL);

      for (length = 0; length < fgraph->vertex_count && path[length] != NULL;
	   ++length)
	;

      // Traverse the component.
      b = 0; e = 2;
      do
	{
	  if (! check_fusible (deps, fgraph, path, b, e))
	    {
	      printf ("2: NOT FUSIBLE\n");
	      // Prune the graph.
	      negclos = compute_negclos (deps, path[b]);
	      posclos = compute_posclos (deps, path[e]);
	      for (i = 0; negclos[i] != NULL; ++i)
		for (j = 0; posclos[j] != NULL; ++j)
		  {
		    screen[ls_oset_idx_p (program->nb_statements,
					  negclos[i]->id + 1,
					  posclos[j]->id + 1) - 1] = 1;
		    screen[ls_oset_idx_t (program->nb_statements,
					  negclos[i]->id + 1,
					  posclos[j]->id + 1) - 1] = 0;
		  }

	      // Build the constraint.
	      cst = fm_vector_alloc (s->nb_cols);
	      fm_vector_set_ineq (cst);
	      for (i = 0; i < e - b; ++i)
		{
		  fm_vector_assign_int_idx (cst, -1, ls_oset_idx_p
					    (program->nb_statements,
					     path[i]->id + 1,
					     path[i + 1]->id + 1));
		}
	      fm_vector_assign_int_idx (cst, e - b, cst->size - 1);
	      if (ls_oset_needed_cst (cst, screen))
		fm_system_add_line (s, cst);
	      // Remove path(b).
	      ls_graph_remove_vertex (fgraph, path[b]);
	      // Go to the next subpath to check for.
	      ++b;
	      if (e - b < 2)
		e = b + 2;
	    }
	  else
	    ++e;
	}
      while (e < length);
    }

  return s;
}


// Compute the solution polytope.
s_ls_space_t*
ls_fsspace2_compute_solution (s_graph_t* deps,
			     CandlProgram* program,
			     CandlDependence* dependences,
			     s_ls_options_t* options)
{
  int i, count;
  int nb_vars = program->nb_statements * (program->nb_statements - 1);
  int* screen = XMALLOC(int, nb_vars);
  for (i = 0; i < nb_vars; ++i)
    screen[i] = LS_OSET_UNSET;

  // Build the fusion graph.
  if (options->verbose)
    fprintf (options->out_file, ".... Compute fusion graph\n");
  s_graph_t* F = ls_fsspace2_build_fusiongraph (deps, program, dependences,
					       options, screen);

  // Build \FSet.
  if (options->verbose)
    fprintf (options->out_file, ".... Explore fusion graph\n");
  s_fm_system_t* s = ls_fsspace2_build_fset (deps, program, options, screen, F);

  if (options->verbose)
    {
      s_fm_compsol_t* cs = fm_compsol_init_sys (s);
      s_fm_system_t* st = fm_solution_to_system (cs->poly);
      fprintf (options->out_file,
	       ".... FSet: %d variables (%d total), %d constraints\n",
	       st->nb_cols - 2,
	       s->nb_cols - 2, st->nb_lines);
      fm_system_free (st);
      fm_compsol_free (cs);
    }

  s_ls_space_t* space = ls_space_alloc ();
  space->u_compacted = XMALLOC(s_fm_compsol_t*, 2);
  space->u_compacted[0] = fm_compsol_init_sys (s);
  space->u_compacted[1] = NULL;
  space->dimension = 1;
  space->u_polyhedron = XMALLOC(s_fm_solution_t*, 2);
  space->u_polyhedron[0] = fm_compsol_expand (space->u_compacted[0]);
  space->u_polyhedron[1] = NULL;
  space->polyhedron = fm_system_to_solution (s);

  fm_system_free (s);

  return space;
}


/**
 * \brief Build the set of legal fusion structure.
 *
 *
 */
s_ls_space_t*
ls_fsspace2_build (CandlProgram* program,
		  CandlDependence* dependences,
		  s_ls_options_t* options)
{
  int i, j, k;
  s_ls_space_t* space = NULL;
  s_graph_t* graph;
  CandlDependence* tmp;

  if (options->verbose)
    fprintf(options->out_file, "... Build legal fusion structures V2\n");

/*   // Preprocess the dependence graph. */
/*   CandlDependence* depgraph = // dependences; */
/*     ls_schedspace_depgraph_preprocess (program, dependences); */
/*   candl_dependence_pprint (stdout, depgraph); */

/*   // Build the dependence graph from Candl's representation. */
/*   graph = ls_graph_alloc (); */
/*   s_vertex_t** vertex = XMALLOC(s_vertex_t*, program->nb_statements); */
/*   int uid = 0; */
/*   for (i = 0; i < program->nb_statements; ++i) */
/*     vertex[i] = ls_graph_create_vertex (graph, program->statement[i], uid++); */

/*   for (tmp = depgraph; tmp; tmp = tmp->next) */
/*     ls_graph_create_edge (graph, vertex[tmp->source->label], */
/* 			  vertex[tmp->target->label], tmp); */

/*   // Compute the solution polytope. */
/*   space = ls_fsspace2_compute_solution (graph, program, depgraph, options); */

/*   if (options->verbose) */
/*     fprintf(options->out_file, "... All systems solved\n"); */

/*   // Initial PIP check. */
/*   for (i = 0; i < space->size; ++i) */
/*     if (! fm_piptools_check_sol (space->u_polyhedron[i], FM_PIPTOOLS_INT)) */
/*       { */
/* 	if (options->verbose) */
/* 	  fprintf(options->out_file, */
/* 		  "=> Ending computation: no point at dimension %d\n", */
/* 		  i + 1); */
/* 	exit (1); */
/*       } */

/*   // Internal conflict/dependence graph is no longer needed. */
/*   ls_graph_free (graph); */

/*   // Copy space for compatibility with former heuritics. */
/*   space->polyhedron = space->u_polyhedron[0]; */

  return NULL;
}
