/*
 * heuristic-exhaust.c: this file is part of the LetSee project.
 *
 * LetSee, the LEgal Transformation SpacE Explorator.
 *
 * Copyright (C) 2006,2007,2008 Louis-Noel Pouchet
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * The complete GNU General Public Licence Notice can be found as the
 * `COPYING' file in the root directory.
 *
 * Author:
 * Louis-Noel Pouchet <Louis-Noel.Pouchet@inria.fr>
 *
 */
#if HAVE_CONFIG_H
# include <letsee/config.h>
#endif

#include <letsee/heuristic-exhaust.h>

// Static upper bound on number of tries.
#define LS_MAX_TRANSFO_COUNT 5000000
#define LS_MAX_DRAW 50000000



/**
 * Exhaustive polytope explorer heuristic for multidimensionnal
 * schedules. Recursive procedure called for each schedule dimension.
 *
 *
 */
static
void
ls_heuristic_exhaust_multi (CandlProgram* program,
			    s_ls_options_t* options,
			    s_ls_space_t* space,
			    int count,
			    s_fm_vector_t** prototype)
{
  s_fm_solution_t* sol = space->u_polyhedron[count];
  s_fm_rational_t* lb = NULL;
  s_fm_rational_t* Ub = NULL;
  z_type_t* i = XMALLOC(z_type_t, sol->size);
  z_type_t* min = XMALLOC(z_type_t, sol->size);
  z_type_t* max = XMALLOC(z_type_t, sol->size);
  int ii;
  unsigned idx = 0;
  s_fm_vector_t* draw = prototype[count];
  s_fm_vector_t* draw_array[2]; draw_array[1] = NULL;

  cycles_t best = 0;
  cycles_t orig = 0, tmp;
  int id = 0;
  unsigned long long int badpath = 0;
  int transfo_count = 0;
  int ret;
  unsigned long int violation = 0;

  int bound_limit = sol->size;
  // Compute the index of the first coefficient to complete.
  int max_coef;
  for (ii = 0, max_coef = 0; ii < program->nb_statements; ++ii)
    max_coef += program->statement[ii]->depth;
  bound_limit = max_coef;

  // Use the entire polytope.
  bound_limit = sol->size;

  for (ii = 0; ii < sol->size; ++ii)
    {
      Z_INIT(i[ii]);
      Z_INIT(min[ii]);
      Z_INIT(max[ii]);
    }

  fm_solver_compute_min (&lb, sol->solution[idx].positive,
			 draw, idx, FM_MINLEXICO_INT);
  fm_solver_compute_max (&Ub, sol->solution[idx].negative,
			 draw, idx, FM_MINLEXICO_INT);
  min[idx] = lb->num;
  max[idx] = Ub->num;

  for (Z_ASSIGN(i[idx], min[idx]); Z_CMP(i[idx], <=, max[idx]);
       Z_INC(i[idx], i[idx]))
    {
      if (space->size > LS_MAX_TRANSFO_COUNT)
	break;
      fm_vector_assign_int_idx (draw, i[idx], idx + 1);
      if (idx + 1 == bound_limit)
	{
	  if (count + 1 == space->dimension)
	    {
	      if (options->create_schedfiles)
		{
		  if (transfo_count++ > LS_MAX_TRANSFO_COUNT)
		    break;
		  ret = ls_transformations_generate_multi 
		    (prototype, program, options, space->size);
#ifndef HAVE_LIBPOCC_UTILS
		  if (options->compile_line != NULL)
#endif
		    {
		      draw_array[0] = draw;
#ifdef HAVE_LIBPOCC_UTILS
		      tmp = ls_explorer_exectrans (program, options,
						   prototype, space->size);
#else
		      tmp = ls_explorer_exectrans (program, options,
						   draw_array, space->size);
#endif
		      if (best == 0)
			{
			  best = tmp;
			  id = space->size;
			}
		      else
			if (best > tmp && tmp != 0)
			  {
			    best = tmp;
			    id = space->size;
			  }
		    }
		}
#ifndef HAVE_LIBPOCC_UTILS
	      if (options->verbose)
		{
		  fprintf (options->out_file, "*");
		  fflush (options->out_file);
		}
#endif
	      // Safety check.
	      if (! ret)
		{
		  space->size++;
		}
	      else
		{
		  transfo_count--;
		  violation++;
		}
	    }
	  else
	    {
	      // Recursive call to the next dimension.
	      ls_heuristic_exhaust_multi (program, options, space,
					  count + 1, prototype);
	    }
	  while (idx > 0 && (Z_CMP(i[idx], ==, max[idx]) ||
			     Z_CMP(min[idx], >, max[idx])))
	    --idx;
	}
      else
	{
	  ++idx;
	  fm_solver_compute_min (&lb, sol->solution[idx].positive,
				 draw, idx, FM_MINLEXICO_INT);
	  fm_solver_compute_max (&Ub, sol->solution[idx].negative,
				 draw, idx, FM_MINLEXICO_INT);
	  Z_ASSIGN(min[idx], lb->num);
	  Z_ASSIGN(max[idx], Ub->num);
	  Z_ASSIGN(i[idx], min[idx]);
	  Z_DEC(i[idx], i[idx]);
	  int has_hole = 0;
	  while (idx > 0 && (Z_CMP(min[idx], >, max[idx]) ||
			     Z_CMP(i[idx], ==, max[idx])))
	    {
	      if (Z_CMP(min[idx], >, max[idx]))
		has_hole = 1;
	      --idx;
	    }
	  if (has_hole)
	    ++badpath;
	}
    }

  if (options->verbose)
    {
      if (orig != 0)
	{
	  fprintf (options->out_file, "[LetSee] Original cycles: %lld\n", orig);
	  fprintf (options->out_file,
		   "[LetSee] Best transformation cycles: %lld\n", best);
	  fprintf (options->out_file,
		   "[LetSee] Best transformation Id: %d\n", id);
	}
/*       fprintf (options->out_file, ".... Unterminated paths: %Ld\n", badpath); */
/*       fprintf (options->out_file, "... Heuristic terminated\n"); */
    }

  // Be clean.
  for (ii = 0; ii < sol->size; ++ii)
    {
      Z_CLEAR(i[ii]);
      Z_CLEAR(min[ii]);
      Z_CLEAR(max[ii]);
    }

  free (i);
  free (min);
  free (max);
  fm_rational_free (lb);
  fm_rational_free (Ub);
/*   fm_vector_free (draw); */
}


/**
 * Exhaustive polytope explorer heuristic for monodimensional schedules.
 *
 *
 */
static
void
ls_heuristic_exhaust_mono (CandlProgram* program,
			   s_ls_options_t* options,
			   s_ls_space_t* space)
{
  s_fm_solution_t* sol = space->polyhedron;
  s_fm_rational_t* lb = NULL;
  s_fm_rational_t* Ub = NULL;
  z_type_t* i = XMALLOC(z_type_t, sol->size);
  z_type_t* min = XMALLOC(z_type_t, sol->size);
  z_type_t* max = XMALLOC(z_type_t, sol->size);
  int ii;
  unsigned idx = 0;
  s_fm_vector_t* draw = fm_vector_alloc (sol->size + 1);
  s_fm_vector_t* draw_array[2]; draw_array[1] = NULL;

  cycles_t best = 0;
  cycles_t orig = 0, tmp;
  int id = 0;
  unsigned long long int badpath = 0;

  int bound_limit = sol->size;

  // Compute the index of the first coefficient to complete. The
  // traversal goes from coef #0 to bound_limit. If bound limit is
  // equal to the number of variables in the system, then it counts
  // all points in the polytope.

  // Code sample to count the number of iterators.
  int max_coef;
  for (ii = 0, max_coef = 0; ii < program->nb_statements; ++ii)
    max_coef += program->statement[ii]->depth;
  bound_limit = max_coef;

  // Count points in the entire polytope.
  bound_limit = sol->size;

  for (ii = 0; ii < sol->size; ++ii)
    {
      Z_INIT(i[ii]);
      Z_INIT(min[ii]);
      Z_INIT(max[ii]);
    }


  if (options->verbose)
    {
      fprintf (options->out_file, "[LetSee] Using Exhaustive heuristic (1d)\n");
      if (options->create_schedfiles)
	fprintf (options->out_file, 
		 "[LetSee] Compile and run the original code\n");
    }

  // Execute the original code.
  orig = ls_explorer_execorig (options);


  for (ii = 0; ii < sol->size; ++ii)
    {
      Z_INIT(i[ii]);
      Z_INIT(min[ii]);
      Z_INIT(max[ii]);
    }

  fm_solver_compute_min (&lb, sol->solution[idx].positive,
			 draw, idx, FM_MINLEXICO_INT);
  fm_solver_compute_max (&Ub, sol->solution[idx].negative,
			 draw, idx, FM_MINLEXICO_INT);
  min[idx] = lb->num;
  max[idx] = Ub->num;

  if (options->verbose)
    fprintf (options->out_file, "[LetSee] Explore polyhedron...\n");

  // Loop on each point.
  for (Z_ASSIGN(i[idx], min[idx]); Z_CMP(i[idx], <=, max[idx]);
       Z_INC(i[idx], i[idx]))
    {
      fm_vector_assign_int_idx (draw, i[idx], idx + 1);
      if (idx + 1 == bound_limit)
	{
	  if (options->create_schedfiles)
	    {
	      ls_transformations_generate (draw, program, options,
					   space->size);
	      if (options->compile_line != NULL)
		{
		  // Execute the transformation.
		  draw_array[0] = draw;
		  tmp = ls_explorer_exectrans (program, options,
					       draw_array, space->size);
		  if (best == 0)
		    {
		      best = tmp;
		      id = space->size;
		    }
		  else
		    if (best > tmp && tmp != 0)
		      {
			best = tmp;
			id = space->size;
		      }
		}
	    }
	  if (options->verbose && options->create_schedfiles)
	    {
	      fprintf (options->out_file, "*");
	      fflush (options->out_file);
	    }
	  (space->size)++;
	  while (idx > 0 && (Z_CMP(i[idx], ==, max[idx]) ||
			     Z_CMP(min[idx], >, max[idx])))
	    --idx;
	}
      else
	{
	  ++idx;
	  fm_solver_compute_min (&lb, sol->solution[idx].positive,
				 draw, idx, FM_MINLEXICO_INT);
	  fm_solver_compute_max (&Ub, sol->solution[idx].negative,
				 draw, idx, FM_MINLEXICO_INT);
	  Z_ASSIGN(min[idx], lb->num);
	  Z_ASSIGN(max[idx], Ub->num);
	  Z_ASSIGN(i[idx], min[idx]);
	  Z_DEC(i[idx], i[idx]);
	  int has_hole = 0;
	  while (idx > 0 && (Z_CMP(min[idx], >, max[idx]) ||
			     Z_CMP(i[idx], ==, max[idx])))
	    {
	      // Integer hole.
	      if (Z_CMP(min[idx], >, max[idx]))
		has_hole = 1;
	      --idx;
	    }
	  if (has_hole)
	    ++badpath;
	}
    }


  if (options->verbose)
    {
      if (orig != 0)
	{
	  fprintf (options->out_file, "\n.... Original cycles: %lld\n", orig);
	  fprintf (options->out_file,
		   ".... Best transformation cycles: %lld\n", best);
	  fprintf (options->out_file, ".... Best transformation Id: %d\n", id);
	}
      fprintf (options->out_file, "[LetSee] Unterminated paths: %Ld\n", 
	       badpath);
      fprintf (options->out_file, "[LetSee] Heuristic terminated\n");
    }

  // Be clean.
  for (ii = 0; ii < sol->size; ++ii)
    {
      Z_CLEAR(i[ii]);
      Z_CLEAR(min[ii]);
      Z_CLEAR(max[ii]);
    }

  free (i);
  free (min);
  free (max);
  fm_rational_free (lb);
  fm_rational_free (Ub);
  fm_vector_free (draw);
}



/**
 * Exhaustive polytope explorer heuristic. The total number of
 * different integer points in the polytope(s) is filled in the
 * space->size field.
 *
 *
 */
void
ls_heuristic_exhaust (s_ls_space_t* space,
		      CandlProgram* program,
		      s_ls_options_t* options)
{
  int i;
  s_fm_compsol_t* cs;

  if (options->create_schedfiles == 0)
    {
      for (i = 0; i < space->dimension; ++i)
	{
	  space->polyhedron = space->u_polyhedron[i];
	  space->size = 0;
	  ls_heuristic_exhaust_mono (program, options, space);
	  fprintf (options->out_file,
		   "[LetSee] Number of points at dimension %d: %Ld\n",
		   i + 1, space->size);
	  space->size = 0;
	}
    }
  else
    {
      if (space->polyhedron == NULL || options->type == LS_TYPE_MULTI)
	{
	  s_fm_vector_t* prototype[space->dimension + 1];

	  for (i = 0; i < space->dimension; ++i)
	    prototype[i] = fm_vector_alloc (space->u_polyhedron[0]->size + 1);
	  prototype[space->dimension] = NULL;

	  ls_heuristic_exhaust_multi (program, options, space, 0, prototype);

	  for (i = 0; i < space->dimension; ++i)
	    fm_vector_free (prototype[i]);
	}
      else
	ls_heuristic_exhaust_mono (program, options, space);
    }
}


