/*
 * heuristic-plutom.c: this file is part of the LetSee project.
 *
 * LetSee, the LEgal Transformation SpacE Explorator.
 *
 * Copyright (C) 2008 Louis-Noel Pouchet
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * The complete GNU General Public Licence Notice can be found as the
 * `COPYING' file in the root directory.
 *
 * Author:
 * Louis-Noel Pouchet <Louis-Noel.Pouchet@inria.fr>
 *
 */
#if HAVE_CONFIG_H
# include <letsee/config.h>
#endif

#include <letsee/common.h>
#include <letsee/heuristic-plutom.h>


int
ls_heuristic_plutom_tcb (CandlProgram* program,
			s_fm_solution_t* sol,
			s_ls_options_t* options)
{
  return sol->size;
}

s_fm_vector_t* G_VECT = NULL;
s_fm_vector_t* G_LAST = NULL;
s_fm_vector_t* G_REF = NULL;

int
ls_heuristic_plutom_tgp (s_fm_vector_t* draw,
			CandlProgram* program,
			s_ls_options_t* options)
{
  if (G_REF == NULL || fm_vector_equal (G_REF, G_LAST))
    {
      G_VECT = fm_vector_dup (draw);
      if (G_LAST)
	fm_vector_free (G_LAST);
      return LS_ENUMERATE_BREAK;
    }
  if (G_LAST)
    fm_vector_free (G_LAST);
  G_LAST = fm_vector_dup (draw);

  return LS_ENUMERATE_CONTINUE;
}


s_fm_vector_t*
ls_heuristic_plutom_pick (s_ls_space_t* space,
			  s_ls_options_t* options,
			  s_fm_solution_t* sol,
			  s_fm_vector_t* p1)
{
  int size = 0;
  G_REF = p1;
  G_LAST = NULL;
  G_VECT = NULL;
  ls_heuristics_enumerate (space->program, options, sol,
			   &size, ls_heuristic_plutom_tcb,
			   ls_heuristic_plutom_tgp);

  return G_VECT;
}


s_fm_vector_t*
ls_heuristic_plutom_schedule (s_ls_space_t* space,
			      s_ls_options_t* options,
			      s_fm_vector_t* p)
{
  int i;
  int size, pos;
  z_type_t one; Z_INIT(one); Z_ASSIGN_SI(one, 1);

  // Compute the size of the schedule.
  for (i = 0, size = 0; i < space->program->nb_statements; ++i)
    {
      size += space->program->statement[i]->depth;
      size += space->program->context->NbColumns - 2;
      ++size;
    }
  s_fm_vector_t* ret = fm_vector_alloc (size + 1);

  // Put the identity schedule in it.
  for (i = 0, pos = 1; i < space->program->nb_statements; ++i)
    {
      fm_vector_assign_int_idx (ret, one, pos);
      pos += space->program->statement[i]->depth;
    }

  Z_CLEAR(one);

  return ret;
}


s_graph_t*
ls_heuristic_plutom_compute_depgraph (s_ls_space_t* space,
				      s_ls_options_t* options,
				      s_fm_vector_t* p)
{
  // Compute the schedule generated by pluto. (Fake for the moment, we
  // assume id sched is produced.
  s_fm_vector_t* schedule = ls_heuristic_plutom_schedule (space, options, p);

  CandlDependence* dep;
  int dimension;
  int uid = 0;
  s_fm_system_t* sys;
  CandlStatement* st1;
  CandlStatement* st2;
  s_fm_vector_t* v;
  int i;
  z_type_t one; Z_INIT(one); Z_ASSIGN_SI(one, 1);
  z_type_t mone; Z_INIT(mone); Z_ASSIGN_SI(mone, -1);
  s_graph_t* g = ls_graph_alloc ();
  int offset;

  Z_ASSIGN_SI(options->lb, 0);
  Z_ASSIGN_SI(options->Ub, 100);
  Z_ASSIGN_SI(options->plb, 0);
  Z_ASSIGN_SI(options->pUb, 0);
  Z_ASSIGN_SI(options->clb, 0);
  Z_ASSIGN_SI(options->cUb, 0);


  for (dep = space->dependences; dep; dep = dep->next)
    {
      sys = ls_farkas_build_system (space->program, options, dep, &dimension,
				    LS_FARKAS_GEONE);
      if (dep->source->label <= dep->target->label)
	{
	  st1 = dep->source;
	  st2 = dep->target;
	}
      else
	{
	  st1 = dep->target;
	  st2 = dep->source;
	}
      // Fill the iterator part.
      v = fm_vector_alloc (sys->nb_cols);
      fm_vector_assign_int_idx (v, one, 1);
      fm_vector_assign_int_idx (v, mone, v->size - 1);
      fm_system_add_line (sys, v);
      offset = st1->depth;
      for (i = 1; i < offset; ++i)
	{
	  v = fm_vector_alloc (sys->nb_cols);
	  fm_vector_assign_int_idx (v, one, i + 1);
	  fm_system_add_line (sys, v);
	}
      if (st1 != st2)
	{
	  v = fm_vector_alloc (sys->nb_cols);
	  fm_vector_assign_int_idx (v, one, i + 1);
	  fm_vector_assign_int_idx (v, mone, v->size - 1);
	  fm_system_add_line (sys, v);
	  offset += st2->depth;
	  for (; i < offset; ++i)
	    {
	      v = fm_vector_alloc (sys->nb_cols);
	      fm_vector_assign_int_idx (v, one, i + 1);
	      fm_system_add_line (sys, v);
	    }
	}
      // Fill the parameter part.
      offset += space->program->context->NbColumns - 2;
      for (; i < offset; ++i)
	{
	  v = fm_vector_alloc (sys->nb_cols);
	  fm_vector_assign_int_idx (v, one, i + 1);
	  fm_system_add_line (sys, v);
	}
      if (st1 != st2)
	{
	  offset += space->program->context->NbColumns - 2;
	  for (; i < offset; ++i)
	    {
	      v = fm_vector_alloc (sys->nb_cols);
	      fm_vector_assign_int_idx (v, one, i + 1);
	      fm_system_add_line (sys, v);
	    }
	}
      // Fill the constant part.
      offset++;
      for (; i < offset; ++i)
	{
	  v = fm_vector_alloc (sys->nb_cols);
	  fm_vector_assign_int_idx (v, one, i + 1);
	  fm_system_add_line (sys, v);
	}
      if (st1 != st2)
	{
	  offset++;
	  for (; i < offset; ++i)
	    {
	      v = fm_vector_alloc (sys->nb_cols);
	      fm_vector_assign_int_idx (v, one, i + 1);
	      fm_system_add_line (sys, v);
	    }
	}
      if (! fm_piptools_check_int (sys))
	{
	  CANDL_DEP_TAG(dep, solved) = 0;
	  ls_graph_create_vertex (g, dep, uid++);
	}
      fm_system_free (sys);
    }

  ls_graph_print (stdout, g);

  return g;
}


static void
fm_system_print_pluto (FILE* stream, s_fm_system_t* s)
{
  unsigned i, j;
  s_fm_vector_t* v;

  if (s == NULL)
    fprintf (stream, "(null)\n");
  else
    {
      fprintf (stream, "%d %d\n", s->nb_lines, s->nb_cols);
      for (i = 0; i < s->nb_lines; ++i)
	{
	  v = s->lines[i];
	  if (v == NULL)
	    fprintf (stream, "(null)");
	  else
	    {
	      for (j = 0; j < v->size - 1; ++j)
		{
		  fm_rational_print (stream, &(v->vector[j]));
		  fprintf (stream, " ");
		}
	      fm_rational_print (stream, &(v->vector[j]));
	    }
	  fprintf (stream, "\n");
	}
    }

  return FM_STATUS_OK;
}


/**
 *
 *
 */
void
ls_heuristic_plutom_build_fst (s_ls_space_t* space,
			       s_ls_options_t* options,
			       s_fm_vector_t** point,
			       int** tile_factor,
			       int count)
{
  int i, j;
  int nbdim;
  s_fm_system_t* sys;
  CandlStatement* s;
  z_type_t one; Z_INIT(one); Z_ASSIGN_SI(one, 1);

  // Compute the depth specified.
  for (nbdim = 0; point[nbdim]; ++nbdim)
    ;

  // Dump the file.
  char str[256];
  char buff[16];
  strcpy(str, options->transfo_dir);
  strcat(str, "/transformation_");
  sprintf(buff, "%d", count);
  strcat(str, buff);
  strcat(str, ".sched");
  FILE* out = fopen (str, "w+");

  // Output the number of statements.
  fprintf (out, "# number of statements\n");
  fprintf (out, "%d\n", space->program->nb_statements);
  // Output the tile depth.
  fprintf (out, "# tile depth\n");
  fprintf (out, "%d\n", nbdim);

  // Iterate on all statements.
  for (i = 0; i < space->program->nb_statements; ++i)
    {
      // Compute the scattering for the statement.
      s = space->program->statement[i];
      sys = fm_system_alloc (2 * (nbdim - 1) + 1, s->domain->NbColumns);
      for (j = 0; j < nbdim; ++j)
	{
	  fm_rational_copy
	    (&(sys->lines[2 * j]->vector[sys->nb_cols - 1]),
	     &(point[j]->vector[i + 1]));
	  if (j % 2 == 1)
	    fm_vector_assign_int_idx (sys->lines[2 * j - 1], one, j);
	}
      // Dump it.
      fprintf (out, "# scattering\n");
      fm_system_print_pluto (out, sys);
/*       // Dump the tile level spec. */
/*       fprintf (out, "# number of level\n"); */
/*       fprintf (out, "%d\n", 1); */
      // Dump the tile parameters.
      for (j = 0; j < nbdim; ++j)
	{
	  fprintf (out, "# tiling depth %d\n", j + 1);
	  if (s->depth > 1)
	    fprintf (out, "%d\n", tile_factor[i][j]);
	  // Do not tile 1d statements.
	  else
	    fprintf (out, "%d\n", 0);
	}
      fm_system_free (sys);
    }

  // Be clean.
  fclose (out);
  Z_CLEAR(one);
}


/**
 *
 *
 */
void
ls_heuristic_plutom_traverse_dim (s_ls_space_t* space,
				  s_ls_options_t* options,
				  s_fm_solution_t* soldim1,
				  s_fm_vector_t** point,
				  int** tile_factor,
				  int count)
{


}



/**
 *
 *
 */
void
ls_heuristic_plutom_normalize_vector (s_fm_vector_t* v)
{
  z_type_t val; Z_INIT(val);
  z_type_t tmp; Z_INIT(val);
  int i, j, k;
  int nb_comp;
  z_type_t* comps = XMALLOC(z_type_t, v->size - 1);
  for (i = 0; i < v->size - 1; ++i)
    Z_INIT(comps[i]);

  // Rule 1: cst <- cst - min (cst_i)
  Z_ASSIGN(val, v->vector[1].num);
  for (j = 2; j < v->size; ++j)
    if (Z_CMP(val, >, v->vector[j].num))
      Z_ASSIGN(val, v->vector[j].num);
  for (j = 1; j < v->size; ++j)
    Z_SUB(v->vector[j].num, v->vector[j].num, val);

  // Rule 2: Normalization.
  // Count the number of different components.
  for (j = 1, nb_comp = 0; j < v->size; ++j)
    {
      for (k = 0;
	   k < nb_comp && Z_CMP(v->vector[j].num, !=, comps[k]); ++k)
	;
      if (k == nb_comp)
	Z_ASSIGN(comps[nb_comp++], v->vector[j].num);
    }
  // Order the components.
  for (j = 0; j < nb_comp; ++j)
    for (k = j + 1; k < nb_comp; ++k)
      {
	if (Z_CMP(comps[j], >, comps[k]))
	  {
	    Z_ASSIGN(tmp, comps[j]);
	    Z_ASSIGN(comps[j], comps[k]);
	    Z_ASSIGN(comps[k], tmp);
	  }
      }
  // Normalize the vector.
  for (k = 0, Z_ASSIGN_SI(tmp, 0); k < nb_comp; ++k, Z_INC(tmp, tmp))
    for (j = 1; j < v->size; ++j)
      if (Z_CMP(v->vector[j].num, ==, comps[k]))
	Z_ASSIGN(v->vector[j].num, tmp);

  // Recompute the vector key (useful for remove_duplicates).
  fm_vector_compute_key (&(v->key), v);

  Z_CLEAR(val);
  Z_CLEAR(tmp);
  for (i = 0; i < v->size - 1; ++i)
    Z_CLEAR(comps[i]);
  XFREE(comps);
}


/**
 *
 *
 */
void
ls_heuristic_plutom_build_tilenotile (s_ls_space_t* space,
				      s_ls_options_t* options,
				      s_fm_vector_t** point,
				      int dim,
				      int** tile_factor,
				      int* count)
{
  int i, j;
  int size = 0;
  int mask;
  int sts[space->program->nb_statements];
  z_type_t val; Z_INIT(val);

  /// Bypass the tile/no tile function.
  ls_heuristic_plutom_build_fst (space, options, point, tile_factor,
				 (*count)++);
  return;

  for (i = 0; i < space->program->nb_statements; ++i)
    sts[i] = -1;

  // Get statements alone in their component.
  for (i = 0; i < space->program->nb_statements; ++i)
    {
      Z_ASSIGN(val, point[dim]->vector[i + 1].num);
      for (j = i + 1; j < space->program->nb_statements + 1; ++j)
	if (Z_CMP(val, ==, point[dim]->vector[j + 1].num))
	  break;
      if (j == space->program->nb_statements + 1)
	// Put it only if its depth is < 2.
	if (space->program->statement[i]->depth < 2)
	  sts[size++] = i;
    }

  // For each of them, generate a tile/no tile.
  for (i = 1; i <= pow (2, size); ++i)
    {
      for (j = 0; j < size; ++j)
	{
	  mask = 1;
	  if (i & mask)
	    tile_factor[sts[j]][dim] = 256;
	  else
	    tile_factor[sts[j]][dim] = 0;
	  mask <<= 2;
	}
      ls_heuristic_plutom_build_fst (space, options, point, tile_factor,
				     (*count)++);
      if (options->verbose)
	fprintf (options->out_file, ".");
    }

  // Restore the full tile factors.
  for (i = 0; i < space->program->nb_statements; ++i)
    tile_factor[dim][i] = 256;

  // Be clean.
  Z_CLEAR(val);
}



#define NB_PICKED_FIRST_DIM 1

/**
 *
 *
 */
int
ls_heuristic_plutom_traverse (s_ls_space_t* space,
			      s_ls_options_t* options,
			      s_fm_solution_t* soldim1,
			      int mode)
{
  s_fm_vector_t* p1 = NULL;
  s_fm_vector_t* p2 = NULL;
  s_fm_vector_t* point[10];
  s_fm_vector_t* pp1;
  s_fm_vector_t* pp2;
  int i, j, k;
  int count = 0;
  int maxdepth = 2;
  int** tile_factor = XMALLOC(int*, space->program->nb_statements);
  s_fm_vector_t** sols1;
  s_fm_vector_t** sols2;
  int sols_count1 = 0;
  int sols_count2 = 0;

  // Backup and cancel verbosity.
  int verb = options->verbose;
  options->verbose = 0;

  // Specify the tile factor(s).
  for (i = 0; i < space->program->nb_statements; ++i)
    {
      tile_factor[i] = XMALLOC(int, maxdepth);
      for (j = 0; j < maxdepth; ++j)
	tile_factor[i][j] = 256;
    }

  if (mode == LS_HEURISTIC_PLUTOM1 || mode == LS_HEURISTIC_PLUTOM)
    {
      sols1 = XMALLOC(s_fm_vector_t*, 64);
      for (i = 0; i < options->rtries; ++i)
	{
	  do
	    {
	      // Pick a point in the first dim.
	      p1 = ls_heuristic_plutom_pick (space, options, soldim1, p1);
	      // We are out of points.
	      if (p1 == NULL)
		break;
	      pp1 = fm_vector_dup (p1);
	      // Normalize it.
	      ls_heuristic_plutom_normalize_vector (pp1);
	      printf ("base: ");
	      fm_vector_print (stdout, p1); printf ("\n");
	      printf ("normalized: ");
	      fm_vector_print (stdout, pp1); printf ("\n");
	      // Check if the normalized solution was previously drawn.
	      for (k = 0; k < sols_count1; ++k)
		if (fm_vector_equal (pp1, sols1[k]))
		  {
		    fm_vector_free (pp1);
		    break;
		  }
	      // It wasn't.
	      if (k >= sols_count1)
		{
		  // Buffer management.
		  if (sols_count1 && sols_count1 % 64 == 0)
		    sols1 = XREALLOC(s_fm_vector_t*, sols1, sols_count1 + 64);
		  // Store the point.
		  sols1[sols_count1++] = pp1;
		  // Build the fst file.
		  point[0] = pp1; point[1] = NULL;
		  ls_heuristic_plutom_build_tilenotile
		    (space, options, point, 0, tile_factor, &count);
		}
	    }
	  while (k < sols_count1 - 1);
	  // We are out of points.
	  if (p1 == NULL)
	    break;
	}
    }
  else if (mode == LS_HEURISTIC_PLUTOM2)
    {
      sols1 = XMALLOC(s_fm_vector_t*, 64);
      sols2 = XMALLOC(s_fm_vector_t*, 64);
      for (j = 0; j < NB_PICKED_FIRST_DIM; ++j)
	{
	  do
	    {
	      // Draw a point.
	      p1 = ls_heuristic_plutom_pick (space, options, soldim1, p1);
	      // We are out of points.
	      if (p1 == NULL)
		break;
	      pp1 = fm_vector_dup (p1);
	      // Normalize it.
	      ls_heuristic_plutom_normalize_vector (pp1);

	      // Check if the normalized point was previously draw.
	      for (k = 0; k < sols_count1; k += 2)
		if (fm_vector_equal (pp1, sols1[k]))
		  {
		    fm_vector_free (pp1);
		    break;
		  }
	    }
	  // Repeat until a new point is found.
	  while (k < sols_count1);

	  // We are out of points.
	  if (p1 == NULL)
	    break;

	  // Buffer management.
	  if (sols_count1 && sols_count1 % 64 == 0)
	    sols1 = XREALLOC(s_fm_vector_t*, sols1, sols_count1 + 64);
	  // Store the point for the first dimension.
	  sols1[sols_count1++] = fm_vector_dup (pp1);

	  // Compute the graph of unsatisfied dependences provided the
	  // first dimension.
	  s_graph_t* g = ls_heuristic_plutom_compute_depgraph
	    (space, options, pp1);
	  // Compute the space for dim2.
	  s_fm_compsol_t* cs = ls_fdspace_compute_one (g, space->program,
						       options, 2);
	  s_fm_solution_t* soldim2 = fm_compsol_expand (cs);
	  fm_compsol_free (cs);

	  // Ensure we have the correct space dimension (in case of
	  // partial load of a space from the command line).
	  if (space->dimension < 2)
	    space->dimension = 2;
	  // Add the space normalization constraints, and propagate them.
	  Z_ASSIGN_SI(options->pUb, space->program->nb_statements);
	  space->u_polyhedron[1] = soldim2;
	  letsee_debug(add normalization space dim 2);
	  ls_fdspace_add_normalization_constraints (space->program, options,
						    space->u_polyhedron[1]);
	  letsee_debug(normalize space dim 2);
	  ls_fusionspace_normalize_space (space->program, options, space,
					  0, space->program->nb_statements);
	  space->u_compacted[1] = fm_compsol_init_sol (space->u_polyhedron[1]);
	  letsee_debug(optimize space dim 2);
	  ls_schedspace_optimize_space (space->program, options, space);
/* 	  fm_solution_print (stdout, space->u_polyhedron[1]); */
	  ls_fusionspace_normalize_space (space->program, options, space,
					  0, space->program->nb_statements);
/* 	  fm_solution_count (space->u_polyhedron[1], */
/* 			     space->u_polyhedron[1]->size, FM_SOLVER_VERBOSE); */
	  fm_solution_print (stdout, space->u_polyhedron[1]);
	  soldim2 = space->u_polyhedron[1];

	  // Pick points for the second dimension.
	  for (i = 0; i < options->rtries; ++i)
	    {
	      do
		{
		  // Pick a point in the second dim.
		  p2 = ls_heuristic_plutom_pick (space, options, soldim2, p2);
		  // We are out of points.
		  if (p2 == NULL)
		    break;
		  pp2 = fm_vector_dup (p2);
		  // Normalize it.
		  ls_heuristic_plutom_normalize_vector (pp2);

		  // Check if it was previously draw.
		  for (k = 1; k < sols_count2; k += 2)
		    if (fm_vector_equal (pp2, sols2[k]))
		      {
			fm_vector_free (pp2);
			break;
		      }
		  // It wasn't.
		  if (k >= sols_count2)
		    {
		      // Buffer management.
		      if (sols_count2 && sols_count2 % 64 == 0)
			sols2 = XREALLOC(s_fm_vector_t*, sols2,
					 sols_count2 + 64);
		      // Store the point for the first dimension.
		      sols2[sols_count2++] = pp2;
		      // Build the fst file.
		      point[0] = pp1; point[1] = pp2; point[2] = NULL;
		      ls_heuristic_plutom_build_fst (space, options, point,
						     tile_factor,
						     count++);
		      if (verb)
			fprintf (options->out_file, ".");
		    }
		}
	      while (k < sols_count2 - 1);
	      // We are out of points.
	      if (p2 == NULL)
		break;
	    }
	}
    }
  if (verb)
    fprintf (options->out_file, "\n");

  // Be clean.
  for (i = 0; i < space->program->nb_statements; ++i)
    XFREE(tile_factor[i]);
  XFREE(tile_factor);

  // Restore verbosity.
  options->verbose = verb;

  return count;
}


void
ls_heuristic_plutom (s_ls_space_t* space,
		     CandlProgram* program,
		     s_ls_options_t* options,
		     int mode)
{
  int size = 0;
  int fs = 0;
  if (options->verbose)
    fprintf (options->out_file, "... Using Plutom heuristic\n");

  // Optimize the space for exploration.
  if (options->normalize_space)
    {
      if (options->verbose)
	fprintf (options->out_file,
		 "... Optimize search space\n");

/*       ls_fdspace_add_normalization_constraints (program, options, */
/* 						space->u_polyhedron[0]); */
      Z_ASSIGN_SI(options->plb, 0);
      Z_ASSIGN_SI(options->pUb, program->nb_statements - 1);
      int count = space->u_polyhedron[0]->size;
/*       space->u_polyhedron[0] = */
/* 	ls_fdspace_add_normalization_constraints_2 (program, options, */
/* 						    space->u_polyhedron[0]); */
      fm_solution_print (stdout, space->u_polyhedron[0]);
      fm_compsol_free (space->u_compacted[0]);
      space->u_compacted[0] = fm_compsol_init_sol (space->u_polyhedron[0]);
      ls_schedspace_optimize_space (program, options, space);
/*       fm_solution_print (stdout, space->u_polyhedron[0]); */
      fm_solution_cut (space->u_polyhedron[0], count);
      ls_fusionspace_normalize_space (program, options, space,
				      options->plb, options->pUb);
/*       fm_solution_print (stdout, space->u_polyhedron[0]); */
/*       fm_solution_count (space->u_polyhedron[0], */
/* 			 space->u_polyhedron[0]->size, FM_SOLVER_VERBOSE); */
    }

  if (options->verbose)
    fprintf (options->out_file,
	     ".... Exploring legal fusion schedules\n", size);

  space->size = ls_heuristic_plutom_traverse
    (space, options, space->u_polyhedron[0], mode);
}
