/*
 * graph.h: this file is part of the LetSee project.
 *
 * LetSee, the LEgal Transformation SpacE Explorator.
 *
 * Copyright (C) 2007,2008 Louis-Noel Pouchet
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * The complete GNU General Public Licence Notice can be found as the
 * `COPYING' file in the root directory.
 *
 * Author:
 * Louis-Noel Pouchet <Louis-Noel.Pouchet@inria.fr>
 *
 */
#ifndef LETSEE_GRAPH_H
# define LETSEE_GRAPH_H

# include <letsee/common.h>
# include <letsee/options.h>
# include <letsee/space.h>
# include <letsee/transformations.h>
# include <fm/solution.h>
# include <fm/system.h>
# include <fm/solver.h>
# include <fm/compsol.h>
# include <fm/list.h>
# include <candl/candl.h>
# include <stdio.h>


#define LS_GRAPH_COLOR_RESET  1
#define LS_GRAPH_COLOR_UPDATE 2

#define for_all_edges(edgeiter, graph)					\
for (edgeiter = graph->first_edge; edgeiter != NULL; edgeiter = edgeiter->next)

#define for_all_nodes(nodeiter, graph)					\
for (nodeiter = graph->root; nodeiter != NULL; nodeiter = nodeiter->next)

BEGIN_C_DECLS

// Edge.
struct s_edge
{
				// Fixed by construction:
  struct s_vertex*	src;
  struct s_vertex*	dst;

				// User data
  void*			data;
  int			label;
  int			tag;

  struct s_edge*	next;
};
typedef struct s_edge s_edge_t;


// Vertex.
struct s_vertex
{
  int			id;     // UID. Do not modify!
  int			walk;	// walking bit (useful for DFS)

				// Fixed by construction:
  int			d_in;	// interior degree
  int			d_out;  // exterior degree
  s_fm_list_t*   	in;	// list of pointers on outgoing edges
  s_fm_list_t*         	out;	// list of pointers on incoming edges

				// Fixed by algorithms:
  int			color;  // color bit
  int			scc;    // scc bit

				// User data
  int			label;
  int			tried;
  int			tag;
  void*			data;

  struct s_vertex*	next;
};

typedef struct s_vertex s_vertex_t;


#define LS_GRAPH_TAG_UNDEFINED -1

// Graph.
// Graph implementation:
//	  - a root node (nodes are chained)
//	  - a first edge (edges are chained)
struct s_graph
{
  s_vertex_t*		root;
  s_edge_t*		first_edge;
  int			edge_count;
  int			vertex_count;


  int			tag;	// Useful for algortithms.
  void*			usr;
};

typedef struct s_graph s_graph_t;


// Visitor function pointer.
typedef void (*visit_fun_t) (s_graph_t*, s_vertex_t*);


// Graph structure methods.
extern
s_graph_t*
ls_graph_alloc ();

extern
void
ls_graph_free (s_graph_t* g);

extern
void
ls_graph_empty (s_graph_t* g);

extern
s_graph_t*
ls_graph_dup (s_graph_t* g);


// Graph manipulation methods.
extern
s_vertex_t*
ls_graph_create_vertex (s_graph_t* g, void* data, int id);

extern
s_edge_t*
ls_graph_create_edge (s_graph_t* g,
		      s_vertex_t* src,
		      s_vertex_t* dst,
		      void* data);

extern
int
ls_graph_remove_edge (s_graph_t* g, s_edge_t* e);

extern
int
ls_graph_remove_vertex (s_graph_t* g, s_vertex_t* v);


// Graph (DOT) printer.
extern
void
ls_graph_print (FILE* out_file, s_graph_t* g);


// Graph algorithms.
extern
int
ls_graph_color (s_graph_t* g, int mask);

extern
int
ls_graph_max_color (s_graph_t* g);

extern
int
ls_graph_second_max_color (s_graph_t* g);

extern
void
ls_graph_transpose (s_graph_t* g);

extern
void
ls_graph_dfs (s_graph_t* g,
	      visit_fun_t prefix,
	      visit_fun_t infix,
	      visit_fun_t suffix);

extern
void
ls_graph_dfs_from (s_graph_t* g,
		   s_vertex_t* v,
		   visit_fun_t prefix,
		   visit_fun_t infix,
		   visit_fun_t suffix);

extern
void
ls_graph_undirected_dfs (s_graph_t* g,
			 visit_fun_t prefix,
			 visit_fun_t infix,
			 visit_fun_t suffix);

extern
void
ls_graph_undirected_dfs_from (s_graph_t* g,
			      s_vertex_t* v,
			      visit_fun_t prefix,
			      visit_fun_t infix,
			      visit_fun_t suffix);

extern
int
ls_graph_compute_scc (s_graph_t* g);

extern
s_graph_t*
ls_graph_scc_build_dag (s_graph_t* g);

extern
int
ls_graph_topological_sort (s_graph_t* g);

extern
void
ls_graph_print_scc (s_graph_t* g, int nb_scc);

extern
void
ls_graph_topological_apply (s_graph_t* g);

extern
void
ls_graph_topological_apply_depth (s_graph_t* g);


END_C_DECLS


#endif // LETSEE_GRAPH_H
