/*
 * heuristics.h: this file is part of the LetSee project.
 *
 * LetSee, the LEgal Transformation SpacE Explorator.
 *
 * Copyright (C) 2006,2007 Louis-Noel Pouchet
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * The complete GNU General Public Licence Notice can be found as the
 * `COPYING' file in the root directory.
 *
 * Author:
 * Louis-Noel Pouchet <Louis-Noel.Pouchet@inria.fr>
 *
 */
#ifndef LETSEE_HEURISTICS_H
# define LETSEE_HEURISTICS_H

# include <letsee/explorer.h>
# include <letsee/options.h>
# include <fm/vector.h>


# define LS_ENUMERATE_BREAK	1
# define LS_ENUMERATE_CONTINUE	2
# define LS_ENUMERATE_DUPLICATE	3


BEGIN_C_DECLS

// Enum for the coefficient type.
enum e_coeftype
{
  LS_CT_ITER,
  LS_CT_PARAM,
  LS_CT_CONST
};
typedef enum e_coeftype e_coeftype_t;


// Enum for the crossover mode.
enum e_crossover_mode
{
  LS_GA_CM_DIMENSION = 1,
  LS_GA_CM_INDVAR = 2,
  LS_GA_CM_SCC = 3,
  LS_GA_CM_OTHER = 4
};
typedef enum e_crossover_mode e_crossover_mode_t;


#define LS_HEURISTICS_CORDEG_FREEDIM	0.85 /* Defines a 'free'
						(i.e. almost scalar)
						dimension */
#define LS_HEURISTICS_CORDEG_ITERDIM	0.5  /* Defines a dimension where
						iterators are highly
						correlated */


typedef int (*t_gen_point)(s_fm_vector_t*, CandlProgram*, s_ls_options_t*);
typedef int (*t_compute_bound)(CandlProgram*, s_fm_solution_t*,
			       s_ls_options_t*);


/**
 * Exhaustive polytope enumerator heuristic for a single polytope.
 *
 *
 */
extern
void
ls_heuristic_enumerate (CandlProgram* program,
			s_ls_options_t* options,
			s_fm_solution_t* sol,
			int* size,
			t_compute_bound tcb,
			t_gen_point tgp);


/**
 * Random polytope enumerator heuristic for a single polytope.
 *
 *
 */
extern
void
ls_heuristics_random_enumerate (CandlProgram* program,
				s_ls_options_t* options,
				s_fm_solution_t* sol,
				int* size,
				t_compute_bound tcb,
				t_gen_point tgp,
				int random_tries);


/**
 * Compute the degree of correlation of variables, for a given
 * schedule row (dim).  The higher the degree, the less 'constrained'
 * the dimension (beware it is the converse of the usual meaning of
 * 'correlation').
 *
 * degree \in [0,1]
 *
 */
extern
double
ls_heuristics_correlationdegree (s_ls_space_t* space,
				 int dim);


/**
 * Modify bounds if necessary to ensure the existence of an integer
 * solution.
 *
 */
extern
int
ls_heuristics_narrow_intbounds (s_fm_rational_t* lb,
				s_fm_rational_t* ub,
				s_fm_solution_t* polyhedron,
				int idx,
				s_fm_vector_t* p);


/**
 * Fix and/or complete a schedule, with minimal corrections.
 * Return a 'correction distance'
 *
 */
extern
int
ls_heuristics_fixcomplete (s_ls_options_t* options,
			   s_ls_space_t* space,
			   s_fm_vector_t** schedule,
			   int dim);


/**
 * Helper functions for schedules manipulated as array of vectors.
 *
 */
/**
 * Print a schedule.
 *
 */
extern
void
ls_explorer_transfo_print (s_fm_vector_t** transfo);


/**
 * Copy a schedule in a NULL terminated array.
 *
 */
extern
void
ls_explorer_transfo_copy (s_fm_vector_t*** dest, s_fm_vector_t** transfo);


/**
 * Remove all schedules more than xx% slower than the best one.
 *
 */
extern
void
ls_explorer_heuristic_filter_multi (s_ls_options_t* options,
				    s_ls_explorer_t* data);

/**
 * Check if a schedule was already drawn.
 *
 */
extern
int
ls_explorer_transfo_exists_multi (s_ls_explorer_t* data,
				  s_fm_vector_t** draw);



END_C_DECLS


#endif // LETSEE_HEURISTICS_H
