/*
 * past_api.h: This file is part of the PAST project.
 *
 * PAST: the PoCC Abstract Syntax Tree
 *
 * Copyright (C) 2011 Louis-Noel Pouchet
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 3
 * of the License, or (at your option) any later version.
 *
 * The complete GNU Lesser General Public Licence Notice can be found
 * as the `COPYING.LESSER' file in the root directory.
 *
 * Author:
 * Louis-Noel Pouchet <pouchet@cse.ohio-state.edu>
 *
 */
#ifndef PAST_PAST_API_H
# define PAST_PAST_API_H

# include <past/common.h>
# include <past/symbols.h>
# include <past/past.h>

BEGIN_C_DECLS

/**
 * Count the number of for loops in the (sub-)tree dominated by 'root'.
 *
 */
extern
int
past_count_for_loops (s_past_node_t* root);

/**
 * Count the number of statements (past_cloogstmt and past_statement)
 * in a tree.
 *
 */
extern
int
past_count_statements (s_past_node_t* root);

/**
 * Count the number of nodes of type 'type' in a tree.
 *
 */
extern
int
past_count_nodetype (s_past_node_t* root, cs_past_node_type_t* type);

/**
 * Return the maximal loop depth in the (sub-)tree dominated by 'root'.
 *
 */
extern
int
past_max_loop_depth (s_past_node_t* root);

/**
 * Return the loop depth of a node, relative to the whole tree.
 *
 */
extern
int
past_loop_depth (s_past_node_t* root);

/**
 * Return a statement/cloogstmt surrounded by 'depth' loop(s) in the
 * (sub-)tree dominated by 'root'.
 *
 */
extern
s_past_node_t*
past_find_statement_at_depth (s_past_node_t* root, int depth);

/**
 * Return a NULL-terminated array of the outer loops in the (sub-)tree
 * dominated by 'root'.
 *
 */
extern
s_past_node_t**
past_outer_loops (s_past_node_t* root);

/**
 * Return a NULL-terminated array of the inner loops in the (sub-)tree
 * dominated by 'root'.
 *
 */
extern
s_past_node_t**
past_inner_loops (s_past_node_t* root);

/**
 * Return true if the (sub-)tree dominated by 'root' contains a loop.
 *
 */
extern
int
past_contain_loop (s_past_node_t* root);

/**
 * Return true if the node is an outer loop in the whole tree.
 *
 */
extern
int
past_is_outer_for_loop (s_past_node_t* node);

/**
 * Return true if the node is an inner loop in the whole tree.
 *
 */
extern
int
past_is_inner_for_loop (s_past_node_t* node);

/**
 * Rebuild the symbol table, and attach it to the root node.
 *
 */
extern
void
past_rebuild_symbol_table (s_past_node_t* node);

/**
 * Optimize 'for' loop bounds, by hoisting them as much as possible.
 *
 */
extern
void
past_optimize_loop_bounds (s_past_node_t* root);

END_C_DECLS


#endif // PAST_PAST_API_H
