/*
 * past.c: This file is part of the PAST project.
 *
 * PAST: the PoCC Abstract Syntax Tree
 *
 * Copyright (C) 2011 Louis-Noel Pouchet
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 3
 * of the License, or (at your option) any later version.
 *
 * The complete GNU Lesser General Public Licence Notice can be found
 * as the `COPYING.LESSER' file in the root directory.
 *
 * Author:
 * Louis-Noel Pouchet <pouchet@cse.ohio-state.edu>
 *
 */
#if HAVE_CONFIG_H
# include <past/config.h>
#endif

#include <assert.h>
#include <past/common.h>
#include <past/past.h>

/**
 *
 *
 */
int
past_node_is_a (s_past_node_t* node, cs_past_node_type_t* type)
{
  int i = 0;
  if (!node || !type)
    return 0;
  while (node->type->freefun[i] && i < PAST_NODE_HIERARCHY_HEIGHT)
    if (node->type->freefun[i++] == type->freefun[0])
      return 1;
  return 0;
}

static
void
past_node_init (s_past_node_t* node,
		cs_past_node_type_t* type,
		past_visitor_fun_t visitor)
{
  node->type = type;
  node->visitor = visitor;
  node->parent = NULL;
  node->next = NULL;
  node->metainfo = NULL;
  node->usr = NULL;
}

static
void
past_node_freeattr (s_past_node_t* node)
{
  if (node)
    {
      XFREE(node->metainfo);
    }
}

static
set_parent_sibling_nodes (s_past_node_t* node, s_past_node_t* parent)
{
  for (; node; node->parent = parent, node = node->next)
    ;
}


/**
 * past_root
 *
 *
 */
static void
past_root_free (s_past_node_t* node)
{
  assert (past_node_is_a (node, past_root));
  s_past_root_t* pr = (s_past_root_t*) node;
  symbol_table_free (pr->symboltable);
  past_node_freeattr (node);
  XFREE(pr);
}
static void
past_root_visitor (s_past_node_t* node,
		   past_fun_t prefix,
		   void* prefix_data,
		   past_fun_t suffix,
		   void* suffix_data)
{
  assert (past_node_is_a (node, past_root));
  PAST_DECLARE_TYPED(root, r, node);
  past_visitor (r->body, prefix, prefix_data, suffix, suffix_data);
}
PAST_DECLARE_NODE_IN_HIERARCHY_UNIT_1(root);

s_past_root_t* past_root_create (s_symbol_table_t* symboltable,
				 s_past_node_t* body)
{
  s_past_root_t* n = XMALLOC(s_past_root_t, 1);
  past_node_init (&(n->node), past_root, past_root_visitor);
  n->symboltable = symboltable;
  n->body = body;
  set_parent_sibling_nodes (n->body, (s_past_node_t*) n);

  return n;
}

s_past_node_t* past_node_root_create (s_symbol_table_t* symboltable,
				      s_past_node_t* body)
{
  return &past_root_create(symboltable, body)->node;
}


/**
 * past_variable
 *
 *
 */
static void
past_variable_free (s_past_node_t* node)
{
  assert (past_node_is_a (node, past_variable));
  s_past_variable_t* pv = (s_past_variable_t*) node;
  past_node_freeattr (node);
  XFREE(pv);
}
static void
past_variable_visitor (s_past_node_t* node,
		       past_fun_t prefix,
		       void* prefix_data,
		       past_fun_t suffix,
		       void* suffix_data)
{
  assert (past_node_is_a (node, past_variable));
}
PAST_DECLARE_NODE_IN_HIERARCHY_UNIT_1(variable);

s_past_variable_t* past_variable_create (s_symbol_t* symbol)
{
  s_past_variable_t* n = XMALLOC(s_past_variable_t, 1);
  past_node_init (&(n->node), past_variable, past_variable_visitor);
  n->symbol = symbol;
  n->usr = NULL;

  return n;
}

s_past_node_t* past_node_variable_create (s_symbol_t* symbol)
{
  return &past_variable_create (symbol)->node;
}


/**
 * past_value
 *
 *
 */
static void
past_value_free (s_past_node_t* node)
{
  assert (past_node_is_a (node, past_value));
  s_past_value_t* pv = (s_past_value_t*) node;
  past_node_freeattr (node);
  XFREE(pv);
}
static void
past_value_visitor (s_past_node_t* node,
		    past_fun_t prefix,
		    void* prefix_data,
		    past_fun_t suffix,
		    void* suffix_data)
{
  assert (past_node_is_a (node, past_value));
}
PAST_DECLARE_NODE_IN_HIERARCHY_UNIT_1(value);

s_past_value_t* past_value_create (int type, u_past_value_data_t val)
{
  s_past_value_t* n = XMALLOC(s_past_value_t, 1);
  past_node_init (&(n->node), past_value, past_value_visitor);
  n->type = type;
  n->value = val;

  return n;
}

s_past_node_t* past_node_value_create (int type, u_past_value_data_t val)
{
  return &past_value_create (type, val)->node;
}

s_past_value_t* past_value_create_from_int (int val)
{
  u_past_value_data_t uval;
  uval.intval = val;
  return past_value_create (e_past_value_int, uval);
}

s_past_node_t* past_node_value_create_from_int (int val)
{
  return &past_value_create_from_int (val)->node;
}


/**
 * past_for
 *
 *
 */
static void
past_for_free (s_past_node_t* node)
{
  assert (past_node_is_a (node, past_for));
  s_past_for_t* pf = (s_past_for_t*) node;
  past_deep_free (pf->init);
  past_deep_free (pf->test);
  past_deep_free (pf->increment);
  past_deep_free ((s_past_node_t*) pf->iterator);
  past_deep_free (pf->body);
  past_node_freeattr (node);
  XFREE(pf);
}
static void
past_for_visitor (s_past_node_t* node,
		  past_fun_t prefix,
		  void* prefix_data,
		  past_fun_t suffix,
		  void* suffix_data)
{
  assert (past_node_is_a (node, past_for));
  PAST_DECLARE_TYPED(for, r, node);
  past_visitor (r->init, prefix, prefix_data, suffix, suffix_data);
  past_visitor (r->test, prefix, prefix_data, suffix, suffix_data);
  past_visitor (r->increment, prefix, prefix_data, suffix, suffix_data);
  past_visitor (r->body, prefix, prefix_data, suffix, suffix_data);
}
PAST_DECLARE_NODE_IN_HIERARCHY_UNIT_1(for);

s_past_for_t* past_for_create (s_past_node_t* init,
			       s_past_node_t* test,
			       s_past_variable_t* iterator,
			       s_past_node_t* increment,
			       s_past_node_t* body,
			       void* usr)
{
  s_past_for_t* n = XMALLOC(s_past_for_t, 1);
  past_node_init (&(n->node), past_for, past_for_visitor);
  n->init = init;
  n->test = test;
  n->iterator = iterator;
  n->increment = increment;
  n->body = body;
  n->type = e_past_unknown_loop;
  n->usr = usr;
  set_parent_sibling_nodes (n->init, (s_past_node_t*) n);
  set_parent_sibling_nodes (n->test, (s_past_node_t*) n);
  set_parent_sibling_nodes (n->increment, (s_past_node_t*) n);
  set_parent_sibling_nodes (n->body, (s_past_node_t*) n);

  return n;
}

s_past_node_t* past_node_for_create (s_past_node_t* init,
				     s_past_node_t* test,
				     s_past_variable_t* iterator,
				     s_past_node_t* increment,
				     s_past_node_t* body,
				     void* usr)
{
  return &past_for_create (init, test, iterator, increment, body, usr)->node;
}


/**
 * past_parfor
 *
 *
 */
static void
past_parfor_free (s_past_node_t* node)
{
  assert (past_node_is_a (node, past_parfor));
  s_past_parfor_t* pf = (s_past_parfor_t*) node;
  past_deep_free (pf->init);
  past_deep_free (pf->test);
  past_deep_free (pf->increment);
  past_deep_free ((s_past_node_t*) pf->iterator);
  past_deep_free (pf->body);
  int i;
  for (i = 0; pf->private_vars && pf->private_vars[i]; ++i)
    past_deep_free ((s_past_node_t*)pf->private_vars[i]);
  past_node_freeattr (node);
  XFREE(pf);
}
static void
past_parfor_visitor (s_past_node_t* node,
		     past_fun_t prefix,
		     void* prefix_data,
		     past_fun_t suffix,
		     void* suffix_data)
{
  assert (past_node_is_a (node, past_parfor));
  PAST_DECLARE_TYPED(parfor, r, node);
  past_visitor (r->init, prefix, prefix_data, suffix, suffix_data);
  past_visitor (r->test, prefix, prefix_data, suffix, suffix_data);
  past_visitor (r->increment, prefix, prefix_data, suffix, suffix_data);
  past_visitor (r->body, prefix, prefix_data, suffix, suffix_data);
}
PAST_DECLARE_NODE_IN_HIERARCHY_UNIT_2(parfor,for);

s_past_parfor_t* past_parfor_create (s_past_node_t* init,
				     s_past_node_t* test,
				     s_past_variable_t* iterator,
				     s_past_node_t* increment,
				     s_past_node_t* body,
				     void* usr)
{
  s_past_parfor_t* n = XMALLOC(s_past_parfor_t, 1);
  past_node_init (&(n->node), past_parfor, past_parfor_visitor);
  n->init = init;
  n->test = test;
  n->iterator = iterator;
  n->increment = increment;
  n->body = body;
  n->type = e_past_unknown_loop;
  n->private_vars = NULL;
  n->usr = usr;
  set_parent_sibling_nodes (n->init, (s_past_node_t*) n);
  set_parent_sibling_nodes (n->test, (s_past_node_t*) n);
  set_parent_sibling_nodes (n->increment, (s_past_node_t*) n);
  set_parent_sibling_nodes (n->body, (s_past_node_t*) n);

  return (s_past_parfor_t*)n;
}

s_past_node_t* past_node_parfor_create (s_past_node_t* init,
					s_past_node_t* test,
					s_past_variable_t* iterator,
					s_past_node_t* increment,
					s_past_node_t* body,
					void* usr)
{
  return &past_parfor_create (init, test, iterator, increment, body, usr)->node;
}


/**
 * past_binary
 *
 *
 */
static void
past_binary_free (s_past_node_t* node)
{
  assert (past_node_is_a (node, past_binary));
  s_past_binary_t* pf = (s_past_binary_t*) node;
  past_deep_free (pf->lhs);
  past_deep_free (pf->rhs);
  past_node_freeattr (node);
  XFREE(pf);
}
static void
past_binary_visitor (s_past_node_t* node,
		     past_fun_t prefix,
		     void* prefix_data,
		     past_fun_t suffix,
		     void* suffix_data)
{
  assert (past_node_is_a (node, past_binary));
  PAST_DECLARE_TYPED(binary, r, node);
  past_visitor (r->lhs, prefix, prefix_data, suffix, suffix_data);
  past_visitor (r->rhs, prefix, prefix_data, suffix, suffix_data);
}
PAST_DECLARE_NODE_IN_HIERARCHY_UNIT_1(binary);

#define declare_bin_free_ops(type)					\
  static void past_##type##_free (s_past_node_t* node)			\
   { past_binary_free (node); }						\
  static void past_##type##_visitor (s_past_node_t* node,		\
				     past_fun_t prefix,			\
				     void* prefix_data,			\
				     past_fun_t suffix,			\
				     void* suffix_data)			\
   { past_binary_visitor (node, prefix, prefix_data, suffix, suffix_data); } \
  PAST_DECLARE_NODE_IN_HIERARCHY_UNIT_2(type, binary);

declare_bin_free_ops(add);
declare_bin_free_ops(sub);
declare_bin_free_ops(mul);
declare_bin_free_ops(div);
declare_bin_free_ops(mod);
declare_bin_free_ops(min);
declare_bin_free_ops(max);
declare_bin_free_ops(ceild);
declare_bin_free_ops(floord);
declare_bin_free_ops(and);
declare_bin_free_ops(or);
declare_bin_free_ops(equal);
declare_bin_free_ops(assign);
declare_bin_free_ops(geq);
declare_bin_free_ops(leq);
declare_bin_free_ops(gt);
declare_bin_free_ops(lt);
declare_bin_free_ops(arrayref);

s_past_binary_t* past_binary_create (cs_past_node_type_t* type,
				     s_past_node_t* lhs,
				     s_past_node_t* rhs)
{
  s_past_binary_t* n = XMALLOC(s_past_binary_t, 1);
  if (type)
    past_node_init (&(n->node), type, past_binary_visitor);
  else
    past_node_init (&(n->node), past_binary, past_binary_visitor);
  n->lhs = lhs;
  n->rhs = rhs;
  set_parent_sibling_nodes (n->lhs, (s_past_node_t*) n);
  set_parent_sibling_nodes (n->rhs, (s_past_node_t*) n);

  return n;
}

s_past_node_t* past_node_binary_create (cs_past_node_type_t* type,
					s_past_node_t* lhs,
					s_past_node_t* rhs)
{
  return &past_binary_create (type, lhs, rhs)->node;
}


/**
 * past_unary
 *
 *
 */
static void
past_unary_free (s_past_node_t* node)
{
  assert (past_node_is_a (node, past_unary));
  s_past_unary_t* pf = (s_past_unary_t*) node;
  past_deep_free (pf->expr);
  past_node_freeattr (node);
  XFREE(pf);
}
static void
past_unary_visitor (s_past_node_t* node,
		    past_fun_t prefix,
		    void* prefix_data,
		    past_fun_t suffix,
		    void* suffix_data)
{
  assert (past_node_is_a (node, past_unary));
  PAST_DECLARE_TYPED(unary, r, node);
  past_visitor (r->expr, prefix, prefix_data, suffix, suffix_data);
}
PAST_DECLARE_NODE_IN_HIERARCHY_UNIT_1(unary);

#define declare_un_free_ops(type)					\
  static void past_##type##_free (s_past_node_t* node)			\
    { past_unary_free (node); }						\
  static void past_##type##_visitor (s_past_node_t* node,		\
				     past_fun_t prefix,			\
				     void* prefix_data,			\
				     past_fun_t suffix,			\
				     void* suffix_data)			\
   { past_unary_visitor (node, prefix, prefix_data, suffix, suffix_data); } \
PAST_DECLARE_NODE_IN_HIERARCHY_UNIT_2(type, unary);

declare_un_free_ops(round);
declare_un_free_ops(floor);
declare_un_free_ops(ceil);
declare_un_free_ops(sqrt);
declare_un_free_ops(inc_before);
declare_un_free_ops(inc_after);
declare_un_free_ops(dec_before);
declare_un_free_ops(dec_after);

s_past_unary_t* past_unary_create (cs_past_node_type_t* type,
				   s_past_node_t* expr)
{
  s_past_unary_t* n = XMALLOC(s_past_unary_t, 1);
  if (type)
    past_node_init (&(n->node), type, past_unary_visitor);
  else
    past_node_init (&(n->node), past_unary, past_unary_visitor);
  n->expr = expr;
  set_parent_sibling_nodes (n->expr, (s_past_node_t*) n);

  return n;
}

s_past_node_t* past_node_unary_create (cs_past_node_type_t* type,
				       s_past_node_t* expr)
{
  return &past_unary_create (type, expr)->node;
}


/**
 * past_block
 *
 *
 */
static void
past_block_free (s_past_node_t* node)
{
  assert (past_node_is_a (node, past_block));
  s_past_block_t* pf = (s_past_block_t*) node;
  past_deep_free (pf->body);
  past_node_freeattr (node);
  XFREE(pf);
}
static void
past_block_visitor (s_past_node_t* node,
		    past_fun_t prefix,
		    void* prefix_data,
		    past_fun_t suffix,
		    void* suffix_data)
{
  assert (past_node_is_a (node, past_block));
  PAST_DECLARE_TYPED(block, r, node);
  past_visitor (r->body, prefix, prefix_data, suffix, suffix_data);
}
PAST_DECLARE_NODE_IN_HIERARCHY_UNIT_1(block);

s_past_block_t* past_block_create (s_past_node_t* body)
{
  s_past_block_t* n = XMALLOC(s_past_block_t, 1);
  past_node_init (&(n->node), past_block, past_block_visitor);
  n->body = body;
  set_parent_sibling_nodes (n->body, (s_past_node_t*) n);

  return n;
}

s_past_node_t* past_node_block_create (s_past_node_t* body)
{
  return &past_block_create (body)->node;
}


/**
 * past_if
 *
 *
 */
static void
past_if_free (s_past_node_t* node)
{
  assert (past_node_is_a (node, past_if));
  s_past_if_t* pf = (s_past_if_t*) node;
  past_deep_free (pf->condition);
  past_deep_free (pf->then_clause);
  past_deep_free (pf->else_clause);
  past_node_freeattr (node);
  XFREE(pf);
}
static void
past_if_visitor (s_past_node_t* node,
		 past_fun_t prefix,
		 void* prefix_data,
		 past_fun_t suffix,
		 void* suffix_data)
{
  assert (past_node_is_a (node, past_if));
  PAST_DECLARE_TYPED(if, r, node);
  past_visitor (r->condition, prefix, prefix_data, suffix, suffix_data);
  past_visitor (r->then_clause, prefix, prefix_data, suffix, suffix_data);
  past_visitor (r->else_clause, prefix, prefix_data, suffix, suffix_data);
}
PAST_DECLARE_NODE_IN_HIERARCHY_UNIT_1(if);

s_past_if_t* past_if_create (s_past_node_t* cond,
			     s_past_node_t* then_clause,
			     s_past_node_t* else_clause)
{
  s_past_if_t* n = XMALLOC(s_past_if_t, 1);
  past_node_init (&(n->node), past_if, past_if_visitor);
  n->condition = cond;
  n->then_clause = then_clause;
  n->else_clause = else_clause;
  set_parent_sibling_nodes (n->condition, (s_past_node_t*) n);
  set_parent_sibling_nodes (n->then_clause, (s_past_node_t*) n);
  set_parent_sibling_nodes (n->else_clause, (s_past_node_t*) n);

  return n;
}

s_past_node_t* past_node_if_create (s_past_node_t* cond,
				    s_past_node_t* then_clause,
				    s_past_node_t* else_clause)
{
  return &past_if_create (cond, then_clause, else_clause)->node;
}


/**
 * past_affineguard
 *
 *
 */
static void
past_affineguard_free (s_past_node_t* node)
{
  assert (past_node_is_a (node, past_affineguard));
  s_past_affineguard_t* pf = (s_past_affineguard_t*) node;
  past_deep_free (pf->condition);
  past_deep_free (pf->then_clause);
  past_node_freeattr (node);
  XFREE(pf);
}
static void
past_affineguard_visitor (s_past_node_t* node,
			  past_fun_t prefix,
			  void* prefix_data,
			  past_fun_t suffix,
			  void* suffix_data)
{
  assert (past_node_is_a (node, past_affineguard));
  PAST_DECLARE_TYPED(affineguard, r, node);
  past_visitor (r->condition, prefix, prefix_data, suffix, suffix_data);
  past_visitor (r->then_clause, prefix, prefix_data, suffix, suffix_data);
}
PAST_DECLARE_NODE_IN_HIERARCHY_UNIT_1(affineguard);

s_past_affineguard_t* past_affineguard_create (s_past_node_t* cond,
					       s_past_node_t* then_clause)
{
  s_past_affineguard_t* n = XMALLOC(s_past_affineguard_t, 1);
  past_node_init (&(n->node), past_affineguard, past_affineguard_visitor);
  n->condition = cond;
  n->then_clause = then_clause;
  set_parent_sibling_nodes (n->condition, (s_past_node_t*) n);
  set_parent_sibling_nodes (n->then_clause, (s_past_node_t*) n);

  return n;
}

s_past_node_t* past_node_affineguard_create (s_past_node_t* cond,
					     s_past_node_t* then_clause)
{
  return &past_affineguard_create (cond, then_clause)->node;
}


/**
 * past_cloogstmt
 *
 *
 */
static void
past_cloogstmt_free (s_past_node_t* node)
{
  assert (past_node_is_a (node, past_cloogstmt));
  s_past_cloogstmt_t* pf = (s_past_cloogstmt_t*) node;
  if (pf->stmt_name)
    XFREE(pf->stmt_name);
  past_deep_free (pf->substitutions);
  past_deep_free (pf->references);
  past_node_freeattr (node);
  XFREE(pf);
}
static void
past_cloogstmt_visitor (s_past_node_t* node,
			 past_fun_t prefix,
			 void* prefix_data,
			 past_fun_t suffix,
			 void* suffix_data)
{
  assert (past_node_is_a (node, past_cloogstmt));
  s_past_cloogstmt_t* pf = (s_past_cloogstmt_t*) node;
  past_visitor (pf->substitutions, prefix, prefix_data, suffix, suffix_data);
}
PAST_DECLARE_NODE_IN_HIERARCHY_UNIT_1(cloogstmt);

s_past_cloogstmt_t* past_cloogstmt_create (void* cloogdomain,
					   void* cloogstatement,
					   s_past_variable_t* stmt_name,
					   int stmt_number,
					   s_past_node_t* substitutions)
{
  s_past_cloogstmt_t* n = XMALLOC(s_past_cloogstmt_t, 1);
  past_node_init (&(n->node), past_cloogstmt, past_cloogstmt_visitor);
  n->cloogdomain = cloogdomain;
  n->cloogstatement = cloogstatement;
  n->stmt_name = stmt_name;
  n->stmt_number = stmt_number;
  n->substitutions = substitutions;
  n->references = NULL;
  set_parent_sibling_nodes (n->substitutions, (s_past_node_t*) n);
  set_parent_sibling_nodes ((s_past_node_t*) n->stmt_name, (s_past_node_t*) n);

  return n;
}

s_past_node_t* past_node_cloogstmt_create (void* cloogdomain,
					   void* cloogstatement,
					   s_past_variable_t* stmt_name,
					   int stmt_number,
					   s_past_node_t* substitutions)
{
  return &past_cloogstmt_create (cloogdomain, cloogstatement, stmt_name,
				 stmt_number, substitutions)->node;
}


/**
 * past_statement
 *
 *
 */
static void
past_statement_free (s_past_node_t* node)
{
  assert (past_node_is_a (node, past_statement));
  s_past_statement_t* pf = (s_past_statement_t*) node;
  past_deep_free (pf->body);
  past_node_freeattr (node);
  XFREE(pf);
}
static void
past_statement_visitor (s_past_node_t* node,
			past_fun_t prefix,
			void* prefix_data,
			past_fun_t suffix,
			void* suffix_data)
{
  assert (past_node_is_a (node, past_statement));
  PAST_DECLARE_TYPED(statement, r, node);
  past_visitor (r->body, prefix, prefix_data, suffix, suffix_data);
}
PAST_DECLARE_NODE_IN_HIERARCHY_UNIT_1(statement);

s_past_statement_t* past_statement_create (s_past_node_t* body)
{
  s_past_statement_t* n = XMALLOC(s_past_statement_t, 1);
  past_node_init (&(n->node), past_statement, past_statement_visitor);
  n->body = body;
  set_parent_sibling_nodes (n->body, (s_past_node_t*) n);

  return n;
}

s_past_node_t* past_node_statement_create (s_past_node_t* body)
{
  return &past_statement_create (body)->node;
}


/**
 * past_funcall
 *
 *
 */
static void
past_funcall_free (s_past_node_t* node)
{
  assert (past_node_is_a (node, past_funcall));
  s_past_funcall_t* pf = (s_past_funcall_t*) node;
  past_deep_free ((s_past_node_t*) pf->name);
  past_deep_free (pf->args_list);
  past_node_freeattr (node);
  XFREE(pf);
}
static void
past_funcall_visitor (s_past_node_t* node,
		      past_fun_t prefix,
		      void* prefix_data,
		      past_fun_t suffix,
		      void* suffix_data)
{
  assert (past_node_is_a (node, past_funcall));
  PAST_DECLARE_TYPED(funcall, r, node);
  past_visitor (r->args_list, prefix, prefix_data, suffix, suffix_data);
}
PAST_DECLARE_NODE_IN_HIERARCHY_UNIT_1(funcall);

s_past_funcall_t* past_funcall_create (s_past_variable_t* name,
				       s_past_node_t* args_list)
{
  s_past_funcall_t* n = XMALLOC(s_past_funcall_t, 1);
  past_node_init (&(n->node), past_funcall, past_funcall_visitor);
  n->name = name;
  n->args_list = args_list;

  return n;
}

s_past_node_t* past_node_funcall_create (s_past_variable_t* name,
					 s_past_node_t* args_list)
{
  return &past_funcall_create (name, args_list)->node;
}


/**
 * past_vardecl
 *
 *
 */
static void
past_vardecl_free (s_past_node_t* node)
{
  assert (past_node_is_a (node, past_vardecl));
  s_past_vardecl_t* pf = (s_past_vardecl_t*) node;
  past_deep_free ((s_past_node_t*) pf->name);
  past_deep_free ((s_past_node_t*) pf->type);
  past_node_freeattr (node);
  XFREE(pf);
}
static void
past_vardecl_visitor (s_past_node_t* node,
		      past_fun_t prefix,
		      void* prefix_data,
		      past_fun_t suffix,
		      void* suffix_data)
{
  assert (past_node_is_a (node, past_vardecl));
  PAST_DECLARE_TYPED(vardecl, r, node);
}
PAST_DECLARE_NODE_IN_HIERARCHY_UNIT_1(vardecl);

s_past_vardecl_t* past_vardecl_create (s_past_variable_t* name,
				       s_past_variable_t* type)
{
  s_past_vardecl_t* n = XMALLOC(s_past_vardecl_t, 1);
  past_node_init (&(n->node), past_vardecl, past_vardecl_visitor);
  n->name = name;
  n->type = type;

  return n;
}

s_past_node_t* past_node_vardecl_create (s_past_variable_t* name,
					 s_past_variable_t* type)
{
  return &past_vardecl_create (name, type)->node;
}



/**
 * past_ternary_cond
 *
 *
 */
static void
past_ternary_cond_free (s_past_node_t* node)
{
  assert (past_node_is_a (node, past_ternary_cond));
  s_past_ternary_cond_t* pf = (s_past_ternary_cond_t*) node;
  past_deep_free ((s_past_node_t*) pf->cond);
  past_deep_free ((s_past_node_t*) pf->true_clause);
  past_deep_free ((s_past_node_t*) pf->false_clause);
  past_node_freeattr (node);
  XFREE(pf);
}
static void
past_ternary_cond_visitor (s_past_node_t* node,
			   past_fun_t prefix,
			   void* prefix_data,
			   past_fun_t suffix,
			   void* suffix_data)
{
  assert (past_node_is_a (node, past_ternary_cond));
  PAST_DECLARE_TYPED(ternary_cond, r, node);
  past_visitor (r->cond, prefix, prefix_data, suffix, suffix_data);
  past_visitor (r->true_clause, prefix, prefix_data, suffix, suffix_data);
  past_visitor (r->false_clause, prefix, prefix_data, suffix, suffix_data);
}
PAST_DECLARE_NODE_IN_HIERARCHY_UNIT_1(ternary_cond);

s_past_ternary_cond_t* past_ternary_cond_create (s_past_node_t* cond,
						 s_past_node_t* true_clause,
						 s_past_node_t* false_clause)
{
  s_past_ternary_cond_t* n = XMALLOC(s_past_ternary_cond_t, 1);
  past_node_init (&(n->node), past_ternary_cond, past_ternary_cond_visitor);
  n->cond = cond;
  n->true_clause = true_clause;
  n->false_clause = false_clause;

  return n;
}


s_past_node_t* past_node_ternary_cond_create (s_past_node_t* cond,
					      s_past_node_t* true_clause,
					      s_past_node_t* false_clause)
{
  return &(past_ternary_cond_create (cond, true_clause, false_clause)->node);
}


/**
 *
 * cloning a node/tree (deep copy).
 *
 *
 */
s_past_node_t* past_clone (s_past_node_t* s)
{
  if (! s)
    return NULL;

  s_past_node_t* ret = NULL;;
  s_past_node_t* val;
  s_past_node_t** stmt = &val;

  // Traverse the clast.
  for ( ; s; s = s->next)
    {
      if (past_node_is_a (s, past_root))
	{
	  PAST_DECLARE_TYPED(root, pr, s);
	  *stmt = past_node_root_create (pr->symboltable,
					 past_clone (pr->body));
	}
      else if (past_node_is_a (s, past_parfor))
	{
	  PAST_DECLARE_TYPED(parfor, pf, s);
	  *stmt = past_node_parfor_create
	    (past_clone (pf->init),
	     past_clone (pf->test),
	     (s_past_variable_t*) past_clone ((s_past_node_t*) pf->iterator),
	     past_clone (pf->increment),
	     past_clone (pf->body),
	     pf->usr);
	  PAST_DECLARE_TYPED(parfor, newpf, *stmt);
	  newpf->type = pf->type;
	  if (pf->private_vars)
	    {
	      int i;
	      for (i = 0; pf->private_vars[i]; ++i)
		;
	      s_past_variable_t** newvars = XMALLOC(s_past_variable_t*, i + 1);
	      for (i = 0; pf->private_vars[i]; ++i)
		{
		  s_past_node_t* newvar =
		    past_clone ((s_past_node_t*) pf->private_vars[i]);
		  newvars[i] = (s_past_variable_t*) newvar;
		}
	      newvars[i] = NULL;
	      newpf->private_vars = newvars;
	    }
	}
      else if (past_node_is_a (s, past_for))
	{
	  PAST_DECLARE_TYPED(for, pf, s);
	  *stmt = past_node_for_create
	    (past_clone (pf->init),
	     past_clone (pf->test),
	     (s_past_variable_t*) past_clone ((s_past_node_t*) pf->iterator),
	     past_clone (pf->increment),
	     past_clone (pf->body),
	     pf->usr);
	  PAST_DECLARE_TYPED(for, newpf, *stmt);
	  newpf->type = pf->type;
	}
      else if (past_node_is_a (s, past_if))
	{
	  PAST_DECLARE_TYPED(if, pf, s);
	  *stmt = past_node_if_create (past_clone (pf->condition),
					past_clone (pf->then_clause),
					past_clone (pf->else_clause));
	}
      else if (past_node_is_a (s, past_affineguard))
	{
	  PAST_DECLARE_TYPED(affineguard, pf, s);
	  *stmt = past_node_affineguard_create (past_clone (pf->condition),
						past_clone (pf->then_clause));
	}
      else if (past_node_is_a (s, past_block))
	{
	  PAST_DECLARE_TYPED(block, pf, s);
	  *stmt = past_node_block_create (past_clone (pf->body));
	}
      else if (past_node_is_a (s, past_statement))
	{
	  PAST_DECLARE_TYPED(statement, pf, s);
	  *stmt = past_node_statement_create (past_clone (pf->body));
	}
      else if (past_node_is_a (s, past_cloogstmt))
	{
	  PAST_DECLARE_TYPED(cloogstmt, pf, s);
	  *stmt = past_node_cloogstmt_create
	    (pf->cloogdomain,
	     pf->cloogstatement,
	     (s_past_variable_t*)past_clone ((s_past_node_t*)pf->stmt_name),
	     pf->stmt_number,
	     past_clone (pf->substitutions));
	}
      else if (past_node_is_a (s, past_binary))
	{
	  PAST_DECLARE_TYPED(binary, pb, s);
	  *stmt = past_node_binary_create (s->type,
					   past_clone (pb->lhs),
					   past_clone (pb->rhs));
	}
      else if (past_node_is_a (s, past_unary))
	{
	  PAST_DECLARE_TYPED(unary, pb, s);
	  *stmt = past_node_unary_create (s->type,
					  past_clone (pb->expr));
	}
      else if (past_node_is_a (s, past_funcall))
	{
	  PAST_DECLARE_TYPED(funcall, pf, s);
	  s_symbol_t* newsym;
	  if (pf->name->symbol->is_char_data)
	    newsym = symbol_add_from_char (NULL, pf->name->symbol->data);
	  else
	    newsym = symbol_add_from_data (NULL, pf->name->symbol->data);
	  s_past_variable_t* newvar = past_variable_create (newsym);
	  *stmt = past_node_funcall_create (newvar, past_clone (pf->args_list));
	}
      else if (past_node_is_a (s, past_vardecl))
	{
	  PAST_DECLARE_TYPED(vardecl, pf, s);
	  s_symbol_t* newsym1;
	  if (pf->name->symbol->is_char_data)
	    newsym1 = symbol_add_from_char (NULL, pf->name->symbol->data);
	  else
	    newsym1 = symbol_add_from_data (NULL, pf->name->symbol->data);
	  s_symbol_t* newsym2;
	  if (pf->type->symbol->is_char_data)
	    newsym2 = symbol_add_from_char (NULL, pf->type->symbol->data);
	  else
	    newsym2 = symbol_add_from_data (NULL, pf->type->symbol->data);
	  s_past_variable_t* newvar1 = past_variable_create (newsym1);
	  s_past_variable_t* newvar2 = past_variable_create (newsym2);
	  *stmt = past_node_vardecl_create (newvar1, newvar2);
	}
      else if (past_node_is_a (s, past_variable))
	{
	  PAST_DECLARE_TYPED(variable, pv, s);
	  s_symbol_t* newsym;
	  if (pv->symbol->is_char_data)
	    newsym = symbol_add_from_char (NULL, pv->symbol->data);
	  else
	    newsym = symbol_add_from_data (NULL, pv->symbol->data);
	  *stmt = past_node_variable_create (newsym);
	}
      else if (past_node_is_a (s, past_value))
	{
	  PAST_DECLARE_TYPED(value, pv, s);
	  *stmt = past_node_value_create (pv->type, pv->value);
	}
      else if (past_node_is_a (s, past_ternary_cond))
	{
	  PAST_DECLARE_TYPED(ternary_cond, pf, s);
	  *stmt = past_node_ternary_cond_create (past_clone (pf->cond),
						 past_clone (pf->true_clause),
						 past_clone (pf->false_clause));
	}
      else
	fprintf (stderr, "[PAST] Clone: unknown node\n");
      if (ret == NULL)
	ret = *stmt;
      stmt = &((*stmt)->next);
    }
  *stmt = NULL;

  return ret;
}



/**
 *
 * Free node/tree (deep free).
 *
 *
 */
void
past_deep_free (s_past_node_t* node)
{
  s_past_node_t* next;
  while (node)
    {
      next = node->next;
      node->type->freefun[0] (node);
      node = next;
    }
}


/**
 * Shallow copy.
 *
 */
s_past_node_t* past_copy (s_past_node_t* node)
{
  /// FIXME: Implement this.
  assert (0);
  return node;
}



/**
 * Set parent pointer for the full tree.
 *
 */
static
void set_parent_pref (s_past_node_t* node, void* data)
{
  if (past_node_is_a (node, past_binary))
    {
      PAST_DECLARE_TYPED(binary, n, node);
      assert (n->lhs);
      assert (n->rhs);
      set_parent_sibling_nodes (n->lhs, node);
      set_parent_sibling_nodes (n->rhs, node);
    }
  else if (past_node_is_a (node, past_unary))
    {
      PAST_DECLARE_TYPED(unary, n, node);
      assert (n->expr);
      set_parent_sibling_nodes (n->expr, node);
    }
  else if (past_node_is_a (node, past_for))
    {
      PAST_DECLARE_TYPED(for, n, node);
      if (n->body)
	set_parent_sibling_nodes (n->body, node);
      if (n->init)
	set_parent_sibling_nodes (n->init, node);
      if (n->test)
	set_parent_sibling_nodes (n->test, node);
      if (n->increment)
	set_parent_sibling_nodes (n->increment, node);
    }
  else if (past_node_is_a (node, past_if))
    {
      PAST_DECLARE_TYPED(if, n, node);
      if (n->condition)
	set_parent_sibling_nodes (n->condition, node);
      if (n->then_clause)
	set_parent_sibling_nodes (n->then_clause, node);
      if (n->else_clause)
	set_parent_sibling_nodes (n->else_clause, node);
    }
  else if (past_node_is_a (node, past_affineguard))
    {
      PAST_DECLARE_TYPED(affineguard, n, node);
      if (n->condition)
	set_parent_sibling_nodes (n->condition, node);
      if (n->then_clause)
	set_parent_sibling_nodes (n->then_clause, node);
    }
  else if (past_node_is_a (node, past_block))
    {
      PAST_DECLARE_TYPED(block, n, node);
      if (n->body)
	set_parent_sibling_nodes (n->body, node);
    }
  else if (past_node_is_a (node, past_root))
    {
      PAST_DECLARE_TYPED(root, n, node);
      if (n->body)
	set_parent_sibling_nodes (n->body, node);
      node->parent = NULL;
    }
  else if (past_node_is_a (node, past_statement))
    {
      PAST_DECLARE_TYPED(statement, n, node);
      if (n->body)
	set_parent_sibling_nodes (n->body, node);
    }
  else if (past_node_is_a (node, past_funcall))
    {
      PAST_DECLARE_TYPED(funcall, n, node);
      if (n->args_list)
	set_parent_sibling_nodes (n->args_list, node);
    }
  else if (past_node_is_a (node, past_ternary_cond))
    {
      PAST_DECLARE_TYPED(ternary_cond, n, node);
      if (n->cond)
	set_parent_sibling_nodes (n->cond, node);
      if (n->true_clause)
	set_parent_sibling_nodes (n->true_clause, node);
      if (n->false_clause)
	set_parent_sibling_nodes (n->false_clause, node);
    }
}

void past_set_parent (s_past_node_t* root)
{
  past_visitor (root, set_parent_pref, NULL, NULL, NULL);
}


/**
 * Generic visitor.
 *
 */
void past_visitor (s_past_node_t* node,
		   past_fun_t prefix,
		   void* prefix_data,
		   past_fun_t suffix,
		   void* suffix_data)
{
  while (node)
    {
      if (prefix)
	prefix (node, prefix_data);
      node->visitor (node,
		     prefix, prefix_data,
		     suffix, suffix_data);
      if (suffix)
	suffix (node, suffix_data);
      node = node->next;
    }
}

static
void
find_node_in_siblings (s_past_node_t** node,
		       s_past_node_t* tofind,
		       s_past_node_t** dest)
{
  while (*node)
    {
      if (*node == tofind)
	{
	  *dest = (s_past_node_t*) node;
	  return;
	}
      node = &((*node)->next);
    }
}

static
void find_parent_ptr (s_past_node_t* node, void* data)
{
  s_past_node_t** args = (s_past_node_t**) data;
  if (past_node_is_a (node, past_binary))
    {
      PAST_DECLARE_TYPED(binary, n, node);
      assert (n->lhs);
      assert (n->rhs);
      find_node_in_siblings (&(n->lhs), args[0], &(args[1]));
      find_node_in_siblings (&(n->rhs), args[0], &(args[1]));
    }
  else if (past_node_is_a (node, past_unary))
    {
      PAST_DECLARE_TYPED(unary, n, node);
      assert (n->expr);
      find_node_in_siblings (&(n->expr), args[0], &(args[1]));
    }
  else if (past_node_is_a (node, past_for))
    {
      PAST_DECLARE_TYPED(for, n, node);
      if (n->body)
	find_node_in_siblings (&(n->body), args[0], &(args[1]));
      if (n->init)
	find_node_in_siblings (&(n->init), args[0], &(args[1]));
      if (n->test)
	find_node_in_siblings (&(n->test), args[0], &(args[1]));
      if (n->increment)
	find_node_in_siblings (&(n->increment), args[0], &(args[1]));
    }
  else if (past_node_is_a (node, past_if))
    {
      PAST_DECLARE_TYPED(if, n, node);
      if (n->condition)
	find_node_in_siblings (&(n->condition), args[0], &(args[1]));
      if (n->then_clause)
	find_node_in_siblings (&(n->then_clause), args[0], &(args[1]));
      if (n->else_clause)
	find_node_in_siblings (&(n->else_clause), args[0], &(args[1]));
    }
  else if (past_node_is_a (node, past_affineguard))
    {
      PAST_DECLARE_TYPED(affineguard, n, node);
      if (n->condition)
	find_node_in_siblings (&(n->condition), args[0], &(args[1]));
      if (n->then_clause)
	find_node_in_siblings (&(n->then_clause), args[0], &(args[1]));
    }
  else if (past_node_is_a (node, past_block))
    {
      PAST_DECLARE_TYPED(block, n, node);
      if (n->body)
	find_node_in_siblings (&(n->body), args[0], &(args[1]));
    }
  else if (past_node_is_a (node, past_root))
    {
      PAST_DECLARE_TYPED(root, n, node);
      if (n->body)
	find_node_in_siblings (&(n->body), args[0], &(args[1]));
    }
  else if (past_node_is_a (node, past_statement))
    {
      PAST_DECLARE_TYPED(statement, n, node);
      if (n->body)
	find_node_in_siblings (&(n->body), args[0], &(args[1]));
    }
  else if (past_node_is_a (node, past_funcall))
    {
      PAST_DECLARE_TYPED(funcall, n, node);
      if (n->name && ((s_past_node_t*)(n->name)) == args[0])
	args[1] = (s_past_node_t*)&(n->name);
      if (n->args_list)
	find_node_in_siblings (&(n->args_list), args[0], &(args[1]));
    }
  else if (past_node_is_a (node, past_vardecl))
    {
      PAST_DECLARE_TYPED(vardecl, n, node);
      if (n->name && ((s_past_node_t*)(n->name)) == args[0])
	args[1] = (s_past_node_t*)&(n->name);
      if (n->type && ((s_past_node_t*)(n->type)) == args[0])
	args[1] = (s_past_node_t*)&(n->type);
    }
  else if (past_node_is_a (node, past_cloogstmt))
    {
      PAST_DECLARE_TYPED(cloogstmt, n, node);
      if (n->substitutions)
	find_node_in_siblings (&(n->substitutions), args[0], &(args[1]));
    }
  else if (past_node_is_a (node, past_ternary_cond))
    {
      PAST_DECLARE_TYPED(ternary_cond, n, node);
      if (n->cond)
	find_node_in_siblings (&(n->cond), args[0], &(args[1]));
      if (n->true_clause)
	find_node_in_siblings (&(n->true_clause), args[0], &(args[1]));
      if (n->false_clause)
	find_node_in_siblings (&(n->false_clause), args[0], &(args[1]));
    }
}


s_past_node_t**
past_node_get_addr (s_past_node_t* node)
{
  if (node->parent == NULL || past_node_is_a (node, past_root))
    return NULL;
  void* data[2];
  data[0] = node;
  data[1] = NULL;
  past_visitor (node->parent, find_parent_ptr, (void*) data, NULL, NULL);
  return (s_past_node_t**) data[1];
}


void past_replace_node (s_past_node_t* old,
			s_past_node_t* new)
{
  if (old->parent == NULL || old == new || past_node_is_a (old, past_root))
    return;
  s_past_node_t* parent = old->parent;
  new->parent = parent;
  s_past_node_t** addr = past_node_get_addr (old);
  assert (addr);
  *addr = new;
}


s_past_node_t*
past_for_to_parfor (s_past_node_t* node)
{
  if (past_node_is_a (node, past_parfor))
    return node;
  if (past_node_is_a (node, past_for))
    {
      PAST_DECLARE_TYPED(for, pf, node);
      s_past_parfor_t* newpf = past_parfor_create
	(pf->init, pf->test, pf->iterator, pf->increment, pf->body, pf->usr);
      newpf->type = pf->type;
      past_replace_node (node, (s_past_node_t*) newpf);
      newpf->node.next = node->next;
      XFREE(node);
      return (s_past_node_t*)newpf;
    }

  return node;
}
