/*
 * pprint.c: This file is part of the PAST project.
 *
 * PAST: the PoCC Abstract Syntax Tree
 *
 * Copyright (C) 2011 Louis-Noel Pouchet
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 3
 * of the License, or (at your option) any later version.
 *
 * The complete GNU Lesser General Public Licence Notice can be found
 * as the `COPYING.LESSER' file in the root directory.
 *
 * Author:
 * Louis-Noel Pouchet <pouchet@cse.ohio-state.edu>
 *
 */
#if HAVE_CONFIG_H
# include <past/config.h>
#endif

#include <assert.h>
#include <past/common.h>
#include <past/past.h>
#include <past/symbols.h>
#include <past/pprint.h>
#include <past/past_api.h>

#define PAST_INDENT_STEP 2

static
void
past_pprint_binary (FILE* out, int indent,
		    s_past_node_t* s, char* str,
		    s_symbol_table_t* symboltable,
		    past_metainfo_fun_t mprint,
		    past_pprint_extensions_fun_t past_pprint_ext)
{
  s_past_binary_t* pb = (s_past_binary_t*) s;
  int need_parenth = 0;
  if (! past_node_is_a (s, past_assign) &&
      ! past_node_is_a (s, past_leq))
    need_parenth = 1;
  if (need_parenth)
    fprintf (out, "(");
  past_pprint_node (out, indent, pb->lhs, symboltable, mprint, past_pprint_ext);
  fprintf (out, " ");
  fprintf (out, "%s", str);
  fprintf (out, " ");
  past_pprint_node (out, indent, pb->rhs, symboltable, mprint, past_pprint_ext);
  if (need_parenth)
    fprintf (out, ")");
}

static
void
past_pprint_binaryfunc (FILE* out, int indent,
			s_past_node_t* s, char* str,
			s_symbol_table_t* symboltable,
			past_metainfo_fun_t mprint,
			past_pprint_extensions_fun_t past_pprint_ext)
{
  s_past_binary_t* pb = (s_past_binary_t*) s;
  fprintf (out, "%s", str);
  fprintf (out, "(");
  past_pprint_node (out, indent, pb->lhs, symboltable, mprint, past_pprint_ext);
  fprintf (out, ", ");
  past_pprint_node (out, indent, pb->rhs, symboltable, mprint, past_pprint_ext);
  fprintf (out, ")");
}

static
void
past_pprint_arrayref (FILE* out, int indent,
		      s_past_node_t* s,
		      s_symbol_table_t* symboltable,
		      past_metainfo_fun_t mprint,
		      past_pprint_extensions_fun_t past_pprint_ext)
{
  PAST_DECLARE_TYPED(binary, pa, s);
  past_pprint_node (out, indent, pa->lhs, symboltable, mprint, past_pprint_ext);
  fprintf (out, "[");
  past_pprint_node (out, indent, pa->rhs, symboltable, mprint, past_pprint_ext);
  fprintf (out, "]");
}


static
void
past_pprint_unaryfunc (FILE* out, int indent,
		       s_past_node_t* s, char* str,
		       s_symbol_table_t* symboltable,
		       past_metainfo_fun_t mprint,
		       past_pprint_extensions_fun_t past_pprint_ext)
{
  s_past_unary_t* pu = (s_past_unary_t*) s;
  fprintf (out, "%s", str);
  fprintf (out, "(");
  past_pprint_node (out, indent, pu->expr, symboltable, mprint,
		    past_pprint_ext);
  fprintf (out, ")");
}


static
void
past_pprint_variable (FILE* out, int indent,
		      s_past_node_t* s,
		      s_symbol_table_t* symboltable,
		      past_metainfo_fun_t mprint,
		      past_pprint_extensions_fun_t past_pprint_ext)
{
  s_past_variable_t* pv = (s_past_variable_t*) s;
  if (pv->symbol->is_char_data)
    fprintf (out, "%s", (char*) pv->symbol->data);
  else
    fprintf (out, "%p", pv->symbol->data);
}

static
void
past_pprint_vardecl (FILE* out, int indent,
		     s_past_node_t* s,
		     s_symbol_table_t* symboltable,
		     past_metainfo_fun_t mprint,
		     past_pprint_extensions_fun_t past_pprint_ext)
{
  PAST_DECLARE_TYPED(vardecl, pf, s);
  past_pprint_variable (out, indent, (s_past_node_t*)pf->type,
			symboltable, mprint, past_pprint_ext);
  fprintf (out, " ");
  past_pprint_variable (out, indent, (s_past_node_t*)pf->name,
			symboltable, mprint, past_pprint_ext);
}

static
void
past_pprint_ternary_cond (FILE* out, int indent,
			  s_past_node_t* s,
			  s_symbol_table_t* symboltable,
			  past_metainfo_fun_t mprint,
			  past_pprint_extensions_fun_t past_pprint_ext)
{
  PAST_DECLARE_TYPED(ternary_cond, pf, s);
  fprintf (out, "(");
  past_pprint_node (out, indent, (s_past_node_t*)pf->cond,
		    symboltable, mprint, past_pprint_ext);
  fprintf (out, " ? ");
  past_pprint_node (out, indent, (s_past_node_t*)pf->true_clause,
		    symboltable, mprint, past_pprint_ext);
  fprintf (out, " : ");
  past_pprint_node (out, indent, (s_past_node_t*)pf->false_clause,
		    symboltable, mprint, past_pprint_ext);
  fprintf (out, ")");
}


static
void
past_pprint_funcall (FILE* out, int indent,
		     s_past_node_t* s, char* str,
		     s_symbol_table_t* symboltable,
		     past_metainfo_fun_t mprint,
		     past_pprint_extensions_fun_t past_pprint_ext)
{
  PAST_DECLARE_TYPED(funcall, pf, s);
  past_pprint_variable (out, indent, (s_past_node_t*)pf->name,
			symboltable, mprint, past_pprint_ext);
  fprintf (out, "(");
  if (pf->args_list)
    {
      s_past_node_t* next;
      s_past_node_t* cur = pf->args_list;
      for (next = cur->next; cur; cur = next, next = cur->next)
	{
	  cur->next = NULL;
	  past_pprint_node (out, indent, cur, symboltable, mprint,
			    past_pprint_ext);
	  cur->next = next;
	  if (next)
	    fprintf (out, ", ");
	}
    }
  fprintf (out, ")");
}


static
void
past_pprint_for (FILE* out, int indent,
		 s_past_node_t* s,
		 s_symbol_table_t* symboltable,
		 past_metainfo_fun_t mprint,
		 char* newlb,
		 char* newub,
		 past_pprint_extensions_fun_t past_pprint_ext)
{
  s_past_for_t* pf = (s_past_for_t*) s;
  fprintf (out, "%*s", indent, "");
  if (mprint)
    {
      mprint (s, out);
      if (s->metainfo)
	fprintf (out, "%*s", indent, "");
    }
  fprintf (out, "for (");
  if (! newlb)
    past_pprint_node (out, indent, pf->init, symboltable, mprint,
		      past_pprint_ext);
  else
    {
      past_pprint_node (out, indent, (s_past_node_t*)pf->iterator, symboltable,
			mprint, past_pprint_ext);
      fprintf (out, " = ");
      fprintf (out, "%s", newlb);
    }
  fprintf (out, "; ");
  if (! newub)
    past_pprint_node (out, indent, pf->test, symboltable, mprint,
		      past_pprint_ext);
  else
    {
      if (past_node_is_a (pf->test, past_binary))
	{
	  PAST_DECLARE_TYPED(binary, pb, pf->test);
	  past_pprint_node (out, indent, pb->lhs, symboltable, mprint,
			    past_pprint_ext);
	  if (past_node_is_a (pf->test, past_leq))
	    fprintf (out, " <= ");
	  else if (past_node_is_a (pf->test, past_lt))
	    fprintf (out, " < ");
	  else if (past_node_is_a (pf->test, past_equal))
	    fprintf (out, " == ");
	  else
	    assert(0);
	  fprintf (out, "%s", newub);
	}
    }
  fprintf (out, "; ");
  past_pprint_node (out, indent, pf->increment, symboltable, mprint,
		    past_pprint_ext);
  fprintf (out, ") {\n");
  past_pprint_node (out, indent + PAST_INDENT_STEP, pf->body, symboltable,
		    mprint, past_pprint_ext);
  fprintf (out, "%*s", indent, "");
  fprintf (out, "}\n");
}

static
void
traverse_private_iter (s_past_node_t* node, void* args)
{
  if (past_node_is_a (node, past_for))
    {
      PAST_DECLARE_TYPED(for, pf, node);
      int i;
      s_past_variable_t** privatevars = args;
      for (i = 0; privatevars[i] &&
	     !symbol_equal (privatevars[i]->symbol, pf->iterator->symbol); ++i)
	;
      if (! privatevars[i])
	privatevars[i] = pf->iterator;
    }
}
static
void find_lb_decls (s_past_node_t* root, s_past_node_t* cur,
		    s_past_variable_t** privatevars)
{
  s_past_node_t* tmp = cur->parent;
  s_past_node_t* tmp2 = cur;
  s_past_node_t* last = cur;
  while (tmp)
    {
      if (past_node_is_a (tmp, past_for) ||
	  past_node_is_a (tmp, past_if) ||
	  past_node_is_a (tmp, past_affineguard) ||
	  past_node_is_a (tmp, past_block))
	{
	  if (past_node_is_a (tmp, past_for))
	    tmp2 = ((s_past_for_t*)tmp)->body;
	  else if (past_node_is_a (tmp, past_affineguard))
	    tmp2 = ((s_past_affineguard_t*)tmp)->then_clause;
	  else if (past_node_is_a (tmp, past_block))
	    tmp2 = ((s_past_block_t*)tmp)->body;
	  else if (past_node_is_a (tmp, past_if))
	    tmp2 = ((s_past_if_t*)tmp)->then_clause;
	  for (; tmp2 && tmp2 != last; tmp2 = tmp2->next)
	    {
	      if (past_node_is_a (tmp2, past_statement))
		{
		  PAST_DECLARE_TYPED(statement, ps, tmp2);
		  if (past_node_is_a (ps->body, past_assign))
		    {
		      PAST_DECLARE_TYPED(binary, pb, ps->body);
		      if (past_node_is_a (pb->lhs, past_vardecl))
			{
			  PAST_DECLARE_TYPED(vardecl, pf, pb->lhs);
			  int i;
			  for (i = 0; privatevars[i] &&
				 !symbol_equal (privatevars[i]->symbol,
						pf->name->symbol); ++i)
			    ;
			  if (! privatevars[i])
			    privatevars[i] = pf->name;
			}
		    }
		}
	    }
	  if (past_node_is_a (tmp, past_if))
	    tmp2 = ((s_past_if_t*)tmp)->else_clause;
	  for (; tmp2 && tmp2 != last; tmp2 = tmp2->next)
	    {
	      if (past_node_is_a (tmp2, past_statement))
		{
		  PAST_DECLARE_TYPED(statement, ps, tmp2);
		  if (past_node_is_a (ps->body, past_assign))
		    {
		      PAST_DECLARE_TYPED(binary, pb, ps->body);
		      if (past_node_is_a (pb->lhs, past_vardecl))
			{
			  PAST_DECLARE_TYPED(vardecl, pf, pb->lhs);
			  int i;
			  for (i = 0; privatevars[i] &&
				 !symbol_equal (privatevars[i]->symbol,
						pf->name->symbol); ++i)
			    ;
			  if (! privatevars[i])
			    privatevars[i] = pf->name;
			}
		    }
		}
	    }
	  last = tmp;
	  tmp = tmp->parent;
	}
      else
	break;
    }
}
static
void
past_pprint_parfor (FILE* out, int indent,
		    s_past_node_t* s,
		    s_symbol_table_t* symboltable,
		    past_metainfo_fun_t mprint,
		    past_pprint_extensions_fun_t past_pprint_ext)
{
  s_past_parfor_t* pf = (s_past_parfor_t*) s;
  char* omp_init = NULL;
  char* omp_test = NULL;

  // Collect loop iterators and loop bounds surrouned by the loop.
  s_past_node_t* tmp;
  int nb_maybe_priv = 0;
  for (tmp = pf->body; tmp; tmp = tmp->next)
    nb_maybe_priv += past_count_for_loops (tmp);
  s_past_variable_t* privatevars[nb_maybe_priv + 1];
  int i;
  for (i = 0; i < nb_maybe_priv + 1; ++i)
    privatevars[i] = NULL;

  // Find the root node, to collect other loop bounds.
  s_past_node_t* r;
  for (r = s; r->parent; r = r->parent)
    ;
  for (tmp = r; tmp; tmp = tmp->next)
    nb_maybe_priv += past_count_nodetype (tmp, past_vardecl);
  s_past_variable_t* firstprivate[nb_maybe_priv + 1];
  for (i = 0; i < nb_maybe_priv + 1; ++i)
    firstprivate[i] = NULL;
  past_visitor (pf->body, traverse_private_iter, privatevars, NULL, NULL);
  find_lb_decls (r, pf->body, firstprivate);
  s_past_node_t* parent;
  for (parent = s->parent; parent && ! past_node_is_a (parent, past_parfor);
       parent = parent->parent)
    ;
  char* extrafirstprivate[3] = { NULL, NULL, NULL };

  // If the loop is an inner-loop, then print a vectorization pragma.
  if (privatevars[0] == NULL)
    {
      fprintf (out, "%*s#pragma ivdep\n", indent, "");
      fprintf (out, "%*s#pragma vector always\n", indent, "");
      fprintf (out, "%*s#pragma simd\n", indent, "");
    }
  // Otherwise, print an OpenMP pragma iff the loop is not surrounded
  // by another parfor AND is not a point-loop.
  else if (! parent && pf->type != e_past_point_loop)
    {
      // Do not omp-ify point loops.
      if (pf->type == e_past_point_loop)
	{
	  past_pprint_for (out, indent, s, symboltable, mprint, omp_init,
			   omp_test, past_pprint_ext);
	  return;
	}

      // Do not omp-ify loops surrounded by more than one loop.
      s_past_node_t* parent;
      int num_enclosing_for = 0;
      int num_enclosing_parfor = 0;
      for (parent = s->parent; parent; parent = parent->parent)
	{
	  if (past_node_is_a (parent, past_for))
	    ++num_enclosing_for;
	  if (past_node_is_a (parent, past_parfor))
	    ++num_enclosing_parfor;
	}
      if (num_enclosing_for > 1)
	{
	  past_pprint_for (out, indent, s, symboltable, mprint, omp_init,
			   omp_test, past_pprint_ext);
	  return;
	}

      // Do not nest omp-ified loops.
      if (num_enclosing_parfor > 0)
	{
	  past_pprint_for (out, indent, s, symboltable, mprint, omp_init,
			   omp_test, past_pprint_ext);
	  return;
	}

      // Omp-ify the loop.
      if (pf->init)
	{
	  if (past_node_is_a (pf->init, past_binary))
	    {
	      PAST_DECLARE_TYPED(binary, pb, pf->init);
	      if (! past_node_is_a (pb->rhs, past_variable) &&
		  ! past_node_is_a (pb->rhs, past_value))
		{
		  fprintf (out, "%*s", indent, "");
		  fprintf (out, "lb1 = ");
		  past_pprint_node (out, indent, pb->rhs, symboltable, mprint,
				    past_pprint_ext);
		  fprintf (out, ";\n");
		  omp_init = "lb1";
		  for (i = 0; extrafirstprivate[i]; ++i)
		    ;
		  extrafirstprivate[i] = "lb1";
		}
	    }
	}
      if (pf->test)
	{
	  if (past_node_is_a (pf->test, past_binary))
	    {
	      PAST_DECLARE_TYPED(binary, pb, pf->test);
	      if (past_node_is_a (pf->test, past_leq) ||
		  past_node_is_a (pf->test, past_lt))
		{
		  if (! past_node_is_a (pb->rhs, past_variable) &&
		      ! past_node_is_a (pb->rhs, past_value))
		    {
		      fprintf (out, "%*s", indent, "");
		      fprintf (out, "ub1 = ");
		      past_pprint_node (out, indent, pb->rhs, symboltable,
					mprint, past_pprint_ext);
		      fprintf (out, ";\n");
		      omp_test = "ub1";
		      for (i = 0; extrafirstprivate[i]; ++i)
			;
		      extrafirstprivate[i] = "ub1";
		    }
		}
	    }
	}
      fprintf (out, "%*s#pragma omp parallel for ", indent, "");
      // Put them in the private clause, if any.
      if (privatevars[0])
	{
	  fprintf (out, "private(");
	  for (i = 0; privatevars[i]; ++i)
	    {
	      past_pprint_variable (out, indent, (s_past_node_t*)privatevars[i],
				    symboltable, mprint, past_pprint_ext);
	      if (privatevars[i + 1])
		fprintf (out, ", ");
	    }
	  fprintf (out, ")");
	}
      if (firstprivate[0] || extrafirstprivate[0])
	{
	  fprintf (out, " firstprivate(");
	  for (i = 0; firstprivate[i]; ++i)
	    {
	      past_pprint_variable (out, indent,
				    (s_past_node_t*)firstprivate[i],
				    symboltable, mprint, past_pprint_ext);
	      if (firstprivate[i + 1])
		fprintf (out, ", ");
	    }
	  if (extrafirstprivate[0])
	    {
	      if (firstprivate[0])
		fprintf (out, ", ");
	      if (extrafirstprivate[1])
		fprintf (out, "%s, %s", extrafirstprivate[0],
			 extrafirstprivate[1]);
	      else
		fprintf (out, "%s", extrafirstprivate[0]);
	    }
	  fprintf (out, ")");
	}
      fprintf (out, "\n");
    }
  past_pprint_for (out, indent, s, symboltable, mprint, omp_init, omp_test,
		   past_pprint_ext);
}

static
void
past_pprint_inc_after (FILE* out, int indent,
		       s_past_node_t* s,
		       s_symbol_table_t* symboltable,
		       past_metainfo_fun_t mprint,
		       past_pprint_extensions_fun_t past_pprint_ext)
{
  s_past_unary_t* pu = (s_past_unary_t*) s;
  if (! past_node_is_a (pu->expr, past_variable))
    fprintf (out, "(");
  past_pprint_node (out, indent, pu->expr, symboltable, mprint,
		    past_pprint_ext);
  if (! past_node_is_a (pu->expr, past_variable))
    fprintf (out, ")");
  fprintf (out, "++");
}

static
void
past_pprint_dec_after (FILE* out, int indent,
		       s_past_node_t* s,
		       s_symbol_table_t* symboltable,
		       past_metainfo_fun_t mprint,
		       past_pprint_extensions_fun_t past_pprint_ext)
{
  s_past_unary_t* pu = (s_past_unary_t*) s;
  if (past_node_is_a (pu->expr, past_variable))
    fprintf (out, "(");
  past_pprint_node (out, indent, pu->expr, symboltable, mprint,
		    past_pprint_ext);
  if (past_node_is_a (pu->expr, past_variable))
    fprintf (out, ")");
  fprintf (out, "--");
}

static
void
past_pprint_value (FILE* out, int indent,
		   s_past_node_t* s,
		   s_symbol_table_t* symboltable,
		   past_metainfo_fun_t mprint,
		   past_pprint_extensions_fun_t past_pprint_ext)
{
  s_past_value_t* pt = (s_past_value_t*) s;
  switch (pt->type)
    {
    case e_past_value_int:
      fprintf (out, "%d", pt->value.intval);
      break;
    case e_past_value_float:
      fprintf (out, "%f", pt->value.floatval);
      break;
    default:
      fprintf (stderr, "[PAST] pprint: other data types not yet implemented\n");
      assert (0);
    }
}

static
void
past_pprint_affineguard (FILE* out, int indent,
			 s_past_node_t* s,
			 s_symbol_table_t* symboltable,
			 past_metainfo_fun_t mprint,
			 past_pprint_extensions_fun_t past_pprint_ext)
{
  s_past_affineguard_t* pa = (s_past_affineguard_t*) s;
  fprintf (out, "%*s", indent, "");
  if (mprint)
    {
      mprint (s, out);
      fprintf (out, "%*s", indent, "");
    }
  fprintf (out, "if (");
  past_pprint_node (out, indent, pa->condition, symboltable, mprint,
		    past_pprint_ext);
  fprintf (out, ") {\n");
  past_pprint_node (out, indent + PAST_INDENT_STEP,
		    pa->then_clause, symboltable, mprint, past_pprint_ext);
  fprintf (out, "%*s", indent, "");
  fprintf (out, "}\n");
}

static
void
past_pprint_if (FILE* out, int indent,
		s_past_node_t* s,
		s_symbol_table_t* symboltable,
		past_metainfo_fun_t mprint,
		past_pprint_extensions_fun_t past_pprint_ext)
{
  s_past_if_t* pa = (s_past_if_t*) s;
  fprintf (out, "%*s", indent, "");
  if (mprint)
    {
      mprint (s, out);
      fprintf (out, "%*s", indent, "");
    }
  fprintf (out, "if (");
  past_pprint_node (out, indent, pa->condition, symboltable, mprint,
		    past_pprint_ext);
  fprintf (out, ") {\n");
  past_pprint_node (out, indent + PAST_INDENT_STEP,
		    pa->then_clause, symboltable, mprint, past_pprint_ext);
  fprintf (out, "%*s", indent, "");
  fprintf (out, "}\n");
  if (pa->else_clause)
    {
      fprintf (out, "%*s", indent, "");
      fprintf (out, "else {\n");
      past_pprint_node (out, indent + PAST_INDENT_STEP,
			pa->else_clause, symboltable, mprint, past_pprint_ext);
      fprintf (out, "%*s", indent, "");
      fprintf (out, "}\n");
    }
}


static
void
past_pprint_cloogstmt (FILE* out, int indent,
		       s_past_node_t* s,
		       s_symbol_table_t* symboltable,
		       past_metainfo_fun_t mprint,
		       past_pprint_extensions_fun_t past_pprint_ext)
{
  s_past_cloogstmt_t* pa = (s_past_cloogstmt_t*) s;
  fprintf (out, "%*s", indent, "");
  fprintf (out, "%s(", (char*) pa->stmt_name->symbol->data);
  s_past_node_t* subs;
  for (subs = pa->substitutions; subs; subs = subs->next)
    {
      s_past_node_t* tmp = subs->next;
      subs->next = NULL;
      past_pprint_node (out, indent, subs, symboltable, mprint,
			past_pprint_ext);
      subs->next = tmp;
      if (subs->next)
	fprintf (out, ", ");
    }
  fprintf (out, ");");
  if (mprint)
    mprint (s, out);
  fprintf (out, "\n");
}

static
void
past_pprint_statement (FILE* out, int indent,
		       s_past_node_t* s,
		       s_symbol_table_t* symboltable,
		       past_metainfo_fun_t mprint,
		       past_pprint_extensions_fun_t past_pprint_ext)
{
  s_past_statement_t* ps = (s_past_statement_t*) s;
  fprintf (out, "%*s", indent, "");
  past_pprint_node (out, indent, ps->body, symboltable, mprint,
		    past_pprint_ext);
  fprintf (out, "; ");
  if (mprint)
    mprint (s, out);
  fprintf (out, "\n");
}


static
void
past_pprint_block (FILE* out, int indent,
		   s_past_node_t* s,
		   s_symbol_table_t* symboltable,
		   past_metainfo_fun_t mprint,
		   past_pprint_extensions_fun_t past_pprint_ext)
{
  s_past_block_t* pa = (s_past_block_t*) s;
  fprintf (out, "%*s", indent, "");
  if (mprint)
    {
      mprint (s, out);
      fprintf (out, "%*s", indent, "");
    }
  fprintf (out, "{\n");
  past_pprint_node (out, indent + PAST_INDENT_STEP, pa->body, symboltable,
		    mprint, past_pprint_ext);
  fprintf (out, "%*s", indent, "");
  fprintf (out, "}\n");
}



void
past_pprint_node (FILE* out,
		  int indent,
		  s_past_node_t* node,
		  s_symbol_table_t* symboltable,
		  past_metainfo_fun_t mprint,
		  past_pprint_extensions_fun_t past_pprint_ext)
{
  s_past_node_t* s;
  for (s = node; s; s = s->next)
    {
      if (past_node_is_a (s, past_parfor))
	past_pprint_parfor (out, indent, s, symboltable, mprint,
			    past_pprint_ext);
      else if (past_node_is_a (s, past_for))
	past_pprint_for (out, indent, s, symboltable, mprint, NULL, NULL,
			 past_pprint_ext);

#define pprint_single_node(type)				\
      else if (past_node_is_a (s, past_##type))		\
	past_pprint_##type (out, indent, s, symboltable, mprint, past_pprint_ext)

      pprint_single_node(variable);
      pprint_single_node(value);
      pprint_single_node(affineguard);
      pprint_single_node(statement);
      pprint_single_node(cloogstmt);
      pprint_single_node(if);
      pprint_single_node(block);
      pprint_single_node(inc_after);
      pprint_single_node(dec_after);

#define pprint_bin_node(type, str)			\
      else if (past_node_is_a (s, past_##type))	\
	past_pprint_binary (out, indent, s, str, symboltable, mprint, past_pprint_ext)
      pprint_bin_node(add, "+");
      pprint_bin_node(sub, "-");
      pprint_bin_node(mul, "*");
      pprint_bin_node(div, "/");
      pprint_bin_node(mod, "%");
      pprint_bin_node(and, "&&");
      pprint_bin_node(or, "||");
      pprint_bin_node(equal, "==");
      pprint_bin_node(assign, "=");
      pprint_bin_node(geq, ">=");
      pprint_bin_node(leq, "<=");
      pprint_bin_node(gt, ">");
      pprint_bin_node(lt, "<");

#define pprint_binfun_node(type, str)			\
      else if (past_node_is_a (s, past_##type))	\
	past_pprint_binaryfunc (out, indent, s, str, symboltable, mprint, past_pprint_ext)

      pprint_binfun_node(min, "min");
      pprint_binfun_node(max, "max");
      pprint_binfun_node(ceild, "ceild");
      pprint_binfun_node(floord, "floord");

#define pprint_unfun_node(type, str)			\
      else if (past_node_is_a (s, past_##type))	\
	past_pprint_unaryfunc (out, indent, s, str, symboltable, mprint, past_pprint_ext)

      pprint_unfun_node(round, "round");
      pprint_unfun_node(floor, "floor");
      pprint_unfun_node(ceil, "ceil");
      pprint_unfun_node(sqrt, "sqrt");
      pprint_unfun_node(inc_before, "++");
      pprint_unfun_node(dec_before, "--");

      else if (past_node_is_a (s, past_arrayref))
	past_pprint_arrayref (out, indent, s, symboltable, mprint,
			      past_pprint_ext);
      else if (past_node_is_a (s, past_vardecl))
	past_pprint_vardecl (out, indent, s, symboltable, mprint,
			     past_pprint_ext);
      else if (past_node_is_a (s, past_ternary_cond))
	past_pprint_ternary_cond (out, indent, s, symboltable, mprint,
				  past_pprint_ext);
      else if (past_pprint_ext &&
	       past_pprint_ext (out, indent, s, symboltable, mprint))
	;
      else
	fprintf (stderr, "[PAST] Pretty-print: Unsupported node type\n");
    }
}


/**
 * Default C pretty-printer.
 *
 *
 */
void
past_pprint (FILE* out, s_past_node_t* root)
{
  if (past_node_is_a (root, past_root))
    {
      s_past_root_t* pr = (s_past_root_t*) root;
      past_pprint_node (out, 0, pr->body, pr->symboltable, NULL, NULL);
    }
  else
    past_pprint_node (out, 0, root, NULL, NULL, NULL);
}


/**
 * C pretty-printer using 'mprint' to print meta-info, if any.
 * prototype: void mprint (s_past_node_t* node, FILE* out).
 *
 */
void
past_pprint_metainfo (FILE* out, s_past_node_t* root,
		      past_metainfo_fun_t mprint)
{
  if (past_node_is_a (root, past_root))
    {
      s_past_root_t* pr = (s_past_root_t*) root;
      past_pprint_node (out, 0, pr->body, pr->symboltable, mprint, NULL);
    }
  else
    past_pprint_node (out, 0, root, NULL, mprint, NULL);
}



/**
 * C pretty-printer using 'mprint' to print meta-info, if any.
 * prototype: void mprint (s_past_node_t* node, FILE* out).
 * Uses 'past_pprint_ext' to print user-defined nodes, if any.
 *
 */
void
past_pprint_extended_metainfo (FILE* out, s_past_node_t* root,
			       past_metainfo_fun_t mprint,
			       past_pprint_extensions_fun_t past_pprint_ext)
{
  if (past_node_is_a (root, past_root))
    {
      s_past_root_t* pr = (s_past_root_t*) root;
      past_pprint_node (out, 0, pr->body, pr->symboltable, mprint,
			past_pprint_ext);
    }
  else
    past_pprint_node (out, 0, root, NULL, mprint, past_pprint_ext);
}
