/*
 * symbols.c: This file is part of the PAST project.
 *
 * PAST: the PoCC Abstract Syntax Tree
 *
 * Copyright (C) 2011 Louis-Noel Pouchet
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 3
 * of the License, or (at your option) any later version.
 *
 * The complete GNU Lesser General Public Licence Notice can be found
 * as the `COPYING.LESSER' file in the root directory.
 *
 * Author:
 * Louis-Noel Pouchet <pouchet@cse.ohio-state.edu>
 *
 */
#if HAVE_CONFIG_H
# include <past/config.h>
#endif

#include <past/common.h>
#include <past/symbols.h>


/**
 *
 *
 */
s_symbol_t*
symbol_malloc ()
{
  s_symbol_t* ret = XMALLOC(s_symbol_t, 1);
  ret->data = NULL;
  ret->num_refs = 1;
  ret->is_char_data = 0;
  ret->type = past_unknown;
  ret->is_attached_to_table = 0;
  ret->next = NULL;
  ret->prev = NULL;

  return ret;
}

/**
 *
 *
 */
s_symbol_table_t*
symbol_table_malloc ()
{
  s_symbol_table_t* ret = XMALLOC(s_symbol_table_t, 1);
  ret->symbols = NULL;

  return ret;
}

/**
 *
 *
 */
void
symbol_free (s_symbol_t* symbol)
{
  if (symbol)
    {
      if (symbol->is_char_data && symbol->data)
	XFREE(symbol->data);
      XFREE(symbol);
    }
}

/**
 *
 *
 */
void
symbol_table_free (s_symbol_table_t* table)
{
  if (table)
    {
      s_symbol_t* s;
      s_symbol_t* next;
      for (s = table->symbols; s; )
	{
	  next = s->next;
	  symbol_free (s);
	  s = next;
	}
      XFREE(table);
    }
}

/**
 *
 *
 */
s_symbol_t*
symbol_create (e_symbol_type_t type, int is_char_data, void* data)
{
  s_symbol_t* ret = symbol_malloc ();
  ret->type = type;
  ret->is_char_data = is_char_data;
  if (is_char_data)
    ret->data = strdup (data);
  else
    ret->data = data;

  return ret;
}

/**
 *
 *
 */
s_symbol_t*
symbol_add (s_symbol_table_t* table, s_symbol_t* symbol)
{
  // If there's no symbol table, simply return the symbol.
  if (table == NULL)
    {
      symbol->is_attached_to_table = 0;
      return symbol;
    }

  s_symbol_t* s;
  symbol->is_attached_to_table = 1;
  if ((s = symbol_find (table, symbol)))
    {
      (s->num_refs)++;
      return s;
    }
  else
    {
      symbol->next = table->symbols;
      if (table->symbols)
	table->symbols->prev = symbol;
      table->symbols = symbol;
      return symbol;
    }
}

/**
 *
 *
 */
s_symbol_t*
symbol_add_from_data (s_symbol_table_t* table, void* data)
{
  s_symbol_t* s = symbol_malloc ();
  s->is_char_data = 0;
  s->data = data;
  s_symbol_t* ret = symbol_add (table, s);
  if (ret != s)
    XFREE(s);
  return ret;
}

/**
 *
 *
 */
s_symbol_t*
symbol_add_from_char (s_symbol_table_t* table, const char* data)
{
  s_symbol_t* s = symbol_malloc ();
  s->is_char_data = 1;
  s->data = strdup (data);
  s_symbol_t* ret = symbol_add (table, s);
  if (ret != s)
    symbol_free (s);
  return ret;
}

/**
 *
 *
 */
void
symbol_remove (s_symbol_table_t* table, s_symbol_t* symbol)
{
  s_symbol_t* s;
  if ((s = symbol_find (table, symbol)))
    {
      if (s->num_refs > 1)
	(s->num_refs)--;
      else
	{
	  if (s->prev)
	    {
	      s->prev->next = s->next;
	      if (s->next)
		s->next->prev = s->prev;
	    }
	  else
	    {
	      table->symbols = s->next;
	      if (s->next)
		s->next->prev = NULL;
	    }
	  symbol_free (s);
	}
    }
}

/**
 *
 *
 */
s_symbol_t*
symbol_find (s_symbol_table_t* table, s_symbol_t* symbol)
{
  s_symbol_t* s;
  for (s = table->symbols; s; s = s->next)
    {
      if (symbol_equal (s, symbol))
	return s;
    }
  return NULL;
}

/**
 *
 *
 */
int
symbol_equal (s_symbol_t* s1, s_symbol_t* s2)
{
  if (s1 == s2)
    return 1;
  if (s1 == NULL || s2 == NULL)
    return 0;
  if (s1->is_char_data && s2->is_char_data)
    return ! strcmp (s1->data, s2->data);
  else
    return s1->data == s2->data;
}

/**
 *
 *
 */
int
symbol_type_equal (s_symbol_t* s1, s_symbol_t* s2)
{
  if (s1 == s2)
    return 1;
  if (s1->is_char_data && s2->is_char_data)
    return s1->type == s2->type && ! strcmp (s1->data, s2->data);
  else
    return s1->type == s2->type && s1->data == s2->data;
}

/**
 *
 *
 */
s_symbol_t*
symbol_find_from_data (s_symbol_table_t* table, void* data)
{
  s_symbol_t* s = symbol_malloc ();
  s->is_char_data = 0;
  s->data = data;
  s_symbol_t* ret = symbol_find (table, s);
  symbol_free (s);

  return ret;
}

/**
 *
 *
 */
s_symbol_t*
symbol_find_from_char (s_symbol_table_t* table, char* data)
{
  s_symbol_t* s = symbol_malloc ();
  s->is_char_data = 1;
  s->data = strdup (data);
  s_symbol_t* ret = symbol_find (table, s);
  symbol_free (s);

  return ret;
}
