/*
 * depgraph.c: this file is part of the PolyFeat project.
 *
 * PolyFeat, a Polyhedral Feature extractor
 *
 * Copyright (C) 2011 Louis-Noel Pouchet
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 3
 * of the License, or (at your option) any later version.
 *
 * The complete GNU Lesser General Public Licence Notice can be found
 * as the `COPYING.LESSER' file in the root directory.
 *
 * Author:
 * Louis-Noel Pouchet <pouchet@cse.ohio-state.edu>
 */
#if HAVE_CONFIG_H
# include <polyfeat/config.h>
#endif

#include <polyfeat/depgraph.h>

#ifndef CANDL_SUPPORTS_SCOPLIB
# define CANDL_SUPPORTS_SCOPLIB
#endif
#include <candl/program.h>
#include <candl/dependence.h>
#include <candl/options.h>
#include <candl/ddv.h>
#include <polylib/polylib64.h>
#include <fm/system.h>
#include <fm/compsol.h>

#define min(a, b) (a < b ? a : b)



s_polyfeat_dep_t*
polyfeat_depgraph_compute_dep_feature(CandlDependence* dep,
				      CandlProgram* cprogram)
{
  s_polyfeat_dep_t* ret = polyfeat_dep_alloc ();

  int nb_param = dep->source->domain->NbColumns - dep->source->depth - 2;
  // Depth.
  ret->depth = dep->depth;

  // 'Dimensionality' of the dependence polyhedron.
  int i, j;
  s_fm_system_t* s = fm_system_alloc (dep->domain->NbRows,
				      dep->domain->NbColumns);
  for (i = 0; i < dep->domain->NbRows; ++i)
    for (j = 0; j < dep->domain->NbColumns; ++j)
      s->lines[i]->vector[j].num = dep->domain->p[i][j];
  s_fm_compsol_t* cs = fm_compsol_init_sys (s);
  int iterdim = cs->poly ? cs->poly->size : 0;
  ret->dim = iterdim - nb_param -
    min(dep->source->depth, dep->target->depth) + 1;
  if (ret->dim < 0)
    ret->dim = 0;
  fm_compsol_free (cs);
  fm_system_free (s);

  // Self.
  ret->self = dep->source == dep->target;

  // Type.
  ret->type = dep->type;

  // Kind.
  CandlDependence* next = dep->next;
  dep->next = NULL;
  for (i = 0; i < min(dep->source->depth, dep->target->depth) &&
	 dep->source->index[i] == dep->target->index[i]; ++i)
    ;
  if (i == min(dep->source->depth, dep->target->depth) && i > 0)
    --i;
  CandlDDV* ddv = candl_ddv_extract_in_loop(cprogram, dep,
					    dep->source->index[i]);
  if (ddv)
    {
      for (i = 0; i < ddv->length; ++i)
	if (ddv->data[i].type != candl_dv_scalar &&
	    ddv->data[i].type != candl_dv_eq)
	  break;
      if (i == ddv->length)
	ret->kind = 1;
      else
	ret->kind = 2;
    }
  else
    // Loop-independent dependence.
    ret->kind = 1;
  /// FIXME: deal with parametric dependence.

  dep->next = next;

  return ret;
}


void
polyfeat_depgraph_extract(FILE* outfile, scoplib_scop_p scop, int use_rar)
{
  //printf ("[PolyFeat] Extract Polyhedral Dependence Graph features...\n");
  CandlOptions* coptions = candl_options_malloc ();
  if (use_rar)
    coptions->rar = 1;
  CandlProgram* cprogram = candl_program_convert_scop (scop, NULL);
  CandlDependence* deps = candl_dependence (cprogram, coptions);

  // Features.
  CandlDependence* tmp;
  int i, j, k, count;
  
  for (tmp = deps, count = 0; tmp; tmp = tmp->next, ++count)
    ;
  
  // Create the adjacency matrix.
  CandlDependence*
    adjmat[cprogram->nb_statements][cprogram->nb_statements][count+1];
  for (i = 0; i < cprogram->nb_statements; ++i)
    for (j = 0; j < cprogram->nb_statements; ++j)
      for (k = 0; k <= count; ++k)
	adjmat[i][j][k] = NULL;
  for (tmp = deps, count = 0; tmp; tmp = tmp->next, ++count)
    {
      for (i = 0; adjmat[tmp->source->label][tmp->target->label][i]; ++i)
	;
      adjmat[tmp->source->label][tmp->target->label][i] = tmp;
    }

  // Number of nodes and edges
  fprintf (outfile, "# Number of nodes and edges\n");
  fprintf (outfile, "%d %d\n", cprogram->nb_statements, count);
  fprintf (outfile, "#\n");

  // Compute and dump the edge features (real valued).
  fprintf (outfile, "# Edge feature vector\n");
  for (tmp = deps, count = 1; tmp; tmp = tmp->next, ++count)
    {
      s_polyfeat_dep_t* f =
	polyfeat_depgraph_compute_dep_feature (tmp, cprogram);
      fprintf (outfile, "%d: ", count);
      polyfeat_dep_print (outfile, f);
      fprintf (outfile, "\n");
      polyfeat_dep_free (f);
    }

  // Dump the adjacency matrix.
  fprintf (outfile, "# Adjacency matrix\n");
  for (i = 0; i < cprogram->nb_statements; ++i)
    {
      for (j = 0; j < cprogram->nb_statements; ++j)
	if (adjmat[i][j][0] != NULL)
	  {
	    fprintf (outfile, "%d,%d: ", i+1, j+1);
	    for (k = 0; adjmat[i][j][k]; ++k)
	      {
		for (tmp = deps, count = 1; tmp && tmp != adjmat[i][j][k];
		     tmp = tmp->next, ++count)
		  ;
		fprintf (outfile, "%d ", count);
	      }
	    fprintf (outfile, "\n");
	  }
    }

  candl_dependence_free (deps);
  candl_program_free (cprogram);
  candl_options_free (coptions);
}
