/*
 * driver-clastops.c: this file is part of the PoCC project.
 *
 * PoCC, the Polyhedral Compiler Collection package
 *
 * Copyright (C) 2009 Louis-Noel Pouchet
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2.1
 * of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * The complete GNU General Public Licence Notice can be found as the
 * `COPYING.LESSER' file in the root directory.
 *
 * Author:
 * Louis-Noel Pouchet <Louis-Noel.Pouchet@inria.fr>
 *
 */
#if HAVE_CONFIG_H
# include <pocc-utils/config.h>
#endif

# include <pocc/driver-clastops.h>
//# define CLOOG_SUPPORTS_SCOPLIB
//# include <cloog/cloog.h>
//# include <cloog/clast.h>
# include <pragmatize/pragmatize.h>
# include <clasttools/pprint.h>
# include <clasttools/clastext.h>


static
void
traverse_print_clast_user_statement_extended_defines (struct clast_stmt* s,
						      FILE* out)
{
  // Traverse the clast.
  for ( ; s; s = s->next)
    {
      if (CLAST_STMT_IS_A(s, stmt_for) ||
	  CLAST_STMT_IS_A(s, stmt_parfor) ||
	  CLAST_STMT_IS_A(s, stmt_vectorfor))
	{
	  struct clast_stmt* body;
	  if (CLAST_STMT_IS_A(s, stmt_for))
	    body = ((struct clast_for*)s)->body;
	  else if  (CLAST_STMT_IS_A(s, stmt_parfor))
	    body = ((struct clast_parfor*)s)->body;
	  else if  (CLAST_STMT_IS_A(s, stmt_vectorfor))
	    body = ((struct clast_vectorfor*)s)->body;
	  traverse_print_clast_user_statement_extended_defines (body, out);
	}
      else if (CLAST_STMT_IS_A(s, stmt_guard))
	traverse_print_clast_user_statement_extended_defines
	  (((struct clast_guard*)s)->then, out);
      else if (CLAST_STMT_IS_A(s, stmt_block))
	traverse_print_clast_user_statement_extended_defines
	  (((struct clast_block*)s)->body, out);
      else if (CLAST_STMT_IS_A(s, stmt_user_extended))
	{
	  struct clast_user_stmt_extended* ue =
	    (struct clast_user_stmt_extended*) s;
	  fprintf (out, "%s\n", ue->define_string);
	}
    }
}



void
pocc_driver_clastops (scoplib_scop_p program,
		      struct clast_stmt* root,
		      s_pocc_options_t* poptions,
		      s_pocc_utils_options_t* puoptions)
{
  CloogOptions* coptions = poptions->cloog_options;

  /* (1) Run the pragmatizer, if required. */
  if (poptions->pragmatizer)
    {
      if (! poptions->quiet)
	printf ("[PoCC] Insert OpenMP and vectorization pragmas\n");
      pragmatize (program, root);
    }

  /* (2) Generate statements macros. */
  FILE* body_file = poptions->output_file;
  int st_count = 1;
  scoplib_statement_p stm;
  int i;
  int nb_scatt = 0;
  for (stm = program->statement; stm; stm = stm->next)
    {
      fprintf (body_file, "#define S%d(", st_count++);
      for (i = 0; i < stm->nb_iterators; ++i)
	{
	  fprintf (body_file, "%s", stm->iterators[i]);
	  if (i < stm->nb_iterators - 1)
	    fprintf (body_file, ",");
	}
      fprintf (body_file, ") %s\n", stm->body);
      nb_scatt = stm->schedule->NbRows > nb_scatt ?
	stm->schedule->NbRows : nb_scatt;
    }
  /* We now can have statement definition overriden by the array
     contraction. Those are stored in clast_user_statement_extended
     nodes only, the #define is in the cuse->define_string, they must
     be collected and pretty-printed here. */
  traverse_print_clast_user_statement_extended_defines (root, body_file);

  /* (3) Generate loop counters. */
  fprintf (body_file,
	   "\t register int lbv, ubv, lb, ub, lb1, ub1, lb2, ub2;\n");
  int done = 0;
  for (i = 0; i < nb_scatt; ++i)
    {
      /// FIXME: Deactivate this, as pluto may generate OpenMP pragmas
      /// using some unused variables. We'll let the compiler remove useless
      /// variables.
	{
	  if (! done++)
	    fprintf (body_file, "\t register int ");
	  else
	    fprintf (body_file, ", ");
	  fprintf(body_file, "c%d, c%dt, newlb_c%d, newub_c%d", i, i, i, i);
	}
    }
  fprintf (body_file, ";\n\n");

  fflush (body_file);
  fprintf (body_file, "#pragma scop\n");

  /* (4) Run the extended CLAST pretty-printer, if needed. */
  if (poptions->pragmatizer)
    clasttols_clast_pprint_debug (body_file, root, 0, coptions);
  else
    // Pretty-print the code with CLooG default pretty-printer.
    clast_pprint (body_file, root, 0, coptions);

  fprintf (body_file, "#pragma endscop\n");

  /// FIXME: This is a BUG: this should be enabled.
/*   /\* (8) Delete the clast. *\/ */
/*   cloog_clast_free (root); */
}
