
/*
 * ExpressionLibrary.cpp: This file is part of the Parametric Tiling project.
 * 
 * Parametric Tiling: A CLAST-to-CLAST parametric tiling software
 * 
 * Copyright (C) 2011 Sanket Tavargeri
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 3
 * of the License, or (at your option) any later version.
 * 
 * The complete GNU Lesser General Public Licence Notice can be found
 * as the `COPYING.LESSER' file in the root directory.
 * 
 * Author:
 * Sanket Tavargeri <sanket.tavargeri@gmail.com>
 * 
 */

#if HAVE_CONFIG_H
# include <ptile/config.h>
#endif


# include <ptile/ExpressionLibrary.hpp>

// This class will have the general utility functions operating on the expressions

  void ExpressionLibrary::AddWInterTileLoopIteratorTermInExpression(vector<Expression*> *exprs, string name)
  {
    //Find a term identical to one of the existing iterator. Make the coefficient 0 and set the new name
    // For each expression
    for (int i = 0; i < exprs->size(); i++)
      {
	// #terms = #terms + 1 because of the w iterator term
	Term *terms = new Term[exprs->at(i)->size + 1];
	int count = 0;

	// InterTileIterators
	bool wTermAdded = false;
	for (int j = 0; j < exprs->at(i)->size; j++)
	  {
	    if (exprs->at(i)->terms[j].type == interTile_loop_iterator && wTermAdded == false)
	      {
		terms[count++].SetIdentityTermFrom(exprs->at(i)->terms[j]);
		terms[count - 1].coefficient = 0;
		terms[count - 1].name = name;
				
		wTermAdded = true;
	      }//if (exprs->at(i)->type == interTile_loop_iterator)

	    terms[count++].CloneFrom(exprs->at(i)->terms[j]);
	  }//for (int j = 0; j < exprs->at(i)->size; j++)

	exprs->at(i)->terms = terms;
	exprs->at(i)->size = exprs->at(i)->size + 1;
      }// for (int i = 0; i < exprs->size(); i++)
  }// void AddWInterTileLoopIteratorTermInExpression(exprs->at(i), "w")

  Term ExpressionLibrary::AddTerms(Term term1, Term term2)
  {
    Term res_term;
    res_term.type = term1.type;
    res_term.name = term1.name;
    // If the terms are equal then add the coefficients
    if (term1.coefficient == 0 && term2.coefficient == 0)
      {
        res_term.coefficient = 0.0;
      }
    else if (term1.coefficient == 0)
      {
	res_term.CloneFrom(term2);
      }// else if term1.coefficient = 0
    else if (term2.coefficient == 0)
      {
        res_term.CloneFrom(term1);
      }// else if term2.coefficient == 0
    else if (AreTermsEqual(term1, term2))
      {
	res_term.coefficient = term1.coefficient + term2.coefficient;
      }// if the terms are equal
    else
      {
        //printf("Adding the following 2 terms\n");
        term1.Print();
        term2.Print();
	// Multiply the numerator of one with the denominator of the other. Do this in 2 ways and then add the terms

	res_term.coefficient = 1;
	res_term.num_polynomials_in_numerator = term1.num_polynomials_in_numerator * term2.num_polynomials_in_denominator
	  + term2.num_polynomials_in_numerator * term1.num_polynomials_in_denominator;

	res_term.num_polynomials_in_denominator = term1.num_polynomials_in_denominator * term2.num_polynomials_in_denominator;
	res_term.AllocateMemory();
	int count = 0;
	// num1 * den2
	for (int i = 0; i < term1.num_polynomials_in_numerator; i++)
	  {
	    // Set coefficients appropriately
	    term1.numerator[i].coefficient *= term1.coefficient;            
	    for (int j = 0; j < term2.num_polynomials_in_denominator; j++)
	      {
		res_term.numerator[count++] = MultiplyPolynomials(term1.numerator[i], term2.denominator[j]);
	      }// for j
	  }// for i

	// num2 * den1
	for (int i = 0; i < term2.num_polynomials_in_numerator; i++)
	  {
	    // Set coefficients appropriately
	    term2.numerator[i].coefficient *= term2.coefficient;            
	    for (int j = 0; j < term1.num_polynomials_in_denominator; j++)
	      {
		res_term.numerator[count++] = MultiplyPolynomials(term2.numerator[i], term1.denominator[j]);
	      }// for j
	  }// for i

	//Reset count
	count = 0;
	// den1 * den2
	for (int i = 0; i < term1.num_polynomials_in_denominator; i++)
	  {
	    for (int j = 0; j < term2.num_polynomials_in_denominator; j++)
	      {
		res_term.denominator[count++] = MultiplyPolynomials(term1.denominator[i], term2.denominator[j]);
	      }// for j
	  }// for i
        
        //printf("Resulting term\n");
        res_term.Print();
      }// if the terms are not equal    

    return res_term;
  }// AddTerms()


  void ExpressionLibrary::SetSignOfRightMostLoopIterator(Expression *expr, int sign, vector<Expression> MinValues, vector<Expression> MaxValues)
  {
    //cout<<"Expression"<<endl;
    expr->Print();
    //cout<<"Calling Find_Index_Of_RightmostLoopIterator()"<<endl;
	
    int r = Find_Index_Of_RightmostLoopIterator(*expr);
    //cout<<"r = "<<r<<endl;
    int existingSign = FindSignOfCoefficient(expr->terms[r]);
    int neg_pols = 0;
    //cout<<"existingSign = "<<existingSign<<endl;
    if (existingSign == sign)
      {
	// Find if all the polynomials also have coefficients positive. If so, we are done.
	for (int i = 0; i < expr->terms[r].num_polynomials_in_numerator; i++)
	  {
	    int poly_sign = FindSignOfCoefficient(expr->terms[r].numerator[i]);
	    if (poly_sign == -1)
	      {
		// Apply relaxation
		// TODO: Temporary fix. Equate the coefficient of the 'offending' polynomial to 0. Need to make it tight by computing lexicographic minimum/maximum for the tile iterator.
		expr->terms[r].numerator[i].coefficient = 0;
		// neg_pols++; 
	      }// polynomial sign is positive
	  }// for int i 
      }// if existingSign == sign
    else if (existingSign == (-1 * sign))
      {
	// Change the sign of the term
	expr->terms[r].coefficient *= -1;
                        
	// Find if all the polynomials also have coefficients positive. If so, we are done.
	for (int i = 0; i < expr->terms[r].num_polynomials_in_numerator; i++)
	  {
	    // Flip the signs of the polynomials as well
	    expr->terms[r].numerator[i].coefficient *= -1;                  
	    int poly_sign = FindSignOfCoefficient(expr->terms[r].numerator[i]);
	    if (poly_sign == -1)
	      {
		//printf("The term in question\n");
		expr->terms[r].Print();
		//printf("Coefficient: %f\n", expr->terms[r].numerator[i].coefficient);
		//printf("Sign is negative!\n");
		// TODO: Temporary fix. Equate the coefficient of the 'offending' polynomial to 0. Need to make it tight by computing lexicographic minimum/maximum for the tile iterator.
		expr->terms[r].numerator[i].coefficient = 0;
		// neg_pols++;
		// Apply relaxation
	      }// polynomial sign is positive
	  }// for int i
      }// else if the existing sign is opposite to what is desired



    if (neg_pols > 0)
      {
        //cout<<"Relaxation required"<<endl;
	// Split the term into 2 terms. One with positive polynomials and one with negative polynomials
	// term1 will be the original one, term2 will be the relaxed part - constant.
	Term term1(expr->terms[r].num_polynomials_in_numerator - neg_pols, expr->terms[r].num_polynomials_in_denominator);
	Term term2(neg_pols, expr->terms[r].num_polynomials_in_denominator);
     
	// term1 will go into the original equation
	term1.type = expr->terms[r].type;
	term1.coefficient = expr->terms[r].coefficient;

	// term2 will be constant
	term2.type = constant;
	term2.coefficient = expr->terms[r].coefficient;

	int count1 = 0, count2 = 0;
	for (int i = 0; i < expr->terms[r].num_polynomials_in_numerator; i++)
	  {
	    int poly_sign = FindSignOfCoefficient(expr->terms[r].numerator[i]);
	    if (poly_sign >= 0)
	      {
                term1.numerator[count1++].CloneFrom(expr->terms[r].numerator[i]);
	      }// if the sign is 0 or positive, add to the original term
	    else
	      {
                term2.numerator[count2++].CloneFrom(expr->terms[r].numerator[i]);
	      }// else 
	  }// for int i

	// Copy denominators as they are
	for (int i = 0; i < expr->terms[r].num_polynomials_in_denominator; i++)
	  {
	    term1.denominator[i].CloneFrom(expr->terms[r].denominator[i]);
	    term2.denominator[i].CloneFrom(expr->terms[r].denominator[i]);
	  }// for i

	// Replace the old term with the new term
	expr->terms[r] = term1;
	// Now multiply term2 with the min or max and then add to the original expression
      

	// If the term has to be made positive, then make the negative term min-value
	// else make it max value
	Expression expr2;
	if (sign == 1)
	  {
	    expr2 = MinValues.at(r);
	    //TODO: looks incorrect. Mostly MultiplyExpressionWithTerm(expr2, term2);
	    MultiplyExpressionWithTerm(&MinValues.at(r), term2);
	  }// if (sign == 1)
	else if (sign == -1)
	  {
	    expr2 = MaxValues.at(r);
	    //TODO: looks incorrect. Mostly MultiplyExpressionWithTerm(expr2, term2);
	    MultiplyExpressionWithTerm(&MaxValues.at(r), term2);
	  }// else if 

	bool isResultNull = true;
	//printf("Original expression\n");
	expr->Print();
	//printf("Expression2\n");
	expr2.Print();
	(*expr) = AddExpressions((*expr), expr2);
	//printf("Resultant expression\n");
	expr->Print();
      }// if neg_pols > 0

  }// void SetSignOfRightMostLoopIterator(Expression *expr, int sign)


   // Returns +1 if the coefficient of the term is > 0, -1 if negative, 0 if it's 0.
  int ExpressionLibrary::FindSignOfCoefficient(Term term)
  {
    if (term.coefficient < 0)
      return -1;
    else if (term.coefficient > 0)
      return 1;
    else 
      return 0;
  }// FindSignOfCoefficient

  // Returns +1 if the coefficient of the term is > 0, -1 if negative, 0 if it's 0.
  int ExpressionLibrary::FindSignOfCoefficient(Polynomial polynomial)
  {
    if (polynomial.coefficient < 0)
      return -1;
    else if (polynomial.coefficient > 0)
      return 1;
    else 
      return 0;
  }// FindSignOfCoefficient

  Expression ExpressionLibrary::AddExpressions(Expression expr1, Expression expr2)
  {
    vector<Term> termVector;
     
    // Process loop iterators and the parameters
    int count;
    int maxCount;
    if (expr1.size < expr2.size)
      {
	maxCount = expr1.size;
      }//if (expr1.size > expr2.size)
    else
      {
	maxCount = expr2.size;
      }// else

    //printf("maxCount - %d\n", maxCount);
    for (count = 0; count < maxCount; count++)
      {
	if ( (expr1.terms[count].type == intraTile_loop_iterator) || (expr1.terms[count].type == interTile_loop_iterator) ||  (expr1.terms[count].type == loop_iterator) || (expr1.terms[count].type == parameter))
          {
            if (expr1.terms[count].type == expr2.terms[count].type)
              {
		Term res_term = AddTerms(expr1.terms[count], expr2.terms[count]);
		termVector.push_back(res_term);
              }//if the types of the terms are the same
            else
              {
                //Add both the terms to the result
                break;
		// termVector.push_back(expr1.terms[count]);
		// termVector.push_back(expr2.terms[count]);
              }//if the types are not the same 
          }//if loop_iterator or parameter
	else
          {
            // Now constant terms start. 
            //break
            break;
          }//else
      }// for count loop      

    // Now simply add the constant terms to the resultant expression
    for (int i = count; i < expr1.size; i++)
      {
        termVector.push_back(expr1.terms[i]);
      }// for i

    for (int i = count; i < expr2.size; i++)
      {
	termVector.push_back(expr2.terms[i]);
      }// for i

    Expression res_expr;
    res_expr.size = termVector.size();
    res_expr.AllocateMemory();
    // Now copy the terms from the vector to the array
    for (int i = 0; i < res_expr.size; i++)
      {
        res_expr.terms[i].CloneFrom(termVector.at(i));
      }// for int i 

    return res_expr;
  }// AddExpressions()

  Expression ExpressionLibrary::Add_Rows(Expression expr1, Expression expr2, bool *isResultNull)
  {
    Expression expr_res;
    (*isResultNull) = true;
    int r1 = Find_Index_Of_RightmostLoopIterator(expr1);
    int r2 = Find_Index_Of_RightmostLoopIterator(expr2);

    //cout<<"Inside Add_Rows()"<<endl;
    //cout<<"Expression1"<<endl;
    expr1.Print();
    //cout<<"Done printing Expression1"<<endl;
    //cout<<"Expression2"<<endl;
    expr2.Print();
    //cout<<"Done printing Expression2"<<endl;
    // Proceed only if the expressions are the upper bound and the lower bounds of the right most  loop iterator
    //cout<<"r1 = "<<r1<<endl;
    //cout<<"r2 = "<<r2<<endl;
    if (r1 == r2)
      {
        // Proceed only if the signs are different
        if ( FindSignOfCoefficient(expr1.terms[r1]) !=
             FindSignOfCoefficient(expr2.terms[r2]))
	  {
	    //cout<<"The signs are different"<<endl;
            Term reciprocal1;
            reciprocal1.MakeReciprocal(expr1.terms[r1]);
 
            Term reciprocal2;
            reciprocal2.MakeReciprocal(expr2.terms[r2]);
            
            // Pass by reference
            MultiplyExpressionWithTerm(&expr1, reciprocal1);
            MultiplyExpressionWithTerm(&expr2, reciprocal2);
            expr_res = AddExpressions(expr1, expr2);
            (*isResultNull) = false;
	  }// if (signs are opposite)
      }// if (r1 == r2)

    //cout<<"Returning from Add_Rows"<<endl;
    return expr_res;
  }// Add_Rows() function

  void ExpressionLibrary::SetCoefficientOfRightmostIteratorOne(Expression *expr)
  {
    int r = Find_Index_Of_RightmostLoopIterator(*expr);
    Term reciprocal;
    reciprocal.MakeReciprocal(expr->terms[r]);
            
    // Pass by reference
    MultiplyExpressionWithTerm(expr, reciprocal);
  }// static void SetCoefficientOfRightmostIteratorOne(Expression *expr)

  void ExpressionLibrary::MultiplyPolynomialsWithPolynomial(Polynomial *array, int size, Polynomial poly)
  {
    for (int i = 0; i < size; i++)
      {
	array[i].CloneFrom(MultiplyPolynomials(array[i], poly));
      }// for i
  }// MultiplyPolynomialsWithPolynomial()


  vector<Expression*>* ExpressionLibrary::CopyExpressions(vector<Expression*> *exprs)
  {
    vector<Expression*> *expressionVector = new vector<Expression*>();
    if (exprs != NULL)
      {
	for (int i = 0; i < exprs->size(); i++)
	  {
	    Expression *newExpr = new Expression;
	    newExpr->CloneFrom(*(exprs->at(i)));
	    expressionVector->push_back(newExpr);
	  }//for (int i = 0; i < exprs->size(); i++)
      }//if (exprs != NULL)

    return expressionVector;
  }// static vector<Expression*> * CopyExpressions(vector<Expression*> *exprs)

  void ExpressionLibrary::PrintExpressions(vector<Expression*> *exprs)
  {
    if (exprs != NULL)
      {
	//cout<<"#Expressions: "<<exprs->size()<<endl;
	// Print the expressions
	for (int i = 0; i < exprs->size(); i++)
	  {
	    exprs->at(i)->Print();
	  }// for (int i = 0; i < exprs->size(); i++)
      }//if (exprs != NULL)
    else
      {
	//cout<<"Expressions are NULL"<<endl;
      }//else
  }// static void PrintExpressions()

  void ExpressionLibrary::PrintExpressions(vector<Expression> *exprs)
  {
    if (exprs != NULL)
      {
	//cout<<"#Expressions: "<<exprs->size()<<endl;
	// Print the expressions
	for (int i = 0; i < exprs->size(); i++)
	  {
	    exprs->at(i).Print();
	  }// for (int i = 0; i < exprs->size(); i++)
      }//if (exprs != NULL)
    else
      {
	//cout<<"Expressions are NULL"<<endl;
      }//else
  }// static void PrintExpressions()

  bool ExpressionLibrary::ArePolynomialsEqual(Polynomial poly1, Polynomial poly2)
  {
    if (poly1.coefficient == poly2.coefficient)
      {
	if (poly1.size == poly2.size)
	  {
	    for (int i = 0; i < poly1.size; i++)
	      {
		if (poly1.exponents[i] != poly2.exponents[i])
		  {
		    return false;
		  }// if exponents are not equal
	      }// for i

	    return true;
	    // Now that the exponents of both the polynomilas are the same return true
	  }// if sizes are euqal
      }// if coefficients are the same

    return false;
  }// ArePolynomialsEqual()

  void ExpressionLibrary::CancelPolynomials(Polynomial *num, Polynomial *denom)
  {
    if (num != NULL && denom != NULL)
      {
	if (num->size == denom->size)
	  {
	    for (int i = 0; i < num->size; i++)
	      {
		if (num->exponents[i] > denom->exponents[i])
		  {
		    num->exponents[i] -= denom->exponents[i];
		    denom->exponents[i] = 0;
		  }// if
		else
		  {
		    denom->exponents[i] -= num->exponents[i];
		    num->exponents[i] = 0;
		  }// else
	      }// for i
	  }// if
      }// if
  }// CancelPolynomials()

  void ExpressionLibrary::GetCommonPolynomial(Polynomial *poly_array, int size, Polynomial *common)
  {
    common->coefficient = 1;
    if (poly_array != NULL && common != NULL)
      {
	common->size = poly_array[0].size;
	common->AllocateMemory();

	// Initialize
	for (int i = 0; i < common->size; i++)
	  {
	    common->exponents[i] = poly_array[0].exponents[i];
	  }// for i

	common->CopyExponentNames(poly_array[0].names);

        // Find the common by taking the minimum of all the exponents of all the polynomials
	for (int i = 1; i < size; i++)
	  {
	    for (int j = 0; j < common->size; j++)
	      {
		if (poly_array[i].exponents[j] < common->exponents[j])
		  {
		    common->exponents[j] = poly_array[i].exponents[j];
		  }// if
	      }// for int j
	  }// for i

        // Subtract common from the polynomials
        for (int i = 0; i < size; i++)
	  {
            for (int j = 0; j < common->size; j++)
	      {
                poly_array[i].exponents[j] -= common->exponents[j];
	      }// for j
	  }// for i
      }// if
  }//GetCommonPolynomial()


  // Returns the index of the rightmost loop iterator. If not available returns -1
  bool ExpressionLibrary::AreSetsOfPolynomialsEqual(Polynomial *poly1, int size1, Polynomial *poly2, int size2)
  {
    if (poly1 != NULL && poly2 != NULL)
      {
	if (size1 == size2)
	  {
	    // Here need to consider all the permutations of the polynomials
	    for (int i = 0; i < size1; i++)
              {
		if (!ArePolynomialsEqual(poly1[i], poly2[i]))
		  {
		    return false;
		  }// if polynomials not equal
              }// for int i

	    return true;
	  }//if size1 == size2
      }// if
    else if (poly1 == NULL && poly2 == NULL)
      {
	return true;
      }// else if

    return false;
  }//AreSetsOfPolynomialsEqual()

  // This method will take the common terms from the numerator and the denominator and cancel if possible and howmuchever is possible
  void ExpressionLibrary::SimplifyTerm(Term *term)
  {
    if (term->coefficient != 0)
      {
	Polynomial common_poly_num, common_poly_denom;
	bool isNumerator, isDenominator;

	if (AreSetsOfPolynomialsEqual(term->numerator, term->num_polynomials_in_numerator,
				      term->denominator, term->num_polynomials_in_denominator))
	  {
	    term->num_polynomials_in_numerator = 1;
	    term->num_polynomials_in_denominator = 1;
	    term->numerator[0] = term->numerator[0].GetIdentityPolynomial(term->numerator[0].names);
	    term->denominator[0] = term->denominator[0].GetIdentityPolynomial(term->numerator[0].names);
	  }// if
	else
	  {
	    GetCommonPolynomial(term->numerator, term->num_polynomials_in_numerator, &common_poly_num);
	    GetCommonPolynomial(term->denominator, term->num_polynomials_in_denominator, &common_poly_denom);

	    CancelPolynomials(&common_poly_num, &common_poly_denom);

	    if (AreSetsOfPolynomialsEqual(term->numerator, term->num_polynomials_in_numerator,
					  term->denominator, term->num_polynomials_in_denominator))
	      {
		term->num_polynomials_in_numerator = 1;
		term->num_polynomials_in_denominator = 1;
		term->AllocateMemory();
		term->numerator[0] = common_poly_num;
		term->denominator[0] = common_poly_denom;
	      }// if (ArePolynomialsEqual(numerator, denominator))
	    else
	      {
		MultiplyPolynomialsWithPolynomial(term->numerator, term->num_polynomials_in_numerator, common_poly_num);
		MultiplyPolynomialsWithPolynomial(term->denominator, term->num_polynomials_in_denominator, common_poly_denom);
	      }//else
	  }// else
      }// if term->coefficient != 0
  }// SimplifyTerm()

  // This will compare everything except for the coefficients
  bool ExpressionLibrary::AreTermsEqual(Term term1, Term term2)
  {
    if (term1.type == term2.type)
      {
	if ((term1.num_polynomials_in_numerator  == term2.num_polynomials_in_numerator) &&
	    (term2.num_polynomials_in_denominator == term2.num_polynomials_in_denominator))
	  {

	    // Numerator is a sum of products so is the denominator. Need to consider all the permutations of them
	    for (int i = 0; i < term1.num_polynomials_in_numerator; i++)
	      {
		if (!ArePolynomialsEqual(term1.numerator[i], term2.numerator[i]))
		  {
		    return false;
		  } // if Not equal
	      }//for i

	    for (int i = 0; i < term2.num_polynomials_in_denominator; i++)
	      {
		if (!ArePolynomialsEqual(term1.denominator[i], term2.denominator[i]))
		  {
		    return false;
		  } // if Not equal
	      }//for i

	    // Now that the numerator and denominators are equal return true

	    return true;
	  }// if sizes are equal
      }// if types are equal

    return false;
  }// AreTermsEqual()

  int ExpressionLibrary::Find_Index_Of_RightmostLoopIterator(Expression expr)
  {
    int index = -1;
    for (int i = 0; i < expr.GetSize(); i++)

      {
	if ((expr.terms[i].type == interTile_loop_iterator || expr.terms[i].type == intraTile_loop_iterator || expr.terms[i].type == loop_iterator)
	    && expr.terms[i].coefficient != 0)
	  {
	    index = i;
	  }//if type = loop-iterator
      }// for loop

    return index;
  }//Find_Index_Of_RightmostLoopIterator

  // Passing expression by reference
  void ExpressionLibrary::MultiplyExpressionWithTerm(Expression *expr, Term term)
  {
    for (int i = 0; i < (*expr).size; i++)
      {
        (*expr).terms[i] = MultiplyTerms((*expr).terms[i], term);
      }// for int i = 0;
  }// Multiply()

  Term ExpressionLibrary::MultiplyTerms(Term term1, Term term2)
  {
    //The type of the product term will be the type of term1
    Term product;
    product.type = term1.type;
    product.coefficient = term1.coefficient * term2.coefficient;

    // The product is given the name of the LHS by convention.
    // This imposes the restriction on the way the MultiplyTerms() can be called
    product.name = term1.name;

    if (product.coefficient != 0)
      {

	// If the terms are inverses of each other, then cancel them
	Term reciprocal;
	reciprocal.MakeReciprocal(term1);
	if (AreTermsEqual(reciprocal, term2))
	  {
	    product.num_polynomials_in_numerator = 1;
	    product.num_polynomials_in_denominator = 1;
	    product.AllocateMemory();
	    product.numerator[0] = term1.numerator[0].GetIdentityPolynomial(term1.numerator[0].names);
	    product.denominator[0] = term1.denominator[0].GetIdentityPolynomial(term1.denominator[0].names);
	  }// if one is the reciprocal of the other
	else
	  {
	    product.num_polynomials_in_numerator = term1.num_polynomials_in_numerator * term2.num_polynomials_in_numerator;
	    product.num_polynomials_in_denominator = term1.num_polynomials_in_denominator * term2.num_polynomials_in_denominator;
	    product.AllocateMemory();

	    // Multiply the one sum of products with another sum of products
	    for (int i = 0; i < term1.num_polynomials_in_numerator; i++)
	      {
		for (int j = 0; j < term2.num_polynomials_in_numerator; j++)
		  {
		    product.numerator[i * term2.num_polynomials_in_numerator + j] = MultiplyPolynomials(term1.numerator[i], term2.numerator[j]);
		  }
	      }

	    for (int i = 0; i < term1.num_polynomials_in_denominator; i++)
	      {
		for (int j = 0; j < term2.num_polynomials_in_denominator; j++)
		  {
		    product.denominator[i * term2.num_polynomials_in_denominator + j] = MultiplyPolynomials(term1.denominator[i], term2.denominator[j]);
		  }
	      }
	  }// else part when one is not the reciprocal of the other
      }// if (product.coefficient != 0)
    else
      {
	// The coefficient is 0. Still add the polynomials for the correctness elsewhere
        product.num_polynomials_in_numerator = 1;
        product.num_polynomials_in_denominator = 1;
        product.AllocateMemory();
        product.numerator[0] = term1.numerator[0].GetIdentityPolynomial(term1.numerator[0].names);
        product.denominator[0] = term1.denominator[0].GetIdentityPolynomial(term1.numerator[0].names);
      }//else

    SimplifyTerm(&product);
    return product;
  }//MultiplyTerms()

  // This method assumes that the polynomials to be multiplied are of the same length. This is because while adding the corresponding exponents from the two polynomials, it's essential to ascertain that they refer to the same variable.
  Polynomial ExpressionLibrary::MultiplyPolynomials(Polynomial poly1, Polynomial poly2)
  {
    Polynomial poly_res;
    poly_res.size = poly1.size;
    poly_res.AllocateMemory();
    // Copy the exponent names from the first polynomial
    poly_res.CopyExponentNames(poly1.names);

    if (poly1.size == poly2.size)
      {
	poly_res.coefficient = poly1.coefficient * poly2.coefficient;

	// if (poly_res.coefficient != 0)
	{
	  for (int i = 0; i < poly1.size; i++)
	    {
	      poly_res.exponents[i] = poly1.exponents[i] + poly2.exponents[i];
	    }//for int i = 0;
	}// if (coefficient != 0)
      }//if sizes are equal

    return poly_res;
  }// MultiplyPolynomials

  void ExpressionLibrary::ConvertExpressionsToInequalities(vector<Expression> *exprs)
  {

    // For each expression
    for (int i = 0; i < exprs->size(); i++)
      {
        /* Make the coefficient of the right most loop iterator 1
	   by taking its reciprocal and multiplying with the other terms*/

	Expression expr = exprs->at(i);
	int r = Find_Index_Of_RightmostLoopIterator(expr);
	Term reciprocal;
	reciprocal.MakeReciprocal(expr.terms[r]);

	// Pass by reference
	MultiplyExpressionWithTerm(&expr, reciprocal);

	// Make the sign of the rightmost iterator negative and rest positive.
	if (expr.terms[r].coefficient > 0)
	  {
	    // If the coefficient is negative
	    for (int j = 0; j < expr.size; j++)
	      {
		// Flip the sign
		expr.terms[j].coefficient *= -1;
	      }//for (int j = 0; j < expr->size; j++)
	  }// if (expr->terms[r]->coefficient < 0)
      }//	for (int i = 0; i < exprs->size(); i++)


  }// static void ConvertExpressionsToInequalities(vector<Expression> exprs)

  void ExpressionLibrary::ConvertExpressionsToInequalities(vector<Expression*> *exprs)
  {
    // For each expression
    for (int i = 0; i < exprs->size(); i++)
      {
        /* Make the coefficient of the right most loop iterator 1
	   by taking its reciprocal and multiplying with the other terms*/

	Expression* expr = exprs->at(i);
	int r = Find_Index_Of_RightmostLoopIterator((*expr));
	Term reciprocal;
	reciprocal.MakeReciprocal(expr->terms[r]);

	// Pass by reference
	MultiplyExpressionWithTerm(expr, reciprocal);

	// Make the sign of the rightmost iterator negative and rest positive.
	if (expr->terms[r].coefficient > 0)
	  {
	    // If the coefficient is negative
	    for (int j = 0; j < expr->size; j++)
	      {
		// Flip the sign
		expr->terms[j].coefficient *= -1;
	      }//for (int j = 0; j < expr->size; j++)
	  }// if (expr->terms[r]->coefficient < 0)
      }//	for (int i = 0; i < exprs->size(); i++)


  }//static void ConvertExpressionsToInequalities(vector<Expression*> *exprs)



  float ExpressionLibrary::GetCoefficientOfRightmostIterator(Expression *expr)
  {
    float coefficient = 0;

    for (int i = 0; i < expr->size; i++)
      {
	//Examine the individual terms
	if (expr->terms[i].type == intraTile_loop_iterator ||
	    expr->terms[i].type == interTile_loop_iterator ||
	    expr->terms[i].type == loop_iterator)
	  {
	    // If the coefficient is non zero
	    if (expr->terms[i].coefficient != 0)
	      {
		coefficient = expr->terms[i].coefficient;
	      }// if (expr->terms[i].coefficient != 0)
	  }//if (expr->terms[i]->type == loop_iterator)
      }//for (int i = 0; i < expr->size; i++)

    return coefficient;
  }//float GetCoefficientOfRightmostIterator(Expression *expr)

  int ExpressionLibrary::FindNumberOfTermsOfType(Expression *expr, TypeOfTerm type)
  {
    int NumberOfIterators = 0;
    for (int i = 0; i < expr->size; i++)
      {
	//Examine the individual terms
	if (expr->terms[i].type == type)
	  {
	    NumberOfIterators++;
	  }//if (expr->terms[i]->type == loop_iterator)
      }//for (int i = 0; i < expr->size; i++)

    return NumberOfIterators;
  }//int FindNumberOfIterators(exprs->at(i))


  void ExpressionLibrary::AddItraTileLoopIteratorTermsInExpression(Expression *expr, int count,
						       int NumberOfIterators, int NumberOfParameters)
  {
    string *exponent_names = expr->terms[0].numerator[0].names;
    Term *term = new Term[expr->size + count];
    int j = 0;
    /* First add the loop iterators then intra tile loop iterators and then the parameters */

    // Adding loop iterators
    int i = 0;
    for (i = 0; i < expr->size; i++, j++)
      {
	if (expr->terms[j].type != parameter && expr->terms[j].type != constant)
	  {
	    term[i].CloneFrom(expr->terms[j]);
	  }//if (expr->terms[j]->type != parameter)
	else
	  {
	    break;
	  }//else
      }//for (int i = 0; i < expr->size; i++)

    // Adding intra tile loop iterators
    int local_count = 0;
    for (; local_count < count; i++, local_count++)
      {
	term[i].SetIdentityNumeratorDenominator(NumberOfIterators + NumberOfParameters,
						exponent_names);
	//Add the polynomial
	/*	Polynomial *num = new Polynomial(NumberOfIterators + NumberOfParameters);
		Polynomial *denom = new Polynomial(NumberOfIterators + NumberOfParameters);
		expr->terms[j].SetNumeratorDenominator(1, num, 1, denom);
	*/
	term[i].type = intraTile_loop_iterator;
      }//for (int i = 0; i < count; i++)

    // Adding the parameters
    for ( ; i < expr->size + count; i++, j++)
      {
	term[i].CloneFrom(expr->terms[j]);
      }//for (; i < expr->size + count; i++)

    expr->size += count;

    // Switch the terms
    expr->terms = term;
    // Set the coefficients of the intra tile loops same as the original iterators
    int interTileMark = 0, intraTileMark = 0;
    for (i = 0; i < expr->size; i++)
      {
	if (expr->terms[i].type == intraTile_loop_iterator)
	  {
	    intraTileMark = i;
	  }//if (expr->terms[i].type == intraTile_loop_iterator)
      }//for (i = 0; i < expr->size; i++)

    intraTileMark = count;

    for (i = 0; i < count; i++, interTileMark++, intraTileMark++)
      {
	expr->terms[intraTileMark].coefficient = expr->terms[interTileMark].coefficient;
	expr->terms[intraTileMark].name = expr->terms[interTileMark].name;
	expr->terms[interTileMark].name = expr->terms[interTileMark].name +
	  GetTileIteratorSuffix(1, 1);
      }//for (i = 0; i < count; i++, interTileMark++, intraTileMark++)
  }//static void AddItraTileLoopIteratorTermsInExpression(Expression *expr, int count)

  string ExpressionLibrary::GetTileIteratorSuffix(int CurrentLevel, int TotalLevels)
  {
    stringstream oss;
    oss<<(TotalLevels - CurrentLevel + 1);
    string suffix = "t" + oss.str();
    return suffix;
  }//static string GetTileIteratorSuffix(int level)

  string ExpressionLibrary::GetTileSizePrefix(int CurrentLevel, int TotalLevels)
  {
    stringstream oss;
    oss<<(TotalLevels - CurrentLevel + 1);
    string prefix = "T" + oss.str();
    return prefix;
  }//static string GetTileIteratorSuffix(int level)


  void ExpressionLibrary::SubstituteIteratorWithIteratorPlusTileSize(vector<Expression*> *exprs)
  {
    // If the expression is: j >= i + N equivalently -i + j - N >= 0
    // Substitute i = it * Ti + ii and j = jt * Tj + jj.
    // The resultant equation would look like -
    // -it*Ti + jt*Tj - ii + jj - N >= 0
    // Note that the intra tile iterators (ii, jj) would appear in the same order as the inter tile iterators (it, jt) which in turn
    // would appear in the order in which they appeared in the original equation
    // Furthermore, we also safely assume that the individual terms wouldn't have any polynomials associated with them.
    // This is because the way cloog/scopLib input appears. Every terms is either a loop iterator or a parameter or a constant
    // and its associated coefficient would appear in the input

    // For each expression
    for (int i = 0; i < exprs->size(); i++)
      {
	int NumberOfIterators = FindNumberOfTermsOfType(exprs->at(i), loop_iterator);
	int NumberOfParameters = FindNumberOfTermsOfType(exprs->at(i), parameter);
	int iteratorNumber = 0;
	int originalSize = exprs->at(i)->size;

	for (int j = 0; j < originalSize; j++)
	  {
	    if (exprs->at(i)->terms[j].type == loop_iterator)
	      {
		//Substitute i with it
		exprs->at(i)->terms[j].type = interTile_loop_iterator;
		// i.e it * Ti
		exprs->at(i)->terms[j].numerator[0].SetCoefficientAt(iteratorNumber, 1);
		// Add the itra tile loop iterator after the existing loop iterators either iter or intra or the untiled
		iteratorNumber++;
	      }//if (exprs->at(i)->terms[j]->type == loop_iterator)
	  }//for (int j = 0; j < exprs->at(i)->size; j++)

	AddItraTileLoopIteratorTermsInExpression(exprs->at(i), iteratorNumber,
						 NumberOfIterators, NumberOfParameters);
      }//for (int i = 0; i < expr->size(); i++)
  }//void SubstituteIteratorWithIteratorPlusTileSize(exprs)


  void ExpressionLibrary::PutTermsInOrder(vector<Expression*> *exprs)
  {
    // For each expression
    for (int i = 0; i < exprs->size(); i++)
      {
	Term *terms = new Term[exprs->at(i)->size];
	int count = 0;

	// InterTileIterators
	for (int j = 0; j < exprs->at(i)->size; j++)
	  {
	    if (exprs->at(i)->terms[j].type == interTile_loop_iterator)
	      {
		terms[count++].CloneFrom(exprs->at(i)->terms[j]);
	      }//if (exprs->at(i)->type == interTile_loop_iterator)
	  }//for (int j = 0; j < exprs->at(i)->size; j++)

	// Parameters
	for (int j = 0; j < exprs->at(i)->size; j++)
	  {
	    if (exprs->at(i)->terms[j].type == parameter)
	      {
		terms[count++].CloneFrom(exprs->at(i)->terms[j]);
	      }//if (exprs->at(i)->type == parameter)
	  }//for (int j = 0; j < exprs->at(i)->size; j++)

	// Constants
	for (int j = 0; j < exprs->at(i)->size; j++)
	  {
	    if (exprs->at(i)->terms[j].type == constant)
	      {
		terms[count++].CloneFrom(exprs->at(i)->terms[j]);
	      }//if (exprs->at(i)->type == parameter)
	  }//for (int j = 0; j < exprs->at(i)->size; j++)

	exprs->at(i)->terms = terms;
      }//for (int i = 0; i < exprs->size(); i++)
  }//void PutTermsInOrder(vector<Expression*> *exprs)

  void ExpressionLibrary::ReplaceIntraTileIteratorWithBounds(vector<Expression*> *exprs)
  {
    // For each expression
    for (int i = 0; i < exprs->size(); i++)
      {

	int NumberOfIterators = FindNumberOfTermsOfType(exprs->at(i), interTile_loop_iterator);
	int NumberOfParameters = FindNumberOfTermsOfType(exprs->at(i), parameter);
	int iteratorNumber = 0;

	float negativeComponent = 0;
	// For each term
	for (int j = 0; j < exprs->at(i)->size; j++)
	  {
	    if (exprs->at(i)->terms[j].type == intraTile_loop_iterator)
	      {
		exprs->at(i)->terms[j].type = constant;
		exprs->at(i)->terms[j].name = "";
		// If the coefficient = 0, eliminate it
		if (exprs->at(i)->terms[j].coefficient <= 0)
		  {
		    exprs->at(i)->terms[j].coefficient = 0;
		  }//if (exprs->at(i)->terms[j] <= 0)
		else
		  {
		    // If the coefficient is positive then replace it with the bound Ti - 1
		    exprs->at(i)->terms[j].numerator[0].SetCoefficientAt(iteratorNumber, 1);

		    // Subtract the coefficient from the constant
		    negativeComponent += exprs->at(i)->terms[j].coefficient;
		  }// else

		iteratorNumber++;
	      }//if (exprs->at(i)->terms[j].type == intraTile_loop_iterator)
	    else if (exprs->at(i)->terms[j].type == constant)
	      {
		// Subtract the coefficient from the constant
		exprs->at(i)->terms[j].coefficient -= negativeComponent;
		negativeComponent = 0;
	      }//else if (exprs->at(i)->terms[j].type == constant)
	  }//for (int j = 0; j < exprs->at(i)->size; j++)
      }//for (int i = 0; i < exprs->size(); i++)

    PutTermsInOrder(exprs);
  }//void ReplaceIntraTileIteratorWithBounds(vector<Expression*> *exprs)


