/*
 * Term.cpp: This file is part of the Parametric Tiling project.
 * 
 * Parametric Tiling: A CLAST-to-CLAST parametric tiling software
 * 
 * Copyright (C) 2011 Sanket Tavargeri
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 3
 * of the License, or (at your option) any later version.
 * 
 * The complete GNU Lesser General Public Licence Notice can be found
 * as the `COPYING.LESSER' file in the root directory.
 * 
 * Author:
 * Sanket Tavargeri <sanket.tavargeri@gmail.com>
 * 
 */

#if HAVE_CONFIG_H
# include <ptile/config.h>
#endif

# include <ptile/Term.hpp>

using namespace std;


  // Methods
  Term::Term()
  {
    Term(1, 1);
  }

  Term::Term(int num_count, int denom_count)
  {
    coefficient = 1;
    num_polynomials_in_numerator = num_count;
    num_polynomials_in_denominator = denom_count;
    AllocateMemory();
  }

  void Term::SetIdentityNumeratorDenominator()
  {
    num_polynomials_in_numerator = 0;
    num_polynomials_in_denominator = 0;
  }//SetIdentityNumeratorDenominator()

  void Term::SetIdentityNumeratorDenominator(int exponentsInPolynomial, string *exponent_names)
  {
    Polynomial *num = new Polynomial(exponentsInPolynomial);
    Polynomial *denom = new Polynomial(exponentsInPolynomial);
    num->CopyExponentNames(exponent_names);
    denom->CopyExponentNames(exponent_names);
    SetNumeratorDenominator(1, num, 1, denom);
  }//void SetIdentityNumberatorDenominator(int exponentsInPolynomial)

  void Term::SetNumeratorDenominator(int num_polynomials_in_numerator,
			       Polynomial* numerator,
			       int num_polynomials_in_denominator,
			       Polynomial* denominator)
  {
    this->num_polynomials_in_numerator = num_polynomials_in_numerator;
    this->num_polynomials_in_denominator = num_polynomials_in_denominator;
    this->numerator = numerator;
    this->denominator = denominator;
  }//SetNumeratorDenominator()


  void Term::SetIdentityTermFrom(Term term)
  {	
    type = term.type;
    name = term.name;
    coefficient = 0;
    int exponentsInPolynomial = term.numerator[0].size;
    string *exponent_names = term.numerator[0].names;
    SetIdentityNumeratorDenominator(exponentsInPolynomial, exponent_names);
  }// void SetIdentityNumberatorDenominatorFromTerm()

  void Term::SetTypeOfTerm(TypeOfTerm type)
  {
    this->type = type;
  }

  void Term::CloneFrom(Term term)
  {
    coefficient = term.coefficient;
    type = term.type;
    name = term.name;

    // Set the numerator and denominator even if the coefficient is equal to 0

    // if (coefficient != 0)
    {
      num_polynomials_in_numerator = term.num_polynomials_in_numerator;
      num_polynomials_in_denominator = term.num_polynomials_in_denominator;
      AllocateMemory();
      for (int i = 0; i < num_polynomials_in_numerator; i++)
	{
	  numerator[i].CloneFrom(term.numerator[i]);
	}// for int i

      for (int i = 0; i < num_polynomials_in_denominator; i++)
        {
	  denominator[i].CloneFrom(term.denominator[i]);
        }// for int i
    }// if (coefficient != 0)

  }// CloneFrom()

  void Term::AllocateMemory()
  {
    numerator = new Polynomial[num_polynomials_in_numerator];
    denominator = new Polynomial[num_polynomials_in_denominator];
  }//  AllocateMemory()

  TypeOfTerm Term::GetType(int inputType)
  {

    TypeOfTerm type;
    switch (inputType)
      {
      case 1:
	type = loop_iterator;
	break;
      case 2:
	type = parameter;
	break;
      case 3:
	type = constant;
	break;
      case 4:
	type =  intraTile_loop_iterator;
	break;
      case 5:
	type = interTile_loop_iterator;
	break;
      default:
	type = constant;
      }// switch

    return type;
  }// GetType


  int Term::GetTypeFromTypeOfTerm(TypeOfTerm inputType)
  {
    int type;
    switch (inputType)
      {
      case loop_iterator:
	type = 1;
	break;
      case parameter:
	type = 2;
	break;
      case constant:
	type = 3;
	break;
      case intraTile_loop_iterator:
	type = 4;
	break;
      case interTile_loop_iterator:
	type = 5;
	break;
      default:
	type = 1;
      }// switch

    return type;
  }// GetType


  string Term::GetTypeStringFromTypeOfTerm(TypeOfTerm inputType)
  {
    string type;
    switch (inputType)
      {
      case loop_iterator:
	type = "loop_iterator";
	break;
      case parameter:
	type = "parameter";
	break;
      case constant:
	type = "constant";
	break;
      case intraTile_loop_iterator:
	type = "intraTile_loop_iterator";
	break;
      case interTile_loop_iterator:
	type = "interTile_loop_iterator";
	break;
      default:
	type = "constant";
      }// switch

    return type;
  }// GetType

  void Term::MakeReciprocal(Term term)
  {
    coefficient = 1/term.coefficient;
    name = term.name;
    // Make coefficient positive
    if (coefficient < 0)
      {
	coefficient *= -1;
      }// if (coefficient < 0)

    type = term.type;
    // if (coefficient != 0)
    {
      num_polynomials_in_numerator = term.num_polynomials_in_denominator;
      num_polynomials_in_denominator = term.num_polynomials_in_numerator;
      AllocateMemory();
      for (int i = 0; i < num_polynomials_in_numerator; i++)
	{
	  numerator[i].CloneFrom(term.denominator[i]);
	}// for int i

      for (int i = 0; i < num_polynomials_in_denominator; i++)
        {
	  denominator[i].CloneFrom(term.numerator[i]);
        }// for int i
    }// if (coefficient != 0)
  }//GetReciprocal()

  void Term::Print()
  {
    //printf("Term\n");

    // print type as type + 1 because the input is taken as 1, 2 and 3 and stored as 0, 1 and 2 respectively
    //printf("coefficient - %f, type - %s, num_polynomials_in_numerator - %d, num_polynomials_in_denominator - %d\n",
    //         coefficient, GetTypeStringFromTypeOfTerm(type).c_str(), num_polynomials_in_numerator, num_polynomials_in_denominator);

    // Print the name of the term
    if (name.empty() == false)
      {
	//printf("name = %s\n", name.c_str());
      }// if (name.empty() == false)
    else
      {
	//printf("name is empty\n");
      }// else

    // Print even if the coefficient = 0
    // if (coefficient != 0)
    {
      for (int i = 0; i < num_polynomials_in_numerator; i++)
	{
	  numerator[i].Print();
	}// for i

      for (int i = 0; i < num_polynomials_in_denominator; i++)
	{
	  denominator[i].Print();
	}// for i
    }//if (coefficient != 0)

     //printf("\n");
  }// Print()

