/*
 * solution.c: this file is part of the FM project.
 *
 * FM, a fast and optimized C implementation of Fourier-Motzkin
 * projection algorithm.
 *
 * Copyright (C) 2006,2007,2008 Louis-Noel Pouchet
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 3
 * of the License, or (at your option) any later version.
 *
 * The complete GNU Lesser General Public Licence Notice can be found
 *  as the `COPYING.LESSER' file in the root directory.
 *
 * Author:
 * Louis-Noel Pouchet <Louis-Noel.Pouchet@inria.fr>
 *
 */
#if HAVE_CONFIG_H
# include <fm/config.h>
#endif

#include <fm/common.h>
#include <fm/solution.h>



s_fm_solution_t*
fm_solution_alloc (size_t size)
{
  unsigned i;
  s_fm_solution_t* s;

  s = XMALLOC(s_fm_solution_t, 1);
  s->solution = XMALLOC(s_fm_ball_t, size);

  for (i = 0; i < size; ++i)
    {
      s->solution[i].positive = NULL;
      s->solution[i].negative = NULL;
    }

  s->size = size;

  return s;
}


void
fm_solution_free (s_fm_solution_t* s)
{
  unsigned i;

  if (s == NULL)
    return;

  for (i = 0; i < s->size; ++i)
    {
      fm_list_free (s->solution[i].positive, (free_fun_t) fm_vector_free);
      fm_list_free (s->solution[i].negative, (free_fun_t) fm_vector_free);
    }
  XFREE(s->solution);
  XFREE(s);
}


s_fm_solution_t*
fm_solution_dup (s_fm_solution_t* s)
{
  if (s == NULL)
    return NULL;

  unsigned i;
  s_fm_list_t* l;
  s_fm_vector_t* tmp;
  s_fm_solution_t* res = fm_solution_alloc (s->size);

  for (i = 0; i < s->size; ++i)
    {
      for (l = s->solution[i].positive; l != NULL; l = l->next)
	{
	  tmp = fm_vector_dup (l->data);
	  fm_list_add_head (&(res->solution[i].positive), tmp);
	}
      for (l = s->solution[i].negative; l != NULL; l = l->next)
	{
	  tmp = fm_vector_dup (l->data);
	  fm_list_add_head (&(res->solution[i].negative), tmp);
	}
    }

  return res;
}


void
fm_solution_print (FILE* stream, s_fm_solution_t* s)
{
  unsigned i;
  s_fm_list_t* tmp;

  if (s == NULL)
    {
      fprintf (stream, "(Empty solution)\n", i);
      return;
    }

  for (i = 0; i < s->size; ++i)
    {
      fprintf (stream, "At index %d\n", i);
      if (s->solution[i].positive != NULL)
	{
	  fprintf (stream, "- Positives:\n");
	  for (tmp = s->solution[i].positive; tmp != NULL; tmp = tmp->next)
	    {
	      fm_vector_print (stream, tmp->data);
	      fprintf (stream, "\n");
	    }
	}
      if (s->solution[i].negative != NULL)
	{
	  fprintf (stream, "- Negatives:\n");
	  for (tmp = s->solution[i].negative; tmp != NULL; tmp = tmp->next)
	    {
	      fm_vector_print (stream, tmp->data);
	      fprintf (stream, "\n");
	    }
	}
    }
}


/**
 *
 * Convert a solution_t to a system_t.
 *
 */
s_fm_system_t*
fm_solution_to_system (s_fm_solution_t* s)
{
  if (s == NULL || s->size < 1)
    return NULL;

  s_fm_system_t* system;
  unsigned i;
  unsigned size = 0, count = 0;
  unsigned stop = 1;
  unsigned max_mu = s->size;
  s_fm_list_t* l;
  s_fm_list_t* tmp;

  // Compute the size of the system.
  for (i = 0; i < max_mu; ++i)
    {
      for (l = s->solution[i].positive; l != NULL; l = l->next, size++)
	;
      for (l = s->solution[i].negative; l != NULL; l = l->next, size++)
	;
    }

  // Allocate the system structure.
  system = XMALLOC(s_fm_system_t, 1);
  system->nb_lines = size;
  system->nb_cols = max_mu + 2;
  system->lines = XMALLOC(s_fm_vector_t*, system->nb_lines);
  system->allocated = system->nb_lines;

  // Fill the system.
  l = s->solution[0].positive;
  for (i = 0; i < max_mu; ++i, l = s->solution[i].positive, stop = 1)
    do
      {
	for (tmp = l; tmp != NULL; tmp = tmp->next)
	  {
	    system->lines[count] = fm_vector_alloc (max_mu + 2);
	    fm_vector_expand (system->lines[count++], tmp->data);
	  }
	l = s->solution[i].negative;
      }
    while (stop--);

  return system;
}


/**
 *
 * Convert a system_t to a solution_t.
 *
 */
s_fm_solution_t*
fm_system_to_solution (s_fm_system_t* s)
{
  if (s == NULL)
    return NULL;

  int i;
  int count;
  s_fm_solution_t* res = fm_solution_alloc (s->nb_cols - 2);
  s_fm_vector_t* v;

  for (i = 0; i < s->nb_lines; ++i)
    {
      if (! fm_vector_is_empty (s->lines[i]))
	{
	  count = s->nb_cols - 2;
	  while (count > 1 && Z_CMP_SI(s->lines[i]->vector[count].num, ==, 0))
	    --count;
	  v = fm_vector_alloc (count + 2);
	  fm_vector_shrink (v, s->lines[i], count);
	  fm_vector_normalize_idx (v, v, count);
	  fm_solution_add_unique_line_at (res, v, count);
	}
      else
	if (Z_CMP_SI(s->lines[i]->vector[s->nb_cols - 1].num, <, 0))
	  {
	    /* The system is empty (there is an (in)equality x >= 0
	       where x < 0). */
	    fm_solution_free (res);
	    return NULL;
	  }
    }

  return res;
}


/**
 *
 * Convert the first 'idx' variables of a solution_t to a system_t.
 *
 */
s_fm_system_t*
fm_solution_to_system_at (s_fm_solution_t* s, int idx)
{
  if (s == NULL || s->size < 1)
    {
      assert(! "Input system is empty");
      return NULL;
    }

  s_fm_system_t* system;
  unsigned i;
  unsigned size = 0, count = 0;
  unsigned stop = 1;
  unsigned max_mu = idx;
  s_fm_list_t* l;
  s_fm_list_t* tmp;

  // Compute the size of the system.
  for (i = 0; i < max_mu; ++i)
    {
      for (l = s->solution[i].positive; l != NULL; l = l->next, size++)
	;
      for (l = s->solution[i].negative; l != NULL; l = l->next, size++)
	;
    }

  // Allocate the system.
  system = fm_system_alloc (size, max_mu + 2);

  // Fill the system.
  l = s->solution[0].positive;
  for (i = 0; i < max_mu; ++i, l = s->solution[i].positive, stop = 1)
    do
      {
	for (tmp = l; tmp != NULL; tmp = tmp->next)
	  {
	    system->lines[count] = fm_vector_alloc (max_mu + 2);
	    fm_vector_expand (system->lines[count++], tmp->data);
	  }
	l = s->solution[i].negative;
      }
    while (stop--);

  return system;
}


/**
 * Add a vector to the solution set. If v is already present in s, or if
 * v (is)does subsume(d by) a vector of s, returns 0, else return 1.
 *
 */
int
fm_solution_add_unique_line_at (s_fm_solution_t* s,
				s_fm_vector_t* v,
				unsigned idx)
{
  if (s == NULL || idx > s->size + 1)
    {
      assert (! "Error in arguments");
      return FM_STATUS_ERROR;
    }

  cmp_fun_t f = (cmp_fun_t) fm_vector_do_subsume;
  s_fm_list_t* mem;
  s_fm_list_t* res;

  if (Z_CMP_SI(v->vector[idx].num, ==, 1))
    {
      mem = s->solution[idx - 1].positive;
      res = fm_list_add_head_unique (&(s->solution[idx - 1].positive), v, f);
    }
  else if (Z_CMP_SI(v->vector[idx].num, ==, -1))
    {
      mem = s->solution[idx - 1].negative;
      res = fm_list_add_head_unique (&(s->solution[idx - 1].negative), v, f);
    }
  else
    {
      fm_vector_print(stdout, v); printf("\nidx: %d\n", idx);
      assert (! "Error: vector[idx] is not 1 or -1");
      return FM_STATUS_ERROR;
    }

  return mem != res;
}


int
fm_solution_add_line_at (s_fm_solution_t* s, s_fm_vector_t* v, unsigned idx)
{
  if (s == NULL || idx > s->size)
    {
      assert (! "Error in arguments");
      return FM_STATUS_ERROR;
    }
  if (Z_CMP_SI(v->vector[idx].num, ==, 1))
    fm_list_add_head (&(s->solution[idx - 1].positive), v);
  else if (Z_CMP_SI(v->vector[idx].num, ==, -1))
    fm_list_add_head (&(s->solution[idx - 1].negative), v);
  else
    {
      assert (! "Error: vector[idx] is not 1 or -1");
      return FM_STATUS_ERROR;
    }

  return FM_STATUS_OK;
}


/**
 * Find implicit equalities in the system.
 *
 *
 */
int
fm_solution_equalities_find (s_fm_solution_t* s)
{
  if (s == NULL)
    return 0;

  s_fm_vector_t* tmp;

  int i, j, k;
  int sz = 0;
  s_fm_list_t* tmpp;
  s_fm_list_t* tmpn;
  s_fm_list_t* pred;
  s_fm_vector_t* tmpvp;
  s_fm_vector_t* tmpvn;

  for (i = 0; i < s->size; ++i)
    {
      for (tmpp = s->solution[i].positive; tmpp; tmpp = tmpp->next)
	{
	  tmpvp = tmpp->data;
	  fm_vector_opp (tmpvp, tmpvp);
	  // Skip real equalities for positive sign.
	  if (Z_CMP_SI(tmpvp->vector[0].num, ==, 0))
	    {
	      ++sz;
	      continue;
	    }
	  for (tmpn = s->solution[i].negative; tmpn; tmpn = tmpn->next)
	    {
	      tmpvn = tmpn->data;
	      // Skip real equalities for negative sign.
	      if (Z_CMP_SI(tmpvn->vector[0].num, ==,0))
		continue;
	      if (fm_vector_equal (tmpvp, tmpvn))
		{
		  fm_vector_free (tmpvn);
		  Z_ASSIGN_SI (tmpvp->vector[0].num, 0);
		  if (tmpn == s->solution[i].negative)
		    s->solution[i].negative = tmpn->next;
		  else
		    pred->next = tmpn->next;
		  XFREE(tmpn);
		  ++sz;
		  break;
		}
	      pred = tmpn;
	    }

	  // Count real equalities for negative sign.
	  for (tmpn = s->solution[i].negative; tmpn; tmpn = tmpn->next)
	    {
	      tmpvn = tmpn->data;
	      if (Z_CMP_SI(tmpvn->vector[0].num, ==, 0))
		++sz;
	    }

	  fm_vector_opp (tmpvp, tmpvp);
	}

    }

  return sz;
}


void
fm_solution_cut (s_fm_solution_t* s, int dim)
{

  if (!s || dim > s->size)
    {
      printf ("ERROR IN ARGUMENTS\n");
      return;
    }

  int i;

  for (i = dim; i < s->size; ++i)
    {
      fm_list_free (s->solution[i].positive, (free_fun_t) fm_vector_free);
      fm_list_free (s->solution[i].negative, (free_fun_t) fm_vector_free);
    }
  s->size = dim;
}



/**
 * Check if a point (represented by a vector_t) is in a polyhedron.
 *
 */
int
fm_solution_point_included (s_fm_solution_t* s, s_fm_vector_t* v)
{
  int i, j, k;
  s_fm_rational_t* val;
  s_fm_rational_t* tmp;
  s_fm_vector_t* data;
  int ret = 1;
  s_fm_list_t* l;

  if (s->size != v->size - 1)
    return 0;

  val = fm_rational_alloc ();
  tmp = fm_rational_alloc ();

  // Simply check if the point does not violate any constraint.
  for (i = 0; i < s->size; ++i)
    {
      for (k = 0, l = s->solution[i].positive; k < 2;
	   l = s->solution[i].negative, ++k)
	for (; l; l = l->next)
	  {
	    fm_rational_init (tmp);
	    data = l->data;
	    for (j = 1; j < data->size - 1; ++j)
	      {
		fm_rational_mul (val, &(data->vector[j]), &(v->vector[j]));
		fm_rational_add (tmp, tmp, val);
	      }
	    fm_rational_add (tmp, tmp, &(data->vector[data->size - 1]));
	    if (Z_CMP_SI(tmp->num, <, 0) ||
		((Z_CMP_SI (data->vector[0].num, ==, 0) &&
		  Z_CMP_SI (tmp->num, !=, 0))))
	      {
		ret = 0;
		break;
	      }
	  }
    }
  fm_rational_free (val);
  fm_rational_free (tmp);

  return ret;
}

#ifdef HAVE_LIBPIPLIB

/**
 * Remove redundant constraints from a polyhedron defined by a set of
 * hyperplanes..
 *
 * See M. Le Fur Irisa research report.
 *
 */
s_fm_solution_t*
fm_solution_simplify (s_fm_solution_t* s,
		      int simplify_mode)
{
  s_fm_system_t* system = fm_solution_to_system (s);
  fm_solution_free (s);

  fm_system_simplify (system, simplify_mode);

  s = fm_system_to_solution (system);
  fm_system_free (system);

  return s;
}

#endif // ! HAVE_LIBPIPLIB


/**
 * Helper to count points in a polytope.
 *
 */
static
void
count_points (s_fm_solution_t* sol, s_fm_vector_t* draw, int mask, void* data)
{
  (*((unsigned long long int*) data))++;

  if (mask & FM_SOLUTION_PRINT_POINT)
    {
      fm_vector_print (stdout, draw);
      printf ("\n");
    }
}


/**
 * Count the number of integer points in the polytope.
 *
 *
 */
unsigned long long int
fm_solution_count (s_fm_solution_t* sol, int bound_limit, int mask)
{
  unsigned long long int res = 0;

  if (mask & FM_SOLVER_VERBOSE)
    fprintf (stderr, ".... Explore polyhedron\n");

  fm_solution_traverse (sol, bound_limit, mask, count_points, (void*) (&res));

  if (mask & FM_SOLVER_VERBOSE)
    fprintf (stderr, "... Polytope has %Lu point(s)\n", res);

  return res;
}


/**
 * Count the number of integer points in the polytope.
 *
 *
 */
void
fm_solution_traverse (s_fm_solution_t* sol, int bound_limit, int mask,
		      point_fun_t f, void* data)
{
  s_fm_rational_t* lb = NULL;
  s_fm_rational_t* Ub = NULL;
  z_type_t* i = XMALLOC(z_type_t, sol->size);
  z_type_t* min = XMALLOC(z_type_t, sol->size);
  z_type_t* max = XMALLOC(z_type_t, sol->size);
  int ii;
  unsigned idx = 0;
  s_fm_vector_t* draw = fm_vector_alloc (sol->size + 1);
  unsigned long long int badpath = 0;
  unsigned long long int ret = 0;

  for (ii = 0; ii < sol->size; ++ii)
    {
      Z_INIT(i[ii]);
      Z_INIT(min[ii]);
      Z_INIT(max[ii]);
    }

  for (ii = 0; ii < sol->size; ++ii)
    {
      Z_INIT(i[ii]);
      Z_INIT(min[ii]);
      Z_INIT(max[ii]);
    }

  fm_solver_compute_min (&lb, sol->solution[idx].positive,
			 draw, idx, FM_MINLEXICO_INT);
  fm_solver_compute_max (&Ub, sol->solution[idx].negative,
			 draw, idx, FM_MINLEXICO_INT);
  min[idx] = lb->num;
  max[idx] = Ub->num;


  // Loop on each point.
  for (Z_ASSIGN(i[idx], min[idx]); Z_CMP(i[idx], <=, max[idx]);
       Z_INC(i[idx], i[idx]))
    {
      fm_vector_assign_int_idx (draw, i[idx], idx + 1);
      if (idx + 1 == bound_limit)
	{
	  ret++;
	  if (f != NULL)
	    f (sol, draw, mask, data);
	  while (idx > 0 && (Z_CMP(i[idx], ==, max[idx]) ||
			     Z_CMP(min[idx], >, max[idx])))
	    --idx;
	}
      else
	{
	  ++idx;
	  fm_solver_compute_min (&lb, sol->solution[idx].positive,
				 draw, idx, FM_MINLEXICO_INT);
	  fm_solver_compute_max (&Ub, sol->solution[idx].negative,
				 draw, idx, FM_MINLEXICO_INT);
	  Z_ASSIGN(min[idx], lb->num);
	  Z_ASSIGN(max[idx], Ub->num);
	  Z_ASSIGN(i[idx], min[idx]);
	  Z_DEC(i[idx], i[idx]);
	  int has_hole = 0;
	  while (idx > 0 && (Z_CMP(min[idx], >, max[idx]) ||
			     Z_CMP(i[idx], ==, max[idx])))
	    {
	      // Integer hole.
	      if (Z_CMP(min[idx], >, max[idx]))
		has_hole = 1;
	      --idx;
	    }
	  if (has_hole)
	    ++badpath;
	}
    }

  // Be clean.
  for (ii = 0; ii < sol->size; ++ii)
    {
      Z_CLEAR(i[ii]);
      Z_CLEAR(min[ii]);
      Z_CLEAR(max[ii]);
    }

  free (i);
  free (min);
  free (max);
  fm_rational_free (lb);
  fm_rational_free (Ub);
  fm_vector_free (draw);

  if (mask & FM_SOLVER_VERBOSE)
    fprintf
      (stderr,
       "... Polytope has %Lu point(s), %Lu integer hole(s) were traversed\n",
       ret, badpath);
}
