/*
 * system.c: this file is part of the FM project.
 *
 * FM, a fast and optimized C implementation of Fourier-Motzkin
 * projection algorithm.
 *
 * Copyright (C) 2006,2007,2008 Louis-Noel Pouchet
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 3
 * of the License, or (at your option) any later version.
 *
 * The complete GNU Lesser General Public Licence Notice can be found
 *  as the `COPYING.LESSER' file in the root directory.
 *
 * Author:
 * Louis-Noel Pouchet <Louis-Noel.Pouchet@inria.fr>
 *
 */
#if HAVE_CONFIG_H
# include <fm/config.h>
#endif

#include <fm/common.h>
#include <fm/system.h>
#include <fm/solution.h>
#include <fm/solver.h>


s_fm_system_t*
fm_system_alloc (size_t nb_lines, size_t nb_cols)
{
  s_fm_system_t* s;
  unsigned i;

  s = XMALLOC(s_fm_system_t, 1);
  s->lines = XMALLOC(s_fm_vector_t*, nb_lines);
  for (i = 0; i < nb_lines; ++i)
    s->lines[i] = fm_vector_alloc(nb_cols);

  s->nb_lines = s->allocated = nb_lines;
  s->nb_cols = nb_cols;

  return s;
}


s_fm_system_t*
fm_system_dup (s_fm_system_t* s)
{
  s_fm_system_t* res;
  unsigned i;

  if (s == NULL)
    return NULL;

  res = XMALLOC(s_fm_system_t, 1);
  res->lines = XMALLOC(s_fm_vector_t*, s->nb_lines);
  for (i = 0; i < s->nb_lines; ++i)
    {
      res->lines[i] = fm_vector_alloc(s->nb_cols);
      fm_vector_assign(res->lines[i], s->lines[i]);
    }

  res->nb_lines = s->nb_lines;
  res->allocated = res->nb_lines;
  res->nb_cols = s->nb_cols;

  return res;
}


void
fm_system_free (s_fm_system_t* s)
{
  if (s == NULL)
    return;

  unsigned i;

  for (i = 0; i < s->nb_lines; ++i)
    fm_vector_free(s->lines[i]);

  XFREE(s->lines);
  XFREE(s);
}


s_fm_system_t*
fm_system_read (FILE* stream)
{
  s_fm_system_t* s = XMALLOC(s_fm_system_t, 1);
  char buf[2048];
  unsigned i;
  s->nb_lines = 0;

  /* Skip blank lines. */
  while (fgets(buf, 2048, stream) == 0)
    ;

  /* Get system size. */
  while ((*buf == '#' || *buf == '\n') ||
	 (sscanf (buf, "%u %u", &(s->nb_lines), &(s->nb_cols)) < 2))
    fgets (buf, 2048, stream);

  if (s->nb_lines == 0)
    {
      XFREE(s);
      return NULL;
    }

  s->lines = XMALLOC(s_fm_vector_t*, s->nb_lines);
  for (i = 0; i < s->nb_lines; ++i)
    {
      s->lines[i] = fm_vector_alloc (s->nb_cols);
      fm_vector_read (stream, s->lines[i], s->nb_cols);
    }
  s->allocated = s->nb_lines;

  return s;
}


int
fm_system_print (FILE* stream, s_fm_system_t* s)
{
  unsigned i;

  if (s == NULL)
    fprintf (stream, "(null)\n");
  else
    {
      fprintf (stream, "%d %d\n", s->nb_lines, s->nb_cols);
      for (i = 0; i < s->nb_lines; ++i)
	{
	  fm_vector_print (stream, s->lines[i]);
	  fprintf (stream, "\n");
	}
    }

  return FM_STATUS_OK;
}



int
fm_system_sort (s_fm_system_t* s, unsigned idx, unsigned* n1, unsigned* n2)
{
  s_fm_vector_t* tmp;
  unsigned i, j;

  // Sort the (+) rows first.
  for (i = 0; i < s->nb_lines; ++i)
    for (j = 1; j < s->nb_lines; ++j)
      if (Z_CMP_SI(s->lines[j]->vector[idx].num, >, 0))
	{
	  tmp = s->lines[j];
	  s->lines[j] = s->lines[j - 1];
	  s->lines[j - 1] = tmp;
	}
  // Compute n1, the number of (+) rows.
  for (i = 0, *n1 = 0;
       i < s->nb_lines && Z_CMP_SI(s->lines[i]->vector[idx].num, >, 0);
       ++i, (*n1)++)
    ;

  // Then sort the (-) rows.
  for (i = *n1; i < s->nb_lines; ++i)
    for (j = *n1 + 1; j < s->nb_lines; ++j)
      if (Z_CMP_SI(s->lines[j]->vector[idx].num, <, 0))
	{
	  tmp = s->lines[j];
	  s->lines[j] = s->lines[j - 1];
	  s->lines[j - 1] = tmp;
	}

  // Compute n2, the number of (-) rows.
  for (i = *n1, *n2 = 0;
       i < s->nb_lines && Z_CMP_SI(s->lines[i]->vector[idx].num, <, 0);
       ++i, (*n2)++)
    ;
}


void
fm_system_normalize_ineq (s_fm_system_t* s)
{
  s_fm_vector_t** lines;
  unsigned i;
  unsigned eq_count = 0;
  unsigned count = 0;

  for (i = 0; i < s->nb_lines; ++i)
    if (Z_CMP_SI(s->lines[i]->vector[0].num, ==, 0))
      ++eq_count;

  if (eq_count)
    {
      s_fm_vector_t** lines = XMALLOC(s_fm_vector_t*, s->nb_lines + eq_count);
      for (i = 0; i < s->nb_lines; ++i)
	{
	  lines[count++] = s->lines[i];
	  if (Z_CMP_SI(s->lines[i]->vector[0].num, ==, 0))
	    {
	      lines[count] = fm_vector_dup (s->lines[i]);
	      fm_vector_opp (lines[count], lines[count]);
	      fm_vector_set_ineq (lines[count]);
	      fm_vector_set_ineq (lines[count - 1]);
	      ++count;
	    }
	  else
	    fm_vector_set_ineq (lines[count - 1]);
	}
      XFREE(s->lines);
      s->lines = lines;
      s->nb_lines += eq_count;
      s->allocated = s->nb_lines;
    }
}


s_fm_system_t*
fm_system_to_z (s_fm_system_t* s)
{
  if (s == NULL)
    return NULL;

  int i;
  s_fm_system_t* res = fm_system_alloc (s->nb_lines, s->nb_cols);

  for (i = 0; i < s->nb_lines; ++i)
    fm_vector_to_z (res->lines[i], s->lines[i]);

  return res;
}


s_fm_system_t*
fm_system_swap_column (s_fm_system_t* s, unsigned col_src, unsigned col_dst)
{
  if (s == NULL)
    return NULL;
  if (col_src == col_dst)
    return s;

  if (col_src >= s->nb_cols || col_dst >= s->nb_cols)
    assert (! "col_dst or col_src too high");

  unsigned i;
  s_fm_rational_t* rat = fm_rational_alloc ();

  for (i = 0; i < s->nb_lines; ++i)
    {
      fm_rational_assign (rat, s->lines[i]->vector[col_src].num,
			  s->lines[i]->vector[col_src].denum);
      fm_rational_assign (&(s->lines[i]->vector[col_src]),
			  s->lines[i]->vector[col_dst].num,
			  s->lines[i]->vector[col_dst].denum);
      fm_rational_assign (&(s->lines[i]->vector[col_dst]),
			  rat->num,
			  rat->denum);
    }
  return s;
}


s_fm_system_t*
fm_system_split (s_fm_system_t* s, unsigned col)
{
  if (s == NULL)
    return NULL;

  unsigned i, j;
  unsigned count = 0;
  // Count the number of "independent" lines.

  for (i = 0; i < s->nb_lines; ++i)
    {
      for (j = col + 1; j < s->nb_cols - 1 &&
	     Z_CMP_SI(s->lines[i]->vector[j].num, ==, 0); ++j)
	;
      if (j == s->nb_cols - 1)
	++count;
    }

  s_fm_system_t* out = fm_system_alloc (count, col + 2);

  for (i = 0; i < s->nb_lines; ++i)
    {
      for (j = col + 1; j < s->nb_cols - 1 &&
	     Z_CMP_SI(s->lines[i]->vector[j].num, ==, 0); ++j)
	;
      if (j == s->nb_cols - 1)
	{
	  for (j = 0; j <= col; ++j)
	    fm_rational_copy (&(out->lines[i]->vector[j]),
			      &(s->lines[i]->vector[j]));
	  fm_rational_copy (&(out->lines[i]->vector[col + 1]),
			    &(s->lines[i]->vector[s->nb_cols - 1]));
	}
    }

  return out;
}



int
fm_system_equalities_find (s_fm_system_t* s)
{
  if (s == NULL)
    return 0;

  s_fm_vector_t* tmp;

  int i, j, k;
  int size = s->nb_lines;
  int sz = 0;

  for (i = 0; i < size; ++i)
    {
      // Skip real equalities.
      if (Z_CMP_SI(s->lines[i]->vector[0].num, ==,0))
	{
	  sz++;
	  continue;
	}

      fm_vector_opp (s->lines[i], s->lines[i]);
      for (j = i + 1; j < size; ++j)
	if (fm_vector_equal (s->lines[i], s->lines[j]))
	  {
	    Z_ASSIGN_SI (s->lines[i]->vector[0].num, 0);
	    fm_vector_free (s->lines[j]);
	    for (k = j + 1; k < size; ++k)
	      s->lines[k - 1] = s->lines[k];
	    --size;
	    break;
	  }
      fm_vector_opp (s->lines[i], s->lines[i]);
    }

  sz += s->nb_lines - size;

  s->lines = XREALLOC(s_fm_vector_t*, s->lines, size);
  s->nb_lines = size;
  s->allocated = size;

  fm_system_equalities_sort (s);

  return sz;
}


void
fm_system_equalities_sort (s_fm_system_t* s)
{
  if (s == NULL)
    return;

  int i, j, count = 0;
  s_fm_vector_t* tmp;

  for (i = 0; i < s->nb_lines; ++i)
    if (Z_CMP_SI(s->lines[i]->vector[0].num, ==, 0))
      {
	tmp = s->lines[i];
	for (j = i - 1; j >= count; --j)
	  s->lines[j + 1] = s->lines[j];
	s->lines[count] = tmp;
	++count;
      }

/*   printf ("Found %d equalities (%d inequalities remain)\n", */
/* 	  count, s->nb_lines - count); */
}


int
fm_system_remove_duplicate (s_fm_system_t* s)
{
  int i, j, k;
  int nblines = s->nb_lines;
  int tmp = 0;
  for (i = 0; i < s->nb_lines; ++i)
    for (j = i + 1; j < s->nb_lines; ++j)
      if (fm_vector_equal (s->lines[i], s->lines[j]))
	{
	  fm_vector_free (s->lines[j]);
	  for (k = j; k < s->nb_lines - 1; ++k)
	    s->lines[k] = s->lines[k + 1];
	  (s->nb_lines)--;
	  --j;
	}

  if (s->nb_lines < nblines)
    {
      s->lines = XREALLOC(s_fm_vector_t*, s->lines, s->nb_lines);
      s->allocated = s->nb_lines;
    }

  return FM_STATUS_OK;
}


int
fm_system_remove_line (s_fm_system_t* s, unsigned idx)
{


}


int
fm_system_add_line_at (s_fm_system_t* s, s_fm_vector_t* v, unsigned idx)
{


}

int
fm_system_remove_column (s_fm_system_t* s, int pos)
{
  int i, j;
  s_fm_vector_t* tmp;

  for (i = 0; i < s->nb_lines; ++i)
    {
      tmp = s->lines[i];
      s->lines[i] = fm_vector_alloc (s->nb_cols -1);
      for (j = 0; j < pos; ++j)
	fm_rational_copy (&(s->lines[i]->vector[j]), &(tmp->vector[j]));
      for (++j; j < s->nb_cols; ++j)
	fm_rational_copy (&(s->lines[i]->vector[j - 1]), &(tmp->vector[j]));
      fm_vector_free (tmp);
    }
  (s->nb_cols)--;

  return FM_STATUS_OK;
}


int
fm_system_add_line (s_fm_system_t* s, s_fm_vector_t* v)
{
  // Assume we're going to add several lines, so bufferize the allocation.
  if (s->allocated == s->nb_lines)
    {
      if (s->allocated == 0)
	s->allocated = FM_SYSTEM_ALLOC_BUFFER_SIZE;
      else
	s->allocated *= 2;
      s->lines = XREALLOC (s_fm_vector_t*, s->lines, s->allocated);
    }
  s->lines[(s->nb_lines)++] = v;
  fm_vector_compute_key(&(v->key), v);

  return FM_STATUS_OK;
}


int
fm_system_add_column (s_fm_system_t* s, int pos)
{
  int i, j;
  s_fm_vector_t* tmp;

  for (i = 0; i < s->nb_lines; ++i)
    {
      tmp = s->lines[i];
      s->lines[i] = fm_vector_alloc (s->nb_cols + 1);
      for (j = 0; j < pos; ++j)
	fm_rational_copy (&(s->lines[i]->vector[j]), &(tmp->vector[j]));
      for (++j; j < s->nb_cols + 1; ++j)
	fm_rational_copy (&(s->lines[i]->vector[j]), &(tmp->vector[j - 1]));
      fm_vector_free (tmp);
    }
  (s->nb_cols)++;

  return FM_STATUS_OK;
}



/**
 * Check if a point is in a polyhedron.
 *
 */
int
fm_system_point_included (s_fm_system_t* s, s_fm_vector_t* v)
{
  int i, j;
  s_fm_rational_t* val;
  s_fm_rational_t* tmp;
  int ret = 1;

  if (s->nb_lines == 0)
    return 1;
  if (s->nb_cols != v->size + 1)
    return 0;

  val = fm_rational_alloc ();
  tmp = fm_rational_alloc ();

  // Simply check if the point does not violate any constraint.
  for (i = 0; i < s->nb_lines; ++i)
    {
      fm_rational_init (tmp);
      for (j = 1; j < s->nb_cols - 1; ++j)
	{
	  fm_rational_mul (val, &(s->lines[i]->vector[j]), &(v->vector[j]));
	  fm_rational_add (tmp, tmp, val);
	}
      fm_rational_add (tmp, tmp, &(s->lines[i]->vector[s->nb_cols - 1]));
      if (Z_CMP_SI (tmp, <, 0) ||
	  ((Z_CMP_SI (s->lines[i]->vector[0].num, ==, 0) &&
	    Z_CMP_SI (tmp, !=, 0))))
	{
	  ret = 0;
	  break;
	}
    }
  fm_rational_free (val);
  fm_rational_free (tmp);

  return ret;
}



void
fm_system_subst_in_vector (s_fm_vector_t* v,
			   s_fm_vector_t* v1,
			   s_fm_vector_t* pattern,
			   s_fm_solution_t* redeq,
			   int iv1)
{
  int i;
  int idx = 0;
  s_fm_rational_t* rtmp = fm_rational_alloc ();

  for (i = 1; i < pattern->size - 1; ++i)
    {
      fm_rational_copy (rtmp, &(v1->vector[iv1]));
      fm_rational_mul (rtmp, rtmp, &(pattern->vector[i]));
      if (redeq->solution[i - 1].positive != NULL)
	idx++;
      // Representation for pattern is x = k, so [ 0 1 -k ] (in fact [ 0 -k ])
      fm_rational_opp (rtmp, rtmp);
      fm_rational_add (&(v->vector[i - idx]), &(v->vector[i - idx]), rtmp);
    }
  fm_rational_copy (rtmp, &(v1->vector[iv1]));
  fm_rational_mul (rtmp, rtmp, &(pattern->vector[i]));
  fm_rational_opp (rtmp, rtmp);
  fm_rational_add (&(v->vector[v->size - 1]), &(v->vector[v->size - 1]), rtmp);

  fm_rational_free (rtmp);
}


s_fm_system_t*
fm_system_reduce (s_fm_system_t* in, s_fm_solution_t* redeq)
{
  int i;
  int j;
  int idx;
  int count = 0;

  for (i = 0; i < redeq->size; ++i)
    if (redeq->solution[i].positive != NULL)
      ++count;

  s_fm_system_t* s = fm_system_alloc (in->nb_lines, in->nb_cols - count);

  for (i = 0; i < in->nb_lines; ++i)
    {
      fm_rational_copy (&(s->lines[i]->vector[0]), &(in->lines[i]->vector[0]));
      fm_rational_copy (&(s->lines[i]->vector[s->nb_cols - 1]),
			&(in->lines[i]->vector[in->nb_cols - 1]));
      // Apply the reduction contained in redeq, for each line.
      for (idx = 1, j = 1; j < in->nb_cols - 1; ++j)
	{
	  if (redeq->solution[j - 1].positive != NULL)
	    fm_system_subst_in_vector (s->lines[i], in->lines[i],
				       redeq->solution[j - 1].positive->data,
				       redeq, j);
	  else
	    fm_rational_copy (&(s->lines[i]->vector[idx++]),
			      &(in->lines[i]->vector[j]));
	}
    }

  // Remove empty lines in the system.
  for (i = 0, count = 0; i < s->nb_lines; ++i)
    if (fm_vector_is_empty (s->lines[i]))
      ++count;

  s_fm_system_t* ret = XMALLOC(s_fm_system_t, 1);
  ret->lines = XMALLOC(s_fm_vector_t*, s->nb_lines - count);
  ret->nb_lines = s->nb_lines - count;
  ret->nb_cols = s->nb_cols;
  ret->allocated = ret->nb_lines;
  for (i = 0, count = 0; i < s->nb_lines; ++i)
    {
      if (! fm_vector_is_empty (s->lines[i]))
	ret->lines[count++] = s->lines[i];
      else
	{
	  if (fm_vector_is_valid (s->lines[i]))
	    fm_vector_free (s->lines[i]);
	  else
	    {
	      printf ("system infeasible\n");
	      XFREE(ret->lines);
	      XFREE(ret);
	      return NULL;
	    }
	}
    }
  XFREE(s->lines);
  XFREE(s);

  return ret;
}


/**
 * Compute the connected indices.
 *
 * Result is outputted as an int matrix. Its line indice represent the
 * class index (there is at most size_sys classes, when the graph is
 * not connected), and for each class there is at most size_sys
 * elements (when the graph is fully connected). A class is
 * represented by a set of variable indices (in basis 0), and is
 * terminated by '-1'.
 *
 */
int**
fm_system_getconnected (s_fm_system_t* system)
{
  int i, j, k, l;
  int pos;
  int stack_top;
  int count;
  int class_count = 0;

  int size_sys = system->nb_cols - 2;
  int mask[size_sys][size_sys];
  int visited[size_sys];

  int fmstack[size_sys];
  int printed[size_sys];
  int output[size_sys];

  int** res = XMALLOC(int*, size_sys + 1);
  for (i = 0; i < size_sys + 1; ++i)
    res[i] = NULL;

  // Reinitialize the mask.
  for (l = 0; l < size_sys; ++l)
    for (j = 0; j < size_sys; ++j)
      mask[l][j] = 0;
  // Iterate on each line.
  for (k = 0; k < system->nb_lines; ++k)
    // Iterate on each variable.
    for (j = 1; j < system->nb_cols - 1; ++j)
      if (Z_CMP_SI(system->lines[k]->vector[j].num, !=, 0))
	for (l = 1; l < system->nb_cols - 1; ++l)
	  if (Z_CMP_SI(system->lines[k]->vector[l].num, !=, 0))
	    {
	      mask[j - 1][l - 1] = 1;
	      mask[l - 1][j - 1] = 1;
	    }

  // Print the connected components.
  for (j = 0; j < size_sys; ++j)
    {
      visited[j] = 0;
      printed[j] = 0;
    }
  for (j = 0; j < size_sys; ++j)
    {
      {
	pos = j;
	stack_top = 0;
	count = 0;

	fmstack[stack_top] = pos;
	while (stack_top >= 0)
	  {
	    pos = fmstack[stack_top--];
	    for (i = 0; i < size_sys; ++i)
	      {
		if (mask[pos][i])
		  {
		    if (! (printed[i])++)
		      output[count++] = i + 1;
		    if (! (visited[i])++)
		      fmstack[++stack_top] = i;
		  }
	      }
	    visited[pos] = 1;
	  }

	if (count)
	  {
	    // Sort.
	    for (i = 0; i < count; ++i)
	      for (l = i; l < count; ++l)
		if (output[i] > output[l])
		  {
		    k = output[i];
		    output[i] = output[l];
		    output[l] = k;
		  }
	    // Format the result.
	    if (count)
	      {
		res[class_count] = XMALLOC(int, count + 1);
		res[class_count][count] = -1;
		for (i = 0; i < count; ++i)
		  res[class_count][i] = output[i] - 1;
		++class_count;
	      }
	  }
      }
    }

  return res;
}


#ifdef HAVE_LIBPIPLIB


/**
 * Remove redundant constraints from a polyhedron defined by a set of
 * hyperplanes..
 *
 * See M. Le Fur Irisa research report.
 *
 */
s_fm_system_t*
fm_system_simplify (s_fm_system_t* system,
		    int simplify_mode)
{
  if (system == NULL)
    return NULL;

  s_fm_system_t* tmp;
  int i, j;
  unsigned count = 0;
  unsigned nb_const = system->nb_lines;

  if ((simplify_mode & FM_SYSTEM_REDREC_DESCENDANT) > 0)
    {
      for (i = 0; i < system->nb_lines; ++i)
	{
	  if ((simplify_mode & FM_SOLVER_VERBOSE) > 0)
	    {
	      fprintf (stderr, ".");
	      fflush (stderr);
	    }
	  fm_vector_opp (system->lines[i], system->lines[i]);
	  tmp = fm_system_to_z (system);
/* 	  Z_DEC(tmp->lines[i]->vector[system->nb_cols - 1].num, */
/* 		tmp->lines[i]->vector[system->nb_cols - 1].num); */
	  if (! fm_piptools_check_int (tmp))
	    {
	      // The constraint is redundant.
	      fm_vector_free (system->lines[i]);
	      if (i < system->nb_lines - 1)
		{
		  for (j = i; j < system->nb_lines - 1; ++j)
		    system->lines[j] = system->lines[j + 1];
		  --i;
		}
	      ++count;
	      system->nb_lines -= 1;
	    }
	  else
	    fm_vector_opp (system->lines[i], system->lines[i]);
	  fm_system_free (tmp);
	}
    }
  else if ((simplify_mode & FM_SYSTEM_REDREC_IRIGOIN) > 0)
    {
      for (i = system->nb_lines - 1; i >= 0; --i)
	{
	  if ((simplify_mode & FM_SOLVER_VERBOSE) > 0)
	    {
	      fprintf (stderr, ".");
	      fflush (stderr);
	    }
	  fm_vector_opp (system->lines[i], system->lines[i]);
	  tmp = fm_system_to_z (system);
/* 	  Z_DEC(tmp->lines[i]->vector[system->nb_cols - 1].num, */
/* 		tmp->lines[i]->vector[system->nb_cols - 1].num); */
	  if (! fm_piptools_check_int (tmp))
	    {
	      // The constraint is redundant.
	      fm_vector_free (system->lines[i]);
	      for (j = i; j < system->nb_lines - 1; ++j)
		system->lines[j] = system->lines[j + 1];
	      system->nb_lines -= 1;
	      ++count;
	    }
	  else
	    fm_vector_opp (system->lines[i], system->lines[i]);
	  fm_system_free (tmp);
	}
    }

  if ((simplify_mode & FM_SOLVER_VERBOSE) > 0)
    fprintf(stdout, "\n... Removed %d / %d redundant constraints\n",
	    count, nb_const);

  return system;
}

#endif // ! HAVE_LIBPIPLIB
