/*
 * fm.c: this file is part of the FM project.
 *
 * FM, a fast and optimized C implementation of Fourier-Motzkin
 * projection algorithm.
 *
 * Copyright (C) 2006-2008 Louis-Noel Pouchet
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 3
 * of the License, or (at your option) any later version.
 *
 * The complete GNU Lesser General Public Licence Notice can be found
 *  as the `COPYING.LESSER' file in the root directory.
 *
 * Author:
 * Louis-Noel Pouchet <Louis-Noel.Pouchet@inria.fr>
 *
 */
#if HAVE_CONFIG_H
# include <config.h>
#endif

#include <fm/common.h>
#include <fm/system.h>
#include <fm/solution.h>
#include <fm/solver.h>
#include <fm/compsol.h>


void
fms_usage ()
{
  fprintf (stderr, "Usage: fm-solver input-file [output-file]\n");
  exit (1);
}

void
fms_error ()
{
  fprintf (stderr, "fm-solver: Bad input file.\n");
  exit (1);
}



int
main (int argc, char * const argv[])
{
  if (argc < 2 || argc > 3)
    fms_usage ();


  FILE* f = fopen (argv[1], "r");
  if (f == NULL)
    fms_error ();

  // Read the input system.
  s_fm_system_t* system = fm_system_read (f);
  fclose (f);

  if (f == NULL)
    fms_error ();

  // Solve the system.
  s_fm_solution_t* solution = 
    fm_solver (system, FM_SOLVER_FAST | FM_SOLVER_NORMALIZE_EQ | 
	       FM_SOLVER_AUTO_SIMPLIFY);

  if (argc == 3)
    f = fopen (argv[2], "w");
  else
    f = fdopen (dup (fileno (stdout)), "w");

  if (f == NULL)
    fms_usage ();

  // Output the results.
  fm_solution_print (f, solution);
  fclose (f);

  // Be clean.
  fm_system_free (system);
  fm_solution_free (solution);

  return 0;
}

