/*
 * clast2scop.c: this file is part of the ScopTools project.
 *
 * ScopTools, a set of convenience functions to translate to/from ScopLib.
 *
 * Copyright (C) 2010 Louis-Noel Pouchet
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 3
 * of the License, or (at your option) any later version.
 *
 * The complete GNU Lesser General Public Licence Notice can be found
 * as the `COPYING.LESSER' file in the root directory.
 *
 * Author:
 * Louis-Noel Pouchet <pouchet@cse.ohio-state.edu>
 */
#if HAVE_CONFIG_H
# include <scoptools/config.h>
#endif

#include <scoptools/clast2scop.h>
#include <scoptools/common.h>
#include <scoptools/exec.h>
#include <scoptools/cloog_iface.h>
#include <clan/clan.h>
#include <clasttools/pprint.h>
#include <clasttools/clastext.h>


static
int is_space (char c)
{
  if (c == ' ' || c == '\t')
    return 1;
  return 0;
}

static
void
remove_all_spaces (char* s)
{
  int i, j;
  for (i = 0, j = 0; s[i]; ++i)
    if (! is_space (s[i]))
      s[j++] = s[i];
  s[j] = '\0';
}

static
int is_digit_or_number_or_underscore (char c)
{
  if ((c >= 'a' && c <= 'z')
      || (c >= 'A' && c <= 'Z')
      || (c >= '0' && c <= '0')
      || c == '_')
    return 1;
  return 0;
}

static
int
is_id (char* s, int pos)
{
  int sz = 0;
  if ((s[pos] >= 'a' && s[pos] <= 'z')
      || (s[pos] >= 'A' && s[pos] <= 'Z')
      || s[pos] == '_')
    {
      sz = 1;
      if (s[pos + sz] == '.')
	sz += 1;
      else if (s[pos + sz] == '-' && s[pos + sz + 1] == '>')
	sz += 2;
      while (is_digit_or_number_or_underscore (s[pos + sz]))
	sz++;
    }

  return sz;
}

static
int
remove_extra_array_parenthesis (char* s)
{
  int i, j, k;
  int l = strlen (s);

  // Find an id followed by a '['.
  int sz;
  int ret = 0;
  for (i = 0; i < l; ++i)
    {
      sz = is_id (s, i);
      if (sz && s[i + sz] == '[')
	{
	  // Find the end of the sequence of '[]'.
	  // Note: we do not support arrays of arrays, it's fine it's
	  // not affine anyway.
	  for (k = i + sz; k < l; ++k)
	    if (s[k] == ']' && s[k + 1] != '[')
	      break;
	  if (i > 0 && s[i - 1] == '(' && s[k + 1] == ')')
	    {
	      // Remove the enclosing parenthesis
	      sz = k;
	      for (; i <= k; ++i)
		s[i - 1] = s[i];
	      // Shift the rest of the string;
	      for (; k < l - 2; ++k)
		s[k] = s[k + 2];
	      s[k] = '\0';
	      ret = 1;
	      //i += sz - 1;
	    }
	}
    }

  return ret;
}

static
int
is_cast_expression (char* s, int pos, int sz)
{
  char buffer[sz + 1];
  int i;
  for (i = 0; i < sz; ++i)
    buffer[i] = s[pos + i];
  buffer[i] = '\0';
  while (! strncmp (buffer, "const", 5))
    {
      for (i = 5; i < sz && buffer[i]; ++i)
	buffer[i - 5] = buffer[i];
      buffer[i - 5] = '\0';
    }
  if (! strncmp (buffer, "struct", 6))
    {
      for (i = 6; i < sz && buffer[i]; ++i)
	buffer[i - 6] = buffer[i];
      buffer[i - 6] = '\0';
    }
  if (! strcmp (buffer, "double") ||
      ! strcmp (buffer, "int") ||
      ! strcmp (buffer, "float") ||
      ! strcmp (buffer, "short") ||
      ! strcmp (buffer, "char") ||
      ! strcmp (buffer, "longint") ||
      ! strcmp (buffer, "longlongint") ||
      ! strcmp (buffer, "longlong") ||
      ! strcmp (buffer, "longdouble"))
    return 1;
  return 0;
}

static
int
remove_type_cast (char* s)
{
  int i, j, k;
  int l = strlen (s);

  // Find ( id ). remove_all_space will have merged
  // const struct ... into conststruct..., a single id.
  int sz;
  int ret = 0;
  for (i = 0; i < l; ++i)
    {
      if (s[i] == '(')
	{
	  k = i + 1;
	  sz = is_id (s, k);
	  if (sz && is_cast_expression (s, k, sz))
	    {
	      // Skip the potential pointer refs
	      k += sz;
	      while (s[k] == '*')
		++k;
	      if (s[k] == ')')
		{
		  for (j = i; k < l - 1; ++k)
		    s[j++] = s[k + 1];
		  s[j] = '\0';
		  ret = 1;
		}
	    }
	}
    }

  return ret;
}

static
char*
rose_statement_preprocessor (char* stmt)
{
  char* ret = strdup (stmt);
  remove_all_spaces (ret);

  while (remove_extra_array_parenthesis (ret))
    ;

  while (remove_type_cast (ret))
    ;

  return ret;
}

static
char* build_pragma_header (scoplib_scop_p orig_scop)
{
  char* alive = scoplib_scop_tag_content (orig_scop, "<live-out-vars>",
					  "</live-out-vars>");
  char* ret = NULL;
  if (alive)
    {
      char* str = alive;
      int i, j, pos, count = 0;
      char buffer[16];
      char* obuffer = XMALLOC(char, 8192);
      obuffer[0] = '\0';
      int* alive_id = (int*) malloc ((orig_scop->nb_arrays + 1) * sizeof(int));
      while (*str != '\n' && *str)
	{
	  while (*str == ' ') ++str;
	  pos = 0;
	  while (*str >= '0' && *str <= '9')
	    buffer[pos++] = *(str++);
	  buffer[pos] = '\0';
	  if (pos)
	    alive_id[count++] = atoi (buffer);
	  ++str;
	}
      alive_id[count] = -1;
      XFREE(alive);
      if (count)
	sprintf (obuffer, "#pragma live-out ");
      for (i = 0; i < count; ++i)
	{
	  assert (alive_id[i] - 1 < orig_scop->nb_arrays);
	  assert (strlen (obuffer) +
		  strlen (orig_scop->arrays[alive_id[i] - 1]) < 8191);
	  if (i == 0)
	    sprintf (obuffer, "%s%s", obuffer,
		     orig_scop->arrays[alive_id[i] - 1]);
	  else
	    sprintf (obuffer, "%s, %s", obuffer,
		     orig_scop->arrays[alive_id[i] - 1]);
	}
      XFREE(alive_id);
      ret = XMALLOC(char, strlen(obuffer) + 1);
      strcpy (ret, obuffer);
      XFREE(obuffer);
    }

  return ret;
}


static
void
traverse_print_clast_user_statement_extended_defines (struct clast_stmt* s,
						      FILE* out)
{
  // Traverse the clast.
  for ( ; s; s = s->next)
    {
      if (CLAST_STMT_IS_A(s, stmt_for) ||
	  CLAST_STMT_IS_A(s, stmt_parfor) ||
	  CLAST_STMT_IS_A(s, stmt_vectorfor))
	{
	  struct clast_stmt* body;
	  if (CLAST_STMT_IS_A(s, stmt_for))
	    body = ((struct clast_for*)s)->body;
	  else if  (CLAST_STMT_IS_A(s, stmt_parfor))
	    body = ((struct clast_parfor*)s)->body;
	  else if  (CLAST_STMT_IS_A(s, stmt_vectorfor))
	    body = ((struct clast_vectorfor*)s)->body;
	  traverse_print_clast_user_statement_extended_defines (body, out);
	}
      else if (CLAST_STMT_IS_A(s, stmt_guard))
	traverse_print_clast_user_statement_extended_defines
	  (((struct clast_guard*)s)->then, out);
      else if (CLAST_STMT_IS_A(s, stmt_block))
	traverse_print_clast_user_statement_extended_defines
	  (((struct clast_block*)s)->body, out);
      else if (CLAST_STMT_IS_A(s, stmt_user_extended))
	{
	  struct clast_user_stmt_extended* ue =
	    (struct clast_user_stmt_extended*) s;
	  fprintf (out, "%s\n", ue->define_string);
	}
    }
}


typedef char* (*preprocess_string_fun_t) (char*);


scoplib_scop_p
scoptools_clast2scop_ (struct clast_stmt* s,
		       scoplib_scop_p orig_scop,
		       preprocess_string_fun_t statement_preprocessor)
{
  CloogStatement *	statement;
  // Create temporary file.
  FILE* body_file = fopen (".tmpbody.c", "w+");
  if (body_file == NULL)
    scoptools_error ("Cannot create temporary file .tmpbody.c");

  // Dump the statement macros in it.
  int st_count = 1;
  scoplib_statement_p stm;
  char* preprocessed;
  int i;
  for (stm = orig_scop->statement; stm; stm = stm->next)
    {
      fprintf (body_file, "#define S%d(", st_count++);
      for (i = 0; i < stm->nb_iterators; ++i)
	{
	  fprintf (body_file, "%s", stm->iterators[i]);
	  if (i < stm->nb_iterators - 1)
	    fprintf (body_file, ",");
	}
      preprocessed = statement_preprocessor (stm->body);
      fprintf (body_file, ") %s\n", preprocessed);
      XFREE(preprocessed);
    }

  /* We now can have statement definition overriden by the array
     contraction. Those are stored in clast_user_statement_extended
     nodes only, the #define is in the cuse->define_string, they must
     be collected and pretty-printed here. */
  traverse_print_clast_user_statement_extended_defines (s, body_file);

  // Initialize CloogOptions for C code pretty-printing.
  CloogState* cstate = cloog_state_malloc ();
  CloogOptions* coptions = cloog_options_malloc (cstate);
  coptions->language = 'c';

  // Pretty-print the code.
  fprintf (body_file, "#pragma scop\n");
  char* pragma_header = build_pragma_header (orig_scop);
  if (pragma_header)
    {
      fprintf (body_file, "%s\n", pragma_header);
      XFREE(pragma_header);
    }
  clasttols_clast_pprint_debug (body_file, s, 0, coptions);
  fprintf (body_file, "#pragma endscop\n");
  cloog_state_free (cstate);
  cloog_options_free (coptions);
  fclose (body_file);

  // Expand the macros thanks to GCC.
  char* args[4];
  args[0] = STR_SCOPTOOLS_ROOT_DIR "/scoptools/expander.sh";
  args[1] = ".tmpbody.c";
  args[2] = args[3] = NULL;
  scoptools_exec (args, SCOPTOOLS_EXECV_SHOW_OUTPUT);

  // Analyze the code with Clan
  body_file = fopen (".tmpbody.c", "r");
  if (body_file == NULL)
    scoptools_error ("Cannot open temporary file .tmpbody.c");
  clan_options_p clanoptions = clan_options_malloc ();
  scoplib_scop_p newscop = clan_scop_extract (body_file, clanoptions);
  clan_options_free (clanoptions);


  // Delete temporary file.
  args[0] = "rm";
  args[1] = "-f";
  args[2] = ".tmpbody.c";
  scoptools_exec (args, SCOPTOOLS_EXECV_SHOW_OUTPUT);

  /// FIXME: DO IT!
/*   // Copy the option tags from the original code. */
/*   if (newscop->optiontags == NULL) */
/*     newscop->optiontags = strdup (orig_scop->optiontags); */
/*   else */
/*     { */
/*       int size = */
/* 	strlen (orig_scop->optiontags) + strlen (newscop->optiontags) + 2; */
/*       char* newtag = (char*)malloc (size * sizeof(char)); */
/*       strcpy (newtag, orig_scop->optiontags); */
/*       strcat (newtag, "\n"); */
/*       strcat (newtag, newscop->optiontags); */
/*       free (newscop->optiontags); */
/*       newscop->optiontags = newtag; */
/*     } */

  return newscop;
}


/**
 * Convert a CLAST to a Scop, by applying some massaging to the
 * original statements:
 * - remove (for analysis only) any type cast
 * - remove (for analysis only) () in expressions such as (a[i])[j]
 *
 * The ROSE compiler has been reported to systematically generate such
 * constructs, which are not supported in Clan.
 *
 */
scoplib_scop_p
scoptools_clast2scop_rose (struct clast_stmt* s, scoplib_scop_p orig_scop)
{
  return scoptools_clast2scop_ (s, orig_scop, rose_statement_preprocessor);
}


/**
 * Convert a CLAST to a Scop.
 *
 */
scoplib_scop_p
scoptools_clast2scop (struct clast_stmt* s, scoplib_scop_p orig_scop)
{
  return scoptools_clast2scop_ (s, orig_scop, (preprocess_string_fun_t) strdup);
}

/**
 * Convert a CLAST to a Scop.
 *
 */
scoplib_scop_p
scoptools_clast2scop_latest (struct clast_stmt* s, scoplib_scop_p orig_scop)
{
  return scoptools_clast2scop_ (s, orig_scop, (preprocess_string_fun_t) strdup);
}



/**
 * Convert a Scop to a CLAST.
 *
 */
struct clast_stmt*
scoptools_scop2clast (scoplib_scop_p scop)
{
  /* Create an input CloogProgram from the .scop. */
  CloogState* cstate = cloog_state_malloc ();
  CloogOptions* coptions = cloog_options_malloc (cstate);
  coptions->language = 'c';

  /* Create the CLAST associated to the scop. */
  struct clast_stmt* root = scoptools_driver_cloog (scop, coptions);

  /* Be clean. */
  cloog_state_free (coptions->state);
  cloog_options_free (coptions);

  return root;
}
